# Videoteque is a collection of videos. On the file system layer it is
# represented by a directory containing video files.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::VideoMode::Videoteque;

our @ISA = qw(CMSsh::Mode::PublicationCollection);

use strict;
use CMSsh::Mode::PublicationCollection;
use CMSsh::FileSystem::Directory;
use CMSsh::VideoMode::VidFile;
use CMSsh::VideoMode::Messages;
use CMSsh::VideoMode::Config;

my $Msg = CMSsh::VideoMode::Messages->new;
my $Cfg = CMSsh::VideoMode::Config->new;

################################################################################
# Common methods

sub new ($) {
    my ($class_name) = @_;

    my $self = CMSsh::Mode::PublicationCollection->new('/');

    $self->{_vid_dir} = CMSsh::FileSystem::Directory->new(
                            $Cfg->get('ROOT_VIDEO'));
    unless ($self->{_vid_dir}) {
        warn $Msg->get('CANT_INST_FSO', $Cfg->get('ROOT_VIDEO'));
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub get_dir ($) {
    my $self = shift;
    return $self->{_vid_dir};
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

# There are no Presentation/Administration methods, because this publication is
# not supposed to be viewed/created/removed/etc, just listing

sub get_listing ($) {
    my $self = shift;
    my @result;

    my $listing = $self->{_vid_dir}->get_listing;
    unless ($listing) {
        warn $Msg->get('CANT_LIST_DIR', $self->{_vid_dir}->get_path);
        return undef;
    }

    my @vids;

    foreach my $file (@{$listing->{files}}) {
        my $vid_file = CMSsh::VideoMode::VidFile->new($file->get_path);
        unless ($vid_file) {
            warn $Msg->get('CANT_INST_FSO', $file->get_basepath);
            next;
        }

        my $vid_name = $vid_file->get_basename;
        unless (defined $vid_name) {
            warn $Msg->get('CANT_MAP_FSO',
                           $vid_file->get_path, $vid_file->get_name);
            next;
        }

        my $video = CMSsh::VideoMode::Video->new("/$vid_name");
        unless ($video) {
            warn $Msg->get('CANT_INST_PUB', "/$vid_name");
            next;
        }

        push @vids, $video;
    }

    return { CMSsh::VideoMode::Video->get_type => \@vids };
}

################################################################################
# Administration methods

sub create_subvid ($) {
    my ($self, $vid_file) = @_;

    my $name = $vid_file->get_basename;

    if (CMSsh::VideoMode->find_publication("/$name")) {
        warn $Msg->get('OCCUPIED_URI', "/$name");
        return undef;
    }

    $vid_file->move($self->{_vid_dir})
    or do {
        warn $Msg->get('CANT_MOVE_FSO',
                       $vid_file->get_path, $self->{_vid_dir}->get_path);
        return undef;
    };

    return CMSsh::VideoMode::Video->new("/$name");
}

1
