# TemplateHandler does all the job related to templates: it generates a complete
# list of tmpl variables, composes HTML markup and sends reply to the user.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::Presentation::TemplateHandler;

use strict;
use CGI;
use File::Path;
use Template;
use CMSsh::Utils qw(collect_modes_data);
use CMSsh::Presentation::Utils qw(cgi_die);
use CMSsh::Messages;
use CMSsh::Config;

my $Msg = CMSsh::Messages->new;
my $Cfg = CMSsh::Config->new;
my $Cgi = CGI->new;
my @Tmpl_Params = qw(COPYRIGHT_YEARS LICENSE OWNER_NAME OWNER_EMAIL
                     URLKEY_DOWNLOAD USE_HTML5);

sub new ($$) {
    my ($class_name, $view) = @_;
    
    my $self = { _view => $view };

    bless $self, $class_name;
    return $self;
}

sub _get_entire_menu ($$) {
    my ($self, $publication) = @_;

    my @result = collect_modes_data (
        sub {
            my $mode = shift;
            return @{$mode->get_view->get_menu_items};
        },
        sub {
            my $mode = shift;
            return 0 if ($mode->get_name eq $self->{_view}->get_mode->get_name);
            return 0 unless ($mode->get_view);
            return 1;
        }
    );
    push @result, @{$self->{_view}->get_menu_items($publication)};

    @result = sort { $b->{priority} <=> $a->{priority} ||
                     $a->{title} cmp $b->{title} } @result;

    return \@result;
}

# Appends template parameters used in header, footer, etc.
sub _add_params ($$$) {
    my ($self, $vars, $publication) = @_;

# Add the desired config settings first.
    my $mode_cfg = $self->{_view}->get_mode->get_config;
    $vars->{lc($_)} = $mode_cfg->get($_)
        foreach (@Tmpl_Params, @{$self->{_view}->get_tmpl_params});

    $vars->{path} = $self->{_view}->get_path($publication);
    $vars->{menu} = $self->_get_entire_menu($publication);
    $vars->{owner_email} =~ s/(.)/sprintf('%%%2x',ord($1))/ge;
}

sub process ($$$$) {
    my ($self, $tmpl_file, $vars, $publication) = @_;

    my $tt = Template->new({ PRE_CHOMP => 1,
                             INCLUDE_PATH => $Cfg->get('ROOT_TMPL') })
        or cgi_die $Msg->get('TEMPLATE_ERR', $Template::ERROR);

    $self->_add_params($vars, $publication);

    my $html;
    $tt->process("$tmpl_file.tmpl", $vars, \$html)
        or cgi_die $Msg->get('TEMPLATE_ERR', $tt->error);

    print $Cgi->header(-charset => 'utf-8'), $html;
}

1
