# Image is a single publication of this mode. On the file system layer it is
# represented by an original image, a thumbnail of the image and a preview of
# the image.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com> 

package CMSsh::GalleryMode::Image;

our @ISA = qw(CMSsh::Mode::SinglePublication);

use strict;
use CMSsh::Mode::SinglePublication;
use CMSsh::FileSystem::ImgFile;
use CMSsh::GalleryMode::Messages;
use CMSsh::GalleryMode::Config;

my $Msg = CMSsh::GalleryMode::Messages->new;
my $Cfg = CMSsh::GalleryMode::Config->new;

################################################################################
# Common methods

sub new ($$) {
    my ($class_name, $uri) = @_;
    
    my $self = CMSsh::Mode::SinglePublication->new($uri);
    unless ($self) {
        warn $Msg->get('CANT_INST_PUB', $uri);
        return undef;
    }

    my $parent_alb = CMSsh::GalleryMode::Album->new($self->get_parent_uri);
    unless ($parent_alb) {
        warn $Msg->get('CANT_INST_PUB', $self->get_parent_uri);
        return undef;
    }

    foreach (['_img',     $parent_alb->get_img_dir],
             ['_preview', $parent_alb->get_preview_dir],
             ['_thumb',   $parent_alb->get_thumb_dir]) {
        my $img_path = $_->[1]->get_path . '/'. $self->{_name} . '.' .
                           CMSsh::FileSystem::ImgFile->get_ext;
        my $img_file = $self->{$_->[0]}
                     = CMSsh::FileSystem::ImgFile->new($img_path);
        unless ($img_file) {
            warn $Msg->get('CANT_INST_FSO', $img_path);
            return undef;
        }
    }

    bless $self, $class_name;
    return $self;
}

sub get_type ($) {
    my $self = shift;
    return $self->_extract_type(__PACKAGE__);
}

################################################################################
# Presentation methods

sub get_view ($) {
    my $self = shift;
    return { href => $self->{_img}->get_htdocs_path,
             path => $self->{_preview}->get_htdocs_path,
             caption => $self->{_name}, };
}

sub get_preview ($) {
    my $self = shift;
    return { href => $self->{_uri},
             path => $self->{_thumb}->get_htdocs_path,
             caption => $self->{_name}, };
}

sub get_album ($) {
    my $self = shift;

    my $album = CMSsh::GalleryMode::Album->new($self->get_parent_uri);
    unless ($album) {
        warn $Msg->get('CANT_INST_PUB', $self->get_parent_uri);
        return undef;
    }

    return $album;
}

################################################################################
# Administration methods

sub move ($$) {
    my ($self, $dest_alb) = @_;

    my $new_uri = $dest_alb->get_uri . '/' . $self->{_name};
    if (CMSsh::GalleryMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    foreach (['_img',     $dest_alb->get_img_dir],
             ['_preview', $dest_alb->get_preview_dir],
             ['_thumb',   $dest_alb->get_thumb_dir]) {
        my $src_file = $self->{$_->[0]};
        $src_file->move($_->[1])
        or do {
            warn $Msg->get('CANT_MOVE_FSO',
                           $src_file->get_path, $_->[1]->get_path);
            return 0;
        };
    }

    $self->_set_uri($new_uri);

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    if ($new_name =~ /^\./) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }
    
    my $new_uri = $self->get_parent_uri . '/' . $new_name;
    if (CMSsh::GalleryMode->find_publication($new_uri)) {
        warn $Msg->get('OCCUPIED_URI', $new_uri);
        return 0;
    }

    foreach (qw(_img _thumb _preview)) {
        $self->{$_}->rename($new_name)
        or do {
            warn $Msg->get('CANT_RENAME_FSO', $self->{$_}->get_path, $new_name);
            return 0;
        };
    }

    $self->_set_uri($new_uri);
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    foreach (qw(_img _preview _thumb)) {
        $self->{$_}->remove
        or do {
            warn $Msg->get('CANT_REMOVE_FSO', $self->{$_}->get_path);
            return 0;
        };
    }

    return 1;
}

1
