# PublicationManager implements the interface between the administration server
# and the publications. It parses and performs administration requests.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::Administration::PublicationManager;

use strict;
use CMSsh::Messages;
use CMSsh::Config;

my $Msg = CMSsh::Messages->new;
my $Cfg = CMSsh::Config->new;

sub new ($$) {
    my ($class_name, $mode) = @_;

    my $self = { _mode => $mode };
    bless $self, $class_name;
    $self->{_syntax} = {
        mk => {},
        mv => {
            handler => sub { $self->_mv(@_) },
            src => { is_file_path => 0 },
            dst => 1
        },
        rn => {
            handler => sub { $self->_rn(@_) },
            src => { is_file_path => 0, max_number => 1 },
            dst => 1
        },
        rm => {
            handler => sub { $self->_rm(@_) },
            src => { is_file_path => 0 },
        },
    };

    return $self;
}

sub get_symbol () {
    die $Msg->get('ABSTRACT_CLASS');
}
 
sub get_pub_type () {
    die $Msg->get('ABSTRACT_CLASS');
}

sub get_mode ($) {
    my $self = shift;
    return $self->{_mode};
}

sub get_syntax ($) {
    my $self = shift;

    sub deepcopy($) {
        my $this = shift;
        if (not ref $this) {
            return $this;
        } elsif (ref $this eq 'HASH') {
            return { map { $_ => deepcopy($this->{$_}) }
                     grep { ref $this->{$_} ne 'CODE'} keys %$this };
        } else {
            die $Msg->get('UNSUPPORTED_TYPE', ref $this);
        }
    }

   return deepcopy($self->{_syntax});
}

sub process ($$$) {
    my ($self, $oper, $args) = @_;

    my $params = $self->_parse_args($args, $oper);
    $self->{_syntax}->{$oper}->{handler}->($params);
}

sub _parse_args ($$$) {
    my ($self, $args_ref, $oper) = @_;

    my $syntax = $self->{_syntax}->{$oper};

    my @args = @$args_ref;
    die $Msg->get('NOT_ENGH_PARAMS') unless (@args);

    my $has_src = (defined $syntax->{src});
    my $is_src_external = $has_src && $syntax->{src}->{is_file_path};
    my $max_src_number = $has_src && $syntax->{src}->{max_number};
    my $has_dst = (defined $syntax->{dst});

    my $dst = pop (@args) if ($has_dst);

    die $Msg->get('TOO_MANY_PARAMS')
        if ($max_src_number && $max_src_number < @args);
    die $Msg->get('NOT_ENGH_PARAMS') if ($has_src && !@args);

    if ($is_src_external) {
        s/^.*\/([^\/]+)$/$1/ foreach (@args);
    }
    
    return { src => @args ? \@args : undef,
             dst => $dst };
}

sub _get_input_file ($$) {
    my ($self, $file_name) = @_;

    my $full_path = $Cfg->get('ROOT_TMP') . '/' . $file_name;
    my $file = CMSsh::FileSystem::File->new($full_path)
        or die $Msg->get('CANT_INST_FSO', $full_path);

    return $file;
}

sub _find_publications ($$) {
    my ($self, $src_ref) = @_;

    sub glob2patt ($) {
        my $glob_str = shift;

        my %patt_map = (
            '*' => '.*',
            '?' => '.',
            '[' => '[',
            ']' => ']',
        );
        $glob_str =~ s{(.)} { $patt_map{$1} || "\Q$1" }ge;

        return '^' . $glob_str . '$';
    }

    my @result;
    foreach (@{$src_ref}) {
        die $Msg->get('ROOT_PUB_OPER') if ($_ eq '/');

        /^(.*\/)([^\/]+)$/ or die $Msg->get('CANT_PARSE_SRC', $_);

        my $coll_uri = $1;
        my $pattern = glob2patt($2);

        my $collection = $self->{_mode}->get_collection($coll_uri) or
# No parent means no publication
            die $Msg->get('SRC_NOT_FOUND', $_);

        my @publications;
        foreach (@{$collection->get_listing()->{$self->get_pub_type}}) {
            push @publications, $_ if ($_->get_name =~ /$pattern/);
        }

        die $Msg->get('SRC_NOT_FOUND', $_) unless (@publications);
        push(@result, @publications);
    }

    return @result;
}

sub _mk () {
    die $Msg->get('ABSTRACT_CLASS');
}

sub _mv ($$) {
    my ($self, $params) = @_;

    my @src_publications = $self->_find_publications($params->{src});

    my $dest_coll = $self->{_mode}->get_collection($params->{dst})
        or die $Msg->get('WRONG_DST');

    foreach (@src_publications) {
        $_->move($dest_coll)
            or die $Msg->get('CANT_MOVE_PUB', $_->get_uri, $dest_coll->get_uri);
    }
}

sub _rn ($$) {
    my ($self, $params) = @_;

    my @src_publications = $self->_find_publications($params->{src});
    die $Msg->get('TOO_MANY_SOURCES', @src_publications)
        if (@src_publications > 1);

    my $new_name = $params->{dst};
    $src_publications[0]->rename($new_name)
        or die $Msg->get('CANT_RENAME_PUB',
                         $src_publications[0]->get_uri, $new_name);
}

sub _rm ($$) {
    my ($self, $params) = @_;

    my @src_publications = $self->_find_publications($params->{src});
    foreach (@src_publications) {
        $_->remove or die $Msg->get('CANT_REMOVE_PUB', $_->get_uri);
    }
}

1
