/*
 * Copyright (C) 2016 Vitaly Minko <vitaly.minko@gmail.com>
 * Copyright (C) 2014-2016 Savoir-faire Linux Inc.
 *
 * Based on dntnode.cpp from OpenDHT.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <opendht.h>
#include <getopt.h>
#include <readline/readline.h>
#include <readline/history.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>

#include <chrono>
#include <cctype>
#include <fstream>
#include <iostream>
#include <regex>
#include <set>
#include <sstream>
#include <string>
#include <thread>
#include <vector>

using namespace dht;


static const constexpr in_port_t DHT_DEFAULT_PORT = 4222;
static const std::string DEFAULT_STORAGE_FILE = "alternatives.txt";
static const std::string DEFAULT_IDENTITY_FILE = ".alternate.identity";
static const constexpr unsigned ALTERNATIVE_LIFETIME = 60;

struct dht_params {
    bool help {false};
    bool log {false};
    std::string logfile {};
    std::string storage_file {};
    std::string identity_file {};
    in_port_t port {0};
    bool daemonize {false};
    std::pair<std::string, std::string> bootstrap {};
};


/**** LOGGING ****************************************************************/

/**
 * Terminal colors for logging
 */
namespace Color {
    enum Code {
        FG_RED      = 31,
        FG_GREEN    = 32,
        FG_YELLOW   = 33,
        FG_BLUE     = 34,
        FG_DEFAULT  = 39,
        BG_RED      = 41,
        BG_GREEN    = 42,
        BG_BLUE     = 44,
        BG_DEFAULT  = 49
    };
    class Modifier {
        const Code m_code;
    public:
        constexpr Modifier(Code code_) : m_code(code_) {}
        friend std::ostream&
        operator<<(std::ostream& os, const Modifier& mod) {
            return os << "\033[" << mod.m_code << 'm';
        }
    };
}

constexpr const Color::Modifier def(Color::FG_DEFAULT);
constexpr const Color::Modifier red(Color::FG_RED);
constexpr const Color::Modifier yellow(Color::FG_YELLOW);

/**
 * Print va_list to std::ostream (used for logging).
 */
void
printLog(std::ostream& s, char const* m, va_list args)
{
    static constexpr int BUF_SZ = 8192;
    char buffer[BUF_SZ];
    int ret = vsnprintf(buffer, sizeof(buffer), m, args);
    if (ret < 0)
        return;
    s.write(buffer, std::min(ret, BUF_SZ));
    if (ret >= BUF_SZ)
        s << "[[TRUNCATED]]";
    s.put('\n');
}

void
enableLogging(dht::DhtRunner& dht)
{
    dht.setLoggers(
        [](char const* m, va_list args){
            std::cerr << red;
            printLog(std::cerr, m, args);
            std::cerr << def;
        },
        [](char const* m, va_list args){
            std::cout << yellow;
            printLog(std::cout, m, args);
            std::cout << def;
        },
        [](char const* m, va_list args){
            printLog(std::cout, m, args);
        }
    );
}

void
enableFileLogging(dht::DhtRunner& dht, const std::string& path)
{
    auto logfile = std::make_shared<std::fstream>();
    logfile->open(path, std::ios::out);

    dht.setLoggers(
        [=](char const* m, va_list args){ printLog(*logfile, m, args); },
        [=](char const* m, va_list args){ printLog(*logfile, m, args); },
        [=](char const* m, va_list args){ printLog(*logfile, m, args); }
    );
}

void
disableLogging(dht::DhtRunner& dht)
{
    dht.setLoggers(dht::NOLOG, dht::NOLOG, dht::NOLOG);
}

/**** END OF LOGGING *********************************************************/



/**** URL ********************************************************************/

bool
is_valid_url(const std::string& url)
{
    std::regex url_regex(
        "("                              // brackets covering match for protocol (optional) and domain
          "([A-Za-z]{3,9}:(?:\\/\\/)?)"  // match protocol, allow in format http:// or mailto:
          "(?:[-;:&=\\+\\$,\\w]+@)?"     // allow something@ for email addresses
          "[A-Za-z0-9.-]+"               // anything looking at all like a domain, non-unicode domains
          "|"                            // or instead of above
          "(?:www.|[-;:&=\\+\\$,\\w]+@)" // starting with something@ or www.
          "[A-Za-z0-9.-]+"               // anything looking at all like a domain
        ")"
        "("                              // brackets covering match for path, query string and anchor
          "(?:\\/[-\\+~%&\\/.\\w_]*)?"    // allow optional /path
          "(\\?)?(?:[-\\+=&;%@.\\w_]*)"  // allow optional query string starting with ?
          "#?(?:[\\w]*)"                 // allow optional anchor #anchor
        ")?"                             // make URL suffix optional
    );
    return std::regex_match(url, url_regex);
}

class URL : public Value::Serializable<URL>
{
private:
    using BaseClass = Value::Serializable<URL>;

public:
    static const ValueType TYPE;
    URL(std::string s) : m_url(s) {
        if (not is_valid_url(m_url))
            throw std::runtime_error(std::string("'") + m_url
                  + std::string("' is not a valid URL."));
    }
    URL(const Blob& b) {
        msgpack_unpack(unpackMsg(b).get());
    }
    std::string m_url;
    MSGPACK_DEFINE(m_url)
};

const ValueType URL::TYPE(6, "URL", std::chrono::minutes(ALTERNATIVE_LIFETIME));

std::ostream& operator<< (std::ostream& s, const URL& url)
{
    s << "URL: " << url.m_url;
    return s;
}

/**** END OF URL *************************************************************/



/**** STORAGE ****************************************************************/

class Storage {
private:
    typedef std::multimap<std::string, std::string> map_type;
    typedef std::lock_guard<std::mutex> lock_guard;

    std::string m_storage_file;
    map_type m_alternatives;
    std::mutex m_mutex;

public:
    typedef typename map_type::const_iterator const_iterator;

    Storage(std::string& file) : m_storage_file(file)
    {
        unsigned line_number = 0;
        std::string line;
        std::ifstream infile(m_storage_file);
        if (not infile) {
            throw std::runtime_error(std::string("Cannot open file ")
                  + m_storage_file + std::string(" for reading."));
        }
        std::string orig_url;
        while (std::getline(infile, line))
        {
            line_number++;
            /* load all alternatives */
            std::istringstream iss(line);
            std::string url;
            if (!(iss >> url)) {
                std::cerr << "Cannot read URL from " << m_storage_file
                          << ", line " << line_number << "." << std::endl;
                continue;
            }
            if (not is_valid_url (url)) {
                std::cerr << "Error in storage file " << m_storage_file
                          << " at line " << line_number << ", '"
                          << url << "' is not a valid URL." << std::endl;
                continue;
            }
            if (isspace(line.at(0))) {
                if (orig_url.empty()) {
                    std::cerr << "Error in storage file " << m_storage_file
                              << " at line " << line_number << ", '"
                              << "Original URL expected, but an alternative"
                              << " URL found." << std::endl;
                    continue;
                }
                m_alternatives.emplace(orig_url, url);
            } else {
                orig_url = url;
            }
        }
    }

    bool empty() {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.empty();
    }

    const_iterator cbegin() {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.cbegin();
    }

    const_iterator cend() {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.cend();
    }

    const_iterator emplace(std::string key, std::string value) {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.emplace(key, value);
    }

    const_iterator find(std::string key) {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.find(key);
    }

    bool erase(std::string key, std::string value) {
        lock_guard guard(this->m_mutex);
        std::pair<const_iterator, const_iterator> iterpair = m_alternatives.equal_range(key);
        const_iterator it = iterpair.first;
        for (; it != iterpair.second; ++it) {
            if (it->second == value) { 
                this->m_alternatives.erase(it);
                return true;
            }
        }
        return false;
    }

    size_t erase(std::string key) {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.erase(key);
    }

    size_t size() {
        lock_guard guard(this->m_mutex);
        return this->m_alternatives.size();
    }

    ~Storage()
    {
        std::ofstream fs(m_storage_file);
        if (fs) {
            decltype(m_alternatives.equal_range("")) range;

            for (auto key_it = m_alternatives.begin(), end = m_alternatives.end();
                 key_it != end;
                 key_it = m_alternatives.upper_bound(key_it->first))
            {
                fs << key_it->first << std::endl;
                std::pair<const_iterator, const_iterator> iterpair = m_alternatives.equal_range(key_it->first);
                auto alt_it = iterpair.first;
                for (; alt_it != iterpair.second; ++alt_it) {
                    fs << "\t" << alt_it->second << std::endl;
                }
            }
            fs.close();
        }
        else {
            std::cerr << "Cannot open the output file '" << m_storage_file
                      << "' for writing." << std::endl;
        }
    }
};

/**** END OF STORAGE *********************************************************/



/**** COMMAND LINE INTERFACE *************************************************/

class ReadLineConsole
{
private:
    std::atomic_bool m_is_reading;
    std::string m_prompt;
    std::stringstream m_tmp_ss;
    std::ostream &m_stream;
public:
    ReadLineConsole(std::string prompt = ">> ",
                    std::ostream &o = std::cout)
      : m_is_reading(false)
      , m_prompt(prompt)
      , m_tmp_ss()
      , m_stream(o)
    {};

    ReadLineConsole& operator>>(std::string& line) {
        m_is_reading = true;
        char* line_read = readline(m_prompt.c_str());
        if (line_read && *line_read)
            add_history(line_read);
        m_is_reading = false;
        line = line_read ? std::string(line_read) : std::string("\0", 1);
        free(line_read);
        return *this;
    }

    template <typename T>
    ReadLineConsole& operator<<(const T &a) {
        m_tmp_ss << a;
        return *this;
    }

    ReadLineConsole& operator<<(std::ostream& (*pf) (std::ostream&)) {
        if (m_is_reading) {
            std::string line_buffer(rl_line_buffer);
            std::string eraser(line_buffer.size() + m_prompt.size() + 2, ' ');
            eraser.front() = '\r';
            eraser.back() = '\r';
            m_stream << eraser << m_tmp_ss.str() << pf
              << m_prompt << line_buffer;
            rl_redisplay();
        } else {
            m_stream << m_tmp_ss.str() << pf;
        }
        m_tmp_ss.str(std::string());
        return *this;
    }
};

void
print_node_info(const std::shared_ptr<DhtRunner>& dht)
{
    std::cout << "Node " << dht->getNodeId() << " running on port "
              <<  dht->getBoundPort() << std::endl;
    std::cout << "Public key ID is " << dht->getId() << std::endl;
}

void
print_help()
{
    std::cout << "Possible commands:" << std::endl
              << "  h, help    Print this help message." << std::endl
              << "  x, quit    Quit the program." << std::endl
              << "  log        Start/stop printing DHT logs." << std::endl;

    std::cout << std::endl << "DHT node information:" << std::endl
              << "  ll         Print basic information and stats about the current node." << std::endl
              << "  ls         Print basic information about current searches." << std::endl
              << "  ld         Print basic information about currently stored values on this node." << std::endl
              << "  lr         Print the full current routing table of this node" << std::endl;

    std::cout << std::endl << "Operations on the DHT:" << std::endl
              << "  g <URL>               Get alternative links for given <URL>." << std::endl
              << "  p <URL> <AltURL>      Put alternative link <AltURL> for <URL>." << std::endl
              << "  d <URL> [<AltURL>]    Delete alternative link(s) for <URL> from local storage." << std::endl;
}

void
cmd_loop(std::shared_ptr<DhtRunner>& dht,
         dht_params& params,
         std::shared_ptr<Storage>& storage,
         ReadLineConsole& tty)
{
    print_node_info(dht);
    std::cout << " (type 'h' or 'help' for a list of possible commands)"
              << std::endl << std::endl;

    // using the GNU History API
    using_history();

    while (true)
    {
        std::string line;
        tty >> line;
        if (!line.empty() && line[0] == '\0')
            break;

        std::istringstream iss(line);
        std::string op, url, value, index, keystr;
        iss >> op;

        if (op == "x" || op == "exit" || op == "quit") {
            break;
        } else if (op == "h" || op == "help") {
            print_help();
            continue;
        } else if (op == "ll") {
            print_node_info(dht);
            unsigned good4, dubious4, cached4, incoming4;
            unsigned good6, dubious6, cached6, incoming6;
            dht->getNodesStats(AF_INET, &good4, &dubious4, &cached4, &incoming4);
            dht->getNodesStats(AF_INET6, &good6, &dubious6, &cached6, &incoming6);
            tty << "IPv4 nodes : " << good4 << " good, "
                                   << dubious4 << " dubious, "
                                   << incoming4 << " incoming." << std::endl;
            tty << "IPv6 nodes : " << good6 << " good, "
                                   << dubious6 << " dubious, "
                                   << incoming6 << " incoming." << std::endl;
            continue;
        } else if (op == "lr") {
            tty << "IPv4 routing table:" << std::endl;
            tty << dht->getRoutingTablesLog(AF_INET) << std::endl;
            tty << "IPv6 routing table:" << std::endl;
            tty << dht->getRoutingTablesLog(AF_INET6) << std::endl;
            continue;
        } else if (op == "ld") {
            tty << dht->getStorageLog() << std::endl;
            continue;
        } else if (op == "ls") {
            tty << "Searches:" << std::endl;
            tty << dht->getSearchesLog() << std::endl;
            continue;
        } else if (op == "la")  {
            tty << "Reported public addresses:" << std::endl;
            auto addrs = dht->getPublicAddressStr();
            for (const auto& addr : addrs)
                tty << addr << std::endl;
            continue;
        } else if (op == "log") {
            params.log = !params.log;
            if (params.log)
                log::enableLogging(*dht);
            else
                log::disableLogging(*dht);
            continue;
        }

        if (op.empty())
            continue;

        static const std::set<std::string> VALID_OPS {"g", "p", "d"};
        if (VALID_OPS.find(op) == VALID_OPS.cend()) {
            tty << "Unknown command: " << op << std::endl
                << " (type 'h' or 'help' for a list of possible commands)"
                << std::endl;
            continue;
        }

        iss >> url;
        if (not is_valid_url(url)) {
            tty << "Syntax error: '" << url << "' is not a valid URL."
                << std::endl;
            continue;
        }

        auto start = std::chrono::high_resolution_clock::now();
        if (op == "g") {
            std::string rem;
            std::getline(iss, rem);
            dht->get(
                InfoHash::get(url),
                [&tty,start](std::shared_ptr<Value> value) {
                    if (value->type == URL::TYPE.id) {
                        auto now = std::chrono::high_resolution_clock::now();
                        tty << "Get: found URL "
                            << "(after " << print_dt(now-start) << "s)";
                        if (value->isSigned()) {
                            tty << " signed by " << value->owner->getId();
                        }
                        tty << std::endl;
                        try {
                            tty << URL(value->data) << std::endl;
                        } catch(const std::exception&e) {
                            std::cerr << e.what() << std::endl;
                        }
                    }
                    return true;
                },
                [&tty,start](bool ok) {
                    auto end = std::chrono::high_resolution_clock::now();
                    tty << "Get: " << (ok ? "completed" : "failure")
                         << " (took " << print_dt(end-start) << "s)" << std::endl;
                },
                {},
                dht::Where {std::move(rem)}
            );
        }
        else if (op == "p") {
            std::string alt_url;
            iss >> alt_url;
            try {
                storage->emplace(url, alt_url);
                dht->putSigned(
                    InfoHash::get(url),
                    URL(alt_url),
                    [&tty,start](bool ok) {
                        auto end = std::chrono::high_resolution_clock::now();
                        tty << "Put: " << (ok ? "success" : "failure")
                            << " (took " << print_dt(end-start) << "s)"
                            << std::endl;
                    }
                );
            } catch(const std::exception&e) {
                std::cerr << e.what() << std::endl;
            }
        }
        else if (op == "d") {
            std::string alt_url;
            iss >> alt_url;
            if (not alt_url.empty()) {
                if (storage->erase(url, alt_url)) {
                    tty << "The alternative URL has been successfully deleted"
                        << " from local storage." << std::endl;
                }
                else {
                    tty << "Failed to delete the alternative URL from"
                        << " the local storage." << std::endl;
                }
            }
            else {
                std::size_t deleted_num = storage->erase(url);
                tty << "Deleted " << deleted_num << "alternatives."
                    << std::endl;
            }
        }
    }

    tty << std::endl <<  "Stopping node..." << std::endl;
}

/**** END OF COMMAND LINE INTERFACE ******************************************/



/**** HANDLE ARGUMENTS *******************************************************/

void
signal_handler(int sig)
{
    switch(sig) {
    case SIGHUP:
        break;
    case SIGTERM:
        exit(EXIT_SUCCESS);
        break;
    }
}

void
daemonize()
{
    pid_t pid = fork();
    if (pid < 0) exit(EXIT_FAILURE);
    if (pid > 0) exit(EXIT_SUCCESS);

    umask(0);

    pid_t sid = setsid();
    if (sid < 0) {
        exit(EXIT_FAILURE);
    }

    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);

    signal(SIGCHLD,SIG_IGN); /* ignore child */
    signal(SIGTSTP,SIG_IGN); /* ignore tty signals */
    signal(SIGTTOU,SIG_IGN);
    signal(SIGTTIN,SIG_IGN);
    signal(SIGHUP,signal_handler); /* catch hangup signal */
    signal(SIGTERM,signal_handler); /* catch kill signal */
}

void
print_usage()
{
    std::cout << "Usage: alternate [-v [-l logfile]] [-s storage_file] [-i identity_file]" << std::endl
              << "                 [-d] [-p local_port] [-b bootstrap_host[:port]]"
              << std::endl << std::endl
              << "alternate, command line tool for finding spare URLs"
              << std::endl
              << "Version: 0.1" << std::endl
              << "Report bugs to: <vitaly.minko@gmail.com>" << std::endl;
}

dht::crypto::Identity
load_identity_from(std::string& identity_file)
{
    std::ifstream ident_ifstream(identity_file.c_str(), std::ios::in|std::ios::binary);
    if (ident_ifstream.good() && ident_ifstream.is_open())
    {
        /* read private key from the file */
        uint16_t priv_key_size_nbo;
        ident_ifstream.read(reinterpret_cast<char*>(&priv_key_size_nbo),
                            sizeof(priv_key_size_nbo));
        if (ident_ifstream.fail()) {
            throw std::runtime_error(std::string("Failed to read size of private key"
                                                 " from the file ")
                                     + identity_file);
        }
        uint16_t priv_key_size_hbo = ntohs(priv_key_size_nbo);

        std::vector<uint8_t> priv_key_buf(priv_key_size_hbo);
        ident_ifstream.read(reinterpret_cast<char*>(priv_key_buf.data()),
                            priv_key_buf.size());
        if (ident_ifstream.fail()) {
            throw std::runtime_error(std::string("Failed to read private key"
                                                 " from the file ")
                                     + identity_file);
        }
        Blob priv_key_blob(priv_key_buf.begin(), priv_key_buf.end());

        /* read certificate from the file */
        uint16_t cert_size_nbo;
        ident_ifstream.read(reinterpret_cast<char*>(&cert_size_nbo),
                            sizeof(cert_size_nbo));
        if (ident_ifstream.fail()) {
            throw std::runtime_error(std::string("Failed to read size of certificate"
                                                 " from the file ")
                                     + identity_file);
        }
        uint16_t cert_size_hbo = ntohs(cert_size_nbo);

        std::vector<uint8_t> cert_buf(cert_size_hbo);
        ident_ifstream.read(reinterpret_cast<char*>(cert_buf.data()),
                            cert_buf.size());
        if (ident_ifstream.fail()) {
            throw std::runtime_error(std::string("Failed to read certificate"
                                                 " from the file ")
                                     + identity_file);
        }

        Blob cert_blob {cert_buf.begin(), cert_buf.end()};

        ident_ifstream.close();

        try {
            return std::make_pair(std::make_shared<dht::crypto::PrivateKey>(priv_key_blob),
                                  std::make_shared<dht::crypto::Certificate>(cert_blob));
        } catch (const std::exception& e) {
            std::cerr << "Error occurred while parsing the identity file."
                      << std::endl <<  e.what() << std::endl;
            throw;
        }
    } else {
        /* generate new identity and save it to the file */
        auto id = dht::crypto::generateIdentity();
        auto private_key = id.first;
        auto certificate = id.second;

        std::ofstream ident_ofstream(identity_file,
                                     std::ios::trunc | std::ios::binary);

        /* serialize and save the private key */
        Blob priv_key_blob = private_key->serialize();
        uint16_t priv_key_size_hbo = priv_key_blob.size();
        uint16_t priv_key_size_nbo = htons(priv_key_size_hbo);
        ident_ofstream.write(reinterpret_cast<char*>(&priv_key_size_nbo),
                             sizeof(priv_key_size_nbo) );
        ident_ofstream.write(reinterpret_cast<char*>(priv_key_blob.data()),
                              priv_key_blob.size() );
        if (ident_ofstream.fail()) {
            throw std::runtime_error(std::string("Failed to write private key"
                                                 " to the file ")
                                     + identity_file);
        }

        /* serialize and save the certificate */
        Blob cert_blob = certificate->getPacked();
        uint16_t cert_size_hbo = cert_blob.size();
        uint16_t cert_size_nbo = htons(cert_size_hbo);
        ident_ofstream.write( (char*)&cert_size_nbo,
                              sizeof(cert_size_nbo) );
        ident_ofstream.write( (char*)cert_blob.data(),
                              cert_blob.size() );
        if (ident_ofstream.fail()) {
            throw std::runtime_error(std::string("Failed to write certificate"
                                                 " to the file ")
                                     + identity_file);
        }

        return id;
    }
}

/**** END OF HANDLE ARGUMENTS ************************************************/



/**** PARSING ARGUMENTS ******************************************************/

/**
 * Split "[host]:port" or "host:port" to pair<"host", "port">.
 */
std::pair<std::string, std::string>
split_port(const std::string& s)
{
    if (s.empty())
        return {};
    if (s[0] == '[') {
        std::size_t closure = s.find_first_of(']');
        std::size_t found = s.find_last_of(':');
        if (closure == std::string::npos)
            return {s, ""};
        if (found == std::string::npos or found < closure)
            return {s.substr(1,closure-1), ""};
        return {s.substr(1,closure-1), s.substr(found+1)};
    }
    std::size_t found = s.find_last_of(':');
    std::size_t first = s.find_first_of(':');
    if (found == std::string::npos or found != first)
        return {s, ""};
    return {s.substr(0,found), s.substr(found+1)};
}

static const constexpr struct option long_options[] = {
   {"help",          no_argument      , nullptr, 'h'},
   {"port",          required_argument, nullptr, 'p'},
   {"bootstrap",     required_argument, nullptr, 'b'},
   {"verbose",       no_argument      , nullptr, 'v'},
   {"daemonize",     no_argument      , nullptr, 'd'},
   {"logfile",       required_argument, nullptr, 'l'},
   {"storage_file",  required_argument, nullptr, 's'},
   {"identity_file", required_argument, nullptr, 'i'},
   {nullptr,         0                , nullptr,  0}
};

dht_params
parse_args(int argc, char **argv)
{
    dht_params params;
    int opt;
    while (-1 != (opt = getopt_long(argc, argv,
                                    "hdvp:b:l:s:i:", long_options,
                                    nullptr)))
    {
        switch (opt) {
        case 'p': {
                int port_arg = atoi(optarg);
                if (port_arg >= 0 && port_arg < 0x10000)
                    params.port = port_arg;
                else
                    std::cout << "Invalid port: " << port_arg << std::endl;
            }
            break;
        case 'b':
            params.bootstrap = split_port((optarg[0] == '=') ? optarg + 1 : optarg);
            if (not params.bootstrap.first.empty()
                and params.bootstrap.second.empty())
            {
                std::stringstream ss;
                ss << DHT_DEFAULT_PORT;
                params.bootstrap.second = ss.str();
            }
            break;
        case 'h':
            params.help = true;
            break;
        case 'l':
            params.logfile = optarg;
            break;
        case 's':
            params.storage_file = optarg;
            break;
        case 'i':
            params.identity_file = optarg;
            break;
        case 'v':
            params.log = true;
            break;
        case 'd':
            params.daemonize = true;
            break;
        default:
            break;
        }
    }
    return params;
}

/**** END OF PARSING ARGUMENTS ***********************************************/



/**** THREAD FOR ADVERTISING ALTERNATIVES ************************************/

class TimerKiller {
public:
    // returns false if killed:
    template<class R, class P>
    bool wait_for( std::chrono::duration<R,P> const& time ) {
        std::unique_lock<std::mutex> lock(m_mutex);
        return !m_cv.wait_for(lock, time, [&]{return m_terminate;});
    }
    void kill() {
        std::unique_lock<std::mutex> lock(m_mutex);
        m_terminate = true;
        m_cv.notify_all();
    }
private:
    std::condition_variable m_cv;
    std::mutex m_mutex;
    bool m_terminate = false;
};


/* Runs in a separate thread. */
void
advertise_alternatives(std::shared_ptr<TimerKiller>& timer_killer,
                       std::shared_ptr<DhtRunner>& dht,
                       std::shared_ptr<Storage>& storage,
                       ReadLineConsole& tty)
{
    do {
        for (auto it = storage->cbegin(), end = storage->cend();
             it != end;
             ++it)
        {
            try {
                dht->putSigned(
                    InfoHash::get(it->first),
                    URL(it->second),
                    [it,&tty](bool ok) {
                        tty << (ok ? "Successfully advertised " : "Failed to advertise ")
                            << it->first << "->" << it->second << std::endl;
                    }
                );
            } catch(const std::exception&e) {
                tty << e.what() << std::endl;
            }
        }
    } while (timer_killer->wait_for(std::chrono::minutes(ALTERNATIVE_LIFETIME/2)));
}

/**** END OF THREAD FOR ADVERTISING ALTERNATIVES *****************************/


int
main(int argc, char **argv)
{
    auto dht = std::make_shared<DhtRunner>();

    try {
        auto params = parse_args(argc, argv);
        if (params.help) {
            print_usage();
            return 0;
        }
        if (params.daemonize) {
            daemonize();
        }

        dht::crypto::Identity crt {};
        auto ca_tmp = dht::crypto::generateIdentity("DHT Node CA");
        crt = dht::crypto::generateIdentity("DHT Node CA", ca_tmp);

        std::string identity_file = params.identity_file.empty() ?
                                    DEFAULT_IDENTITY_FILE : params.identity_file;
        try {
            dht::crypto::Identity crt = load_identity_from(identity_file);
            dht->run(params.port, crt, true);
        } catch (const std::exception& e) {
            std::cerr << "Could not obtain the identity from the file "
                      << identity_file << "." << std::endl
                      << e.what() << std::endl;
            exit(EXIT_FAILURE);
        }

        if (params.log) {
            if (not params.logfile.empty())
                log::enableFileLogging(*dht, params.logfile);
            else
                log::enableLogging(*dht);
        }

        if (not params.bootstrap.first.empty()) {
            dht->bootstrap(params.bootstrap.first.c_str(),
                           params.bootstrap.second.c_str());
        }

        std::string storage_file = params.storage_file.empty() ?
                                   DEFAULT_STORAGE_FILE : params.storage_file;
        auto storage = std::make_shared<Storage>(storage_file);

        ReadLineConsole tty;

        auto timer_killer = std::make_shared<TimerKiller>();
        std::thread advertising_thread(advertise_alternatives,
                                       std::ref(timer_killer),
                                       std::ref(dht),
                                       std::ref(storage),
                                       std::ref(tty));

        if (params.daemonize) {
            while (true)
                std::this_thread::sleep_for(std::chrono::seconds(30));
        } else {
            cmd_loop(dht, params, storage, tty);
        }

        timer_killer->kill();
        advertising_thread.join();
    }
    catch(const std::exception&e) {
        std::cerr << std::endl <<  e.what() << std::endl;
    }

    std::condition_variable cv;
    std::mutex m;
    std::atomic_bool done {false};

    dht->shutdown([&]()
    {
        std::lock_guard<std::mutex> lk(m);
        done = true;
        cv.notify_all();
    });

    // wait for shutdown
    std::unique_lock<std::mutex> lk(m);
    cv.wait(lk, [&](){ return done.load(); });

    dht->join();

    return 0;
}

