/*
xkeymouse enables synthezing button events from keyboard events
Copyright (C) 2001 Henrik Sandklef

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
ERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


  

#include "xkeymouse.h"


static char *xkm_help[]=
  {
    PACKAGE " - maps keys (with modifiers) to:",
    "\tmouse motion",
    "\tbutton press/release",
    "\tkey press/release",
    "\tlaunch program",
    NULL
  };

static char *xkm_description[]=
  {
    PACKAGE "grabs keys as specified in ithe rc file. These keys", 
    "are mapped to either a mouse button click or mouse motion",
    NULL
  };


static char *xkm_example[]=
  {
    ".... examples are coming soon",
    NULL
  };

static char *xkm_files[]=
  {
    PACKAGE "read files in the following order:",
    "\tgiven by environment variable: " PROGRAM_RC_ENV ,
    "\t" XKEYMOUSE_LOCAL_RC,
    "\t" XKEYMOUSE_SYSTEM_RC,
    NULL
  };

static char *xkm_file_syntax[]=
  {
    " ... se example xkmrc as delivered in this package",
    NULL
  };

static char *xkm_options[]=
  {
    "--display <name>, -d <name>" , "display to connect to",
    "--verbose, -v",                "set verbose mode",
    "--help, -h",                   "prints this help printout",
    "--version, -V",                "prints version number",
    NULL
  };


int
get_modifier_from_mapping_sub(Display *display, char *mod_str) 
{
  XModifierKeymap *mod_keymap;
  KeyCode *keycode_ptr;
  KeySym my_keysym;
  int i, j, max_keys;
  int ret=0;
  static char *mod_list[]={"ShiftMask", "LockMask","ControlMask", 
		  "Mod1Mask", "Mod2Mask", "Mod3Mask", "Mod4Mask", "Mod5Mask" };

  mod_keymap = XGetModifierMapping(display);
  max_keys = mod_keymap->max_keypermod;
  keycode_ptr = mod_keymap->modifiermap;
  for (i = 0; i < 8; i++) {
    for (j = 0; j < max_keys; j++) {
      my_keysym = XKeycodeToKeysym(display, *keycode_ptr, 0);

      /*      xkm_verbose ("\tChecking modier mapping \"%s\" == \"%s\"\n", mod_str, XKeysymToString(my_keysym));*/
      if (XKeysymToString(my_keysym)!=NULL)
	{
	  if (xkm_check (mod_str, XKeysymToString(my_keysym),XKeysymToString(my_keysym)))
	    {
	      ret |= get_modifier_sub(NULL, mod_list[i]);
	      xkm_verbose ("= %s --> %s .... returning MASK=%d\n", mod_str, mod_list[i], ret);
	      return ret;
	    }
	  else
	    {
	      ;
	    }
	}
      else
	{
	  ;
	}
      keycode_ptr++;
    }
  }
  return ret;
}

int
get_modifier_from_mapping(Display *display, char *mod_str) 
{
  XModifierKeymap *mod_keymap;
  KeyCode *keycode_ptr;
  KeySym my_keysym;
  int i, j, max_keys;
  int ret;

  
  /* xkeymouse syntax related translations */
  if (xkm_check(mod_str, "Alt", "Alt"))
    {
      ret = get_modifier_from_mapping_sub(display, "Alt_L");
      ret |= get_modifier_from_mapping_sub(display, "Alt_R");
    }
  else if (xkm_check(mod_str, "Shift", "Shift"))
    {
      ret = get_modifier_from_mapping_sub(display, "Shift_L");
      ret |= get_modifier_from_mapping_sub(display, "Shift_R");
    }
  else if (xkm_check(mod_str, "Control", "Control"))
    {
      ret = get_modifier_from_mapping_sub(display, "Control_L");
      ret |= get_modifier_from_mapping_sub(display, "Control_R");
    }
  else if (xkm_check(mod_str, "Scroll_Lock", "Scroll"))
    {
      ret = get_modifier_from_mapping_sub(display, "Scroll_Lock");
    }
  else if (xkm_check(mod_str, "Caps_Lock", "Caps"))
    {
      ret = get_modifier_from_mapping_sub(display, "Caps_Lock");
    }
  else
    {
      ret = get_modifier_from_mapping_sub(display, mod_str);
    }
  return ret;
}


int
get_modifier_sub(xkm_prog_data *xd,  char *mod_str) 
{
  int ret=0;
  if( xkm_check (mod_str, "ShiftMask", "ShiftMask"  ) )
    {
      ret = ShiftMask;
    }
  else if( xkm_check (mod_str, "LockMask", "LockMask"  ) )
    {
      ret = LockMask;
    }
  else if( xkm_check (mod_str, "ControlMask", "ControlMask"  ) )
    {
      ret = ControlMask;
    }
  else if( xkm_check (mod_str, "Mod1Mask", "Mod1Mask"  ) )
    {
      ret = Mod1Mask;
    }
  else if( xkm_check (mod_str, "Mod2Mask", "Mod2Mask"  ) )
    {
      ret = Mod2Mask;
    }
  else if( xkm_check (mod_str, "Mod3Mask", "Mod3Mask"  ) )
    {
      ret = Mod3Mask;
    }
  else if( xkm_check (mod_str, "Mod4Mask", "Mod4Mask"  ) )
    {
      ret = Mod4Mask;
    }
  else if( xkm_check (mod_str, "Mod5Mask", "Mod5Mask"  ) )
    {
      ret = Mod5Mask;
    }
  else if( xkm_check (mod_str, "AnyModifier", "any"  ) )
    {
      ret = AnyModifier;
    }
  else if( xkm_check (mod_str, "none", "0"  ) )
    {
      ret = 0;
    }
  else 
    {
      /* prevent recursive disaster ... */
      if (xd==NULL)
	ret = 0;
      else
	ret = get_modifier_from_mapping( xd->data_display, mod_str);
    }
  return ret;
}

int 
is_last (xkm_prog_data *xd,  char *mod_strs) 
{
  char *tmp;
  int i ;
  int len = strlen (mod_strs);
  tmp=mod_strs;

  for (i=0;  (i<len) ; i++)
    {
      if (tmp[i]=='+')
	{
	  //	  printf ("FOUND + in %s\n", tmp);
	  return 0;
	}
    }
  return 1;
}

int
get_modifier(xkm_prog_data *xd,  char *mod_strs) 
{
  int ret=0;
  int try_ret;
  char *tmp;
  char mod_head[20]="";
  char mod_tail[40]="";
  int i;
  int last=0;
  int len=strlen(mod_strs);
  int found=0;
  static int _level=0;
  int level=_level;
  xkm_rem_blanks(mod_strs);

  _level++;
  
  xkm_verbose ("%d ---> get_modifier:\"%s\"\n", level, mod_strs);
  tmp=mod_strs;
  
  if (!is_last(xd, tmp)==1)
    {
      for (i=0;  (i<len) ; i++)
	{
	  if (tmp[i]=='+')
	    {
	      
	      found=1;
	      strncpy (mod_head, tmp, i); 
	      tmp+=(sizeof(char)*i+1);
	      
	      strcpy (mod_tail, tmp); 
	      ret |= get_modifier_sub (xd, mod_head);
	      ret |= get_modifier (xd, mod_tail);
	      //	      xkm_verbose ("%d ---- get_modifier FOUND \"%s\"   ret=%d\n", level, tmp, ret);
	      break;
	    }
	}
    }
  else
    {
      ret |= get_modifier_sub (xd, tmp);
      xkm_verbose ("%d ---- get_modifier LAST :\"%s\"  ret=%d\n", level, tmp, ret);
      
    }
  /*	  XKM_TAB(level);
	  xkm_verbose ("\t %d head:\"%s\"   tail:\"%s\"\n", level, mod_head, mod_tail);
  */
  
  xkm_verbose ("%d <--- get_modifier:\"%s\" --- ret=%d\n", level, mod_strs, ret);
  return ret;
}


int
get_modifier2(xkm_prog_data *xd,  char *mod_strs) 
{
  int ret=0;
  int try_ret;
  char *tmp;
  char mod_head[20]="";
  char *mod_tail=NULL;
  int i;
  int last=0;
  int len=strlen(mod_strs);
  int found=0;
  static int _level=0;
  int level=_level;
  xkm_rem_blanks(mod_strs);

  _level++;
  
  xkm_verbose ("%d ---> get_modifier:\"%s\"\n", level, mod_strs);
  tmp=mod_strs;
  
  strcpy (mod_head, mod_strs); 
  for (i=0;  (i<len) ; i++)
    {
      if (tmp[i]=='+')
	{
	  found=1;
	  /* xkm_verbose ("\thead:\"%s\"   tmp:\"%s\"\n", mod_head, tmp);*/
	  strncpy (mod_head, tmp, i); 
	  tmp+=(sizeof(char)*i+1);
	  mod_tail=(char*) malloc (sizeof(char)*strlen(tmp));
	  strcpy (mod_tail, tmp); 
	  
	  xkm_verbose ("\t %d head:\"%s\"   tail:\"%s\"\n", level, mod_head, mod_tail);
	  if ( (try_ret=get_modifier(xd, mod_tail)) != -1)
	    {
	      xkm_verbose ("\t %d PLACE 3 %s\n", level, mod_tail);
	      free (mod_tail);
	      ret |= try_ret;
	    }
	  else
	    {
	      xkm_verbose ("\t %d PLACE 4\n", level);
	      free (mod_tail);
	      ret=-1;
	      xkm_verbose (" ERROR 12\n");
	      exit(0);
	      break;
	    }
	  break;
	}
    }

  /*  if (found==0)
      {*/
      xkm_verbose ("\t %d PLACE 8 really getting \"%s\" \"%s\" \"%s\" \n", level, mod_strs, mod_head, mod_tail);
      if ( (try_ret=get_modifier_sub(xd, mod_strs)) != -1)
	{
	  xkm_verbose ("\t %d PLACE 1 (%s)\n", level, mod_head);
	  ret |= try_ret;
	}
      else
	{
	  xkm_verbose ("\t %d PLACE 2\n", level);
	  //	  break;
	}
      /*    }*/
  xkm_verbose ("%d <--- get_modifier:\"%s\" --- ret=%d\n", level, mod_strs, ret);
  return ret;
}



void
xkm_print_chars ( char **str_arr)
{
  char **tmp;

  for (tmp=str_arr; *tmp ; tmp++)
    {
      fprintf (stdout, "\t%s\n", *tmp);
    }
}

void
xkm_print_opt_chars ( char **str_arr)
{
  int i=0; 
  char **tmp;

  for (tmp=str_arr; *tmp ; tmp++)
    {
	fprintf (stdout, "\t%s", *tmp);
      if (i++%2)
	fprintf (stdout, "\n");
    }
}

void 
xkm_usage ()
{

  fprintf (stdout, "NAME\n ");
  xkm_print_chars (xkm_help);
  fprintf (stdout, "\n");

  fprintf (stdout, "DESCRIPTION\n");
  xkm_print_chars (xkm_description);
  fprintf (stdout, "\n");

  fprintf (stdout, "OPTIONS\n");
  xkm_print_opt_chars (xkm_options);
  fprintf (stdout, "\n");

  fprintf (stdout, "RESOURCE FILES\n");
  xkm_print_chars (xkm_files);
  fprintf (stdout, "\n");

  fprintf (stdout, "RESOURCE FILE SYNTAX\n");
  xkm_print_chars (xkm_file_syntax);
  fprintf (stdout, "\n");

  fprintf (stdout, "EXAMPLE\n");
  xkm_print_chars (xkm_example);
  fprintf (stdout, "\n");
}


void 
xkm_verbose (char *msg, ...)
{
  va_list ap;
  if (xkm_data->verbose) 
    {
      va_start(ap, msg);
      vfprintf ( stderr, msg, ap );
    }
}


void 
xkm_error (char *msg, ...)
{
  va_list ap;
  if (1) 
    {
      va_start(ap, msg);
      vfprintf ( stderr, msg, ap );
    }
}

void 
signal_handler(int sig) 
{
  xkm_verbose ("Inside handler sig=%d\n",sig);
  switch (sig)
    {
    case SIGINT:
      xkm_close_down (xkm_data);
      exit (0);
      break;
    default:
      xkm_error ( "signal_handler error .... unxpected signal (%d)\n .... leaving", sig);
     _exit (sig);
    }
}




/**
 *
 *
 *
 */
grab_data* 
new_grab_data()
{
  grab_data *internal_gd;
  internal_gd=(grab_data *) malloc ((sizeof (grab_data)) +1);
  if (internal_gd==NULL)
    {
      exit (XKEYMOUSE_ALLOC_FAILED);
    }
  internal_gd->code           = 0;
  internal_gd->modifier       = 0;
  internal_gd->type           = -1;
  internal_gd->autorepeat     = XKEYMOUSE_REPEAT_OFF;
  return internal_gd;
}

/*grab_data** 
new_grab_data_list()
{
  int i=0;
  grab_data **gd_list = NULL;
  xkm_verbose ("allocating ", size);
  return gd_list;
}
*/

grab_data **
alloc_grab_data (grab_data **gd_list, xkm_prog_data *xd)
{
  
  grab_data** tmp;
  int size=xd->gdl_size;
  if ( (gd_list==NULL) || (size==0) ) 
    {
      tmp = (grab_data **) calloc ( size+1, sizeof (grab_data));
      if (tmp==NULL)
	{
	  exit (XKEYMOUSE_ALLOC_FAILED);
	}
      tmp[size] = new_grab_data();
    }
  else
    {
      tmp = (grab_data **) realloc ( gd_list, (size+1)*sizeof(grab_data));
      if (tmp==NULL)
	{
	  exit (XKEYMOUSE_ALLOC_FAILED);
	}
      else
	{
	  tmp[size] = new_grab_data();
	}
    }
  xd->gdl_size++;
  return tmp;
}



void
xkm_print_settings(xkm_prog_data *xd)
{
  int i=0;
  for (i=0;i<xd->size;i++)
    {
      xkm_verbose ("grab_data_list[%d]=%d\n",i, xd->grab_data_list[i]->type );
    }  
}

void 
xkm_close_down(xkm_prog_data *xd)
{
  int i=0;
  xkm_verbose ("Ungrabbing keys\n");
  grab_all_data (xd, 0);

  
  for (i=0;i<xd->size;i++)
    {
      if ( xd->grab_data_list[i]->autorepeat==XKEYMOUSE_REPEAT_ON ) 
	{
	  xkm_verbose ("Turning on autorepeat on key %d\n",xd->grab_data_list[i]->code );
	  set_no_repeat ( xd->grab_data_list[i]->code , XKEYMOUSE_REPEAT_ON);
	}
    }  


  xkm_verbose ("Freeing memory\n");
  for (i=0;i<xd->size;i++)
    {
      free(xd->grab_data_list[i]);
    }
  free (xd->grab_data_list);
}

char *
str_strip( char *str )
{
  char ret[ strlen(str)];
  char* p;
  char* ret_char;
  int i;

  for ( p = str; *p != 0; p++ )
    if ( *p != ' ' && *p != '\t' )
      break;

  strcpy( ret, p );
  for ( p = ret; *p != 0; p++ )
    if ( *p == ' ' || *p == '\t' )
      break;
  *p = 0;
  ret_char=(char*) malloc (sizeof (char) * strlen ( ret));
  strcpy (ret_char, ret);

  return ret_char;
}
 


int 
xkm_strip(char *str) 
{
  int i ;
  int nr_of_blanks = 0;

  int blanks_removed = 0 ;


  for (i=0 ; i<=strlen(str) ; i++)
    {
      if ( !blanks_removed && (str[i]==' ') )
	{
	  nr_of_blanks++;
	}
      else
	{
	  blanks_removed=1;
	  break;
	}
    }
  if (nr_of_blanks!=0) 
    {

      for (i=0 ; i<=strlen(str) ; i++)
	{
	  str[i]=str[i+nr_of_blanks];
	}
    }

  for (i=0 ; i<=strlen(str) ; i++)
    {
      if ((str[i]=='\n')) 
	{
	  str[i]='\0';
	}
    }
  return 1;
} 


int 
is_comment_or_blank (char *str) 
{
  if ( (str[0]=='#') || (str[0]=='\n') || (strlen(str)==0))
    { 
      return 1;
    }
  else 
    {
      return 0;
    }
}




int 
get_key (xkm_prog_data* xd,char *_buf)
{
  int tmp_member;
  char *buf=_buf;
  xkm_rem_blanks(buf);
  xkm_verbose ("\tget_key:\"%s\"\n", buf);

  if ( strncmp(buf, "keycode=", strlen("keycode="))==0 )
       {
	 buf+=(sizeof(char)*strlen("keycode="));
	 xkm_verbose ("\tget_key: now:::: \"%s\"\n", buf);
	 if ( sscanf ( buf, "%d", &tmp_member) > 0 )
	   {
	     return tmp_member;
	   }
	 else
	   return -1;
       }
  else
    {
      xkm_verbose ("\tget_key: -- 1 --\n");
      if (strlen(buf)!=1)
	{
	  xkm_verbose ("\tget_key: -- 2 --\n");
	  fprintf (stderr, "SYNTAX ERROR\n\tCouldn't convert "
		   "string \"%s\" to an int.\n",buf);
	}
      else
	{
	  xkm_verbose ("\tget_key: -- 3 --\n");
	  /* is it a letter */
	  tmp_member = XStringToKeysym (buf)  ; 
	  if ( tmp_member == 0 )
	    {
	      fprintf (stderr, "SYNTAX ERROR\n\tCouldn't convert "
		       "string \"%s\" to an int\n",buf);
	      
	    }
	  xkm_verbose ("\tget_key: -- 4 --\n");
	  
	  tmp_member = XKeysymToKeycode (xd->data_display, 
					 tmp_member);
	}
      xkm_verbose ("\tget_key: -- 5 --\n");
      
      xkm_verbose ("\tget_key:\"%s\"  returning=%d\n", buf, tmp_member);
      return tmp_member;
    }
  return 0;
}

int 
add_exec_argument (xkm_prog_data* xd, grab_data *gd, char *str, int action_arg)
{
  if ( action_arg == 1 )
    {
      strcpy (gd->xkm_action.ea.command_line,str);
    }
  else
    {
      xkm_rem_blanks (str);
      if (xkm_check (str, "NoFork", "NoFork"))
	{
	  ;
	}
      else if (xkm_check (str, "Fork", "Fork"))
	{
	  strcat (gd->xkm_action.ea.command_line," &");
	}
      else if (xkm_check (str, "NoAutoRepeat", "NoAutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_OFF;
	}
      else if (xkm_check (str, "AutoRepeat", "AutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_ON;
	}
    }
  xkm_verbose ("\t   motion argument=%s\n", gd->xkm_action.ea.command_line); 
  xkm_verbose ("\t   exec argument=%s\n", gd->xkm_action.ea.command_line); 
}



int 
add_key_argument (xkm_prog_data* xd, grab_data *gd, char *str, int action_arg)
{
  xkm_rem_blanks (str);
  if ( action_arg == 1 )
    {
      ;
    }
  else
    {
      xkm_rem_blanks (str);
      if (xkm_check (str, "NoAutoRepeat", "NoAutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_OFF;
	}
      else if (xkm_check (str, "AutoRepeat", "AutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_ON;
	}
    }
}


int 
add_button_argument (xkm_prog_data* xd, grab_data *gd, char *str, int action_arg)
{
  xkm_rem_blanks (str);
  if ( action_arg == 1 )
    {
      if ( sscanf ( str, "%d", &gd->xkm_action.ba.nr) <= 0 )
	{
	  return -1;
	}
    }
  else 
    {
      
      if (xkm_check (str, "NoAutoRepeat", "NoAutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_OFF;
	}
      else if (xkm_check (str, "AutoRepeat", "AutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_ON;
	}
    }
}


int 
add_motion_argument (xkm_prog_data* xd, grab_data *gd, char *str, int action_arg)
{
  xkm_rem_blanks (str);
  if ( action_arg == 1 )
    {
      if (xkm_check (str, "Up", "Up"))
	{
	  gd->xkm_action.ma.x_direction=XKEYMOUSE_NO_MOTION;
	  gd->xkm_action.ma.y_direction=XKEYMOUSE_MOTION_UP;
	}
      else if (xkm_check (str, "Left", "Left"))
	{
	  gd->xkm_action.ma.x_direction=XKEYMOUSE_MOTION_LEFT;
	  gd->xkm_action.ma.y_direction=XKEYMOUSE_NO_MOTION;
	}
      else if (xkm_check (str, "Down", "Down"))
	{
	  gd->xkm_action.ma.x_direction=XKEYMOUSE_NO_MOTION;
	  gd->xkm_action.ma.y_direction=XKEYMOUSE_MOTION_DOWN;
	}
      else if (xkm_check (str, "Right", "Right"))
	{
	  gd->xkm_action.ma.x_direction=XKEYMOUSE_MOTION_RIGHT;
	  gd->xkm_action.ma.y_direction=XKEYMOUSE_NO_MOTION;
	}
    }
  else 
    {
      if (xkm_check (str, "NoAutoRepeat", "NoAutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_OFF;
	}
      else if (xkm_check (str, "AutoRepeat", "AutoRepeat"))
	{
	  gd->autorepeat=XKEYMOUSE_REPEAT_ON;
	}
      else if (strncmp (str, "Speed=", strlen("Speed="))==0)
	{
	  char *tmp=str;
	  char speed_buf[3];
	  tmp+=strlen("Speed=");
	  if ( sscanf ( tmp, "%d", &gd->xkm_action.ma.speed) <= 0 )
	    {
	      xkm_verbose ("faulty resource file\n");
	      exit(-1);
	    }
	  
	}
    }
}


int 
add_argument (xkm_prog_data* xd, grab_data *gd, char *str, int action_arg)
{
  xkm_verbose ("\tadding type=%d\targument=%s\n", gd->type, str); 
  switch (gd->type)
    {
    case XKEYMOUSE_BUTTON:
      add_button_argument (xd, gd, str, action_arg);
      break;
    case XKEYMOUSE_EXEC:
      add_exec_argument (xd, gd, str, action_arg);
      break;
    case XKEYMOUSE_MOTION:
      add_motion_argument (xd, gd, str, action_arg);
      break;
    case XKEYMOUSE_KEY:
      add_key_argument (xd, gd, str, action_arg);
      break;
    default:
      xkm_verbose ("adding nothing .... :(\n"); 
      break;
    }
}

int 
get_type (xkm_prog_data* xd,char *str)
{
  int ret;
  xkm_rem_blanks(str);
  xkm_verbose ("\tget_type:\"%s\"\n", str);
  if (xkm_check (str, "Button", "b"))
    {
      ret = XKEYMOUSE_BUTTON;
    }
  else if (xkm_check (str, "Motion", "m"))
    {
      ret = XKEYMOUSE_MOTION;
    }
  else if (xkm_check (str, "Exec", "e"))
    {
      ret = XKEYMOUSE_EXEC;
    }
  else if (xkm_check (str, "Key", "k"))
    {
      ret = XKEYMOUSE_KEY;
    }
  else
    {
      xkm_verbose ("--- get_type we AINT got nothing\n", str);
      ret = -1;
    }
  return ret;
}

void
verbose_gd (grab_data *gd)
{
  xkm_verbose ("################### grab data ############# \n");
  xkm_verbose ("#                                         # \n");
  xkm_verbose ("#  key        = %.4d                      # \n", gd->code);
  xkm_verbose ("#  modifier   = %.6d                    # \n", gd->modifier);
  xkm_verbose ("#  autorepeat = %.1d                         # \n", gd->autorepeat);

  switch (gd->type)
    {
    case XKEYMOUSE_BUTTON:
      xkm_verbose ("#  Button:                                # \n");
      xkm_verbose ("#   nr   = %.2d                             # \n", gd->xkm_action.ba.nr);
      break;
    case XKEYMOUSE_EXEC: 
      xkm_verbose ("#  Exec  :                                # \n");
      xkm_verbose ("#   command line= %s        # \n", gd->xkm_action.ea.command_line);
      break;
    case XKEYMOUSE_MOTION:
      xkm_verbose ("#  Motion:                                # \n");
      xkm_verbose ("#   speed    = %.2d                         # \n", gd->xkm_action.ma.speed);
      xkm_verbose ("#   x direction= %.2d                       # \n", gd->xkm_action.ma.x_direction);
      xkm_verbose ("#   y direction= %.2d                       # \n", gd->xkm_action.ma.y_direction);
      break;
    case XKEYMOUSE_KEY:
      xkm_verbose ("#  Key   :                                # \n");
      xkm_verbose ("#   nr    = %.2d                  # \n", gd->xkm_action.ka.keycode);
      break;
    default:
      xkm_verbose ("adding nothing .... :(\n"); 
      break;
    }
  xkm_verbose ("#                                         # \n");
  xkm_verbose ("########################################### \n");

}


void 
init_type( grab_data *gd)
{
  if (gd->type==XKEYMOUSE_EXEC) 
    { 
      gd->autorepeat     = XKEYMOUSE_REPEAT_OFF;
      strcpy(gd->xkm_action.ea.command_line,""); 
    }
  else if (gd->type==XKEYMOUSE_MOTION ) 
    { 
      gd->autorepeat     = XKEYMOUSE_REPEAT_ON;
      gd->xkm_action.ma.speed=1;
      gd->xkm_action.ma.x_direction=XKEYMOUSE_NO_MOTION;
      gd->xkm_action.ma.y_direction=XKEYMOUSE_NO_MOTION;
    }
  else if (gd->type==XKEYMOUSE_BUTTON ) 
    { 
      gd->autorepeat     = XKEYMOUSE_REPEAT_OFF;
      gd->xkm_action.ba.double_click=XKEYMOUSE_SINGLE_CLICK;
    }
  else if (gd->type==XKEYMOUSE_KEY ) 
    { 
      gd->autorepeat     = XKEYMOUSE_REPEAT_ON;
    }
}



int
read_line (xkm_prog_data* xd,char *str, grab_data *gd, int fill)
{
  char *tmp=str;
  char *buf ;
  int tmp_member;
  int type;
  int nr;
  char test_[strlen(str)];

  /* get key */
  buf=strtok (tmp, ",");
  gd->code=get_key( xd, buf);

  /* get modifier */
  buf=strtok (NULL, ",");
  gd->modifier=get_modifier(xd, buf);

  /* get action type */
  buf=strtok (NULL, ",");
  gd->type=get_type (xd, buf);
  init_type(gd);

  /* get action arg  */
  buf=strtok (NULL, ",");
  add_argument (xd, gd, buf,1);

  /* get the rest of the args */
  while ( (buf=strtok (NULL, ","))!=NULL)
    {
      add_argument (xd, gd, buf,0);
    }
  
  verbose_gd(gd);

  return 1;
}


int
xkm_rem_blanks (char *array)
{
  int i=0;
  int j=0;
  int idx=-1;
  int size=strlen(array);

  for (i=0; i<size;i++)
    {
      idx++;
      if ( (array[idx]==' ') ||  (array[idx]=='\t') ||  (array[idx]=='\n') )
	{
	  for (j=idx;j<size-1;j++)
	    {
	      array[j]=array[j+1];
	    }
	  array[j]='\0';
	  idx--;
	}
    }
}



int
rc_lines(xkm_prog_data *xd, int fill)
{
  
  int rc_entries=0;
  int all_entries=0;
  int bytesread;
  int ret=0;
  char tmp[XKEYMOUSE_LINE_LENGTH] ;

   while ( fgets(tmp, XKEYMOUSE_LINE_LENGTH, xd->fd)!=NULL )
    {
      all_entries++;

      bytesread=strlen(tmp);
      if (bytesread==0)
	{
	  break;
	}
 
      if (is_comment_or_blank(tmp)) 
	{
	  ;
	}
      else 
	{
	  xd->grab_data_list=alloc_grab_data (xd->grab_data_list, xd);
	  ret = read_line (xd, tmp,xd->grab_data_list[rc_entries],fill);
	  if (ret == 0)
	    {
	      xkm_verbose ("Error in config file (line %d) \n", all_entries);
	    }
	  else 
	    {
	      rc_entries++;
	    } 
	}
    }
  return rc_entries;
}


int
grab_all_data (xkm_prog_data *xd, int grab)
{
  int i=0;
  xkm_verbose (" ---> grab_all_data\n");
  for (i=0; i<xd->gdl_size ; i++)
    {
      if (grab)
	{      
	  xkm_verbose ("--- grab_all_data i=%d (%d) ---------------------- code=%d\n",i,xd->gdl_size, xd->grab_data_list[i]->code);
	  verbose_gd ( xd->grab_data_list[i]);
	  xkm_verbose ("  --- grab_all_data setting autorepeat to %d\n",xd->grab_data_list[i]->autorepeat);
	  set_no_repeat ( xd->grab_data_list[i]->code , xd->grab_data_list[i]->autorepeat);
	  
	  xkm_verbose ("XGrabKey (disp=%d,window=%d,code=%d,modifier=%d)\n",
			xd->data_display,  
			xd->window,  
			xd->grab_data_list[i]->code,           
			xd->grab_data_list[i]->modifier);
	  
	  
	  XGrabKey (xd->data_display,  
		    xd->grab_data_list[i]->code,           
		    xd->grab_data_list[i]->modifier,
		    xd->window,       
		    False,        
		    GrabModeAsync,
		    GrabModeAsync );
	}
      else 
	{      
	  xkm_verbose ("XUngrabKey (disp=%d,code=%d,modifier=%d,window=%d\n",
			xd->data_display,  
			xd->grab_data_list[i]->code,           
			xd->grab_data_list[i]->modifier,
			xd->window);
	  
	  
	  XUngrabKey (xd->data_display,  
		      xd->grab_data_list[i]->code,           
		      xd->grab_data_list[i]->modifier,
		      xd->window);
	}
    }
  xkm_verbose (" <--- grab_all_data\n");
  return 1;
}



int 
count_rc_lines (xkm_prog_data *xd)
{
  return rc_lines (xd,0);
}


void
add_all_data(xkm_prog_data *xd)
{
  rc_lines (xd,1);
}




/*  
 *  Checks if the first argument equals to any of the other two 
 *   and returns 1 if that's the case, else it returns 0            
 *
 */
int 
xkm_check ( char *arg, char *long_arg , char *short_arg ) 
{
  return ( ! strcmp ( arg, long_arg) || ( ! strcmp ( arg, short_arg) )) ; 
}


/**
 *
 * Parse the command line
 *
 */
int 
xkm_parse_args ( xkm_prog_data *xd, int argc, char **argv)
{
  int i ;
  for (i = 1; i < argc; i++) 
    {
      if( xkm_check (argv[i], "--verbose", "-v"  ) )
	{
	  xd->verbose=1;
	  continue;
	}
      else if( xkm_check (argv[i], "--display", "-d"  ) )
	{
	  xd->display_name=argv[++i];
	  continue;
	}
      else if( xkm_check (argv[i], "--version", "-V"  ) )
	{
	  xkm_version();
	  exit(0);
	}
      else if( xkm_check (argv[i], "--help", "-h"  ) )
	{
	  xkm_usage();
	  exit(0);

	}
      else 
	{
	  fprintf (stderr, "Could not parse arg: \"%s\"\n...skipping the arg continuing\n", argv[i]);
	  i++;
	}
    }
  return 1;
}



int
xkm_init (xkm_prog_data *xd)
{
  xd->program=PROGRAM_NAME;
  xd->version=PROGRAM_VERSION;
  xd->gdl_size=0;
  return 1;
}




int 
xkm_open_rc_file (xkm_prog_data *xd)
{
  char   rc_buf[100];
  char  *rc_file;  

  /* Try environment variable first */
  rc_file=getenv(PROGRAM_RC_ENV);
  if (rc_file!=NULL)
    {
      xkm_verbose ("Trying \"%s\"\n", rc_file);
      xd->fd = fopen (rc_file,"r");
      if (xkm_data->fd==NULL)
	{
	  xkm_verbose ("Couldn't find %s\n", rc_file);
	}
      else
	{
	  xkm_verbose ("found %s (%d)\n", rc_file, xd->fd);
	  return 1;
	}
    }
  
  /* Perhaps we have a local rc file */ 
  rc_file=getenv("HOME");
  if (rc_file!=NULL)
    {
      strcpy (rc_buf, rc_file);
      strcat (rc_buf, "/");
      strcat(rc_buf, XKEYMOUSE_LOCAL_RC);
      xkm_verbose ("Trying \"%s\"\n", rc_buf);
      xd->fd = fopen (rc_buf,"r");
      if (xkm_data->fd==NULL)
	{
	  xkm_verbose ("Couldn't find %s\n", rc_buf);
	}
      else
	{
	  return 1;
	}
    }

  /* OK, let's try the system rc file */
  xkm_verbose ("Trying %s\n", XKEYMOUSE_SYSTEM_RC);
  xd->fd = fopen (XKEYMOUSE_SYSTEM_RC,"r");
  if (xkm_data->fd==NULL)
    {
      xkm_verbose ("Couldn't find %s\n", XKEYMOUSE_SYSTEM_RC);
    }
  else
    {
      return 1;
    }
  return 0;
}


void
set_no_repeat ( int code , int on_or_off) 
{
  XKeyboardControl xkbd_ctl ; 

  XKeyboardState xkbd_state ; 

  XGetKeyboardControl ( xkm_data->data_display, &xkbd_state);
  xkm_verbose ("   Global autorepeat            = %d \n", xkbd_state.global_auto_repeat);
  xkm_verbose ("          autorepeat            = %d \n", xkbd_state.auto_repeats[code]);
  xkm_verbose ("          AutoRepeatModeOn      = %d \n", AutoRepeatModeOn);
  xkm_verbose ("          AutoRepeatModeOff     = %d \n", AutoRepeatModeOff);
  xkm_verbose ("          AutoRepeatModeDefault = %d \n", AutoRepeatModeDefault);


  xkbd_ctl.key_click_percent = 0 ;
  xkbd_ctl.bell_percent      = 0 ;
  xkbd_ctl.bell_pitch        = 0 ;
  xkbd_ctl.bell_duration     = 0 ;
  xkbd_ctl.led               = 0 ;
  xkbd_ctl.led_mode          = 0 ;
  xkbd_ctl.key               = code;
  if ( on_or_off == XKEYMOUSE_REPEAT_OFF) 
    {
      xkm_verbose ("  AutoRepeatModeOff \n");
      xkbd_ctl.auto_repeat_mode  = AutoRepeatModeOff ; 
    }
  else
    { 
      xkm_verbose ("  AutoRepeatModeDefault\n");
      xkbd_ctl.auto_repeat_mode  =  AutoRepeatModeOn ; 
    }
  
  xkm_verbose ("   Handling autorepeat on keycode:%d\n", code);
  XChangeKeyboardControl ( xkm_data->data_display, 
			   KBKey | KBAutoRepeatMode,
			   &xkbd_ctl);
			  
  XGetKeyboardControl ( xkm_data->data_display, &xkbd_state);
  xkm_verbose ("   Global autorepeat            = %d \n", xkbd_state.global_auto_repeat);
  xkm_verbose ("          autorepeat            = %d \n", xkbd_state.auto_repeats[code]);
  xkm_verbose ("          AutoRepeatModeOn      = %d \n", AutoRepeatModeOn);
  xkm_verbose ("          AutoRepeatModeOff     = %d \n", AutoRepeatModeOff);
  xkm_verbose ("          AutoRepeatModeDefault = %d \n", AutoRepeatModeDefault);
}


/*
 * Print version information etc
 * 
 *
 */
void
xkm_version()
{
  fprintf (stderr, "Program : %s (version %s) \n",PACKAGE, VERSION);
  fprintf (stderr, "Copyright (C) 2001,2003 Henrik Sandklef (xkm@sandklef.com)\n");
  fprintf (stderr, "%s and comes with \n", PACKAGE);
  fprintf (stderr, "\tABSOLUTELY NO WARRANTY\n" );
  fprintf (stderr, "This is free software, and you are welcome to redistribute\n");
  fprintf (stderr, "it under conditions as specified in GPL\n");
  fprintf (stderr, "Feel free to make a contribution to the author or the FSF\n");
  fprintf (stderr, "\n");
  fprintf (stderr, "GPL: Gnu General Public License (http://www.gnu.org) \n");
  fprintf (stderr, "FSF: Free Software Foundation (http://www.fsf.org)\n");
  fprintf (stderr, "xkeymouse: http://xkeymouse.sourceforge.net\n");
}
