/*****************************************************************************
 * x11.c: misc x11 functions
 *****************************************************************************
 * $Id: x11.c,v 1.55 2004/09/18 16:46:55 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu and Maupatz
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * Original code:
 *
 * misc x11 functions:  pixmap handling (incl. MIT SHMEM), event
 *                      tracking for the TV widget.
 *
 *  (c) 1998 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 *****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "config.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Simple.h>
#include <X11/Shell.h>

#ifdef HAVE_MITSHM
# include <sys/ipc.h>
# include <sys/shm.h>
# include <X11/extensions/XShm.h>
#endif

#ifdef HAVE_LIBXXF86DGA
# include <X11/extensions/xf86dga.h>
#endif

#ifdef HAVE_XV
# include <X11/extensions/Xv.h>
# include <X11/extensions/Xvlib.h>
#endif

#include "grab.h"
#ifdef HAVE_V4L
#include "videodev.h"
#endif
#include "x11.h"
#include "channel.h"
#include "blackborder.h"
#include "deinterlace.h"
#include "memcpy.h"
#include "fourcc.h"
#include "strtab.h"
#include "divx.h"

#define WINDOW              XtWindow
#define DEL_TIMER(proc)     XtRemoveTimeOut(proc)
#define ADD_TIMER(proc)     XtAppAddTimeOut(app_context,200,proc,NULL)

#ifdef HAVE_XV
# ifdef HAVE_MITSHM
#  define XVPUTIMAGE(dpy,xvp,dr,gc,xv,a,b,c,d,e,f,w,h)                 \
    if (have_shmem)                                                  \
    XvShmPutImage(dpy,xvp,dr,gc,xv,a,b,c,d,e,f,w,h,False);          \
    else                                                             \
    XvPutImage(dpy,xvp,dr,gc,xv,a,b,c,d,e,f,w,h)
# else
#  define XVPUTIMAGE(dpy,xvp,dr,gc,xv,a,b,c,d,e,f,w,h)                 \
        XvPutImage(dpy,xvp,dr,gc,xv,a,b,c,d,e,f,w,h)
# endif
#endif

#ifdef HAVE_MITSHM
# define XPUTIMAGE(dpy,dr,gc,xi,a,b,c,d,w,h)                     \
    if (have_shmem)                                              \
    XShmPutImage(dpy,dr,gc,xi,a,b,c,d,w,h,True);                 \
    else                                                         \
    XPutImage   (dpy,dr,gc,xi,a,b,c,d,w,h)
#else
# define XPUTIMAGE(dpy,dr,gc,xi,a,b,c,d,w,h)                     \
    XPutImage   (dpy,dr,gc,xi,a,b,c,d,w,h)
#endif

//#ifdef USE_LIBXOSD <= xosd inside now
#include <xosd.h> 
//#endif <= xosd inside now

/* ------------------------------------------------------------------------ */
//extern int decode;
extern int cur_capture;

extern XtAppContext app_context;

Display * dpy;
static Screen * scr;
static int scr_num;
static Window root;
Colormap colormap;
XVisualInfo *vinfo;
static video_fmt xv_format = VIDEO_YUYV;

static int have_shmem = 0;
int have_xv = 0;
int have_video_xv = 0;
#ifdef WORDS_BIGENDIAN
#define CPUByteOrder MSBFirst
#else 
#define CPUByteOrder LSBFirst
#endif
static int x11_error = 0;
char xawxvmessage [1024];
char xawxvmesstmp [256];

#ifdef HAVE_XV
extern int disable_xv;
extern XvPortID cmdline_xv_img_port;
static XvImage *xvimage = NULL;
static Atom xvattribute;
static XvImageFormatValues *xvfo = NULL;

static int xvid = -1;
static XvPortID xv_img_port = -1;
static XvAdaptorInfo *xvai = NULL;
static XvAttribute *xvat = NULL;

static int xvver, xvrel, xvreq, xvev, xverr;
static int xvadaptors;
static Atom xv_img_colorkey = -1;
#  ifdef HAVE_VIDEO_XV
extern XvPortID cmdline_xv_video_port;
XvEncodingInfo *xv_video_encodings;
int xv_video_encodings_n;
static XvPortID xv_video_port = -1;
char* xv_video_name=NULL;
static Atom xv_video_encoding = -1;
static Atom xv_video_brightness = -1;
static Atom xv_video_contrast = -1;
static Atom xv_video_saturation = -1;
static Atom xv_video_hue = -1;
static Atom xv_video_volume = -1;
static Atom xv_video_mute = -1;
static Atom xv_video_freq = -1;
static Atom xv_video_colorkey = -1;
static Atom xv_video_double_buffer = -1;

static XvAttribute xvat_brightness,xvat_contrast,xvat_saturation,
  xvat_hue,xvat_volume,xvat_colorkey;

static GC xv_gc = NULL;
#  endif
int video_displayframe_xv (void);
#endif
int video_displayframe_x11 (void);

/* ------------------------------------------------------------------------ */
/* video overlay stuff                                                      */

video_fmt overlay_format;     // used by grab_overlay
video_fmt x11_native_format = 0;  // used by grab_one
video_fmt x11_pixmap_format = 0;  // used by grab_scr
int swidth, sheight;            /* screen  */
int hw_scaling = 1;             /* obsolete variable... */

extern Widget app_shell;
extern Widget tv;
static Widget video, video_parent;
 int wx, wy, wwidth, wheight, wmap;    /* window  */
 int ox, oy, owidth, oheight;    /* overlay */
int width_capture = 0, height_capture = 0;
int width_capture_max = 768, height_capture_max = 576;

static set_overlay overlay_cb;
static XtIntervalId overlay_refresh;
static int did_refresh, oc_count;
static int visibility = VisibilityFullyObscured;
static int conf = 1, move = 1, overlay_on = 0;
static struct OVERLAY_CLIP oc[32];

static XImage *ximage = NULL;
static void *ximage_shm = NULL;        /* used for MIT shmem */
static GC gc;

static char *events[] = {
  "0", "1",
  "KeyPress",
  "KeyRelease",
  "ButtonPress",
  "ButtonRelease",
  "MotionNotify",
  "EnterNotify",
  "LeaveNotify",
  "FocusIn",
  "FocusOut",
  "KeymapNotify",
  "Expose",
  "GraphicsExpose",
  "NoExpose",
  "VisibilityNotify",
  "CreateNotify",
  "DestroyNotify",
  "UnmapNotify",
  "MapNotify",
  "MapRequest",
  "ReparentNotify",
  "ConfigureNotify",
  "ConfigureRequest",
  "GravityNotify",
  "ResizeRequest",
  "CirculateNotify",
  "CirculateRequest",
  "PropertyNotify",
  "SelectionClear",
  "SelectionRequest",
  "SelectionNotify",
  "ColormapNotify",
  "ClientMessage",
  "MappingNotify"
};

extern char *cmd;
extern void do_command(char *);
extern void http_read_clients(void);

video_fmt x11c_to_xaw(int bpp, int r, int g, int b, int bo) {
  int i;
  struct {video_fmt f; int bpp; int r,g,b;
    video_fmt fswap;} conv[]= {
    {VIDEO_RGB03,   8,0x4,0x2,0x1,0},
    {VIDEO_RGB04b,  8,0x3,0x4,0x8,0},   /* used by XFree86 ! */
    {VIDEO_RGB08b,  8,0x7,0x38,0xc0,0}, /* used by XFree86 ! */
    {VIDEO_RGB08,   8,0xc0,0x38,0x7,0},
    {VIDEO_RGB15,  16,0x7c00,0x03e0,0x001f,VIDEO_RGB15X},
    {VIDEO_RGB16,  16,0xf800,0x07e0,0x001f,VIDEO_RGB16X},
    {VIDEO_RGB24,  24,0xff0000,0x00ff00,0x0000ff,VIDEO_RGB24X},
    {VIDEO_RGB32,  32,0x00ff0000,0x0000ff00,0x000000ff,VIDEO_RGB32X},
    {VIDEO_RGB24X, 24,0x0000ff,0x00ff00,0xff0000,VIDEO_RGB24},
    {VIDEO_RGB32X, 32,0x0000ff00,0x00ff0000,0xff000000,VIDEO_RGB32},
    {VIDEO_RGB32P, 32,0xff000000,0x00ff0000,0x0000ff00,VIDEO_RGB32PX},
    {VIDEO_RGB32PX,32,0x000000ff,0x0000ff00,0x00ff0000,VIDEO_RGB32P},
    {0,0,0,0}};
  for(i=0; conv[i].f!=0; i++)
    if(conv[i].bpp==bpp && r==conv[i].r && g==conv[i].g && b==conv[i].b) {
      if(!conv[i].fswap || bo==CPUByteOrder)
	return conv[i].f;
      else
	return conv[i].fswap;
    }
  return 0;
}
  

#ifdef HAVE_XV
/* The other fields are needed for RGB formats...
   see http://www.fourcc.org/rgb.php
 */
video_fmt fmt_xv_to_xaw(XvImageFormatValues *f) {
  switch(f->id) {
  case BI_RGB:
  case RGB:
  case BI_BITFIELD:
    return x11c_to_xaw(f->bits_per_pixel,f->red_mask,f->green_mask,
		       f->blue_mask, f->byte_order);
  case YUY2: return VIDEO_YUYV;
  case YV12: return VIDEO_YVU420;
  case UYVY: return VIDEO_UYVY;
  case I420:
  case IYUV: return VIDEO_YUV420;
  default:   return 0;
  }
}
#endif

static inline video_fmt x11vis_to_xaw(XVisualInfo *v, XPixmapFormatValues *pf,int npf) {
  int i,bpp=0;
  for (i = 0; i < npf; i++)
    if (pf[i].depth == vinfo->depth)
      bpp = pf[i].bits_per_pixel;
  switch(v->class) {
  case TrueColor:
  case StaticColor:
    return x11c_to_xaw(bpp, v->red_mask, v->green_mask, v->blue_mask,
		       ImageByteOrder(dpy));
  case StaticGray:
    switch(v->depth) {
    case 1:
      if(bpp==1 && BitmapBitOrder(dpy)==CPUByteOrder &&
	 (ImageByteOrder(dpy)==CPUByteOrder || BitmapUnit(dpy)==1) &&
	 BitmapPad(dpy)==32)
	return VIDEO_GRAY1;
      if(bpp==1 && BitmapBitOrder(dpy)!=CPUByteOrder &&
	 (ImageByteOrder(dpy)!=CPUByteOrder || BitmapUnit(dpy)==1) &&
	 BitmapPad(dpy)==32)
	return VIDEO_GRAY1X;
      return 0;
    case 4: if(bpp==8) return VIDEO_GRAY4; else return 0;
    case 8: return VIDEO_GRAY8;
    default: return 0;
    }
  case PseudoColor:
    switch(v->depth) {
    case 4: if(bpp==8) return VIDEO_RGB03; else return 0;
    case 8: return VIDEO_HI240;
    default: return 0;
    }
  default:
    return 0;
  }
}

  
/* ------------------------------------------------------------------------ */
void x11_init (void)
{
  XVisualInfo template;
  XPixmapFormatValues *pf;
  int found, i, n, visdef=0;
  video_fmt fmt;
  int pixmap_bytes=0;
#ifdef HAVE_XV
  int val, j;
  long int k;
  int first_xvport;
  int last_xvport;
#endif

  if(x11_pixmap_format) xv_format = x11_pixmap_format;
  dpy = XtDisplay(app_shell);
  scr = XtScreen(app_shell);
  

  for(scr_num = 0; scr_num < ScreenCount(dpy); scr_num++)
    if(ScreenOfDisplay(dpy,scr_num)==scr) break;
  if(scr_num ==  ScreenCount(dpy)) scr_num = XDefaultScreen(dpy);
  swidth = WidthOfScreen(scr);
  sheight = HeightOfScreen(scr);
  root = RootWindowOfScreen(scr);

#ifdef HAVE_MITSHM
  if (XShmQueryExtension (dpy))
    {
      have_shmem = 1;
      if (debug)
        printf("MIT Shm extension available\n\n");
    }
#endif

#ifdef HAVE_XV
  if (!disable_xv)
    {
      strcpy(xawxvmessage,"XV initial informations:\n");
      if (Success !=
          XvQueryExtension (dpy, &xvver, &xvrel, &xvreq, &xvev, &xverr))
        {
          if (debug)
	  {
	    sprintf(xawxvmesstmp, "Server does'nt support Xvideo!\n");
            fprintf(stderr, xawxvmesstmp);
            strcat(xawxvmessage,xawxvmesstmp);
	  }
        }
      else if (Success !=
               XvQueryAdaptors (dpy, root, &xvadaptors,
                                &xvai))
        {
          if (debug)
            fprintf (stderr, "XvQueryAdaptors failed!\n");
        }
      else
        {
           sprintf(xawxvmesstmp, "Xv version = %d.%d\n\n", xvver, xvrel);
           strcat(xawxvmessage,xawxvmesstmp);

	    if (debug)
	    {
	      fprintf(stderr, "Xv version = %d.%d\n", xvver, xvrel);
	      fprintf(stderr, "Xv adaptors available = %d\n\n", xvadaptors);
	    }

          for (i = 0; i < xvadaptors; i++)
            {
              for (j = 0; j < xvai[i].num_formats; j++)
                {
                  if (debug)
                    fprintf (stderr, "Format %d = depth:%d id:%x\n", j,
                             xvai[i].formats[j].depth,
                             (int) xvai[i].formats[j].visual_id);
                }

              /* selects FIRST Xv image port available which supports xv_format */
              if ((xvai[i].type & XvInputMask) && (xvai[i].type & XvImageMask)) {
		first_xvport = xvai[i].base_id;
		last_xvport = xvai[i].base_id+xvai[i].num_ports-1;
		for (k = first_xvport; k <= last_xvport; k++) {
		  if(cmdline_xv_img_port && k!=cmdline_xv_img_port) continue;
		  xvfo = XvListImageFormats (dpy, k, &n);
		  for (j = 0; j < n; j++)
		    if (fmt_xv_to_xaw(&xvfo[j])==xv_format) {xvid=xvfo[j].id; break;}
		  if(xvid!=-1) {
		    /*  Xv port is exclusive for xdtv. */
		    if (XvGrabPort (dpy, k, CurrentTime) != Success) {
		      fprintf (stderr, "XvGrabPort() failed, on XvImage port %ld, may be used "
			       "by another application.\n", k);
		      xvid=-1;
		    } else {
		      xv_img_port = k;
		      fprintf (stderr, "Selected XvImage adaptor with %s support: "
			       "%s on port %ld (grabdisplay)\n",
			       int_to_str(xv_format, video_fmt_names),xvai[i].name,
			       xv_img_port);
		      // Gotten a valid, unsued, Xv port
		      //XvUngrabPort (dpy, xv_img_port, CurrentTime);
		      break;
		    }
		  } else if(debug)
		    fprintf (stderr, "XvImage port %ld does not support %s format\n", k,
			     int_to_str(xv_format, video_fmt_names));
		}
		XFree (xvfo);
	      }
#ifdef HAVE_VIDEO_XV
              /* selects FIRST Xv video port available */
              if ((xvai[i].type & XvInputMask) &&
                  (xvai[i].type & XvVideoMask) && (xv_video_port == -1))
                {
                  first_xvport = xvai[i].base_id;
                  last_xvport = xvai[i].base_id+xvai[i].num_ports-1;
                  for (k = first_xvport; k <= last_xvport; k++)
                    {
		      if(cmdline_xv_video_port &&
			 k != cmdline_xv_video_port)
			continue;
                      /*  Xv port is exclusive for xdtv. */
                      if (XvGrabPort (dpy, k, CurrentTime) == Success)
                        {
                          xv_video_port = k;
                          fprintf (stderr, "  Selected XvVideo: %s on port %ld (overlay)\n\n", xvai[i].name, xv_video_port);
			  xv_video_name = strdup(xvai[i].name);
                          XvUngrabPort (dpy, k, CurrentTime);
                          break;
                        }
                      else
                        fprintf (stderr, "XvGrabPort() failed, XvVideo port may be used by another application.\n");
                    }
                }
#endif // HAVE_VIDEO_XV

            }
        }


#ifdef HAVE_VIDEO_XV
      if (-1 == xv_video_port)
        {
          fprintf (stderr, "No XvVideo port available.\n");
          have_video_xv = 0;
        }
      else
        {
          have_video_xv = 1;
          xvat = XvQueryPortAttributes (dpy, xv_video_port, &n);
          if (xvat != NULL)
            {
              fprintf(stderr, "XV Information:\n");
              for (i = 0; i < n; i++)
                {
		  int cur_val;
		  Atom val_atom;
		  val_atom = XInternAtom(dpy, xvat[i].name, False);

		  fprintf(stderr, "    %s max value = %d\n", xvat[i].name, xvat[i].max_value);
		  fprintf(stderr, "    %s min value = %d\n", xvat[i].name, xvat[i].min_value);
		  if(xvat[i].flags & XvGettable)
		  {
		    if(XvGetPortAttribute(dpy, xv_video_port, val_atom, &cur_val) != Success)
                      fprintf(stderr, "Couldn't get attribute\n");
		    else
		      fprintf(stderr, "    %s cur value = %d\n\n", xvat[i].name, cur_val);
		  } else
		      fprintf(stdout, "    %s cur value not gettable\n\n", xvat[i].name);

                  if (xvat[i].flags & XvSettable)
                    {
                      if (strcmp ("XV_ENCODING", xvat[i].name) == 0)
                        {
                          xv_video_encoding = xvattribute = XInternAtom (dpy, xvat[i].name, True);
                        }

                      if (strcmp ("XV_COLORKEY", xvat[i].name) == 0)
                        {
                          xv_video_colorkey = xvattribute = XInternAtom (dpy, xvat[i].name, True);
			  xvat_colorkey = xvat[i];
                        }

                      if (strcmp ("XV_BRIGHTNESS", xvat[i].name) == 0)
                        {
                          xv_video_brightness = xvattribute = XInternAtom (dpy, xvat[i].name, True);
			  xvat_brightness = xvat[i];
                        }

                      if (strcmp ("XV_CONTRAST", xvat[i].name) == 0)
                        {
                          xv_video_contrast = xvattribute = XInternAtom (dpy, xvat[i].name, True);
			  xvat_contrast = xvat[i];
                        }

                      if (strcmp ("XV_SATURATION", xvat[i].name) == 0)
                        {
                          xv_video_saturation = xvattribute = XInternAtom (dpy, xvat[i].name, True);
			  xvat_saturation = xvat[i];
                        }

                      if (strcmp ("XV_HUE", xvat[i].name) == 0)
                        {
                          xv_video_hue = xvattribute = XInternAtom (dpy, xvat[i].name, True);
			  xvat_hue = xvat[i];
                        }

                      if (strcmp ("XV_FREQ", xvat[i].name) == 0)
                        {
                          xv_video_freq = xvattribute = XInternAtom (dpy, xvat[i].name, True);
                        }

                      if (strcmp ("XV_VOLUME", xvat[i].name) == 0)
                        {
                          xv_video_volume = xvattribute = XInternAtom (dpy, xvat[i].name, True);
			  xvat_volume = xvat[i];
                        }

                      if (strcmp ("XV_MUTE", xvat[i].name) == 0)
                        {
                          xv_video_mute = xvattribute = XInternAtom (dpy, xvat[i].name, True);
                        }

                      if (strcmp ("XV_DOUBLE_BUFFER", xvat[i].name) == 0)
                        {
                          xv_video_double_buffer = xvattribute = XInternAtom (dpy, xvat[i].name, True);
                        }

                      if (strcmp ("XV_SET_DEFAULTS", xvat[i].name) == 0)
                        {
                          xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          //XvSetPortAttribute(dpy, xv_video_port, xvattribute, True);
                        }

                    }

		}
	      XFree(xvat); xvat=NULL;
            }
	  XvQueryEncodings(dpy, xv_video_port, &xv_video_encodings_n,
			   &xv_video_encodings);
	}
#endif // HAVE_VIDEO_XV

      if (-1 == xv_img_port)
        {
          fprintf (stderr, "No XvImage port available with %s format.\n",
		   int_to_str(xv_format, video_fmt_names));
          have_xv = 0;
        }
      else
        {
	  int find_xvsetdefaults=0;
          have_xv = 1;

	  if (debug)
            fprintf (stderr, "XvImage port %ld supports %s format\n\n", xv_img_port, int_to_str(xv_format, video_fmt_names));
          xvat = XvQueryPortAttributes (dpy, xv_img_port, &n);
          if (xvat != NULL)
            {
              /* Set by default */
              for (i = 0; i < n; i++)
	      {
                int cur_val;
                Atom val_atom;
                val_atom = XInternAtom(dpy, xvat[i].name, False);

                if (strcmp ("XV_COLORKEY", xvat[i].name) == 0)
		{
                  sprintf(xawxvmesstmp, "%10s: \nMaximum value = %10d\nMinimum value = %10d\nInitial value = ",
		  		xvat[i].name, xvat[i].max_value, xvat[i].min_value);
                  strcat(xawxvmessage,xawxvmesstmp);
                  if(xvat[i].flags & XvGettable)
		  {
                    if(XvGetPortAttribute(dpy, xv_img_port, val_atom, &cur_val) != Success)
                      {
                        sprintf(xawxvmesstmp, "Not Gettable\n");
                        strcat(xawxvmessage,xawxvmesstmp);
                      }
                      else
                      {
                        sprintf(xawxvmesstmp, "%10d\n", cur_val);
                        strcat(xawxvmessage,xawxvmesstmp);
                      }
                  } else
                      {
                        sprintf(xawxvmesstmp, "Not Gettable\n");
                        strcat(xawxvmessage,xawxvmesstmp);
                      }
                }

	       if (strcmp ("XV_SET_DEFAULTS", xvat[i].name) == 0)
                        {
                          xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          val = xvat[i].max_value;
                          XvSetPortAttribute (dpy, xv_img_port, xvattribute, val);
                          if (debug)
                            printf("Set XV to defaults values");
                        }
              }
              /* Set manually */
	      if (!find_xvsetdefaults)
              for (i = 0; i < n; i++)
                {

                  if (xvat[i].flags & XvSettable)
                    {
                      if (strcmp ("XV_COLORKEY", xvat[i].name) == 0)
                        {
                          xv_img_colorkey = xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          val = 0; /*Assume the original value is near zero*/
                          XvSetPortAttribute (dpy, xv_img_port, xvattribute, val);
                          if (debug)
                            printf("XV_COLORKEY = %d\n", val);

                        }

                      if (strcmp ("XV_BRIGHTNESS", xvat[i].name) == 0)
                        {
                          xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          val = (xvat[i].min_value + xvat[i].max_value) / 2;
                          XvSetPortAttribute (dpy, xv_img_port, xvattribute, val);
                          if (debug)
                            printf("XV_BRIGHTNESS = %d\n", val);
                        }

                      if (strcmp ("XV_CONTRAST", xvat[i].name) == 0)
                        {
                          xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          val = (xvat[i].min_value + xvat[i].max_value) / 2;
                          XvSetPortAttribute (dpy, xv_img_port, xvattribute, val);
                          if (debug)
                            printf("XV_CONTRAST = %d\n", val);
                        }

                      if (strcmp ("XV_SATURATION", xvat[i].name) == 0)
                        {
                          xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          val = (xvat[i].min_value + xvat[i].max_value) / 2;
                          XvSetPortAttribute (dpy, xv_img_port, xvattribute, val);
                          if (debug)
                            printf("XV_SATURATION = %d\n", val);
                        }

                      if (strcmp ("XV_HUE", xvat[i].name) == 0)
                        {
                          xvattribute = XInternAtom (dpy, xvat[i].name, True);
                          /* ATI -1000 to 1000; Nvidia 0..360 in degrees?*/
                          /*Fixme: assume 0 is good ?*/
                          val = 0;
                          XvSetPortAttribute (dpy, xv_img_port, xvattribute, val);
                          if (debug)
                            printf("XV_HUE = %d\n", val);
                        }
                    }
                }
	      XFree(xvat); xvat=NULL;
            }
        }

      if (xvadaptors > 0)
        XvFreeAdaptorInfo (xvai);
    }
  else
#endif
    {
      have_xv = 0;
      have_video_xv = 0;
      strcat(xawxvmessage,"Xv not used. You can't have any info about it\n");      
    }

  /* Now select correct video renderer */
#ifdef HAVE_XV
  if (have_xv)
    {
      video_displayframe = video_displayframe_xv;
    }
  else
#endif
    {
      video_displayframe = video_displayframe_x11;
    }
  
  /* Ask for visual type */
  template.screen = scr_num;
  template.visualid = XVisualIDFromVisual (DefaultVisualOfScreen (scr));
  vinfo = XGetVisualInfo (dpy, VisualIDMask | VisualScreenMask, &template,
			  &found);
  colormap = DefaultColormap (dpy, scr_num);
  pf = XListPixmapFormats (dpy, &n);
  fmt = x11vis_to_xaw(vinfo,pf, n);
  if(x11_native_format && fmt!=x11_native_format) fmt=0;
  if(fmt)
    visdef = 1;
  else {
    XVisualInfo *info;
    int found, vi = -1;
    XFree(vinfo);
    info = XGetVisualInfo(dpy, VisualScreenMask,&template,&found);
    vi = -1;
    /* TODO: consider the capabilities of the TV-card to chooss */
    for (i = 0; vi == -1 && i < found; i++) {
      fmt = x11vis_to_xaw(&info[i],pf, n);
      if(x11_native_format && fmt!=x11_native_format) fmt=0;
      if (fmt)	{vi = i; break;}
    }
    if(vi==-1) {
      fprintf (stderr, "no good visual found\n");
      exit(1);
    }
    template.visualid = info[vi].visualid;
    vinfo = XGetVisualInfo (dpy, VisualIDMask | VisualScreenMask, &template,
			    &found);
    XFree(info);
  }
  if(vinfo->class==PseudoColor || !visdef) {
    colormap = XCreateColormap(dpy,root,vinfo->visual,AllocNone);
    if(vinfo->class==PseudoColor) {
      if(fmt==VIDEO_HI240) {
	unsigned long i,pixels[256];
	XColor colors[225];
	/* does not work if I create directly the colormap with AllocAll */
	XAllocColorCells(dpy,colormap,1,NULL,0,pixels,256);
	for(i=0;i<16;i++) pixels[i]=i;
	for(i=241;i<256;i++) pixels[16+i-241]=i;
	XFreeColors(dpy,colormap,pixels,31,0);
	for(i=0;i<=225;i++) {
	  XColor *color = &colors[i];
	  color->red= ((i/5)%5)*65535.0/4+0.5;
	  color->blue = (i%5)*65535.0/4+0.5;
	  color->green  = (i/25)*65535.0/8+0.5;
	  color->flags= DoRed|DoBlue|DoGreen;
	  color->pixel= i+16;
	}
	XStoreColors(dpy,colormap,colors,225);
      } else if(fmt==VIDEO_RGB03) {
	unsigned long i,pixels[16];
	XColor colors[8];
	XAllocColorCells(dpy,colormap,1,NULL,0,pixels,16);
	for(i=0;i<8;i++) pixels[i]=8+i; XFreeColors(dpy,colormap,pixels,8,0);
	for(i=0;i<=7;i++) {
	  XColor *color = &colors[i];
	  color->blue = ((i&1)?65535:0);
	  color->green  = ((i&2)?65535:0);
	  color->red = ((i&4)?65535:0);
	  color->flags= DoRed|DoBlue|DoGreen;
	  color->pixel= i;
	}
	XStoreColors(dpy,colormap,colors,8);
      }
    }
    app_shell = XtVaAppCreateShell("xdtv","XdTV",
				   applicationShellWidgetClass, dpy,
				   XtNvisual,vinfo->visual,
				   XtNcolormap,colormap,
				   XtNdepth, vinfo->depth,
				   NULL);
  }
  x11_native_format = fmt;
  for (i = 0; i < n; i++)
    {
      if (pf[i].depth == vinfo->depth)
        pixmap_bytes = pf[i].bits_per_pixel / 8;
    }
  XFree(pf);

  if (debug)
    {
      fprintf (stderr, "x11: color depth: "
               "%d bits - pixmap: %d bytes\n",
               vinfo->depth, pixmap_bytes);
      if (vinfo->class == TrueColor)
        {
          fprintf (stderr, "x11: color masks: "
                   "red=0x%08lx green=0x%08lx blue=0x%08lx\n",
                   vinfo->red_mask, vinfo->green_mask, vinfo->blue_mask);
        }

    }
#ifdef HAVE_XV
  if ((!disable_xv) && (have_xv))
    {
      if (debug)
        fprintf(stderr, "Forcing x11_pixmap_format to %s\n", 
		int_to_str(xv_format, video_fmt_names));
      x11_pixmap_format = xv_format;
    }
#endif
  else
    x11_pixmap_format = x11_native_format;
}

int
x11_error_dev_null (Display * dpy, XErrorEvent * event)
{
  x11_error++;
  if (debug > 1)
    fprintf (stderr, " x11-error");
  return 0;
}

void
set_colorkey(Display * dpy, int val)
{
#ifdef HAVE_XV
  if (have_xv && (xv_img_colorkey != -1))
    {
      XvSetPortAttribute (dpy, xv_img_port, xv_img_colorkey, val);
      XvGetPortAttribute (dpy, xv_img_port, xv_img_colorkey, &val);
      if (debug)
        printf("x11: set_colorkey: XV_COLORKEY = %d\n", val);
    }
#endif
}

int
get_colorkey(Display * dpy)
{
  int val;

#ifdef HAVE_XV
  if (have_xv && (xv_img_colorkey != -1))
    {
      XvGetPortAttribute (dpy, xv_img_port, xv_img_colorkey, &val);
      if (debug)
        printf("x11; get_colorkey: XV_COLORKEY = %d\n", val);
    }
  else
#endif
    val = -1;

    return val;
}
/* ------------------------------------------------------------------------ */
/*     xvimage,ximage handling for grab & display                           */


/*******************************Xvimage**********************************/
#ifdef HAVE_XV
XvImage *
x11_create_xvimage (Display * dpy, int width, int height, void **shm)
{
  XvImage *xvimage = NULL;
  unsigned char *xvimage_data;
#ifdef HAVE_MITSHM
  XShmSegmentInfo *shminfo = NULL;
  void *old_handler = NULL;
#endif

  if (debug)
    fprintf (stderr, "video: x11_create_xvimage\n");
#ifdef HAVE_MITSHM
  if (have_shmem)
    {
      x11_error = 0;
      old_handler = XSetErrorHandler (x11_error_dev_null);
      (*shm) = shminfo = malloc (sizeof (XShmSegmentInfo));
      memset (shminfo, 0, sizeof (XShmSegmentInfo));
      xvimage =
        XvShmCreateImage (dpy, xv_img_port, xvid, NULL, width, height, shminfo);
      shminfo->shmid =
        shmget (IPC_PRIVATE, xvimage->data_size, IPC_CREAT | 0777);
      if (-1 == shminfo->shmid)
        {
          perror ("shmget");
          goto oom;
        }
      shminfo->shmaddr = (char *) shmat (shminfo->shmid, 0, 0);
      if ((void *) -1 == shminfo->shmaddr)
        {
          perror ("shmat");
          goto oom;
        }
      xvimage->data = shminfo->shmaddr;
      shminfo->readOnly = False;

      XShmAttach (dpy, shminfo);
      XSync (dpy, False);
      shmctl (shminfo->shmid, IPC_RMID, 0);

      if (x11_error)
        {
          have_shmem = 0;
          shmdt (shminfo->shmaddr);
          free (shminfo);
          shminfo = *shm = NULL;
          XFree (xvimage);
          xvimage = NULL;
        }

    }
  else
    {
      have_shmem = 0;
      free (shminfo);
      shminfo = *shm = NULL;
    }
  XSetErrorHandler (old_handler);
#endif

  if (xvimage == NULL)
    {
      (*shm) = NULL;
      if (NULL == (xvimage_data = malloc (width * height * 2)))
        {
          fprintf (stderr, "out of memory for create xvimage\n");
          goto oom;
        }
      xvimage =
        XvCreateImage (dpy, xv_img_port, xvid, xvimage_data, width, height);
    }
  memset (xvimage->data, 16, xvimage->data_size);

  return xvimage;

oom:;
#ifdef HAVE_MITSHM
  if (shminfo)
    {
      if (shminfo->shmid && shminfo->shmid != -1)
        shmctl (shminfo->shmid, IPC_RMID, 0);
      free (shminfo);
    }
#endif
  if (xvimage)
    XFree (xvimage);
  return NULL;
}

void
x11_destroy_xvimage (Display * dpy, XvImage * xvimage, void *shm)
{

#ifdef HAVE_MITSHM
  XShmSegmentInfo *shminfo = shm;

  if (shminfo)
    {
      XShmDetach (dpy, shminfo);
      shmdt (shminfo->shmaddr);
      shmctl (shminfo->shmid, IPC_RMID, 0);
      free (shminfo);
      XFree (xvimage);
    }
  else
    {
#endif
      if (xvimage->data != NULL)
        XFree (xvimage->data);
      XFree (xvimage);
#ifdef HAVE_MITSHM
    }
#endif
  if (debug)
    fprintf (stderr, "video: x11_destroy_xvimage\n");
}

#endif



/********************************XLIB *************************************/

XImage *
x11_create_ximage (Display * dpy, int width, int height, void **shm)
{
  XImage *ximage = NULL;
  unsigned char *ximage_data;
#ifdef HAVE_MITSHM
  XShmSegmentInfo *shminfo = NULL;
  void *old_handler;
#endif

  if (debug)
    fprintf (stderr, "video: x11_create_ximage\n");
#ifdef HAVE_MITSHM
  if (have_shmem)
    {
      x11_error = 0;
      old_handler = XSetErrorHandler (x11_error_dev_null);
      (*shm) = shminfo = malloc (sizeof (XShmSegmentInfo));
      memset (shminfo, 0, sizeof (XShmSegmentInfo));
      ximage = XShmCreateImage (dpy, vinfo->visual, vinfo->depth,
                                ZPixmap, NULL, shminfo, width, height);
      if (ximage)
        {
          shminfo->shmid = shmget (IPC_PRIVATE,
                                   ximage->bytes_per_line * ximage->height,
                                   IPC_CREAT | 0777);
          if (-1 == shminfo->shmid)
            {
              perror ("shmget");
              goto oom;
            }
          shminfo->shmaddr = (char *) shmat (shminfo->shmid, 0, 0);
          if ((void *) -1 == shminfo->shmaddr)
            {
              perror ("shmat");
              goto oom;
            }
          ximage->data = shminfo->shmaddr;
          shminfo->readOnly = False;

          XShmAttach (dpy, shminfo);
          XSync (dpy, False);
          shmctl (shminfo->shmid, IPC_RMID, 0);
          if (x11_error)
            {
              have_shmem = 0;
              shmdt (shminfo->shmaddr);
              free (shminfo);
              shminfo = *shm = NULL;
              XDestroyImage (ximage);
              ximage = NULL;
            }
        }
      else
        {
          have_shmem = 0;
          free (shminfo);
          shminfo = *shm = NULL;
        }
      XSetErrorHandler (old_handler);
    }
#endif

  if (ximage == NULL)
    {
      (*shm) = NULL;
      if (NULL == (ximage_data = malloc(size_img(x11_native_format, width, height))))
        {
          fprintf (stderr, "out of memory\n");
          goto oom;
        }
      ximage = XCreateImage (dpy, vinfo->visual, vinfo->depth,
                             ZPixmap, 0, ximage_data, width, height, 8, 0);
    }
  memset (ximage->data, 0, ximage->bytes_per_line * ximage->height);
  return ximage;

oom:
#ifdef HAVE_MITSHM
  if (shminfo)
    {
      if (shminfo->shmid && shminfo->shmid != -1)
        shmctl (shminfo->shmid, IPC_RMID, 0);
      free (shminfo);
    }
#endif
  if (ximage)
    XDestroyImage (ximage);
  return NULL;
}

void
x11_destroy_ximage (Display * dpy, XImage * ximage, void *shm)
{
#ifdef HAVE_MITSHM
  XShmSegmentInfo *shminfo = shm;

  if (shminfo)
    {
      XShmDetach (dpy, shminfo);
      XDestroyImage (ximage);
      shmdt (shminfo->shmaddr);
      free (shminfo);
    }
  else
#endif
    XDestroyImage (ximage);

  if (debug)
    fprintf (stderr, "video: x11_destroy_ximage\n");
}

Pixmap
x11_create_pixmap (Widget w, unsigned char *byte_data,
                   int width, int height, char *label)
{
  static XFontStruct *font;
  static XColor color, dummy;

  Pixmap pixmap;
  XImage *pix_ximage;
  XGCValues values;
  GC gc;
  void *shm;

  if (!font)
    {
      font = XLoadQueryFont (dpy, "fixed");
      XAllocNamedColor (dpy, colormap,
                        "yellow", &color, &dummy);
    }

  pixmap = XCreatePixmap (dpy, root, width, height, vinfo->depth);

  values.font = font->fid;
  values.foreground = color.pixel;
  gc = XCreateGC (dpy, pixmap, GCFont | GCForeground, &values);

  if ((pix_ximage = x11_create_ximage (dpy, width, height, &shm)) == NULL)
    {
      fprintf(stderr, "x11_create_pixmap: x11_create_ximage failed\n");
      XFreePixmap (dpy, pixmap);
      XFreeGC (dpy, gc);
      return 0;
    }
  
  memcpy (pix_ximage->data, byte_data, size_img (x11_native_format, width, height));

  XPUTIMAGE (dpy, pixmap, gc, pix_ximage, 0, 0, 0, 0, width, height);

  if (label)
    XDrawString (dpy, pixmap, gc, 5, height - 5, label, strlen (label));

  x11_destroy_ximage (dpy, pix_ximage, shm);

  XFreeGC (dpy, gc);
  return pixmap;
}


/* ------------------------------------------------------------------------ */

static void
add_clip (int x1, int y1, int x2, int y2)
{
  if (oc[oc_count].x1 != x1 || oc[oc_count].y1 != y1 ||
      oc[oc_count].x2 != x2 || oc[oc_count].y2 != y2)
    {
      conf = 1;
    }
  oc[oc_count].x1 = x1;
  oc[oc_count].y1 = y1;
  oc[oc_count].x2 = x2;
  oc[oc_count].y2 = y2;
  oc_count++;
}

static void
get_clips ()
{
  int x1, y1, x2, y2, lastcount;
  XWindowAttributes wts;
  Window me, rroot, parent, *children;
  uint nchildren, i;
  void *old_handler = XSetErrorHandler (x11_error_dev_null);

  if (debug > 1)
    fprintf (stderr, " getclips");
  lastcount = oc_count;
  oc_count = 0;

  if (ox < 0)
    add_clip (0, 0, (uint) (-ox), oheight);
  if (oy < 0)
    add_clip (0, 0, owidth, (uint) (-oy));
  if ((ox + owidth) > swidth)
    add_clip (swidth - ox, 0, owidth, oheight);
  if ((oy + oheight) > sheight)
    add_clip (0, sheight - oy, owidth, oheight);

  me = WINDOW (video);
  for (;;)
    {
      XQueryTree (dpy, me, &rroot, &parent, &children, &nchildren);
      XFree ((char *) children);
      /* fprintf(stderr,"me=0x%x, parent=0x%x\n",me,parent); */
      if (root == parent)
        break;
      me = parent;
    }
  XQueryTree (dpy, root, &rroot, &parent, &children, &nchildren);

  for (i = 0; i < nchildren; i++)
    if (children[i] == me)
      break;

  for (i++; i < nchildren; i++)
    {
      XGetWindowAttributes (dpy, children[i], &wts);
      if (!(wts.map_state & IsViewable))
        continue;

      x1 = wts.x - ox;
      y1 = wts.y - oy;
      x2 = x1 + wts.width + 2 * wts.border_width;
      y2 = y1 + wts.height + 2 * wts.border_width;
      if ((x2 < 0) || (x1 > (int) owidth) || (y2 < 0) || (y1 > (int) oheight))
        continue;

      if (x1 < 0)
        x1 = 0;
      if (y1 < 0)
        y1 = 0;
      if (x2 > (int) owidth)
        x2 = owidth;
      if (y2 > (int) oheight)
        y2 = oheight;
      add_clip (x1, y1, x2, y2);
    }
  XFree ((char *) children);

  if (lastcount != oc_count)
    conf = 1;
  XSetErrorHandler (old_handler);
}

static void
refresh_timer (XtPointer clientData, XtIntervalId * id)
{
  Window win = root;
  XSetWindowAttributes xswa;
  unsigned long mask;
  Window tmp;

  if (!move && wmap && visibility == VisibilityUnobscured)
    {
      if (debug > 1)
        fprintf (stderr, "video: refresh skipped\n");
      return;
    }

  if (debug > 1)
    fprintf (stderr, "video: refresh\n");
  overlay_refresh = 0;
  if (wmap && visibility != VisibilityFullyObscured)
    did_refresh = 1;

  xswa.override_redirect = True;
  xswa.backing_store = NotUseful;
  xswa.save_under = False;
  mask = (CWSaveUnder | CWBackingStore | CWOverrideRedirect);
  tmp = XCreateWindow (dpy, win, 0, 0, swidth, sheight, 0,
                       CopyFromParent, InputOutput, CopyFromParent,
                       mask, &xswa);
  XMapWindow (dpy, tmp);
  XUnmapWindow (dpy, tmp);
  XDestroyWindow (dpy, tmp);
  move = 0;
}

static void
configure_overlay ()
{
#ifdef HAVE_XV
  if (have_video_xv)
    return;
#endif
  if (!overlay_cb)
    return;

  if (debug > 1)
    fprintf (stderr, "video: configure");
  if (wmap && visibility != VisibilityFullyObscured)
    {
      if (visibility == VisibilityPartiallyObscured)
        get_clips ();
      else
        oc_count = 0;

      if (debug > 1)
        fprintf (stderr, " %s\n", conf ? "yes" : "no");
      if (conf)
        {
          overlay_on = 1;
          overlay_cb (ox, oy, owidth, oheight,
                      overlay_format, oc, oc_count);
          if (overlay_refresh)
            DEL_TIMER (overlay_refresh);
          overlay_refresh = ADD_TIMER (refresh_timer);
          conf = 0;
        }
    }
  else
    {
      if (debug > 1)
        fprintf (stderr, " off\n");
      if (conf && overlay_on)
        {
          overlay_on = 0;
          overlay_cb (0, 0, 0, 0, 0, NULL, 0);
          if (overlay_refresh)
            DEL_TIMER (overlay_refresh);
          overlay_refresh = ADD_TIMER (refresh_timer);
          conf = 0;
        }
    }
}


void
video_set_size(int w, int h)
{
  XtVaSetValues (video_parent, XtNwidth, w, XtNheight, h, NULL);
  video_new_size();
}

void
video_new_size (void)
{
  Dimension x, y, w, h;

  XtVaGetValues (video_parent, XtNx, &x, XtNy, &y,
                 XtNwidth, &w, XtNheight, &h, NULL);
		    
  wx = x;
  if (wx > 32768)
    wx -= 65536;

  wy = y;
  if (wy > 32768)
    wy -= 65536;

  wwidth = w;
  if (wwidth > 32768)
    wwidth -= 65536;

  wheight = h;
  if (wheight > 32768)
    wheight -= 65536;

  if (debug > 1)
    fprintf (stderr, "video: shell: size %dx%d+%d+%d\n",
             wwidth, wheight, wx, wy);

  if (wwidth > cur_maxwidth)
    owidth = cur_maxwidth;
  else
    owidth = wwidth;

  if(cur_capture == CAPTURE_GRABDISPLAY)
    if(owidth>width_capture_max) owidth = width_capture_max;
  
  if (wheight > cur_maxheight)
    oheight = cur_maxheight;
  else
    oheight = wheight;

  if(cur_capture == CAPTURE_GRABDISPLAY)
    if(oheight>height_capture_max) oheight = height_capture_max;

  owidth &= ~(WPAD-1);
  oheight &= ~(HPAD-1);
  ox = wx + (wwidth - owidth) / 2;
  oy = wy + (wheight - oheight) / 2;
  ox &= ~(XPAD-1);
  
  if (ox < wx)
    ox += XPAD;

  if (ox + owidth > wx + wwidth)
    owidth -= WPAD;

  if(cur_capture == CAPTURE_GRABDISPLAY) {
    int width_capture2=width_capture, height_capture2=height_capture;
    if(width_capture2==-1) width_capture2=cur_maxwidth;
    if(height_capture2==-1) height_capture2=cur_maxheight;
    if(width_capture2 > 0) {
      owidth = width_capture2;
      if(owidth>cur_maxwidth) owidth = cur_maxwidth;
      ox = wx + (wwidth - owidth) / 2;
      ox = (ox+XPAD/2) & (~(XPAD-1));
     }
    if(height_capture2 > 0) {
      oheight = height_capture2;
      if(oheight>cur_maxheight) oheight = cur_maxheight;
      oy = wy + (wheight - oheight) / 2;
    }
  }
  
  if (ximage)
    {
      x11_destroy_ximage (dpy, ximage, ximage_shm);
      ximage = NULL;
    }

#ifdef HAVE_XV
  if (xvimage)
    {
      x11_destroy_xvimage (dpy, xvimage, ximage_shm);
      xvimage = NULL;
    }

#ifdef HAVE_VIDEO_XV
  if (have_video_xv && (xv_gc != NULL)&&(xv_video_port != -1) &&
      cur_capture==CAPTURE_OVERLAY)
    {
      int sx, sy, dx, dy;
      int sw, sh;
      int dw, dh;
      sx = sy = dx = dy = 0;
      sw = 768;
      sh = 576;
      dw = wwidth;
      dh = wheight;
      if (Success != XvPutVideo(dpy, xv_video_port, WINDOW(video), xv_gc,
                                sx,sy,sw,sh, dx,dy,dw,dh))
        fprintf(stderr, "Xv video_overlay: XvPutVideo failed!\n");
      
      if (debug)
	fprintf(stderr,"Xvideo: video: win=0x%lx, "
		"src=%dx%d+%d+%d dst=%dx%d+%d+%d\n",
		WINDOW(video), sw,sh,sx,sy, dw,dh,dx,dy);
    }
#endif // HAVE_VIDEO_XV

#endif // HAVE_XV

  conf = 1;
  move = 1;
  configure_overlay ();
}

/* ------------------------------------------------------------------------ */

static void
video_event (Widget widget, XtPointer client_data, XEvent * event,
             Boolean * d)
{
  //static unsigned long int i=0;
  //i++;
  //fprintf(stderr, "x11: video_event: got an event %ld\n", i);
  if (widget == video_parent)
    {
      /* shell widget */
      switch (event->type)
        {
        case ConfigureNotify:
#if 0
          wx = event->xconfigure.x;
          wy = event->xconfigure.y;
          wwidth = event->xconfigure.width;
          wheight = event->xconfigure.height;
          if (debug > 1)
            fprintf (stderr, "video: shell: cfg %dx%d+%d+%d\n",
                     wwidth, wheight, wx, wy);
#endif
          video_new_size ();
          break;
        case MapNotify:
          if (debug > 1)
            fprintf (stderr, "video: shell: map\n");
          wmap = 1;
          conf = 1;
          configure_overlay ();
          break;
        case UnmapNotify:
          if (debug > 1)
            fprintf (stderr, "video: shell: unmap\n");
          wmap = 0;
          conf = 1;
          configure_overlay ();
          break;
        default:
          if (debug > 1)
            fprintf (stderr, "video: shell: %s\n", events[event->type]);
        }
      return;

    }
  else
    {
      /* TV widget (+root window) */
      switch (event->type)
        {
        case Expose:
          if (event->xvisibility.window == WINDOW (video))
            {
              /* tv */
              if (!event->xexpose.count)
                {
                  if (did_refresh)
                    {
                      did_refresh = 0;
                      if (debug > 1)
                        fprintf (stderr, "video: tv: last refresh expose\n");
                    }
                  else
                    {
                      if (debug > 1)
                        fprintf (stderr, "video: tv: expose\n");
                      conf = 1;
                      configure_overlay ();
                    }
                }
            }
          break;
        case VisibilityNotify:
          if (event->xvisibility.window == WINDOW (video))
            {
              /* tv */
              visibility = event->xvisibility.state;
              if (debug > 1)
                fprintf (stderr, "video: tv: visibility %d%s\n",
                         event->xvisibility.state,
                         did_refresh ? " (ignored)" : "");
              if (did_refresh)
                {
                  if (event->xvisibility.state != VisibilityFullyObscured)
                    did_refresh = 0;
                }
              else
                {
                  conf = 1;
                  configure_overlay ();
                }
            }
          else
            {
              /* root */
              if (debug > 1)
                fprintf (stderr, "video: root: visibility\n");
            }
          break;
        case MapNotify:
        case UnmapNotify:
        case ConfigureNotify:
          if (event->xvisibility.window != WINDOW (video))
            {
              if (debug > 1)
                fprintf (stderr, "video: root: %s%s\n",
                         events[event->type],
                         did_refresh ? " (ignored)" : "");
              if (!did_refresh)
                configure_overlay ();
            }
          break;
        default:
          if (debug > 1)
            fprintf (stderr, "video: tv(+root): %s\n", events[event->type]);
          break;
        }
    }
}


void
video_overlay (set_overlay cb)
{
#ifdef HAVE_VIDEO_XV
  static int grabbed = 0;
  int sx,sy,dx,dy;
  int sw,sh;
  int dw, dh;

  if (have_video_xv)
    {
      if (cb)
        {
          sw = cur_maxwidth;
          sh = cur_maxheight;
          dw = wwidth;
          dh = wheight;
          sx = sy = dx = dy = 0;
          if (NULL == xv_gc)
            xv_gc = XCreateGC(dpy, WINDOW(video), 0, NULL);

          if (!grabbed)
            {
              if (Success == XvGrabPort(dpy, xv_video_port, CurrentTime))
                grabbed = 1;
              else
                fprintf(stderr, "Xv video_overlay: XvGrabPort failed!\n");
            }

          if (grabbed)
            {
              if (Success != XvPutVideo(dpy, xv_video_port, WINDOW(video), xv_gc,
                                        sx,sy,sw,sh, dx,dy,dw,dh))
                fprintf(stderr, "Xv video_overlay: XvPutVideo failed!\n");
              
              if (debug)
                fprintf(stderr,"Xvideo: video: win=0x%lx, "
                        "src=%dx%d+%d+%d dst=%dx%d+%d+%d\n",
                        WINDOW(video), sw,sh,sx,sy, dw,dh,dx,dy);
            }
          else
            {
              fprintf(stderr,"Xvideo: port %ld busy\n", xv_video_port);
            }
        }
      else
        {
          if (grabbed)
            {
              XClearArea(dpy, WINDOW(video), 0, 0, 0, 0, False);
              XvStopVideo(dpy, xv_video_port, WINDOW(video));
              XvUngrabPort(dpy, xv_video_port, CurrentTime);
              grabbed = 0;
              if (debug)
                fprintf(stderr,"Xvideo: video off\n");
            }
        }
    }
  else
#endif // HAVE_VIDEO_XV
    {
      if (cb)
        {
          overlay_cb = cb;
          conf = 1;
          configure_overlay ();
        }
      else
        {
          if (overlay_cb)
            {
              overlay_on = 0;
              overlay_cb (0, 0, 0, 0, 0, NULL, 0);
              overlay_refresh = ADD_TIMER (refresh_timer);
            }
          overlay_cb = NULL;
        }
    }
}

int
video_displayframe_x11 (void)
{
  static void *image_data;
  int ysbar;
  // owidth and oheight are the size of the overlay (i.e. the TV image) (max value is 768x576)
  // wwidth and wheight are the size of the window whatever size it is

  if (!WINDOW (video))
    return -1;

  if (!gc)
    gc = XCreateGC (dpy, WINDOW (video), 0, NULL);

  if (!ximage)
    {
      ximage =
        x11_create_ximage (dpy, owidth, oheight, &ximage_shm);
      if (NULL == ximage)
        {
          fprintf (stderr, "oops: can't create ximage\n");
          exit (1);
        }
      XClearArea (dpy, WINDOW (video), 0, 0, wwidth, wheight,
                  False);
    }

  image_data = ximage->data;
  if (!get_image2(image_data, x11_native_format, owidth, oheight))
    return -1;
  
  ysbar = get_ybar(oheight);
  
  if (is_divx_in_progress())
    {
      divx_encode(image_data+size_img (x11_native_format, owidth, ysbar));
      
      if (divx_display_frames() == 0)
	return 0;
    }
  
  if (ysbar > 0)
    {
      //XClearArea (dpy, WINDOW(video), 0, 0, owidth, ydbar, False);
      //XClearArea (dpy, WINDOW(video), 0, oheight-ydbar, owidth, ydbar, False);
      XClearArea (dpy, WINDOW(video), ox-wx, oy-wy, owidth, ysbar, False);
      XClearArea (dpy, WINDOW(video), ox-wx, wheight-ysbar-oy+wy, owidth, ysbar, False);
    }
  XPUTIMAGE (dpy, WINDOW (video), gc, ximage,
	     0, 0 + ysbar,
	     ox - wx, oy - wy + ysbar,
	     owidth, oheight-2*ysbar);
  if(cmd) {do_command(cmd);cmd=NULL;}
  if (is_http_in_progress()) http_read_clients();
  return 0;
}

#ifdef HAVE_XV
int
video_displayframe_xv (void)
{
  static void *image_data;
  int ydbar;
  int ysbar;
  // owidth and oheight are the size of the overlay (i.e. the TV image) (max value is 768x576)
  // wwidth and wheight are the size of the window whatever size it is

  if (!WINDOW (video))
    return -1;

  if (!gc)
    gc = XCreateGC (dpy, WINDOW (video), 0, NULL);

  if (!xvimage)
    {
      xvimage =
        x11_create_xvimage (dpy, owidth, oheight,
                            &ximage_shm);
      if (NULL == xvimage)
        {
          fprintf (stderr, "oops: can't create xvimage\n");
          exit (1);
        }
    }

      image_data = xvimage->data;
      if (!get_image2 (image_data, x11_pixmap_format, owidth, oheight))
	return -1;

      ysbar = get_ybar(oheight);
      ydbar = get_ybar(wheight);

      if (is_divx_in_progress())
        {
          divx_encode(image_data+ysbar*owidth*2);

          if (divx_display_frames() == 0)
            return 0;
        }

      //XLockDisplay (dpy);
      if (ydbar>0)
        {
          // add extra 2 lines to clear nasty green or rosa line noisy lines
          XClearArea (dpy, WINDOW(video), 0, 0, wwidth, ydbar+2, False);
          XClearArea (dpy, WINDOW(video), 0, wheight-ydbar-2, wwidth, ydbar+2, False);
        }
          

      XVPUTIMAGE (dpy, xv_img_port, WINDOW (video), gc, xvimage,
                  0, 0+ysbar, owidth, oheight-2*ysbar, // source rect
                  0, 0+ydbar, wwidth, wheight-2*ydbar); // dest rect

      //XFlush (dpy);
      //XUnlockDisplay (dpy);

  if(cmd) {do_command(cmd);cmd=NULL;}
  if (is_http_in_progress()) http_read_clients();
  return 0;
}
#endif

Widget
video_init (void)
{
  x11_init ();

  video_parent = app_shell;

  video = XtVaCreateManagedWidget ("tv", simpleWidgetClass, video_parent,
				   NULL);

  /* TV Widget -- need visibility, expose */
  XtAddEventHandler (video,
                     VisibilityChangeMask |
                     StructureNotifyMask, False, video_event, NULL);

  /* Shell widget -- need map, unmap, configure */
  XtAddEventHandler (video_parent, StructureNotifyMask, True,
		     video_event, NULL);

  /* root window -- need */
  XSelectInput (dpy, root,
                VisibilityChangeMask |
                SubstructureNotifyMask | StructureNotifyMask);

  XtRegisterDrawable (dpy, root, video);

  return video;
}

void
video_lock_size(void)
{
  /* TV Widget -- need visibility, expose */
  XtRemoveEventHandler (video,
                     StructureNotifyMask, False, video_event, NULL);

  /* Shell widget -- need map, unmap, configure */
  XtRemoveEventHandler (video_parent, StructureNotifyMask, True, video_event, NULL);
  //XtVaSetValues(tv, XtNallowShellResize, False, NULL);

}

void
video_unlock_size(void)
{
  /* TV Widget -- need visibility, expose */
  XtAddEventHandler (video,
                     StructureNotifyMask, False, video_event, NULL);

  /* Shell widget -- need map, unmap, configure */
  XtAddEventHandler (video_parent, StructureNotifyMask, True, video_event, NULL);
  //  XtVaSetValues(tv, XtNallowShellResize, True, NULL);
}

void
video_background(void)
{
  Window window;
  //XVisualInfo vinfo;
  
  XColor background, ignored;
  XSetWindowAttributes attr;
  //Colormap              theCmap = 0;

  window = WINDOW(video);

  if (XAllocNamedColor (dpy, colormap, "black", &background, &ignored)
      == 0)
    fprintf (stderr, "video: Cannot allocate color black\n");

  //attr.background_pixel = background.pixel;
  attr.background_pixel = BlackPixel(dpy, scr_num);
  attr.border_pixel     = BlackPixel(dpy, scr_num);
  attr.override_redirect = True;

  XChangeWindowAttributes(dpy,
                          window,
                          CWBackPixel | CWBorderPixel,
                          //|CWOverrideRedirect,
                          &attr);

  /*
    XMatchVisualInfo(display, screen_num, DefaultDepthOfScreen (screen), TrueColor, &vinfo);

    theCmap   = XCreateColormap(display,
                                RootWindow(display, screen_num),
                                vinfo.visual, AllocNone);

    attr.background_pixel = 0;
    attr.border_pixel     = 1;
    attr.colormap         = theCmap;
    XChangeWindowAttributes(display,
                            window,
                            CWBackPixel | CWBorderPixel | CWColormap,
                            &attr);
  */

  XClearWindow(dpy, window);
  XFlush(dpy);
}

void
video_close (void)
{
  XSelectInput (dpy, root, 0);
  XtUnregisterDrawable (dpy, root);

  if (ximage)
    {
      x11_destroy_ximage (dpy, ximage, ximage_shm);
      ximage = NULL;
    }

#ifdef HAVE_XV

  if (xvimage)
    {
      x11_destroy_xvimage (dpy, xvimage, ximage_shm);
      xvimage = NULL;
    }

  if (have_xv)
    {

      if (XvUngrabPort (dpy, xv_img_port, CurrentTime) != Success)
        fprintf (stderr, "XvUngrabPort() failed.\n");
    }

#ifdef HAVE_VIDEO_XV
  if (have_video_xv)
    {
      XvFreeEncodingInfo(xv_video_encodings);
      if (XvUngrabPort (dpy, xv_video_port, CurrentTime) != Success)
        fprintf (stderr, "XvUngrabPort() failed.\n");
    }
#endif

#endif

}

void
video_get_size(int *w, int *h)
{
  *w = owidth;
  *h = oheight;
}

void video_get_capture_size(int *w, int *h)
{
  *w = width_capture;
  *h = height_capture;
}

void video_set_capture_size(int w, int h)
{
  /* -2 means no change, -1 the maximum dimension, 0 no constraint */
  if(w!=-2) width_capture = w;
  if(h!=-2) height_capture = h;
  video_new_size ();
}

#ifdef HAVE_VIDEO_XV

inline static int xv_to_xaw(int x,XvAttribute xat) {
  return (float)(x-xat.min_value)/(xat.max_value-xat.min_value)*65535+0.5;
}

inline static int xaw_to_xv(int x,XvAttribute xat) {
  return (float)x/65535*(xat.max_value-xat.min_value)+xat.min_value+0.5;
}


unsigned long
xv_getfreq(void)
{
  unsigned int freq;

  if (xv_video_freq != -1)
    XvGetPortAttribute(dpy, xv_video_port, xv_video_freq, &freq);
  printf("xv_getfreq: freq = %d\n", freq);
  return (unsigned long)freq;
}

void
xv_setfreq(unsigned long freq)
{
  if (xv_video_freq != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_freq, freq);

  XSync(dpy, False);
  printf("xv_setfreq: freq = %ld\n", freq);
}

int
xv_tuned(void)
{
    /* don't know ... */
    return 0;
}

void
xv_setencoding(int e)
{
  if(cur_capture==CAPTURE_OVERLAY) video_overlay (NULL);
  if (xv_video_encoding != -1 && e != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_encoding, e);
  if(cur_capture==CAPTURE_OVERLAY) video_overlay((set_overlay)1);
}

void
xv_setpicture(int color, int bright, int hue, int contrast)
{
  /*if (xv_video_colorkey != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_colorkey,
    xaw_to_xv(color,xvat_colorkey));*/
  if (xv_video_brightness != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_brightness,
		       xaw_to_xv(bright,xvat_brightness));
  if (xv_video_contrast != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_contrast,
		       xaw_to_xv(contrast,xvat_contrast));
  if (xv_video_saturation != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_saturation, 
		       xaw_to_xv(color,xvat_saturation));
  if (xv_video_hue != -1)
    XvSetPortAttribute(dpy, xv_video_port, xv_video_hue, 
		       xaw_to_xv(hue,xvat_hue));
  XSync(dpy, False);
}

void
xv_getaudio(int *volume, int *mute)
{
  int value;

  if (xv_video_volume != -1)
    {
      XvGetPortAttribute(dpy, xv_video_port, xv_video_volume, &value);
      *volume = xv_to_xaw(value,xvat_volume);
    }

  if (xv_video_mute != -1)
    {
      XvGetPortAttribute(dpy, xv_video_port, xv_video_mute, &value);
      *mute = value;
    }
}

void
xv_setaudio(int mute, int volume)
{
  int value;

  if (xv_video_volume != -1 && volume != -1)
    {
      value = xaw_to_xv(volume,xvat_volume);
      XvSetPortAttribute(dpy, xv_video_port, xv_video_volume, value);
      fprintf(stderr, "xv_setaudio: v4l vol = %d, XV vol = %d\n", volume, value);
    }

  if (xv_video_mute != -1 && mute != -1)
    {
      XvSetPortAttribute(dpy, xv_video_port, xv_video_mute, mute);
      if(mute)
	fprintf(stderr, "xv_setaudio: mute\n");
      else
	fprintf(stderr, "xv_setaudio: unmute\n");
    }
  
  XSync(dpy, False);
}
#endif // HAVE_VIDEO_XV
