/*****************************************************************************
 * $Id: plugin.c,v 1.4 2004/09/25 14:54:20 alainjj Exp $
 * Program under GNU General Public License (see ../COPYING)
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "plugin.h"

#ifndef RTLD_NOW /* For OpenBSD */
#define RTLD_NOW 0
#endif
#ifdef DL_ADD_UNDERSCORE
#define DL_UND(x) ("_" x)
#else
#define DL_UND(x) (x)
#endif

static struct xdtvplug {
  void *plug;
  struct xdtvplug *prev;
} *hPlugins = NULL;

static char *plugdir = PREFIX "/lib/xdtv-plugins";
/* plugfile exists for compatibility reasons, but the new plugins
   should be in plugdir */
static char *plugfile = "libxdtv.so";

void load_plugin (char *name, int exit_on_error)  {
  char *error;
  unsigned int *plugin_api;
  int api_error = 0;
  void *hPlugin;
  
  if(name == NULL) return;
  if(debug) fprintf(stderr, "Loading %s\n", name);
  /* dlopen executes _init in plugfile if it exists */
  hPlugin = dlopen(name, RTLD_NOW);
  if (!hPlugin) {
      error = dlerror();
      if(!exit_on_error && !strstr(error,"No such file"))
	fprintf(stderr, "dlopen: %s\n", error);
      if(exit_on_error) exit(1);
  }  else {
    plugin_api = dlsym(hPlugin, DL_UND("version"));
    if ((error = dlerror()) != NULL) {
      if (debug||1)
	fprintf(stderr, "dlsym: %s\n", error);
      fprintf(stderr,
	      "WARNING: A plugin has been find but is unable to provide its api number.\n"
	      "         the API used by your plugin is too old.\n");
      api_error = 1;
    }  else if (*plugin_api != XDTV_PLUGIN_API) {
      fprintf(stderr,
	      "WARNING: Plugin API is %06x but %06x at least is needed.\n",
	      *plugin_api, XDTV_PLUGIN_API);
      api_error = 1;
    }
    if (api_error) {
      //dlclose(hPlugin); //dlopen has probably registered things...
      hPlugin = NULL;
      fprintf(stderr, "!!!!!!!!! %s BADLY LOADED !!!!!!!!!!!.\n", name);
      if(exit_on_error) exit(1);
      fprintf(stderr, "RECOMPILE OR REMOVE %s\n", name);
      exit(1);
    } else {
      struct xdtvplug *p;
      p=malloc(sizeof(struct xdtvplug));
      p->prev=hPlugins;
      p->plug=hPlugin;
      hPlugins=p;
    }
  }
}

/* load the default plugins */
void load_plugins(void) {
  struct stat buf;
  DIR *dir;
  struct dirent *dir2;
  
  load_plugin(plugfile, 0);
  
  if(plugdir==NULL) return;
  dir=opendir(plugdir);
  if(dir==NULL) {
    if(errno!=ENOENT) perror("opendir");
    return;
  }
  while ((dir2 = readdir(dir)) != NULL) {
    char name[255];
    sprintf(name,"%s/%s",plugdir,dir2->d_name);
    if (stat (name, &buf) == 0 && S_ISREG(buf.st_mode))
      load_plugin(name, 0);
  }
  closedir(dir);
}

void plugins_close(void) {
  while(hPlugins) {
    struct xdtvplug *p=hPlugins;
    dlclose(p->plug);
    hPlugins=p->prev;
    free(p);
  }
}

/* remark: the option "-plugin plugfile" must be before the
   option for plugfile */
static struct _parseopts {
  int (*f)(int argc,char *argv[],int *i);
  struct _parseopts *prev;
}*parseopts = NULL;
void register_parseopts(int (*f)(int argc,char *argv[],int *i)) {
  struct _parseopts *p=malloc(sizeof(struct _parseopts));
  p->prev=parseopts;
  p->f=f;
  parseopts=p;
}
int plugin_parse_opts(int argc,char *argv[],int *i) {
  struct _parseopts *p=parseopts;
  while(p) {if(p->f(argc,argv,i)) return 1; p=p->prev;}
  return 0;
}

static struct _initactions {
  void (*f)(void);
  struct _initactions *prev;
}*initactions = NULL;
void register_initaction(void (*f)(void)) {
  struct _initactions *i=malloc(sizeof(struct _initactions));
  i->prev=initactions;
  i->f=f;
  initactions=i;
}
void plugin_init_actions(void) {
  struct _initactions *i=initactions;
  while(i) {i->f(); i=i->prev;}
}

static struct _idleactions {
  void (*f)(void);
  struct _idleactions *prev;
}*idleactions = NULL;
void register_idleaction(void (*f)(void)) {
  struct _idleactions *i=malloc(sizeof(struct _idleactions));
  i->prev=idleactions;
  i->f=f;
  idleactions=i;
}
void plugin_idle_actions(void) {
  struct _idleactions *i=idleactions;
  while(i) {i->f(); i=i->prev;}
}

void set_plugdir(char *dirname) {
  plugdir=dirname;
}

void set_default_plugfile(char *filename) {
  plugfile=filename;
}
