/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* the verbose one: syslog messages                                           */
/*============================================================================*/

/*
 * logmsg
 *
 * input: type   - the message id
 *	  reason - the reason code for the message
 *	  anchor - pointer to vpnd global data
 *
 * This procedure writes a syslog message
 */

void logmsg(int type,int reason,VPN *anchor)
{
	struct in_addr addr;	/* used for ip to string conversion */


	/* debug message */

	ENTER("logmsg");

	/* print message according to message type given */

	switch(type)
	{
	case STARTUP:
#ifdef COMPRESS
		syslog(LOG_NOTICE,"version %s (compression enabled) starting.",VERSION);
#else
		syslog(LOG_NOTICE,"version %s (no compression) starting.",VERSION);
#endif
		break;
	case SHUTDOWN:
		syslog(LOG_NOTICE,"shutting down.");
		break;
	case SIGNAL:
		syslog(LOG_INFO,"signal %d received.",reason);
		break;
	case SLIPFAIL:
		syslog(LOG_NOTICE,"slip link failed, reason=%d, " \
			"will retry in %d seconds.",reason,anchor->retry);
		break;
	case SLIPUP:
		syslog(LOG_INFO,"slip link established.");
		break;
	case SRVSFAIL:
		syslog(LOG_NOTICE,"server socket failed, reason=%d, " \
			"will retry in %d seconds.",reason,anchor->retry);
		break;
	case SRVSLSTN:
		syslog(LOG_INFO,"listening with address %s on port %d",
			anchor->server_ip,anchor->server_port);
		break;
	case PEERFAIL:
		syslog(LOG_INFO,"peer link failed, reason=%d, " \
			"will retry in %d seconds.",reason,anchor->retry);
		break;
	case CRYPTFAIL:
		syslog(LOG_NOTICE,"crypto init failed, reason=%d, " \
			"will retry in %d seconds.",reason,anchor->retry);
		break;
	case CCONNECT:
		syslog(LOG_INFO,"connected to %s",anchor->server_ip);
		break;
	case DEVFAIL:
		syslog(LOG_INFO,"access to %s failed, " \
			"will retry in %d seconds.",
			anchor->serialdev,anchor->retry);
		break;
	case FORKFAIL:
		syslog(LOG_INFO,"fork failed, will retry in %d seconds.",
			anchor->retry);
		break;
	case INITCHAT:
		syslog(LOG_INFO,"serial line init chat started on %s",
			anchor->serialdev);
		break;
	case SELECTFAIL:
		syslog(LOG_NOTICE,"select failed, dying...");
		break;
	case CHATFAIL:
		syslog(LOG_INFO,"init chat failed, reason %d, " \
			"will retry in %d seconds.",reason,anchor->retry);
		break;
	case DCONNECT:
		syslog(LOG_INFO,"connection established on %s",
			anchor->serialdev);
		break;
	case SPEERFAIL:
		syslog(LOG_INFO,"peer link failed, reason %d, " \
			"closing connection.",reason);
		break;
	case ACPTERR:
		syslog(LOG_INFO,"client connect failed, still waiting.");
		break;
	case ILLEGAL:
		addr.s_addr=reason;
		syslog(LOG_NOTICE,"illegal connect from %s, " \
			"closing connection.",inet_ntoa(addr));
		break;
	case PEEROPT:
		syslog(LOG_NOTICE,"peer socket option setup failed, " \
			"closing connection.");
		break;
	case PEERIOCTL:
		syslog(LOG_NOTICE,"peer socket ioctl failed, " \
			"closing connection.");
		break;
	case SCRYPT:
		syslog(LOG_NOTICE,"crypto init failed, reason=%d, " \
			"closing connection.",reason);
		break;
	case SCONNECT:
		addr.s_addr=reason;
		syslog(LOG_INFO,"connect from %s",inet_ntoa(addr));
		break;
	case KEYFAIL:
		syslog(LOG_NOTICE,"reading master key from file failed, " \
			"terminating...");
		break;
	case FALLBACK:
		syslog(LOG_NOTICE,"warning: transaction id mismatch, " \
			"using fallback data");
		break;
	default:
		syslog(LOG_NOTICE,"unknown message with reason %d",reason);
		break;
	}

	/* debug message */

	LEAVE("logmsg");
}
