/*
 * Copyright (C) 2001, 2002 Apple Computer, Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include <gtk/gtk.h>
#include "KWQFont.h"

#include "KWQString.h"
#include "KWQAssertions.h"
#include "WebCoreTextRendererFactory.h"
#include "KWQFoundationExtras.h"

QFont::QFont()
    :_trait(0)
    ,_size(12.0)
    ,_isPrinterFont(false)
    ,_NSFont(0)
    ,_pangoDescr(0)
{
}

QFont::~QFont()
{
    invalidate();
}

void QFont::invalidate()
{
    KWQRelease(_NSFont);
    _NSFont = 0;

    if (_pangoDescr) {
	pango_font_description_free(_pangoDescr);
	_pangoDescr = 0;
    }
}

QFont::QFont(const QFont &other)
    :_NSFont(0)
     ,_pangoDescr(0)
{
    *this = other;
}

QFont &QFont::operator=(const QFont &other)
{
    invalidate();
    _family = other._family;
    _trait = other._trait;
    _size = other._size;
    _isPrinterFont = other._isPrinterFont;
    
    KWQRetain(other._NSFont);
    _NSFont = other._NSFont;
    
    if (other._pangoDescr)
	_pangoDescr = pango_font_description_copy(other._pangoDescr);

    return *this;
}

QString QFont::family() const
{
    return _family.family().string();
}

void QFont::setFamily(const QString &qfamilyName)
{
    _family.setFamily(qfamilyName);
    invalidate();
}
	

void QFont::setFirstFamily(const KWQFontFamily& family) 
{
    _family = family;
    invalidate();
}

void QFont::setPixelSize(float s)
{
    if (_size != s) {
	_size = s;
	invalidate();
    }
}

void QFont::setWeight(int weight)
{
    if (weight == Bold) {
        if (!(_trait & NSBoldFontMask)){
	    invalidate();
        }
        _trait |= NSBoldFontMask;
    } else if (weight == Normal) {
        if ((_trait & NSBoldFontMask)){
	    invalidate();
        }
        _trait &= ~NSBoldFontMask;
    }
}

void QFont::setPrinterFont(bool isPrinterFont)
{
    _isPrinterFont = isPrinterFont;
}

int QFont::weight() const
{
    return bold() ? Bold : Normal;
}

void QFont::setItalic(bool flag)
{
   if (flag) {
       if (!(_trait & NSItalicFontMask)){
	    invalidate();
       }
       _trait |= NSItalicFontMask;
   } else {
        if ((_trait & NSItalicFontMask)){
	    invalidate();
        }
        _trait &= ~NSItalicFontMask;
    }
}

bool QFont::italic() const
{
    return _trait & NSItalicFontMask;
}

bool QFont::bold() const
{
    return _trait & NSBoldFontMask;
}

bool QFont::isFixedPitch() const 
{
    return WebCoreTextRendererFactory::sharedFactory()->isFontFixedPitch(getNSFont());
}

bool QFont::operator==(const QFont &compareFont) const
{
    return _family == compareFont._family 
	&& _trait == compareFont._trait
        && _size == compareFont._size
	&& _isPrinterFont == compareFont._isPrinterFont;	
}


NSFont* QFont::getNSFont() const
{
    if (!_NSFont) {
        CREATE_FAMILY_ARRAY(*this, families);
        _NSFont = WebCoreTextRendererFactory::sharedFactory()->fontWithFamilies(families,
										(NSFontTraitMask) getNSTraits(),
										getNSSize());
	KWQRetain(_NSFont);
    }
    return _NSFont;
}


PangoFontDescription* QFont::getPangoFont() const
{
    if (_pangoDescr) 
	return _pangoDescr;

    PangoFontDescription* descr = pango_font_description_new();
    pango_font_description_set_family(descr, family().latin1());    
    pango_font_description_set_weight(descr,
				      (weight()== QFont::Bold) ? PANGO_WEIGHT_BOLD : PANGO_WEIGHT_NORMAL);
    pango_font_description_set_style(descr,
				     italic() ? PANGO_STYLE_ITALIC : PANGO_STYLE_NORMAL);
   
    float px = pixelSize();

    if (px < 1) 
	px = 1;

    // This hack is kind of specified in 
    // http://mail.gnome.org/archives/gtk-i18n-list/2003-July/msg00019.html
    // might be something to look after if font sizes 

    float dpi = (float) 96.0f;	// this is academic quess
    pango_font_description_set_size(descr, (int)(px * PANGO_SCALE * 72.0f / dpi));

    _pangoDescr = descr;
    return _pangoDescr;
}
