/*
 * Copyright (c) 2004 Nokia. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Nokia nor the names of its contributors may be
 * used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "KWQCharsets.h"
#include "KWQMap.h"
#include "KWQString.h"

#include "KWQAssertions.h"
#include <glib.h>
#include <ctype.h>

struct CharsetEntry {
    const char *name;
    CFStringEncoding encoding;
    KWQEncodingFlags flags;
};


/* The following autogenerated file includes the charset data. */
#include "KWQCharsetData.c"

typedef guint CFHashCode;
static gboolean encodingNamesEqual(const void *value1, const void *value2);
static CFHashCode encodingNameHash(const void *value);


// these dictionaries will stay until end of the execution 
// ie they leak their internal structure. The data inside is static

static GHashTable *nameToTable = NULL;
static GHashTable *encodingToTable = NULL;

static void buildDictionaries()
{
    nameToTable = g_hash_table_new(encodingNameHash, encodingNamesEqual);
    encodingToTable = g_hash_table_new(g_int_hash, g_int_equal);


    for (int i = 0; table[i].name != NULL; i++) {
	
	g_hash_table_insert(nameToTable,
			    const_cast<char*>(table[i].name),
			    const_cast<CharsetEntry*>(&table[i]));

	// there are 1:1 table[i].name -> encoding, but
	// 1:* encoding -> table[i].name
	// insert only the first occurance of encoding integer, no aliases
	if (g_hash_table_lookup(encodingToTable,&table[i].encoding)) {
	    // skip this
	} else {
	    g_hash_table_insert(encodingToTable,
				const_cast<CFStringEncoding*>(&table[i].encoding),
				const_cast<CharsetEntry*>(&table[i]));
	}
    }
}

CFStringEncoding KWQCFStringEncodingFromIANACharsetName(const char *name, KWQEncodingFlags *flags)
{
    if (nameToTable == NULL) {
        buildDictionaries();
    }

    const CharsetEntry* value =
	static_cast<CharsetEntry*>(g_hash_table_lookup(nameToTable, name));
    
    if (!value) {
        if (flags) {
            *flags = NoEncodingFlags;
        }
        return kCFStringEncodingInvalidId;
    }
    if (flags) {
        *flags = value->flags;
    }

    return value->encoding;
}

const char *KWQCFStringEncodingToIANACharsetName(CFStringEncoding encoding)
{
    if (encodingToTable == NULL) {
        buildDictionaries();
    }
    
    const CharsetEntry* value =
	static_cast<CharsetEntry*>(g_hash_table_lookup(encodingToTable, 
						       static_cast<void*>(&encoding)));

    if (!value)
	return NULL;    

    return value->name;
}

static gboolean encodingNamesEqual(const void *value1, const void *value2)
{
    const char *s1 = static_cast<const char *>(value1);
    const char *s2 = static_cast<const char *>(value2);
    
    while (1) {
        char c1;
        do {
            c1 = *s1++;
        } while (c1 && !isalnum(c1));
        char c2;
        do {
            c2 = *s2++;
        } while (c2 && !isalnum(c2));

        if (tolower(c1) != tolower(c2)) {
            return false;
        }
        
        if (!c1 || !c2) {
            return !c1 && !c2;
        }
    }
}

// Golden ratio - arbitrary start value to avoid mapping all 0's to all 0's
// or anything like that.
const unsigned PHI = 0x9e3779b9U;

// This hash algorithm comes from:
// http://burtleburtle.net/bob/hash/hashfaq.html
// http://burtleburtle.net/bob/hash/doobs.html
static CFHashCode encodingNameHash(const void *value)
{
    const char *s = static_cast<const char *>(value);
    
    CFHashCode h = PHI;

    for (int i = 0; i != 16; ++i) {
        char c;
        do {
            c = *s++;
        } while (c && !isalnum(c));
        if (!c) {
            break;
        }
        h += tolower(c);
	h += (h << 10); 
	h ^= (h << 6); 
    }

    h += (h << 3);
    h ^= (h >> 11);
    h += (h << 15);
 
    return h;
}

