/* $Id: vnet_kernel.c,v 1.1 2003/12/13 18:55:20 pb Exp $ */
/***************************************************************************************
	Copyright 2000-2001 ATMEL Corporation.
	
	This file is part of atmel wireless lan drivers.

    Atmel wireless lan drivers is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Atmel wireless lan drivers is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Atmel wireless lan drivers; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

**************************************************************************************/
#include <linux/module.h>
#include "vnetusba.h"


extern struct usb_driver vnet_driver;

int VNetAllocMem(void **Ptr, USHORT size)
{
	if (in_interrupt()) {
		*Ptr = kmalloc(size, GFP_ATOMIC);
	} else {
		*Ptr = kmalloc(size, GFP_KERNEL);
	}
	if (*Ptr == NULL) {
		return -1;
	}
	return 0;
}

void VNetFree(void *ptr)
{
	kfree(ptr);
}

void VnetTimer(PVNet_ADAPTER Adapter, UCHAR timeout)
{
	if (timeout == 0) {
		del_timer(Adapter->CtrlTimer);
		return;
	}
	Adapter->flags |= TimerReScheduled;

	if (timer_pending(Adapter->CtrlTimer) != 0) {
		mod_timer(Adapter->CtrlTimer,
			  (jiffies + (timeout * HZ) / 10));
	} else {
		Adapter->CtrlTimer->expires =
		    jiffies + (timeout * HZ) / 10;
		add_timer(Adapter->CtrlTimer);
	}
}

void VNetFreeSkb(struct sk_buff *skb)
{
	if (in_interrupt())
		dev_kfree_skb_irq(skb);
	else
		dev_kfree_skb(skb);
}

int rx_kernel_part(PVNet_ADAPTER Adapter, struct sk_buff **skb,
		   USHORT PacketLength, UCHAR DataOffset)
{
	*skb = dev_alloc_skb(PacketLength + 2);

	if (!(*skb))
		return -1;

	skb_reserve(*skb, 2);
	memcpy(skb_put(*skb, PacketLength),
	       (PUCHAR) Adapter->RxBuffUp.WirelessPacket + DataOffset,
	       PacketLength);
	return 0;
}

int VnetUsbReset(PVNet_ADAPTER Adapter)
{
	struct usb_interface *iface = &Adapter->config->interface[0];

	usb_driver_release_interface(&vnet_driver, iface);

	info("Reseting Usb Device\n");
	usb_inc_dev_use(Adapter->usb);
	usb_reset_device(Adapter->usb);
	usb_dec_dev_use(Adapter->usb);
	info("Reset completed.");

	if (usb_get_configuration(Adapter->usb) != 0) {
		info("Get configuration failed");
		return -1;
	}

	if (usb_set_configuration(Adapter->usb, 1) != 0) {
		info("Set configuration failed");
		return -1;
	}

	Adapter->usb->actconfig = Adapter->usb->config;
	Adapter->config = Adapter->usb->actconfig;
	iface = Adapter->config->interface;
	usb_driver_claim_interface(&vnet_driver, iface, Adapter);
	return 0;
}

int VNetSchedule(PVNet_ADAPTER Adapter, ULONG flag)
{
	DECLARE_WAITQUEUE(wait, current);

	if (in_interrupt()) {
		info("scheduling in interrupt?!!!");
		return -1;
	}
	dbgcond(DBG_SCHEDULE, "ctrl_wait %p wait @ %p  Adapter @ %p\n",
		Adapter->ctrl_wait, &wait, Adapter);
	add_wait_queue(Adapter->ctrl_wait, &wait);
	set_current_state(TASK_INTERRUPTIBLE);
	Adapter->flags |= flag;
	schedule();
	set_current_state(TASK_RUNNING);
	remove_wait_queue(Adapter->ctrl_wait, &wait);
	return 0;
}

void VNetIf_queue(struct net_device *dev, UCHAR c)
{
	PVNet_ADAPTER Adapter = (PVNet_ADAPTER) dev->priv;

	if (c) {
		dbgcond(DBG_TX, "waking up the net queue\n");
		netif_wake_queue(dev);
		Adapter->flags &= ~TX_STOPPED;
	} else {
		dbgcond(DBG_TX, "stopping the net queue\n");
		netif_stop_queue(dev);
		Adapter->flags |= TX_STOPPED;
	}
}

int InitAdapter(PVNet_ADAPTER Adapter)
{

#define ALLOC_OR_EXIT(dest)\
  do {\
   if (!(Adapter->dest=kmalloc(sizeof(*(Adapter->dest)),GFP_KERNEL))) {\
     err("alloc of field " #dest " (%d byte) failed", sizeof(*(Adapter->dest)));\
     return -1;\
   }\
  } while (0)


	ALLOC_OR_EXIT(CtrlTimer);
	init_timer(Adapter->CtrlTimer);

	ALLOC_OR_EXIT(netstats);
	ALLOC_OR_EXIT(w_stats);
	ALLOC_OR_EXIT(devreq);
	ALLOC_OR_EXIT(lock);

	ALLOC_OR_EXIT(ctrl_wait);
	init_waitqueue_head(Adapter->ctrl_wait);

	ALLOC_OR_EXIT(remove_wait);
	init_waitqueue_head(Adapter->remove_wait);

	ALLOC_OR_EXIT(TxList);
	INIT_LIST_HEAD(Adapter->TxList);	//empty list for tx packets... 

	/* Ctrl Timer used for (Re)Authentication, (Re)Association requests,... */
	Adapter->CtrlTimer->expires = jiffies + (HZ);	// one second.
	Adapter->CtrlTimer->function = TimerExpired;
	Adapter->CtrlTimer->data = (ULONG) Adapter;

	Adapter->rx_urb = usb_alloc_urb(0);
	Adapter->tx_urb = usb_alloc_urb(0);
	Adapter->ctrl_urb = usb_alloc_urb(0);
	if ((Adapter->rx_urb == NULL) ||
	    (Adapter->tx_urb == NULL) || (Adapter->ctrl_urb == NULL)) {
		err("usb_alloc_urb failed");
		return -1;
	}

	SetParameters(Adapter);
	memset(Adapter->CmdStatus, 0, sizeof(Adapter->CmdStatus));
	memset(&Adapter->Stats, 0, sizeof(Adapter->Stats));
	memset(Adapter->netstats, 0, sizeof(*(Adapter->netstats)));
	memset(Adapter->w_stats, 0, sizeof(*(Adapter->w_stats)));

	init_CRCtable(Adapter);
	return 0;
}
