/* 
 * Copyright (C) 2000, 2001 Jeff Dike (jdike@karaya.com)
 * Licensed under the GPL
 */

#include "linux/config.h"
#include "linux/stddef.h"
#include "linux/sys.h"
#include "linux/sched.h"
#include "linux/wait.h"
#include "linux/kernel.h"
#include "linux/smp_lock.h"
#include "linux/module.h"
#include "linux/slab.h"
#include "asm/signal.h"
#include "asm/uaccess.h"
#include "user_util.h"
#include "kern_util.h"
#include "signal_kern.h"
#include "signal_user.h"
#include "kern.h"

EXPORT_SYMBOL(block_signals);
EXPORT_SYMBOL(unblock_signals);

struct sys_pt_regs *signal_state(void *t)
{
	struct task_struct *task = t;

	return(&task->thread.altstack_regs);
}

int probe_stack(unsigned long sp, int delta)
{
	int n;

	if((get_user(n, (int *) sp) != 0) ||
	   (put_user(n, (int *) sp) != 0) ||
	   (get_user(n, (int *) (sp - delta)) != 0) ||
	   (put_user(n, (int *) (sp - delta)) != 0))
		return(-EFAULT);
	return(0);
}

int have_signals(void *t)
{
	struct task_struct *task;
	int ret;

	if(t == NULL) task = current;
	else task = t;
	ret = (task->thread.signal.state == SIGNAL_PENDING);
	task->thread.signal.state = SIGNAL_NONE;
	return(ret);
}

int copy_siginfo_to_user(siginfo_t *to, siginfo_t *from)
{
	if (!access_ok (VERIFY_WRITE, to, sizeof(siginfo_t)))
		return -EFAULT;
	if (from->si_code < 0)
		return __copy_to_user(to, from, sizeof(siginfo_t));
	else {
		int err;

		/* If you change siginfo_t structure, please be sure
		   this code is fixed accordingly.
		   It should never copy any pad contained in the structure
		   to avoid security leaks, but must copy the generic
		   3 ints plus the relevant union member.  */
		err = __put_user(from->si_signo, &to->si_signo);
		err |= __put_user(from->si_errno, &to->si_errno);
		err |= __put_user((short)from->si_code, &to->si_code);
		/* First 32bits of unions are always present.  */
		err |= __put_user(from->si_pid, &to->si_pid);
		switch (from->si_code >> 16) {
		case __SI_FAULT >> 16:
			break;
		case __SI_CHLD >> 16:
			err |= __put_user(from->si_utime, &to->si_utime);
			err |= __put_user(from->si_stime, &to->si_stime);
			err |= __put_user(from->si_status, &to->si_status);
		default:
			err |= __put_user(from->si_uid, &to->si_uid);
			break;
		/* case __SI_RT: This is not generated by the kernel as of now.  */
		}
		return err;
	}
}

#define _S(nr) (1<<((nr)-1))

#define _BLOCKABLE (~(_S(SIGKILL) | _S(SIGSTOP)))

/*
 * OK, we're invoking a handler
 */	
static int handle_signal(unsigned long signr, struct k_sigaction *ka, 
			 sigset_t *oldset, unsigned long *error, 
			 int *again_out)
{
	struct signal_context *context;
        __sighandler_t handler;
	unsigned long sp;
	sigset_t save;
	int frame_size;

	if(again_out) *again_out = 0;
        if((error != NULL) && (*error != 0)){
		switch (*error) {
		case -ERESTARTNOHAND:
			*error = -EINTR;
			break;

		case -ERESTARTSYS:
			if (!(ka->sa.sa_flags & SA_RESTART)) {
				*error = -EINTR;
				break;
			}
			/* fallthrough */
		case -ERESTARTNOINTR:
			if(again_out) *again_out = 1;
			break;
		}
	}
	handler = ka->sa.sa_handler;
	save = *oldset;

	if (ka->sa.sa_flags & SA_ONESHOT)
		ka->sa.sa_handler = SIG_DFL;

	if (!(ka->sa.sa_flags & SA_NODEFER)) {
		spin_lock_irq(&current->sigmask_lock);
		sigorsets(&current->blocked,&current->blocked,&ka->sa.sa_mask);
		sigaddset(&current->blocked,signr);
		recalc_sigpending(current);
		spin_unlock_irq(&current->sigmask_lock);
	}

	sp = UM_SP(&current->thread.process_regs);

	if((ka->sa.sa_flags & SA_ONSTACK) && (sas_ss_flags(sp) == 0))
		sp = current->sas_ss_sp + current->sas_ss_size;
	
	frame_size = 4 * sizeof(void *) + sizeof(*context) + 
		4 * sizeof(void *) + signal_stack_size;

	if(probe_stack(sp - sizeof(void *), frame_size) < 0){
		if(signr == SIGSEGV){
			struct k_sigaction *ka;

			ka = &current->sig->action[SIGSEGV - 1];
			ka->sa.sa_handler = SIG_DFL;
		}
		force_sig(SIGSEGV, current);
		return(1);
	}

	current->thread.signal.signal = signr;
	current->thread.signal.handler = (unsigned long) handler;
	current->thread.signal.state = SIGNAL_PENDING;

	sp -= 4 * sizeof(void *) + sizeof(*context);
	context = (struct signal_context *) sp;
	if(error != NULL) 
		UM_SET_SYSCALL_RETURN(&current->thread.process_regs, *error);
	context->regs = current->thread.process_regs;
	context->repeat = current->thread.repeat_syscall;
	context->sigs = save;
	context->prev = current->thread.signal_context;
	current->thread.signal_context = context;

	sp -= 4 * sizeof(void *);
	setup_stack(sp, &current->thread.altstack_regs);

	return(0);
}

/*
 * Note that 'init' is a special process: it doesn't get signals it doesn't
 * want to handle. Thus you cannot kill init even with a SIGKILL even by
 * mistake.
 */

static int kern_do_signal(sigset_t *oldset, unsigned long *error, 
			  int *again_out)
{
	siginfo_t info;
	struct k_sigaction *ka;
	int err;

	if (!oldset)
		oldset = &current->blocked;

	if(again_out) *again_out = 0;
	for (;;) {
		unsigned long signr;

		spin_lock_irq(&current->sigmask_lock);
		signr = dequeue_signal(&current->blocked, &info);
		spin_unlock_irq(&current->sigmask_lock);

		if (!signr)
			break;

		if ((current->ptrace & PT_PTRACED) && signr != SIGKILL) {
			/* Let the debugger run.  */
			current->exit_code = signr;
			current->state = TASK_STOPPED;
			notify_parent(current, SIGCHLD);
			schedule();

			/* We're back.  Did the debugger cancel the sig?  */
			if (!(signr = current->exit_code))
				continue;
			current->exit_code = 0;

			/* The debugger continued.  Ignore SIGSTOP.  */
			if (signr == SIGSTOP)
				continue;

			/* Update the siginfo structure.  Is this good?  */
			if (signr != info.si_signo) {
				info.si_signo = signr;
				info.si_errno = 0;
				info.si_code = SI_USER;
				info.si_pid = current->p_pptr->pid;
				info.si_uid = current->p_pptr->uid;
			}

			/* If the (new) signal is now blocked, requeue it.  */
			if (sigismember(&current->blocked, signr)) {
				send_sig_info(signr, &info, current);
				continue;
			}
		}

		ka = &current->sig->action[signr-1];
		if (ka->sa.sa_handler == SIG_IGN) {
			if (signr != SIGCHLD)
				continue;
			/* Check for SIGCHLD: it's special.  */
			while (sys_wait4(-1, NULL, WNOHANG, NULL) > 0)
				/* nothing */;
			continue;
		}

		if (ka->sa.sa_handler == SIG_DFL) {
			int exit_code = signr;

			/* Init gets no signals it doesn't want.  */
			if (current->pid == 1)
				continue;

			switch (signr) {
			case SIGCONT: case SIGCHLD: case SIGWINCH:
				continue;

			case SIGTSTP: case SIGTTIN: case SIGTTOU:
				if (is_orphaned_pgrp(current->pgrp))
					continue;
				/* FALLTHRU */

                        case SIGSTOP: {
                                struct signal_struct *sig;
				current->state = TASK_STOPPED;
				current->exit_code = signr;
                                sig = current->p_pptr->sig;
                                if (sig && !(sig->action[SIGCHLD-1].sa.sa_flags & SA_NOCLDSTOP))
					notify_parent(current, SIGCHLD);
				schedule();
				continue;
			}
			case SIGQUIT: case SIGILL: case SIGTRAP:
			case SIGABRT: case SIGFPE: case SIGSEGV:
			case SIGBUS: case SIGSYS: case SIGXCPU: case SIGXFSZ:
				if (do_coredump(signr, NULL))
					exit_code |= 0x80;
				/* FALLTHRU */

			default:
				sigaddset(&current->pending.signal, signr);
				recalc_sigpending(current);
				current->flags |= PF_SIGNALED;
				do_exit(exit_code);
				/* NOTREACHED */
			}
		}

		/* Whee!  Actually deliver the signal.  */
		err = handle_signal(signr, ka, oldset, error, again_out);
		if(!err) return(1);
	}
	return(0);
}

int do_signal(unsigned long *error, int *again_out)
{
	return(kern_do_signal(NULL, error, again_out));
}

/*
 * Atomically swap in the new signal mask, and wait for a signal.
 */
int
sys_sigsuspend(int history0, int history1, old_sigset_t mask)
{
	sigset_t saveset;

	mask &= _BLOCKABLE;
	spin_lock_irq(&current->sigmask_lock);
	saveset = current->blocked;
	siginitset(&current->blocked, mask);
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	while (1) {
		current->state = TASK_INTERRUPTIBLE;
		schedule();
		if (kern_do_signal(&saveset, NULL, NULL))
			return -EINTR;
	}
}

int
sys_rt_sigsuspend(sigset_t *unewset, size_t sigsetsize)
{
	sigset_t saveset, newset;

	/* XXX: Don't preclude handling different sized sigset_t's.  */
	if (sigsetsize != sizeof(sigset_t))
		return -EINVAL;

	if (copy_from_user(&newset, unewset, sizeof(newset)))
		return -EFAULT;
	sigdelsetmask(&newset, ~_BLOCKABLE);

	spin_lock_irq(&current->sigmask_lock);
	saveset = current->blocked;
	current->blocked = newset;
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	while (1) {
		current->state = TASK_INTERRUPTIBLE;
		schedule();
		if (kern_do_signal(&saveset, NULL, NULL))
			return -EINTR;
	}
}

void signal_deliverer(int sig)
{
	unsigned long handler;
	int signal;

	stop_pid(getpid());
	signal = current->thread.signal.signal;
	handler = current->thread.signal.handler;
	signal_handler(current, handler, signal);
}

int sys_sigreturn(struct sys_pt_regs regs)
{
	struct signal_context *context = current->thread.signal_context;

	sigdelsetmask(&context->sigs, ~_BLOCKABLE);
	spin_lock_irq(&current->sigmask_lock);
	current->blocked = context->sigs;
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);
	current->thread.process_regs = context->regs;
	current->thread.repeat_syscall = context->repeat;
	current->thread.signal_context = context->prev;
	return(UM_SYSCALL_RET(&current->thread.process_regs));
}

void *signal_context(void *t, unsigned long *cr2_out, int *err_out)
{
	struct task_struct *task = t ? t : current;

	if(cr2_out) *cr2_out = task->thread.cr2;
	if(err_out) *err_out = task->thread.err;
	return(&task->thread.signal_context->regs);
}

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-file-style: "linux"
 * End:
 */
