/*
 *	$Id: maple.c,v 1.7 2002/05/01 16:01:26 lethal Exp $
 *	Maple Bus device driver
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/timer.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/smp_lock.h>
#include <linux/completion.h>

#include <asm/page.h>
#include <asm/io.h>
#include <asm/dc_sysasic.h>

#include <linux/maple.h>

/* 
 * Significantly rewritten by Fredrik Hubinette <hubbe@hubbe.net> 2002
 *
 * 2002-2-2: added device detection when module is added - Hubbe
 */

#if 0
#define DPRINTK(format, args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format, args...)
#endif

#if 0
#define POLL_DPRINTK(format, args...) printk(KERN_INFO format,##args)
#else
#define POLL_DPRINTK(format, args...) 
#endif

#if 0
#define IRQ_DPRINTK(format, args...)  printk(KERN_INFO format,##args)
#else
#define IRQ_DPRINTK(format, args...) 
#endif

#if 0
#define SETUP_DPRINTK(format, args...) printk(KERN_DEBUG format,##args)
#else
#define SETUP_DPRINTK(format, args...)
#endif

#if 0
#define DCOLOR(X) ((*(unsigned long *)0xa05f8040)=(X))
#else
#define DCOLOR(X) 
#endif

#define MAPLE_PORTS 4
#define MAPLE_SCANHZ (HZ/100)
#define MAPLE_PNP_INTERVAL HZ

#define MAPLE_MAXPACKETS 8
#define MAPLE_DMA_ORDER	14	/* 16K bytes */
#define MAPLE_DMA_SIZE	(1<<MAPLE_DMA_ORDER)
#define MAPLE_DMA_PAGES	((MAPLE_DMA_ORDER > PAGE_SHIFT) ? MAPLE_DMA_ORDER-PAGE_SHIFT : 0)

/* Maple units:
 * 0 = primary
 * 1-5 = secondaries
 */
static int unitcodes[MAPLE_MAX_UNITS] = { 32, 1, 2, 4, 8, 16 };

static struct maple_port ports[MAPLE_PORTS];
static int maple_active_guns;
static int maple_gunport;
static int maple_active_gunport;

/*** Heleper functions  ***/

static void maple_detect_callback(struct mapleq *);

static int maple_dma_done(void)
{
	return (ctrl_inl(MAPLE_STATE) & 1) == 0;
}


#define MCP(P,U) do{							  \
	if( !(ports[(P)].dev[(U)])  != 					  \
	    !(ports[(P)].known_units & unitcodes[(U)]) ) {		  \
		unsigned long flags;					  \
		printk(KERN_INFO "MF: %d (%d,%d) u=%x\n",__LINE__,(P),(U),ports[(P)].known_units); \
		save_and_cli(flags);					  \
		while(1);						  \
	}								  \
}while(0)

#define CHECKALLP()  do{						\
	int P,U;							\
	for(P=0;P<4;P++) for(U=0;U<MAPLE_MAX_UNITS;U++) MCP(P,U);	\
}while(0)


/*** Low-level maple functions ***
 *
 * This section only handles data transfers and callbacks,
 * it does not handle any devices, auto-detection or
 * timing. See the next section for that.
 */

unsigned long *maple_sendbuf, *maple_sendptr;

static LIST_HEAD(maple_waitq);
static LIST_HEAD(maple_sentq);

void maple_init_mq(struct mapleq *mq)
{
	unsigned long buf;

	mq->done = 1;
	buf = (unsigned long)mq->buf;
	buf = (buf + 31) & ~31;
	mq->recvbuf = (void *)P2SEGADDR(buf);
}

static struct mapleq *maple_allocq(struct maple_device *dev)
{
	struct mapleq *mq;

	mq = kmalloc(sizeof(*mq), GFP_KERNEL);
	if (!mq)
		return NULL;

	maple_init_mq(mq);

	return mq;
}


static void maple_freeq(struct mapleq *mq)
{
	if (!mq)
		return;

	kfree(mq);
}

static unsigned long last_gun_data;

static void maple_dma_irq(int irq, void *dev, struct pt_regs *fp)
{
	struct mapleq *mq;
	struct list_head *lh, tmph;

	DCOLOR(0x0000ff);
	IRQ_DPRINTK("MAPLE DMA IRQ BEGIN\n");

	POLL_DPRINTK("Maple DMA finished %d,  %d.\n",
		     list_empty(&maple_sentq), !maple_dma_done());

	if (list_empty(&maple_sentq) || !maple_dma_done()) return;

#if 1
	if(maple_active_gunport >= 0) {
		int x,y;
		unsigned long tmp=*(unsigned long *)0xA05F80C4;
		if(last_gun_data != tmp) {
			y=tmp>>16;
			x=tmp&0xffff;
			POLL_DPRINTK("Maple gunport reply for port %d: "
				     "%04x %04x\n",
				     maple_active_gunport,
				     x,y);
			ports[maple_active_gunport].gunmode(
				ports[maple_active_gunport].gunmode_data,x,y);
			maple_active_gunport=-1;
		}
	}
#endif

	list_for_each(lh, &maple_sentq) {
		tmph = *lh;
		
		mq = (struct mapleq *)lh;
		mq->done=1;
		POLL_DPRINTK("Maple reply (%d,%d) cmd=%d => %d\n",
			     mq->port, mq->unit,
			     mq->command, mq->recvbuf[0]);

		if(mq->recvbuf[0] == MAPLE_RESPONSE_NONE)
			maple_detect_callback(mq);
		else if(mq->callback)
			mq->callback(mq);

		lh = &tmph;

	}
	
	INIT_LIST_HEAD(&maple_sentq);
	maple_send();

	IRQ_DPRINTK("MAPLE DMA IRQ END\n");
}




/* Call this after one or more calls to maple_add_packet */
/* FIXME: implement scheduling */
static void low_maple_send(int vblank)
{
	int i;
	int maple_packets;
	struct mapleq *mq;
	struct list_head *lh, tmph;
	unsigned long *maple_lastptr;
	unsigned long flags;

	/* Major-league dodgy */
	save_and_cli(flags);

	if (!list_empty(&maple_sentq) ||
	    (maple_active_gunport != -1 && !vblank) ) {
		restore_flags(flags);
		return;
	}

	if (list_empty(&maple_waitq) || !maple_dma_done()) {
		restore_flags(flags);
		return;
	}

	maple_packets = 0;
	maple_sendptr = maple_lastptr = maple_sendbuf;
	maple_active_gunport=-1;

	list_for_each(lh, &maple_waitq) {
		int port, unit, from, to, len;
		unsigned long *lsendbuf;

		mq = (struct mapleq *)lh;

		port = mq->port;
		unit = mq->unit;
		len = mq->length & 0xff;

		if(ports[port].dev[unit] &&
		   ports[port].dev[unit]->lock)
		{
			if(vblank) ports[port].dev[unit]->lock--;
			continue;
		}

		tmph = *lh;

		list_del(lh);
		list_add_tail(lh, &maple_sentq);

		lsendbuf = mq->sendbuf;

		POLL_DPRINTK("Maple sending (%d,%d) cmd=%d len=%d\n",
			     mq->port, mq->unit, mq->command, len);

		from = port<<6;
		to = (port<<6) | unitcodes[unit];

		maple_lastptr = maple_sendptr;

		*maple_sendptr++ = (port<<16) | len;
		*maple_sendptr++ = PHYSADDR(mq->recvbuf);
		*maple_sendptr++ = mq->command | (to<<8) | 
			(from<<16) | (len<<24);

#if 1
		while (len-->0) {
			*maple_sendptr++ = *lsendbuf++;
		}
#else
		if(len > 0)
		{
			memcpy((char *)maple_sendptr,
			       (char *)lsendbuf,
			       sizeof(*maple_sendptr) * (len - 1));

			maple_sendptr+=len;
		}
#endif
		if(mq->length & MAPLEQ_LENGTH_GUN) {
/* 			printk("Setting GUN MODE!\n"); */
			maple_lastptr = maple_sendptr;
			*maple_sendptr++ = (port<<16) | 0x200;
			maple_active_gunport = port;
			break;
		}

		lh = &tmph;
		if (maple_packets++ > MAPLE_MAXPACKETS)
			break;
	}

#if 0
	*maple_lastptr |= 0x80000000; /* Set 'last' bit */
#else
	switch(maple_active_guns)
	{
		case 0:
			maple_active_gunport=-1;
			*maple_lastptr |= 0x80000000; /* Set 'last' bit */
			break;

		default:
			do {
				if(++maple_gunport == MAPLE_PORTS)
					maple_gunport=0;
			} while( ! ports[maple_gunport].gunmode);

			/* FALL THROUGH */
		case 1:
			maple_active_gunport=maple_gunport;
			*maple_sendptr++ = (maple_gunport <<16) | 0x80000200;
			POLL_DPRINTK("Active gunport = %d\n",
				     maple_active_gunport);
	}
#endif

	if (maple_packets>0) {
		for (i=0; i<(1<<MAPLE_DMA_PAGES); i++)
			dma_cache_wback_inv(maple_sendbuf+i*PAGE_SIZE, PAGE_SIZE);
		POLL_DPRINTK("Maple, starting DMA packets=%d\n",maple_packets);
		ctrl_outl(1, MAPLE_STATE);
		DCOLOR(0x00ffff);
	}

	restore_flags(flags);

}

void maple_send(void)
{
	low_maple_send(0);
}

int maple_add_packet(struct mapleq *mq)
{
	int ret=0;
	unsigned long flags;

	save_and_cli(flags);
	if(mq->done) {
		mq->done=0;
		list_add_tail((struct list_head *)mq, &maple_waitq);
	}else{
		DPRINTK("Maple, rejecting package (to %d,%d, cmd=%d)\n",
			mq->port, mq->unit,
			mq->command);
		ret=-EINPROGRESS;
	}
	restore_flags(flags);

	return ret;
}

int maple_del_packet(struct mapleq *mq)
{
	struct list_head *lh;
	unsigned long flags;
	
	save_and_cli(flags);

	list_for_each(lh, &maple_sentq) {
		if (mq == (struct mapleq *)lh) {
			restore_flags(flags);
			return -EBUSY;
		}
	}
	
	list_for_each(lh, &maple_waitq) {
		if (mq == (struct mapleq *)lh)
			list_del(lh);
	}

	restore_flags(flags);

	return 0;
}

void maple_set_gunmode(int port, void (cb)(void *,int,int),void *data)
{
#if 1
	unsigned long flags;

	save_and_cli(flags);

	if( (!ports[port].gunmode) ^ (!cb) ) {
		maple_active_guns += (cb ? 1 : -1);
		if(cb) maple_gunport=port;
	}


	ports[port].gunmode = cb;
	ports[port].gunmode_data = data;


	restore_flags(flags);

	printk("Setting gunmode for port %d to %p(%p) (active guns = %d)\n",port,cb,data,maple_active_guns);
#endif
}

static int init_maple_low(void)
{
	int p, u;

	printk(KERN_INFO "SEGA Dreamcast MAPLE Bus drivers\n");

	/* Allocate DMA buffer */
	maple_sendbuf = (void *)__get_dma_pages(GFP_KERNEL, MAPLE_DMA_PAGES);
	if (maple_sendbuf == NULL)
		return -ENOMEM;
	memset(maple_sendbuf, 0, MAPLE_DMA_SIZE);

	for(p=0;p<MAPLE_PORTS;p++)
	{
		ports[p].port=p;
		ports[p].known_units=0;
		ports[p].units=0;
		ports[p].gunmode=0;
		
		for(u=0;u<MAPLE_MAX_UNITS;u++)
			ports[p].dev[u]=NULL;
	}
	CHECKALLP();

        /* Initialize hardware */
        ctrl_outl(MAPLE_MAGIC, MAPLE_RESET);
        ctrl_outl(0, MAPLE_RESET2);
        ctrl_outl(MAPLE_2MBPS|MAPLE_TIMEOUT(50000), MAPLE_SPEED);
	ctrl_outl(PHYSADDR(maple_sendbuf), MAPLE_DMAADDR);
        ctrl_outl(1, MAPLE_ENABLE);


	if (request_irq(HW_EVENT_MAPLE_DMA, maple_dma_irq, 0,
			"Maple BUS DMA", 0)) {
		DPRINTK("couldn't register DMA int\n");
		goto cleanup;
	}
	return 0;

 cleanup:
 	if (maple_sendbuf)
		free_pages((unsigned long)maple_sendbuf, MAPLE_DMA_PAGES);

	printk(KERN_INFO "maple: Register failed\n");

	return -EINVAL;
}

static void exit_maple_low(void)
{
    free_irq(HW_EVENT_MAPLE_DMA, 0);

    /* twiddle */
    while(!maple_dma_done());

    if (maple_sendbuf)
	    free_pages((unsigned long)maple_sendbuf, MAPLE_DMA_PAGES);
}



/*** Maple Device interface ***
 *
 * This section provides the interface for individual
 * drivers and handles device state polling and auto-detection.
 */

static LIST_HEAD(maple_data_list);
static LIST_HEAD(maple_driver_list);
static DECLARE_WAIT_QUEUE_HEAD(kmapled_wait);
static DECLARE_COMPLETION(kmapled_exited);
static int kmapled_pid = 0;


static int maple_detector_counter;
static struct mapleq maple_detector;
static int detector_new_dev;
static int detector_new_driver;


static void maple_dump_devinfo(struct maple_devinfo *devinfo)
{
	DPRINTK("  function:     0x%08x\n", be32_to_cpu(devinfo->function));
	DPRINTK("   data[0]:     0x%08x\n", be32_to_cpu(devinfo->function_data[0]));
	DPRINTK("   data[1]:     0x%08x\n", be32_to_cpu(devinfo->function_data[1]));
	DPRINTK("   data[2]:     0x%08x\n", be32_to_cpu(devinfo->function_data[2]));
	DPRINTK("  area code:    %d\n", devinfo->area_code);
	DPRINTK("  direction:    %d\n", devinfo->connector_direction);
	DPRINTK("  name:         \"%-30.30s\"\n", devinfo->product_name);
	DPRINTK("  license:      \"%-60.60s\"\n", devinfo->product_license);
	DPRINTK("  stanby power: %d\n", le16_to_cpu(devinfo->standby_power));
	DPRINTK("  max power:    %d\n",  le16_to_cpu(devinfo->max_power));
}


void maple_register_driver(struct maple_driver *driver)
{
	struct list_head *lh = (void *)driver;
	unsigned long flags;
	
	/* Atomic */
	save_and_cli(flags);
	list_add(lh, &maple_driver_list);
	detector_new_driver=1;
	restore_flags(flags);

	wake_up(&kmapled_wait);

	MOD_INC_USE_COUNT;

	printk(KERN_INFO "maple: registered driver: %s (function 0x%lx)\n",
	       driver->name, driver->function);
}


void maple_unregister_driver(struct maple_driver *driver)
{
	struct list_head *lh = (void *)driver;
	list_del(lh);

	MOD_DEC_USE_COUNT;

	printk(KERN_INFO "maple: unregistered driver: %s (function 0x%lx)\n",
	       driver->name, driver->function);
}


static struct maple_device *maple_alloc_dev(int port, int unit)
{
	unsigned long flags;
	struct maple_device *dev;

	SETUP_DPRINTK("Maple, Allocating unit %d, %d\n",port, unit);

	dev = kmalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev)
		return NULL;

	memset(dev, 0, sizeof(*dev));

	dev->port = port;
	dev->unit = unit;
	dev->lock = 0;

	CHECKALLP();

	save_and_cli(flags);
	ports[port].dev[unit]=dev;
	ports[port].known_units |= unitcodes[unit];
	restore_flags(flags);

	CHECKALLP();

	return dev;
}


void maple_getcond_vblank_callback(struct maple_driver_data *data)
{
	if( data->mq.done )
	{
		data->mq.command = MAPLE_COMMAND_GETCOND;
		data->mq.length = 1;
	        ((unsigned long *)data->mq.recvbuf)[0] =
			cpu_to_be32(data->driver->function);
		data->mq.sendbuf = data->mq.recvbuf;

		POLL_DPRINTK("queueing GETCOND for %d,%d,%lx (%s)\n",
			     data->mq.port,
			     data->mq.unit,
			     data->driver->function,
			     data->driver->name);
		
		maple_add_packet(& data->mq );
	}
}

static void maple_driver_cmd_callback(struct mapleq *mq)
{
	struct maple_driver_data *data=(struct maple_driver_data *)mq->mq_privdata;
	if(data->driver->reply)
		data->driver->reply(data);
}

static int maple_connect(struct maple_device *dev,
			 struct maple_driver *driver)
{
	unsigned long flags, func;
	int tmp;
	struct maple_driver_data *data;
	data = kmalloc(sizeof(*data), GFP_KERNEL);

	if(!data) return -ENOMEM;
	data->driver=driver;
	data->dev=dev;
	data->private_data=0;

	/* Init mq as much as possible */
	maple_init_mq(& data->mq );
	data->mq.mq_privdata=data;
	data->mq.port=dev->port;
	data->mq.unit=dev->unit;
	data->mq.callback=maple_driver_cmd_callback;
	
	for(func=data->dev->function, tmp=0;;tmp++) {
		func &= func-1;
		if(data->driver->function > func)
			break;
	}
	data->function_data=be32_to_cpu(dev->devinfo.function_data[tmp]);
	SETUP_DPRINTK("Function data: %lx\n",data->function_data);

	if(! driver->connect(data)) {
		save_and_cli(flags);
		list_add((struct list_head *)data, &maple_data_list);
		restore_flags(flags);
		return 0;
	}

	kfree( data );

	CHECKALLP();

	return 1;
}

static void maple_attach_driver_functions(int port,
					  int unit,
					  unsigned long function)
{
	struct list_head *lh;
	struct maple_driver *driver;
	unsigned long bits_left;
	struct maple_device *dev=ports[port].dev[unit];
	if(!dev) return;

	printk(KERN_INFO "maple(%d,%d): Connected(function 0x%lx)\n",
	       dev->port, dev->unit, function);

	/* First, try all functions of this device */
	for(bits_left=function;bits_left;) {
		int best_bits=0;

		/* Find out the maximum number of functions we can get
		 * from one driver.
		 */
		list_for_each(lh, &maple_driver_list) {
			unsigned long bits, tmp;

			driver = (struct maple_driver *)lh;
			if(driver->function & ~bits_left) continue;
			tmp=bits_left & driver->function;
			bits=0;
			while(tmp) {
				tmp = tmp & (tmp -1);
				bits++;
			}
			if(bits > best_bits)
				best_bits=bits;
		}
		
		/* Now, first try the best drivers,
		 * then the second best, etc.
		 */
		for(;best_bits>0;best_bits--) {
			list_for_each(lh, &maple_driver_list) {
				unsigned long bits, tmp;

				driver = (struct maple_driver *)lh;
				if(driver->function & ~bits_left) continue;
				tmp=bits_left & driver->function;
				bits=0;
				while(tmp) {
					tmp = tmp & (tmp -1);
					bits++;
				}
				if(bits == best_bits &&
				   !maple_connect(dev,driver))	{
					bits_left=bits_left &
						~driver->function;
					goto check_more_bits;
					
				}
			}
		}

		printk(KERN_INFO "maple(%d,%d): No driver for function(s): %lx.\n",
		       dev->port, dev->unit, bits_left);
		maple_dump_devinfo(&dev->devinfo);
		break;

	check_more_bits: ;
	}

	CHECKALLP();
}

static void maple_attach_driver(int port, int unit)
{
	unsigned long function;
	char *p, *recvbuf = maple_detector.recvbuf;

	struct maple_device *dev=maple_alloc_dev(port, unit);

	memcpy(&dev->devinfo, recvbuf+4, sizeof(dev->devinfo));

	memcpy(dev->product_name, dev->devinfo.product_name, 30);
	memcpy(dev->product_license, dev->devinfo.product_license, 60);
	dev->product_name[30] = 0;
	dev->product_license[60] = 0;

	for (p=dev->product_name+29; dev->product_name<=p; p--)
		if (*p==' ') *p = 0; else break;
	
	for (p=dev->product_license+59; dev->product_license<=p; p--)
		if (*p==' ') *p = 0; else break;

	function = be32_to_cpu(dev->devinfo.function);
	dev->function=function;

	maple_attach_driver_functions(port,unit,function);
}


static void maple_detach_driver(int port, int unit)
{
	unsigned long flags;

	struct list_head *lh, tmph;
	struct maple_device *dev=ports[port].dev[unit];

	if(!dev)
		return;

	printk(KERN_INFO "maple(%d,%d): Disconnected (%d,%d)\n",
	       dev->port, dev->unit, port, unit);

	list_for_each(lh, &maple_data_list) {
		struct maple_driver_data *data;
		data=(struct maple_driver_data *)lh;
		if(data->dev == dev) {
			save_and_cli(flags);
			tmph=*lh;
			list_del(lh);
			restore_flags(flags);
			
			data->driver->disconnect(data);

			while( maple_del_packet( & data->mq ) < 0)
				/* yield() */;

			kfree(lh);
			lh=&tmph;
		}
	}
	
	save_and_cli(flags);
	ports[port].dev[unit]=0;
	ports[port].known_units &=~ unitcodes[unit];
	restore_flags(flags);

	CHECKALLP();

	kfree(dev);
}


static void maple_new_driver(void) {
	int u, p;
	struct list_head *lh;
	unsigned long all_functions = 0;
	
	list_for_each(lh, &maple_driver_list) {
		struct maple_driver *driver = (struct maple_driver *)lh;
		all_functions |= driver->function;
	}

	for(p=0;p<MAPLE_PORTS;p++) {
		for(u=0;u<MAPLE_MAX_UNITS;u++) {
			unsigned long f;

			if(!ports[p].dev[u]) continue;
				
			f=ports[p].dev[u]->function & all_functions;

			list_for_each(lh, &maple_data_list) {
				struct maple_driver_data *data;
				data=(struct maple_driver_data *)lh;
				if(ports[p].dev[u] != data->dev)
					continue;

				f &=~ data->driver->function;
			}

			if(f) maple_attach_driver_functions(p,u,f);
		}
	}
}


/* kmapled */
static int kmapled_thread(void *hoge)
{
	lock_kernel();

	daemonize();

	strcpy(current->comm, "kmapled");

	do {
		/*Yawn*/
		/* Time to see if there are any new devices */

		int p, u;

		SETUP_DPRINTK("kmapled: *yawn*\n"); 
		CHECKALLP();

		/* maple_detector contains devinfo for a new device ? */
		if(detector_new_dev) {
			maple_attach_driver(maple_detector.port,
					    maple_detector.unit);
			detector_new_dev=0;
			maple_detector.done=1; /* allow more autodetects */
		}
		
		for(p=0;p<MAPLE_PORTS;p++) {
			if(!(ports[p].known_units & ~ports[p].units))
				continue;

			SETUP_DPRINTK("Maple, port[%d] units=%x  known_units=%x\n",
				p,
				ports[p].units,
				ports[p].known_units);

			for(u=0;u<MAPLE_MAX_UNITS;u++) {
				if(unitcodes[u] & 
				   ports[p].known_units &
				   ~ports[p].units) {
					maple_detach_driver(p,u);
				}
			}
		}

		if(detector_new_driver) {
			detector_new_driver=0;
			maple_new_driver();
		}

		CHECKALLP();

		/* Wait for another event */
		interruptible_sleep_on(&kmapled_wait);
	} while(!signal_pending(current));
	
	unlock_kernel();

	complete_and_exit(&kmapled_exited, 0);
}

/*
 * If there is an unconfigure device on this port,
 * send a DEVINFO request to it.
 */
static int maple_detect_resend(void)
{
	int u,p;
	if(! maple_detector.done ) return 0;

	CHECKALLP();

	p=maple_detector.port;
	if(ports[p].units & ~ports[p].known_units) {
		for(u=0;u<MAPLE_MAX_UNITS;u++) {
			if(unitcodes[u] &
			   ports[p].units &
			   ~ports[p].known_units) {
				SETUP_DPRINTK("Detector resender, sending devinfo to unit 0 on port %d\n",p);
				maple_detector.command=MAPLE_COMMAND_DEVINFO;
				maple_detector.port = p;
				maple_detector.unit = u;
				maple_detector.length = 0;
				maple_add_packet(& maple_detector);
				return 1;
			}
		}
	}
	return 0;
}


/*
 * Autodetect DMA callback.
 * 1) Wake up kmapled when we receive a MAPLE_RESPONSE_DEVINFO
 *    for an unconfigured device.
 * 2) Wake up kmapled if devices have been removed.
 */
static void maple_detect_callback(struct mapleq *mq)
{
	int p=maple_detector.port;
	int u=maple_detector.unit;

/* 	SETUP_DPRINTK("Detector callback.\n"); */

	CHECKALLP();

	switch(mq->recvbuf[0]) {
		case MAPLE_RESPONSE_DEVINFO:
			SETUP_DPRINTK("Detector reply DEVINFO (%d, %d), units = %x.\n",p,u,mq->recvbuf[2] & 0x3f);
			if(!u)
				ports[p].units = mq->recvbuf[2] & 0x3f;
			else
				ports[p].units |= unitcodes[u];
				
			if(!ports[p].dev[u]) {
				maple_detector.done=0;
				detector_new_dev=1;
				SETUP_DPRINTK("Kmapled, wakie wakie (new device) ... ");
				wake_up(&kmapled_wait);
				return;
			}
			break;

		case MAPLE_RESPONSE_NONE:
			SETUP_DPRINTK("Detector reply NONE (%d,%d)\n",p,u);
			if(!u)
				ports[p].units = 0;
			else
				ports[p].units &=~ unitcodes[u];
			break;
	}
	maple_detect_resend();

	/* Devices are gone */
	if(ports[p].known_units & ~ports[p].units) {
		SETUP_DPRINTK("Kmapled, wakie wakie (device gone) ... ");
		wake_up(&kmapled_wait);
	}
}

/* 
 * Autodetect vertical blank callback
 * Check if there are subdevices to configure.
 * If not, check if it is time to send a COMMAND_DEVINFO
 * to a primary device on a port.
 */
static void maple_detect_vblank(void)
{
	int p;

	CHECKALLP();

	if(! maple_detector.done ) return;
	if( maple_detect_resend() ) return;

	maple_detector_counter++;
	if(maple_detector_counter & 0xf) return;
	p = (maple_detector_counter >> 4 ) & 3;

	SETUP_DPRINTK("Detector, sending devinfo to unit 0 on port %d\n",p);
	
	maple_detector.command=MAPLE_COMMAND_DEVINFO;
	maple_detector.port = p;
	maple_detector.unit = 0;
	maple_detector.length = 0;
	maple_detector.callback = maple_detect_callback;
	maple_add_packet(& maple_detector);
}

static void maple_vblank_interupt(int irq, void *dev, struct pt_regs *fp)
{
	struct list_head *lh;
	struct maple_driver_data *data;

	IRQ_DPRINTK("MAPLE VBLANK IRQ BEGIN\n");
	DCOLOR(0xff00ff);

	/* Dodgyness, damn maple! */
	if(maple_active_gunport >=0) maple_dma_irq(0,0,0);


	maple_detect_vblank();


	/* Do task for each device */
	list_for_each (lh, &maple_data_list) {
		data = (struct maple_driver_data *)lh;
		if(data->driver->vblank)
			data->driver->vblank(data);
	}

	/* Scan list and build sending block */
	low_maple_send(1);

	IRQ_DPRINTK("MAPLE VBLANK IRQ END\n");
}


static int __init maple_init(void)
{
	int i;

	if((i=init_maple_low()))
		return i;

	maple_init_mq(& maple_detector );

	/* Start kernel thread */
	kmapled_pid = kernel_thread(kmapled_thread, NULL,
				    CLONE_FS|CLONE_FILES|CLONE_SIGHAND);
	if (kmapled_pid==0)
		goto cleanup;

	if (request_irq(HW_EVENT_VSYNC, maple_vblank_interupt, SA_SHIRQ,
			"Maple BUS", 0)) {
		DPRINTK("couldn't register VBL int\n");
		goto cleanup;
	}

        return 0;

cleanup:
 	if (kmapled_pid) {
		kill_proc(kmapled_pid, SIGTERM, 1);
		wait_for_completion(&kmapled_exited);
	}
	
	exit_maple_low();

	printk(KERN_INFO "maple: Register failed\n");

	return -EINVAL;
}


static void __exit maple_exit(void)
{
	/* XXX: Must do proper clean-up */

	free_irq(HW_EVENT_VSYNC, 0);

	kill_proc(kmapled_pid, SIGTERM, 1);
	wait_for_completion(&kmapled_exited);

	exit_maple_low();
}


module_init(maple_init);
module_exit(maple_exit);

MODULE_LICENSE("GPL");

EXPORT_SYMBOL(maple_init_mq);
EXPORT_SYMBOL(maple_allocq);
EXPORT_SYMBOL(maple_freeq);
EXPORT_SYMBOL(maple_add_packet);
EXPORT_SYMBOL(maple_del_packet);
EXPORT_SYMBOL(maple_send);
EXPORT_SYMBOL(maple_set_gunmode);

EXPORT_SYMBOL(maple_register_driver);
EXPORT_SYMBOL(maple_unregister_driver);
EXPORT_SYMBOL(maple_getcond_vblank_callback);

/*
 * Local variables:
 * c-basic-offset: 8
 * End:
 */
