/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "preferences.h"

#include "cache_maint.h"
#include "editors.h"
#include "filelist.h"
#include "img-view.h"
#include "layout_config.h"
#include "layout_util.h"
#include "utilops.h"
#include "ui_fileops.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"

#include "icons/config.xpm"


#define EDITOR_NAME_MAX_LENGTH 32
#define EDITOR_COMMAND_MAX_LENGTH 1024


typedef struct _ThumbSize ThumbSize;
struct _ThumbSize
{
	gint w;
	gint h;
};

static ThumbSize thumb_size_list[] =
{
	{ 24, 24 },
	{ 32, 32 },
	{ 48, 48 }, 
	{ 64, 64 }, 
	{ 85, 64 }, 
	{ 85, 85 },
	{ 100, 100 }, 
	{ 125, 125 }, 
	{ 150, 150 }, 
	{ 175, 175 }, 
	{ 200, 200 }, 
	{ 256, 256 }, 
	{ 0, 0 }
};

enum {
	FE_ENABLE,
	FE_EXTENSION,
	FE_DESCRIPTION
};

/* config memory values */
static gint startup_path_enable_c;
static gint confirm_delete_c;
static gint enable_delete_key_c;
static gint safe_delete_enable_c;
static gint safe_delete_size_c;
static gint restore_tool_c;
static gint save_window_positions_c;
static gint zoom_mode_c;
static gint two_pass_zoom_c;
static gint fit_window_c;
static gint limit_window_size_c;
static gint zoom_to_fit_expands_c;
static gint max_window_size_c;
static gint progressive_key_scrolling_c;
static gint thumb_max_width_c;
static gint thumb_max_height_c;
static gint enable_thumb_caching_c;
static gint enable_thumb_dirs_c;
static gint use_xvpics_thumbnails_c;
static gint show_dot_files_c;
static gint file_filter_disable_c;
static gint file_sort_case_sensitive_c;
static gint slideshow_delay_c;
static gint slideshow_random_c;
static gint slideshow_repeat_c;

static gint mousewheel_scrolls_c;

static gint enable_in_place_rename_c;

static gint collection_rectangular_selection_c;

static gint tile_cache_max_c;

static gint thumbnail_quality_c;
static gint zoom_quality_c;
static gint dither_quality_c;

static gint zoom_increment_c;

static gint enable_read_ahead_c;

#if 0
static gint place_dialogs_under_mouse_c;
#endif

static gint black_window_background_c;

static gint dupe_custom_threshold_c;

static GtkWidget *configwindow = NULL;
static GtkWidget *startup_path_entry;
static GtkListStore *filter_store = NULL;
static GtkWidget *editor_name_entry[GQVIEW_EDITOR_SLOTS];
static GtkWidget *editor_command_entry[GQVIEW_EDITOR_SLOTS];

static GtkWidget *layout_widget;

static GtkWidget *safe_delete_path_entry;

static gint tree_descend_subdirs_c;

static gint update_on_time_change_c;
static gint exif_rotate_enable_c;


/*
 *-----------------------------------------------------------------------------
 * option widget callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void startup_path_set_current(GtkWidget *widget, gpointer data)
{
	gtk_entry_set_text(GTK_ENTRY(startup_path_entry), layout_get_path(NULL));
}

static void zoom_mode_original_cb(GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		zoom_mode_c = ZOOM_RESET_ORIGINAL;
}

static void zoom_mode_fit_cb(GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		zoom_mode_c = ZOOM_RESET_FIT_WINDOW;
}

static void zoom_mode_none_cb(GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		zoom_mode_c = ZOOM_RESET_NONE;
}

static void zoom_increment_cb(GtkObject *adj, gpointer data)
{
	zoom_increment_c = (gint)(GTK_ADJUSTMENT(adj)->value * 10.0 + 0.01);
}

static void dummy_cancel_cb(GenericDialog *gd, gpointer data)
{
	/* no op, only so cancel button appears */
}

static void cache_clear_ok_cb(GenericDialog *gd, gpointer data)
{
	cache_maintain_home(TRUE);
}

static void cache_clear_cb(GtkWidget *widget, gpointer data)
{
	GenericDialog *gd;

	gd = generic_dialog_new(_("Clear cache"), _("This will remove all thumbnails that have\nbeen saved to disk, continue?"),
				"GQview", "clear_cache", TRUE, dummy_cancel_cb, NULL);
	generic_dialog_add_stock(gd, _("Ok"), GTK_STOCK_OK, cache_clear_ok_cb, TRUE);

	gtk_widget_show(gd->dialog);
}

/*
 *-----------------------------------------------------------------------------
 * sync progam to config window routine (private)
 *-----------------------------------------------------------------------------
 */ 

static void config_window_apply(void)
{
	const gchar *buf;
	gchar *l_conf;
	gint new_style;
	gint i;
	gint refresh = FALSE;

	for(i = 0; i < GQVIEW_EDITOR_SLOTS; i++)
		{
		g_free(editor_name[i]);
		editor_name[i] = NULL;
		buf = gtk_entry_get_text(GTK_ENTRY(editor_name_entry[i]));
		if (buf && strlen(buf) > 0) editor_name[i] = g_strdup(buf);

		g_free(editor_command[i]);
		editor_command[i] = NULL;
		buf = gtk_entry_get_text(GTK_ENTRY(editor_command_entry[i]));
		if (buf && strlen(buf) > 0) editor_command[i] = g_strdup(buf);
		}
	layout_edit_update_all();

	g_free(startup_path);
	startup_path = NULL;
	buf = gtk_entry_get_text(GTK_ENTRY(startup_path_entry));
	if (buf && strlen(buf) > 0) startup_path = remove_trailing_slash(buf);

	g_free(safe_delete_path);
	safe_delete_path = NULL;
	buf = gtk_entry_get_text(GTK_ENTRY(safe_delete_path_entry));
	if (buf && strlen(buf) > 0) safe_delete_path = remove_trailing_slash(buf);

	if (show_dot_files != show_dot_files_c) refresh = TRUE;
	if (file_sort_case_sensitive != file_sort_case_sensitive_c) refresh = TRUE;
	if (file_filter_disable != file_filter_disable_c) refresh = TRUE;

	startup_path_enable = startup_path_enable_c;
	confirm_delete = confirm_delete_c;
	enable_delete_key = enable_delete_key_c;
	safe_delete_enable = safe_delete_enable_c;
	safe_delete_size = safe_delete_size_c;
	restore_tool = restore_tool_c;
	save_window_positions = save_window_positions_c;
	zoom_mode = zoom_mode_c;
	two_pass_zoom = two_pass_zoom_c;
	fit_window = fit_window_c;
	limit_window_size = limit_window_size_c;
	zoom_to_fit_expands = zoom_to_fit_expands_c;
	max_window_size = max_window_size_c;
	progressive_key_scrolling = progressive_key_scrolling_c;
	thumb_max_width = thumb_max_width_c;
	thumb_max_height = thumb_max_height_c;
	enable_thumb_caching = enable_thumb_caching_c;
	enable_thumb_dirs = enable_thumb_dirs_c;
	use_xvpics_thumbnails = use_xvpics_thumbnails_c;
	show_dot_files = show_dot_files_c;
	file_sort_case_sensitive = file_sort_case_sensitive_c;
	file_filter_disable = file_filter_disable_c;

	slideshow_random = slideshow_random_c;
	slideshow_repeat = slideshow_repeat_c;
	slideshow_delay = slideshow_delay_c;

	mousewheel_scrolls = mousewheel_scrolls_c;

	enable_in_place_rename = enable_in_place_rename_c;

	collection_rectangular_selection = collection_rectangular_selection_c;

	tile_cache_max = tile_cache_max_c;

	thumbnail_quality = thumbnail_quality_c;
	zoom_quality = zoom_quality_c;
	dither_quality = dither_quality_c;

	zoom_increment = zoom_increment_c;

	enable_read_ahead = enable_read_ahead_c;

#if 0
	place_dialogs_under_mouse = place_dialogs_under_mouse_c;
#endif

	if (black_window_background != black_window_background_c)
		{
		black_window_background = black_window_background_c;

		layout_colors_update();
		view_window_colors_update();
		}

	update_on_time_change = update_on_time_change_c;
	exif_rotate_enable = exif_rotate_enable_c;

	dupe_custom_threshold = dupe_custom_threshold_c;

	tree_descend_subdirs = tree_descend_subdirs_c;

	l_conf = layout_config_get(layout_widget, &new_style);

	if (new_style != layout_style ||
	    (l_conf == NULL) != (layout_order == NULL) ||
	    !layout_order ||
	    strcmp(buf, layout_order) != 0)
		{
		if (refresh) filter_rebuild();
		refresh = FALSE;

		g_free(layout_order);
		layout_order = l_conf;
		l_conf = NULL;

		layout_style = new_style;

		layout_styles_update();
		}

	g_free(l_conf);

	if (refresh)
		{
		filter_rebuild();
		layout_refresh(NULL);
		}
}

/*
 *-----------------------------------------------------------------------------
 * config window main button callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void config_window_close_cb(GtkWidget *widget, gpointer data)
{
	gtk_widget_destroy(configwindow);
	configwindow = NULL;
	filter_store = NULL;
}

static gint config_window_delete(GtkWidget *w, GdkEventAny *event, gpointer data)
{
	config_window_close_cb(NULL, NULL);
	return TRUE;
}

static void config_window_ok_cb(GtkWidget *widget, gpointer data)
{
	config_window_apply();
	config_window_close_cb(NULL, NULL);
}

static void config_window_apply_cb(GtkWidget *widget, gpointer data)
{
	config_window_apply();
}

/*
 *-----------------------------------------------------------------------------
 * config window setup (private)
 *-----------------------------------------------------------------------------
 */ 

static void check_button_cb(GtkWidget *widget, gpointer data)
{
	gint *value_ptr = data;
	*value_ptr = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
}

static GtkWidget *add_check_button(gint option, gint *option_c, gchar *text, GtkWidget *box)
{
	GtkWidget *button;

	*option_c = option;
	button = gtk_check_button_new_with_label (text);
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), option);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(check_button_cb), option_c);
	gtk_widget_show(button);

	return button;
}

static void check_button_sensitivity_cb(GtkWidget *button, gpointer data)
{
	GtkWidget *widget = data;

	gtk_widget_set_sensitive(widget, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
}

static void add_check_button_sensitivity(GtkWidget *ct_button, GtkWidget *button)
{
	g_signal_connect(G_OBJECT(ct_button), "toggled",
			 G_CALLBACK(check_button_sensitivity_cb), button);
	check_button_sensitivity_cb(ct_button, button);
}

static void spin_button_cb(GtkObject *adj, gpointer data)
{
	gint *value_ptr = data;
	*value_ptr = (gint)GTK_ADJUSTMENT(adj)->value;
}

static GtkWidget *add_spin_button(const gchar *text, gint value, gint *value_c,
				  gfloat min, gfloat max, gfloat increment, guint digits, GtkWidget *box)
{
	GtkWidget *hbox = NULL;
	GtkObject *adj;
	GtkWidget *spin;

	if (text)
		{
		GtkWidget *label;

		hbox = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
		gtk_widget_show(hbox);

		label = gtk_label_new(text);
		gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
		gtk_widget_show(label);

		box = hbox;
		}

	*value_c = value;
	adj = gtk_adjustment_new((gfloat)value, min, max, increment, increment, increment);
        spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, digits);
        gtk_box_pack_start(GTK_BOX(box), spin, FALSE, FALSE, 5);
        g_signal_connect(G_OBJECT(adj), "value_changed",
			 G_CALLBACK(spin_button_cb), value_c);
        gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
        gtk_widget_show(spin);

	return (hbox) ? hbox : spin;
}

static void quality_menu_cb(GtkWidget *widget, gpointer data)
{
	gint *option_c = data;

	*option_c = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "value"));
}

static void add_menu_item(GtkWidget *menu, gchar *text, gint *option_c, gint type)
{
	GtkWidget *item;

	item = gtk_menu_item_new_with_label(text);
	g_object_set_data(G_OBJECT(item), "value", GINT_TO_POINTER(type));
	g_signal_connect(G_OBJECT(item), "activate",
			 G_CALLBACK(quality_menu_cb), option_c);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	gtk_widget_show(item);
}

static void add_quality_menu(gint option, gint *option_c, gchar *text, GtkWidget *box, gint allow_tiles)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *omenu;
	GtkWidget *menu;

	*option_c = option;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	add_menu_item(menu, _("Nearest (worst, but fastest)"), option_c, (gint)GDK_INTERP_NEAREST);
	if (allow_tiles) add_menu_item(menu, _("Tiles"), option_c, (gint)GDK_INTERP_TILES);
	add_menu_item(menu, _("Bilinear"), option_c, (gint)GDK_INTERP_BILINEAR);
	add_menu_item(menu, _("Hyper (best, but slowest)"), option_c, (gint)GDK_INTERP_HYPER);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu),
			(!allow_tiles && *option_c > (gint)GDK_INTERP_TILES) ? *option_c - 1 : *option_c);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);
}

static void add_dither_menu(gint option, gint *option_c, gchar *text, GtkWidget *box)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *omenu;
	GtkWidget *menu;

	*option_c = option;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	add_menu_item(menu, _("None"), option_c, (gint)GDK_RGB_DITHER_NONE);
	add_menu_item(menu, _("Normal"), option_c, (gint)GDK_RGB_DITHER_NORMAL);
	add_menu_item(menu, _("Best"), option_c, (gint)GDK_RGB_DITHER_MAX);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), *option_c);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);
}

static void thumb_size_menu_cb(GtkWidget *widget, gpointer data)
{
	gint n;

	n = GPOINTER_TO_INT(data);

	if (n >= 0 && n < sizeof(thumb_size_list) / sizeof(ThumbSize))
		{
		thumb_max_width_c = thumb_size_list[n].w;
		thumb_max_height_c = thumb_size_list[n].h;
		}
}

static void add_thumb_size_menu(gchar *text, GtkWidget *box)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *omenu;
	GtkWidget *menu;
	gint row;
	gint i;
	gint c;

	thumb_max_width_c = thumb_max_width;
	thumb_max_height_c = thumb_max_height;

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(text);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	row = 0;
	i = 0;
	c = TRUE;
	while (c)
		{
		gint w, h;

		w = thumb_size_list[i].w;
		h = thumb_size_list[i].h;

		if ( w > 0 && h > 0)
			{
			GtkWidget *item;
			gchar *buf;

			buf = g_strdup_printf("%d x %d", w, h);
			item = gtk_menu_item_new_with_label(buf);
			g_free(buf);

			g_signal_connect(G_OBJECT(item), "activate",
					 G_CALLBACK(thumb_size_menu_cb), GINT_TO_POINTER(i));
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
			gtk_widget_show(item);
			if (thumb_max_width >= w && thumb_max_width >= h) row = i;
			}
		else
			{
			c = FALSE;
			}
		i++;
		}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), row);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);
}

static void filter_store_populate(void)
{
	GList *work;

	if (!filter_store) return;

	gtk_list_store_clear(filter_store);

	work = filter_get_list();
	while (work)
		{
		FilterEntry *fe;
		GtkTreeIter iter;

		fe = work->data;
		work = work->next;

		gtk_list_store_append(filter_store, &iter);
		gtk_list_store_set(filter_store, &iter, 0, fe, -1);
		}
}

static void filter_store_ext_edit_cb(GtkCellRendererText *cell, gchar *path_str,
				     gchar *new_text, gpointer data)
{
	GtkWidget *model = data;
	FilterEntry *fe = data;
	GtkTreePath *tpath;
	GtkTreeIter iter;

	if (!new_text || strlen(new_text) < 1) return;

	tpath = gtk_tree_path_new_from_string(path_str);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, tpath);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 0, &fe, -1);

	g_free(fe->extensions);
	fe->extensions = g_strdup(new_text);

	gtk_tree_path_free(tpath);
	filter_rebuild();
}

static void filter_store_desc_edit_cb(GtkCellRendererText *cell, gchar *path_str,
				      gchar *new_text, gpointer data)
{
	GtkWidget *model = data;
	FilterEntry *fe;
	GtkTreePath *tpath;
	GtkTreeIter iter;

	if (!new_text || strlen(new_text) < 1) return;

	tpath = gtk_tree_path_new_from_string(path_str);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, tpath);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 0, &fe, -1);

	g_free(fe->description);
	fe->description = g_strdup(new_text);

	gtk_tree_path_free(tpath);
}

static void filter_store_enable_cb(GtkCellRendererToggle *renderer,
				   gchar *path_str, gpointer data)
{
	GtkWidget *model = data;
	FilterEntry *fe;
	GtkTreePath *tpath;
	GtkTreeIter iter;

	tpath = gtk_tree_path_new_from_string(path_str);
	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, tpath);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 0, &fe, -1);

	fe->enabled = !fe->enabled;

	gtk_tree_path_free(tpath);
	filter_rebuild();
}

static void filter_set_func(GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
			    GtkTreeModel *tree_model, GtkTreeIter *iter, gpointer data)
{
	FilterEntry *fe;

	gtk_tree_model_get(tree_model, iter, 0, &fe, -1);

	switch(GPOINTER_TO_INT(data))
		{
		case FE_ENABLE:
			g_object_set(GTK_CELL_RENDERER(cell),
				     "active", fe->enabled, NULL);
			break;
		case FE_EXTENSION:
			g_object_set(GTK_CELL_RENDERER(cell),
				     "text", fe->extensions, NULL);
			break;
		case FE_DESCRIPTION:
			g_object_set(GTK_CELL_RENDERER(cell),
				     "text", fe->description, NULL);
			break;
		}
}

static void filter_add_cb(GtkWidget *widget, gpointer data)
{
	filter_add_unique("description", ".new", TRUE);
	filter_store_populate();

	/* FIXME: implement the scroll to/select row stuff for tree view */
}

static void filter_remove_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *filter_view = data;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	FilterEntry *fe;

	if (!filter_store) return;
       	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(filter_view));
	if (!gtk_tree_selection_get_selected(selection, NULL, &iter)) return;
	gtk_tree_model_get(GTK_TREE_MODEL(filter_store), &iter, 0, &fe, -1);
	if (!fe) return;

	filter_remove_entry(fe);
	filter_rebuild();
	filter_store_populate();
}

static void filter_default_ok_cb(GenericDialog *gd, gpointer data)
{
	filter_reset();
	filter_add_defaults();
	filter_rebuild();
	filter_store_populate();
}

static void filter_default_cb(GtkWidget *widget, gpointer data)
{
	GenericDialog *gd;

	gd = generic_dialog_new(_("Reset filters"), _("This will reset the file filters to the defaults.\nContinue?"),
				"GQview", "reset_filter", TRUE, dummy_cancel_cb, NULL);
	generic_dialog_add_stock(gd, _("Ok"), GTK_STOCK_OK, filter_default_ok_cb, TRUE);
	gtk_widget_show(gd->dialog);
}

static void filter_disable_cb(GtkWidget* widget, gpointer data)
{
	GtkWidget *frame = data;

	gtk_widget_set_sensitive(frame,
				 !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)));
}

static void editor_default_ok_cb(GenericDialog *gd, gpointer data)
{
	gint i;

	editor_reset_defaults();
	if (!configwindow) return;

	for (i = 0; i < GQVIEW_EDITOR_SLOTS; i++)
		{
		gtk_entry_set_text(GTK_ENTRY(editor_name_entry[i]),
				   (editor_name[i]) ? editor_name[i] : "");
		gtk_entry_set_text(GTK_ENTRY(editor_command_entry[i]),
				   (editor_command[i]) ? editor_command[i] : "");
		}
}

static void editor_default_cb(GtkWidget *widget, gpointer data)
{
	GenericDialog *gd;

	gd = generic_dialog_new(_("Reset editors"), _("This will reset the edit commands to the defaults.\nContinue?"),
				"GQview", "reset_filter", TRUE, dummy_cancel_cb, NULL);
	generic_dialog_add_stock(gd, _("Ok"), GTK_STOCK_OK, editor_default_ok_cb, TRUE);
	gtk_widget_show(gd->dialog);
}

static void editor_help_cb(GtkWidget *widget, gpointer data)
{
	help_window_show("editors");
}

static void safe_delete_view_cb(GtkWidget* widget, gpointer data)
{
	layout_set_path(NULL, gtk_entry_get_text(GTK_ENTRY(safe_delete_path_entry)));
}

static void safe_delete_clear_ok_cb(GenericDialog *gd, gpointer data)
{
	file_util_trash_clear();
}

static void safe_delete_clear_cb(GtkWidget* widget, gpointer data)
{
	GenericDialog *gd;
	GtkWidget *entry;
	gd = generic_dialog_new(_("Clear trash"), _("This will remove the trash contents."),
				"GQview", "clear_trash", TRUE, dummy_cancel_cb, NULL);
	generic_dialog_add_stock(gd, _("Ok"), GTK_STOCK_OK, safe_delete_clear_ok_cb, TRUE);
	entry = gtk_entry_new();
	gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
	if (safe_delete_path) gtk_entry_set_text(GTK_ENTRY(entry), safe_delete_path);
	gtk_box_pack_start(GTK_BOX(gd->vbox), entry, FALSE, FALSE, 0);
	gtk_widget_show(entry);
	gtk_widget_show(gd->dialog);
}

static void config_window_create(void)
{
	GtkWidget *win_vbox;
	GtkWidget *hbox;
	GtkWidget *notebook;
	GtkWidget *frame;
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *vbox1;
	GtkWidget *button;
	GtkWidget *tabcomp;
	GtkWidget *radiobuttongroup;
	GtkWidget *ct_button;
	GtkWidget *table;
	GtkObject *adj;
	GtkWidget *spin;
	GtkWidget *scrolled;
	GtkWidget *filter_view;
	GtkCellRenderer *renderer;
	GtkTreeSelection *selection;
	GtkTreeViewColumn *column;
	gint i;

	configwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_type_hint(GTK_WINDOW(configwindow), GDK_WINDOW_TYPE_HINT_DIALOG);
	g_signal_connect(G_OBJECT (configwindow), "delete_event",
			 G_CALLBACK(config_window_delete), NULL);
	gtk_window_set_resizable(GTK_WINDOW(configwindow), FALSE);
	gtk_window_set_title(GTK_WINDOW(configwindow), _("GQview configuration"));
	gtk_window_set_wmclass(GTK_WINDOW(configwindow), "config", "GQview");
	gtk_container_set_border_width(GTK_CONTAINER (configwindow), 5);

	window_set_icon(configwindow, (const gchar **)config_xpm, NULL);

	win_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(configwindow), win_vbox);
	gtk_widget_show(win_vbox);

	hbox = gtk_hbutton_box_new();
	gtk_box_set_spacing(GTK_BOX(hbox), 10);
	gtk_box_pack_end(GTK_BOX(win_vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_button_new_from_stock(GTK_STOCK_OK);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(config_window_ok_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_widget_show(button);

	button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(config_window_apply_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(config_window_close_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start (GTK_BOX(win_vbox), notebook, TRUE, TRUE, 0);

	/* general options tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("General"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

	frame = gtk_frame_new(_("Initial directory"));
	gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	ct_button = add_check_button(startup_path_enable, &startup_path_enable_c,
				      _("On startup, change to this directory:"), vbox1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);
	add_check_button_sensitivity(ct_button, hbox);

	gtk_widget_realize(configwindow);

	tabcomp = tab_completion_new(&startup_path_entry, startup_path, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tabcomp, TRUE, TRUE, 0);
	gtk_widget_show(tabcomp);

	button = gtk_button_new_with_label (_("Use current"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(startup_path_set_current), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	frame = gtk_frame_new(_("Thumbnails"));
	gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_thumb_size_menu(_("Size:"), vbox1);

	add_quality_menu(thumbnail_quality, &thumbnail_quality_c, _("Quality:"), vbox1, TRUE);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);
	ct_button = add_check_button(enable_thumb_caching, &enable_thumb_caching_c,
				     _("Cache thumbnails"), hbox);
	button = gtk_button_new_with_label(_("Clear cache"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cache_clear_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = add_check_button(enable_thumb_dirs, &enable_thumb_dirs_c,
				  _("Cache thumbnails into .thumbnails"), vbox1);
	add_check_button_sensitivity(ct_button, button);

	button = add_check_button(use_xvpics_thumbnails, &use_xvpics_thumbnails_c,
				  _("Use xvpics thumbnails when found (read only)"), vbox1);
	add_check_button_sensitivity(ct_button, button);

	frame = gtk_frame_new(_("Slide show"));
	gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_spin_button(_("Delay before image change (seconds):"), slideshow_delay, &slideshow_delay_c,
			1.0, 1200.0, 1.0, 0, vbox1);
	
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	add_check_button(slideshow_random, &slideshow_random_c,
			 _("Random"), hbox);
	add_check_button(slideshow_repeat, &slideshow_repeat_c,
			 _("Repeat"), hbox);

	add_check_button(enable_read_ahead, &enable_read_ahead_c,
			 _("Preload next image"), vbox);

	/* image tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("Image"));
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, label);
	
	frame = gtk_frame_new(_("Quality"));
	gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_dither_menu(dither_quality, &dither_quality_c, _("Dithering method:"), vbox1);
	add_quality_menu(zoom_quality, &zoom_quality_c, _("Zoom (scaling):"), vbox1, !(GDK_PIXBUF_TILES_BROKEN));
	add_check_button(two_pass_zoom, &two_pass_zoom_c, _("Two pass zooming"), vbox1);

	frame = gtk_frame_new(_("When new image is selected:"));
	gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	zoom_mode_c = zoom_mode;
	radiobuttongroup = gtk_radio_button_new_with_label (NULL, _("Zoom to original size"));
	if (zoom_mode == ZOOM_RESET_ORIGINAL) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttongroup), 1);
	g_signal_connect(G_OBJECT(radiobuttongroup), "clicked",
			 G_CALLBACK(zoom_mode_original_cb), NULL);
	gtk_box_pack_start(GTK_BOX(vbox1), radiobuttongroup, FALSE, FALSE, 0);
	gtk_widget_show(radiobuttongroup);

	button = gtk_radio_button_new_with_label(gtk_radio_button_get_group(GTK_RADIO_BUTTON(radiobuttongroup)),_("Fit image to window"));
	if (zoom_mode == ZOOM_RESET_FIT_WINDOW) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), 1);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(zoom_mode_fit_cb), NULL);
	gtk_box_pack_start(GTK_BOX(vbox1), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_radio_button_new_with_label(gtk_radio_button_get_group(GTK_RADIO_BUTTON(radiobuttongroup)),_("Leave Zoom at previous setting"));
	if (zoom_mode == ZOOM_RESET_NONE) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), 1);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(zoom_mode_none_cb), NULL);
	gtk_box_pack_start(GTK_BOX(vbox1), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	add_check_button(zoom_to_fit_expands, &zoom_to_fit_expands_c,
			 _("Allow enlargement of image for zoom to fit."), vbox);

	add_spin_button(_("Offscreen cache size (Mb per image):"), tile_cache_max, &tile_cache_max_c,
			0.0, 128.0, 1.0, 0, vbox);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Zoom increment:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	zoom_increment_c = zoom_increment;
	adj = gtk_adjustment_new(zoom_increment_c ? (float)zoom_increment_c / 10.0 : 1.0, 0.1, 4.0, 0.1, 0.1, 0.1);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, 1);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 5);
	g_signal_connect(G_OBJECT(adj), "value_changed",
			 G_CALLBACK(zoom_increment_cb), NULL);
	gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spin), GTK_UPDATE_ALWAYS);
	gtk_widget_show(spin);

	add_check_button(progressive_key_scrolling, &progressive_key_scrolling_c,
			 _("Progressive keyboard scrolling"), vbox);
	add_check_button(mousewheel_scrolls, &mousewheel_scrolls_c,
			 _("Mouse wheel scrolls image"), vbox);

	add_check_button(black_window_background, &black_window_background_c,
			 _("Black background"), vbox);
	add_check_button(update_on_time_change, &update_on_time_change_c,
			 _("Refresh on file change"), vbox);

	/* window tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("Windows"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);

	add_check_button(save_window_positions, &save_window_positions_c,
			 _("Remember window positions"), vbox);
	add_check_button(restore_tool, &restore_tool_c,
			 _("Remember tool state (float/hidden)"), vbox);

	add_check_button(fit_window, &fit_window_c,
			 _("Fit window to image when tools are hidden/floating"), vbox);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	ct_button = add_check_button(limit_window_size, &limit_window_size_c,
				      _("Limit size when auto-sizing window (%):"), hbox);

	spin = add_spin_button(NULL, max_window_size, &max_window_size_c,
			       10.0, 150.0, 1.0, 0, hbox);
	add_check_button_sensitivity(ct_button, spin);

#if 0
	add_check_button(place_dialogs_under_mouse, &place_dialogs_under_mouse_c,
			 _("Place dialogs under mouse"), vbox);
#endif

	add_check_button(tree_descend_subdirs, &tree_descend_subdirs_c,
			 _("Descend folders in tree view"), vbox);

	frame = gtk_frame_new(_("Layout"));
	gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	layout_widget = layout_config_new();
	layout_config_set(layout_widget, layout_style, layout_order);
	gtk_container_add(GTK_CONTAINER(frame), layout_widget);
	gtk_widget_show(layout_widget);


	/* filtering tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER (vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("Filtering"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);

	add_check_button(show_dot_files, &show_dot_files_c,
			 _("Show entries that begin with a dot"), vbox);
	add_check_button(file_sort_case_sensitive, &file_sort_case_sensitive_c,
			 _("Case sensitive sort"), vbox);
	button = add_check_button(file_filter_disable, &file_filter_disable_c,
				  _("Disable File Filtering"), vbox);

	frame = gtk_frame_new(_("Include files of type:"));
	gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vbox1), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	filter_store = gtk_list_store_new(1, G_TYPE_POINTER);
	filter_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(filter_store));
	g_object_unref(filter_store);
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(filter_view));
	gtk_tree_selection_set_mode(GTK_TREE_SELECTION(selection), GTK_SELECTION_SINGLE);

	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(filter_disable_cb), frame);
	gtk_widget_set_sensitive(frame, !file_filter_disable_c);

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, _("Filter"));
	gtk_tree_view_column_set_resizable(column, TRUE);

	renderer = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(renderer), "toggled",
			 G_CALLBACK(filter_store_enable_cb), filter_store);
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func(column, renderer, filter_set_func,
						GINT_TO_POINTER(FE_ENABLE), NULL);
	
	renderer = gtk_cell_renderer_text_new();
	g_signal_connect(G_OBJECT(renderer), "edited",
			 G_CALLBACK(filter_store_ext_edit_cb), filter_store);
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	g_object_set(G_OBJECT(renderer), "editable", (gboolean)TRUE, NULL);
	gtk_tree_view_column_set_cell_data_func(column, renderer, filter_set_func,
						GINT_TO_POINTER(FE_EXTENSION), NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(filter_view), column);

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, _("Description"));
	gtk_tree_view_column_set_resizable(column, TRUE);
	renderer = gtk_cell_renderer_text_new();
	g_signal_connect(G_OBJECT(renderer), "edited",
			 G_CALLBACK(filter_store_desc_edit_cb), filter_store);
	g_object_set(G_OBJECT(renderer), "editable", (gboolean)TRUE, NULL);
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func(column, renderer, filter_set_func,
						GINT_TO_POINTER(FE_DESCRIPTION), NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(filter_view), column);

	filter_store_populate();
	gtk_container_add(GTK_CONTAINER(scrolled), filter_view);
	gtk_widget_show(filter_view);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX(vbox1), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Defaults"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(filter_default_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Remove"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(filter_remove_cb), filter_view);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Add"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(filter_add_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

#if 0
	/* collections tab */

	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("Collections"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
#endif

	/* editor entry tab */

	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("Editors"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);

	table = gtk_table_new(3,9,FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
	gtk_widget_show(table);

	label = gtk_label_new(_("#"));
	gtk_table_attach(GTK_TABLE (table),label, 0, 1, 0, 1,
			 GTK_SHRINK, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	label = gtk_label_new(_("Menu name"));
	gtk_table_attach_defaults(GTK_TABLE (table),label, 1, 2, 0, 1);
	gtk_widget_show(label);
	label = gtk_label_new(_("Command Line"));
	gtk_table_attach_defaults(GTK_TABLE (table),label, 2, 3, 0, 1);
	gtk_widget_show(label);

	for (i = 0; i < GQVIEW_EDITOR_SLOTS; i++)
		{
		gchar *buf;

		buf = g_strdup_printf("%d", i+1);
		label = gtk_label_new(buf);
		g_free(buf);

		gtk_table_attach(GTK_TABLE (table),label, 0, 1, i+1, i+2,
				 GTK_SHRINK, GTK_FILL, 3, 2);
		gtk_widget_show(label);

		editor_name_entry[i] = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(editor_name_entry[i]), EDITOR_NAME_MAX_LENGTH);
		gtk_widget_set_size_request(editor_name_entry[i],80,-1);
		if (editor_name[i]) gtk_entry_set_text(GTK_ENTRY(editor_name_entry[i]),editor_name[i]);
		gtk_table_attach(GTK_TABLE (table),editor_name_entry[i],1,2,i+1,i+2,
				 GTK_FILL, GTK_FILL, 2, 2);
		gtk_widget_show(editor_name_entry[i]);

		editor_command_entry[i] = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(editor_command_entry[i]), EDITOR_COMMAND_MAX_LENGTH);
		gtk_widget_set_size_request(editor_command_entry[i],160,-1);
		tab_completion_add_to_entry(editor_command_entry[i], NULL, NULL);
		if (editor_command[i]) gtk_entry_set_text(GTK_ENTRY(editor_command_entry[i]), editor_command[i]);
		gtk_table_attach(GTK_TABLE (table),editor_command_entry[i],2,3,i+1,i+2,
				 GTK_FILL | GTK_EXPAND, GTK_FILL, 2, 2);
		gtk_widget_show(editor_command_entry[i]);
		}

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Defaults"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(editor_default_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label(_("Help"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(editor_help_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	/* advanced entry tab */

	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);
	label = gtk_label_new(_("Advanced"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);

	frame = gtk_frame_new(NULL);
	gtk_container_set_border_width(GTK_CONTAINER (frame), 5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame),vbox1);
	gtk_widget_show(vbox1);

	add_check_button(confirm_delete, &confirm_delete_c,
			 _("Confirm file delete"), vbox1);
	add_check_button(enable_delete_key, &enable_delete_key_c,
			 _("Enable Delete key"), vbox1);

	ct_button = add_check_button(safe_delete_enable, &safe_delete_enable_c,
				     _("Safe delete"), vbox1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);
	add_check_button_sensitivity(ct_button, hbox);

	label = gtk_label_new(_("Folder:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	tabcomp = tab_completion_new(&safe_delete_path_entry, safe_delete_path, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tabcomp, TRUE, TRUE, 0);
	gtk_widget_show(tabcomp);

	button = gtk_button_new_with_label (_("View"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(safe_delete_view_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label (_("Clear"));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(safe_delete_clear_cb), NULL);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	button = add_spin_button(_("Maximum size (MB)"), safe_delete_size, &safe_delete_size_c,
				 1.0, 500.0, 1.0, 0, vbox1);
	add_check_button_sensitivity(ct_button, button);

	add_check_button(exif_rotate_enable, &exif_rotate_enable_c,
			 _("Auto rotate image using Exif information"), vbox);

	add_spin_button(_("Custom similarity threshold:"), dupe_custom_threshold, &dupe_custom_threshold_c,
			0.0, 100.0, 1.0, 0, vbox);

	add_check_button(collection_rectangular_selection, &collection_rectangular_selection_c,
			 _("Rectangular selection"), vbox);

	add_check_button(enable_in_place_rename, &enable_in_place_rename_c,
			 _("In place renaming"), vbox);

	gtk_widget_show(notebook);

	gtk_widget_show(configwindow);
}

/*
 *-----------------------------------------------------------------------------
 * config window show (public)
 *-----------------------------------------------------------------------------
 */ 

void show_config_window(void)
{
	if (configwindow) return;
	config_window_create();
}

/*
 *-----------------
 * about window
 *-----------------
 */

static GtkWidget *about = NULL;

static gint about_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	gtk_widget_destroy(about);
	about = NULL;

	return TRUE;
}

static void about_window_close(GtkWidget *widget, gpointer data)
{
	if (!about) return;

	gtk_widget_destroy(about);
	about = NULL;
}

static void about_credits_cb(GtkWidget *widget, gpointer data)
{
	help_window_show("credits");
}

void show_about_window(void)
{
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *button;
	GdkPixbuf *pixbuf;

	gchar *buf;

	if (about)
		{
		gdk_window_raise(about->window);
		return;
		}

	about = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_type_hint(GTK_WINDOW(about), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_wmclass(GTK_WINDOW(about), "about", "GQview");
	g_signal_connect(G_OBJECT(about), "delete_event",
			 G_CALLBACK(about_delete_cb), NULL);

	gtk_window_set_title(GTK_WINDOW(about), _("About - GQview"));
	gtk_container_set_border_width(GTK_CONTAINER(about), 5);

	frame = gtk_frame_new(NULL);
	gtk_container_add(GTK_CONTAINER(about), frame);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new (FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_container_add (GTK_CONTAINER(frame),vbox);
	gtk_widget_show(vbox);

	pixbuf = gdk_pixbuf_new_from_xpm_data(logo);
	button = gtk_image_new_from_pixbuf(pixbuf);
	g_object_unref(pixbuf);
	gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
	gtk_widget_show(button);

	buf = g_strdup_printf(_("GQview %s\n\nCopyright (c) %s John Ellis\nwebsite: %s\nemail: %s\n\nReleased under the GNU General Public License"),
			      VERSION,
			      "2004",
			      "gqview.sourceforge.net",
			      "gqview@users.sourceforge.net");
	label = gtk_label_new(buf);
	g_free(buf);

	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label(_("Credits..."));
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(about_credits_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(about_window_close), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_widget_show(button);

	window_set_icon(about, NULL, NULL);
	gtk_widget_show(about);
}

