/*
 * btscanner - Displays the output of Bluetooth scans
 * Copyright (C) 2003 Pentest Limited
 * 
 * Written 2003 by Tim Hurman <timh at pentest.co.uk>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY
 * RIGHTS.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE
 * FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS,
 * COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS SOFTWARE
 * IS DISCLAIMED.
 */

/*
 * btscanner.c: keep a bluetooth card scanning the network to find devices
 * that are discoverable. Heavily based on hcitool.c
 */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>
#include <curses.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>

#include "btscanner.h"

/* usage */
void usage (char *pname) {
	fprintf(stderr, "Usage: %s [options]\n", pname);
	fprintf(stderr, "options\n\t--help\tDisplay help\n\t-i\tHCI Device\n");
}


/* command line options */
static struct option main_options[] = {
	{"help", 0,0, 'h'},
	{"device", 1,0, 'i'},
	{"ouidb", 1,0, 'o'},
	{0, 0, 0, 0}
};


/* handle a signal */
void handlesig (int sig)
{
	int *doloop = pthread_getspecific(doloop_key);
	sig=sig;
	if (doloop) *doloop = 0;
}
void dointerrupt(int sig)
{
	/* do nothing, just interrupt */
	sig=sig;
}


/* main */
int main (int argc, char **argv)
{
	int opt, i;
	struct sigaction act;
	sigset_t sset;
	struct proc_info pi;
	pthread_mutexattr_t pmatt;
	int sendkills;
	char *ouidb_filename = OUIDB_FILE;

	/* init */
	memset(&pi, 0, sizeof(struct proc_info));

	pthread_mutexattr_init(&pmatt);
	pthread_mutex_init(&(pi.proc_info_mutex), &pmatt);
	pthread_mutex_init(&(pi.dhead_mutex), &pmatt);
	pthread_mutex_init(&(pi.dev_mutex), &pmatt);
	pi.dev_id = -1;

	if (0 != (i = pthread_key_create(&doloop_key, NULL))) {
		errno = i;
		perror("Unable to create thread specific key");
		exit(1);
	}

	/* get the args */
	while ((opt=getopt_long(argc, argv, "+i:ho:", main_options, NULL)) != -1)
		switch(opt) {
		case 'i':
			pi.dev_id = hci_devid(optarg);
			if (pi.dev_id < 0) {
				perror("Invalid device");
				exit(1);
			}
			break;
		case 'o':
			ouidb_filename = optarg;
			break;
		case 'h':
		default:
			usage(argv[0]);
			exit(0);
		}

	/* get the default route is no device specified */
	if (pi.dev_id == -1) {
		pi.dev_id = hci_get_route(&(pi.dev_bd));
		if (pi.dev_id < 0) {
			perror("Device is not available.");
			exit(1);
		}
	}
	/* get the device address */
	if (hci_devba(pi.dev_id, &(pi.dev_bd)) < 0) {
		perror("Device is not available");
		exit(1);
	}
	/* we have confirmed that bluetooth is at least working */

	/* read the OUI db */
	if(ouidb_init(&pi, ouidb_filename)) {
		fprintf(stderr, "Unable to open the OUI database\n");
	}

	/* initialise the signals */
	memset (&sset, 0, sizeof(sset));
	sigfillset(&sset);
	sigdelset(&sset, SIGKILL);
	sigdelset(&sset, SIGSTOP);
	sigdelset(&sset, SIGTERM);
	sigdelset(&sset, SIGINT);
	sigdelset(&sset, SIGSEGV);
	sigdelset(&sset, SIGWINCH); /* need for wresize */
	if (-1 == sigprocmask(SIG_SETMASK, &sset, NULL)) {
		perror("main::sigprocmask()");
		exit(0);
	}

	/* initialise the signal handler */
	memset(&act, 0, sizeof(act));
	act.sa_handler = handlesig;
	sigfillset(&(act.sa_mask));
	sigdelset(&(act.sa_mask), SIGKILL);
	sigdelset(&(act.sa_mask), SIGSTOP);
	if (-1 == sigaction(SIGTERM, &act, NULL)) {
		perror("main::sigaction()");
		exit(0);
	}
	if (-1 == sigaction(SIGINT, &act, NULL)) {
		perror("main::sigaction()");
		exit(0);
	}
	act.sa_handler = dointerrupt;
	if (-1 == sigaction(SIGUSR1, &act, NULL)) {
		perror("main::sigaction()");
		exit(0);
	}

	/* start up the threads */
	i = pthread_mutex_lock(&(pi.proc_info_mutex));
	if (0 != i) {
		fprintf(stderr, "main::pthread_mutex_lock(): %s\n", strerror(errno));
		exit(1);
	}

	/* firstly the scanner */
	i = pthread_create(&(pi.thr_ids[0]), NULL, bs_runscan, &pi);
	if (i != 0)  {
		fprintf(stderr, "main::pthread_create(): %s\n", strerror(errno));
		exit(1);
	}
	/* and now the screen */
	i = pthread_create(&(pi.thr_ids[1]), NULL, bs_runscreen, &pi);
	if (i != 0)  {
		fprintf(stderr, "main::pthread_create(): %s\n", strerror(errno));
		exit(1);
	}
	/* lastly the info gatherer */
	i = pthread_create(&(pi.thr_ids[2]), NULL, bs_runinfo, &pi);
	if (i != 0)  {
		fprintf(stderr, "main::pthread_create(): %s\n", strerror(errno));
		exit(1);
	}

	/* unlock the mutex */
	pthread_mutex_unlock(&(pi.proc_info_mutex));

	/* rejoin the threads */
	sendkills = 0;
	while (1) {
		opt = 0;

		i = pthread_mutex_lock(&(pi.proc_info_mutex));
		for (i=0; i < MAX_WORK_THREADS; i++) {
			if (pi.thr_ids[i] != 0 && pi.thr_died[i] == 1) {
				pthread_join(pi.thr_ids[i], NULL);
				pi.thr_ids[i] = 0;
				sendkills = 1;
			}
			if (pi.thr_ids[i] != 0 && pi.thr_died[i] == 0) {
				if(sendkills)
					pthread_kill(pi.thr_ids[i], SIGINT);
				opt++;
			}
		}
		pthread_mutex_unlock(&(pi.proc_info_mutex));

		if (opt == 0)
			break;
		sleep(1);
	}

	/* tidy up */
	ouidb_close(&pi);
	if (pi.scanner_error != NULL) free(pi.scanner_error);
	exit(0);
}

