/**
 * Copyright (c) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef BANDS_H_INCLUDED
#define BANDS_H_INCLUDED

typedef struct band_entry_s band_entry_t;
typedef struct band_s band_t;
typedef struct freqtable_s freqtable_t;

struct band_entry_s {
    const char *name;	/* ascii representation of frequency */
    unsigned int freq;
};

struct band_s {
    const char *name;
    const band_entry_t *channels;
    int count;
    int active;
};

/**
 * Country specific VHF bands.
 */
const band_entry_t band_data_vhf_italy[] = {
    { "A",   53750 }, { "B",  62250 }, { "C",   82250 },
    { "D",  175250 }, { "E", 183750 }, { "F",  192250 },
    { "G",  201250 }, { "H", 210250 }, { "H1", 217250 },
    { "H2", 224250 }
};
const band_entry_t band_data_vhf_ireland[] = {
    { "I1",  45750 }, { "I2",  53750 }, { "I3",  61750 },
    { "I4", 175250 }, { "I5", 183250 }, { "I6", 191250 },
    { "I7", 299250 }, { "I8", 207250 }, { "I9", 215250 }
};

/**
 * For australian VHF and UHF information, see:
 * http://www.aba.gov.au/broadcasters/television.htm
 * For australian analog terrestrial transmissions.
 * Old allocations: AS10=209250, AS11=216250
 * New allocations: AS10=210250, AS11=217250
 */
const band_entry_t band_data_vhf_australia[] = {
    { "AS0",   46250 }, { "AS1",   57250 }, { "AS2",   64250 },
    { "AS3",   86250 }, { "AS4",   95250 }, { "AS5",  102250 },
    { "AS5A", 138250 }, { "AS6",  175250 }, { "AS7",  182250 },
    { "AS8",  189250 }, { "AS9",  196250 }, { "AS9A", 203250 },
    { "AS10", 209250 }, { "AS11", 216250 }, { "AS12", 224250 }
};

/**
 * Why does australia have to be so different? :)
 */
const band_entry_t band_data_uhf_australia[] = {
    { "AU28", 527250 }, { "AU29", 534250 }, { "AU30", 541250 },
    { "AU31", 548250 }, { "AU32", 555250 }, { "AU33", 562250 },
    { "AU34", 569250 }, { "AU35", 576250 }, { "AU36", 583250 },
    { "AU37", 590250 }, { "AU38", 597250 }, { "AU39", 604250 },
    { "AU40", 611250 }, { "AU41", 618250 }, { "AU42", 625250 },
    { "AU43", 632250 }, { "AU44", 639250 }, { "AU45", 646250 },
    { "AU46", 653250 }, { "AU47", 660250 }, { "AU48", 667250 },
    { "AU49", 674250 }, { "AU50", 681250 }, { "AU51", 688250 },
    { "AU52", 695250 }, { "AU53", 702250 }, { "AU54", 709250 },
    { "AU55", 716250 }, { "AU56", 723250 }, { "AU57", 730250 },
    { "AU58", 737250 }, { "AU59", 744250 }, { "AU60", 751250 },
    { "AU61", 758250 }, { "AU62", 765250 }, { "AU63", 772250 },
    { "AU64", 779250 }, { "AU65", 786250 }, { "AU66", 793250 },
    { "AU67", 800250 }, { "AU68", 807250 }, { "AU69", 814250 }
};

/**
 * General PAL/SECAM VHF bands.
 */
const band_entry_t band_data_vhf_e1_e12[] = {
    { "E1",   45250 }, { "E2",   48250 }, { "E3",   55250 },
    { "E4",   62250 }, { "E5",  175250 }, { "E6",  182250 },
    { "E7",  189250 }, { "E8",  196250 }, { "E9",  203250 },
    { "E10", 210250 }, { "E11", 217250 }, { "E12", 224250 }
};
const band_entry_t band_data_vhf_misc[] = {
    { "X",   69250 }, { "Y",   76250 }, { "Z", 83250 },
    { "Z+1", 90250 }, { "Z+2", 97250 }
};
const band_entry_t band_data_vhf_s1_s41[] = {
    { "S1",  105250 }, { "S2",  112250 }, { "S3",  119250 },
    { "S4",  126250 }, { "S5",  133250 }, { "S6",  140250 },
    { "S7",  147250 }, { "S8",  154250 }, { "S9",  161250 },
    { "S10", 168250 }, { "S11", 231250 }, { "S12", 238250 },
    { "S13", 245250 }, { "S14", 252250 }, { "S15", 259250 },
    { "S16", 266250 }, { "S17", 273250 }, { "S18", 280250 },
    { "S19", 287250 }, { "S20", 294250 }, { "S21", 303250 },
    { "S22", 311250 }, { "S23", 319250 }, { "S24", 327250 },
    { "S25", 335250 }, { "S26", 343250 }, { "S27", 351250 },
    { "S28", 359250 }, { "S29", 367250 }, { "S30", 375250 },
    { "S31", 383250 }, { "S32", 391250 }, { "S33", 399250 },
    { "S34", 407250 }, { "S35", 415250 }, { "S36", 423250 },
    { "S37", 431250 }, { "S38", 439250 }, { "S39", 447250 },
    { "S40", 455250 }, { "S41", 463250 }
};
const band_entry_t band_data_vhf_france[] = {
    { "K01",  47750 }, { "K02",  55750 }, { "K03",  60500 },
    { "K04",  63750 }, { "K05", 176000 }, { "K06", 184000 },
    { "K07", 192000 }, { "K08", 200000 }, { "K09", 208000 },
    { "K10", 216000 },

    { "KB", 116750 }, { "KC", 128750 }, { "KD", 140750 },
    { "KE", 159750 }, { "KF", 164750 }, { "KG", 176750 },
    { "KH", 188750 }, { "KI", 200750 }, { "KJ", 212750 },
    { "KK", 224750 }, { "KL", 236750 }, { "KM", 248750 },
    { "KN", 260750 }, { "KO", 272750 }, { "KP", 284750 },
    { "KQ", 296750 },

    { "H01", 303250 }, { "H02", 311250 }, { "H03", 319250 },
    { "H04", 327250 }, { "H05", 335250 }, { "H06", 343250 },
    { "H07", 351250 }, { "H08", 359250 }, { "H09", 367250 },
    { "H10", 375250 }, { "H11", 383250 }, { "H12", 391250 },
    { "H13", 399250 }, { "H14", 407250 }, { "H15", 415250 },
    { "H16", 423250 }, { "H17", 431250 }, { "H18", 439250 },
    { "H19", 447250 }
};
const band_entry_t band_data_vhf_russia[] = {
    { "R1",   49750 }, { "R2",   59250 }, { "R3",   77250 },
    { "R4",   85250 }, { "R5",   93250 }, { "R6",  175250 },
    { "R7",  183250 }, { "R8",  191250 }, { "R9",  199250 },
    { "R10", 207250 }, { "R11", 215250 }, { "R12", 223250 },

    { "SR1",  111250 }, { "SR2",  119250 }, { "SR3",  127250 },
    { "SR4",  135250 }, { "SR5",  143250 }, { "SR6",  151250 },
    { "SR7",  159250 }, { "SR8",  167250 }, { "SR11", 231250 },
    { "SR12", 239250 }, { "SR13", 247250 }, { "SR14", 255250 },
    { "SR15", 263250 }, { "SR16", 271250 }, { "SR17", 279250 },
    { "SR18", 287250 }, { "SR19", 295250 },
};

const band_entry_t band_data_uhf[] = {
    { "U21", 471250 }, { "U22", 479250 }, { "U23", 487250 },
    { "U24", 495250 }, { "U25", 503250 }, { "U26", 511250 },
    { "U27", 519250 }, { "U28", 527250 }, { "U29", 535250 },
    { "U30", 543250 }, { "U31", 551250 }, { "U32", 559250 },
    { "U33", 567250 }, { "U34", 575250 }, { "U35", 583250 },
    { "U36", 591250 }, { "U37", 599250 }, { "U38", 607250 },
    { "U39", 615250 }, { "U40", 623250 }, { "U41", 631250 },
    { "U42", 639250 }, { "U43", 647250 }, { "U44", 655250 },
    { "U45", 663250 }, { "U46", 671250 }, { "U47", 679250 },
    { "U48", 687250 }, { "U49", 695250 }, { "U50", 703250 },
    { "U51", 711250 }, { "U52", 719250 }, { "U53", 727250 },
    { "U54", 735250 }, { "U55", 743250 }, { "U56", 751250 },
    { "U57", 759250 }, { "U58", 767250 }, { "U59", 775250 },
    { "U60", 783250 }, { "U61", 791250 }, { "U62", 799250 },
    { "U63", 807250 }, { "U64", 815250 }, { "U65", 823250 },
    { "U66", 831250 }, { "U67", 839250 }, { "U68", 847250 },
    { "U69", 855250 }
};

const band_entry_t band_data_us_catv[] = {
    { "1",    73250 }, { "2",    55250 }, { "3",    61250 },
    { "4",    67250 }, { "5",    77250 }, { "6",    83250 },
    { "7",   175250 }, { "8",   181250 }, { "9",   187250 },
    { "10",  193250 }, { "11",  199250 }, { "12",  205250 },
    { "13",  211250 }, { "14",  121250 }, { "15",  127250 },
    { "16",  133250 }, { "17",  139250 }, { "18",  145250 },
    { "19",  151250 }, { "20",  157250 }, { "21",  163250 },
    { "22",  169250 }, { "23",  217250 }, { "24",  223250 },
    { "25",  229250 }, { "26",  235250 }, { "27",  241250 },
    { "28",  247250 }, { "29",  253250 }, { "30",  259250 },
    { "31",  265250 }, { "32",  271250 }, { "33",  277250 },
    { "34",  283250 }, { "35",  289250 }, { "36",  295250 },
    { "37",  301250 }, { "38",  307250 }, { "39",  313250 },
    { "40",  319250 }, { "41",  325250 }, { "42",  331250 },
    { "43",  337250 }, { "44",  343250 }, { "45",  349250 },
    { "46",  355250 }, { "47",  361250 }, { "48",  367250 },
    { "49",  373250 }, { "50",  379250 }, { "51",  385250 },
    { "52",  391250 }, { "53",  397250 }, { "54",  403250 },
    { "55",  409250 }, { "56",  415250 }, { "57",  421250 },
    { "58",  427250 }, { "59",  433250 }, { "60",  439250 },
    { "61",  445250 }, { "62",  451250 }, { "63",  457250 },
    { "64",  463250 }, { "65",  469250 }, { "66",  475250 },
    { "67",  481250 }, { "68",  487250 }, { "69",  493250 },
    { "70",  499250 }, { "71",  505250 }, { "72",  511250 },
    { "73",  517250 }, { "74",  523250 }, { "75",  529250 },
    { "76",  535250 }, { "77",  541250 }, { "78",  547250 },
    { "79",  553250 }, { "80",  559250 }, { "81",  565250 },
    { "82",  571250 }, { "83",  577250 }, { "84",  583250 },
    { "85",  589250 }, { "86",  595250 }, { "87",  601250 },
    { "88",  607250 }, { "89",  613250 }, { "90",  619250 },
    { "91",  625250 }, { "92",  631250 }, { "93",  637250 },
    { "94",  643250 }, { "95",   91250 }, { "96",   97250 },
    { "97",  103250 }, { "98",  109250 }, { "99",  115250 }
};

const band_entry_t band_data_us_catv_100[] = {
    { "100", 649250 }, { "101", 655250 }, { "102", 661250 },
    { "103", 667250 }, { "104", 673250 }, { "105", 679250 },
    { "106", 685250 }, { "107", 691250 }, { "108", 697250 },
    { "109", 703250 }, { "110", 709250 }, { "111", 715250 },
    { "112", 721250 }, { "113", 727250 }, { "114", 733250 },
    { "115", 739250 }, { "116", 745250 }, { "117", 751250 },
    { "118", 757250 }, { "119", 763250 }, { "120", 769250 },
    { "121", 775250 }, { "122", 781250 }, { "123", 787250 },
    { "124", 793250 }, { "125", 799250 }
};

const band_entry_t band_data_us_twoway[] = {
    { "T7",   8250 }, { "T8",  14250 }, { "T9",  20250 },
    { "T10", 26250 }, { "T11", 32250 }, { "T12", 38250 },
    { "T13", 44250 }, { "T14", 50250 }
};

const band_entry_t band_data_us_bcast[] = {
    { "2",   55250 }, { "3",   61250 }, { "4",   67250 },
    { "5",   77250 }, { "6",   83250 }, { "7",  175250 },
    { "8",  181250 }, { "9",  187250 }, { "10", 193250 },
    { "11", 199250 }, { "12", 205250 }, { "13", 211250 },
    { "14", 471250 }, { "15", 477250 }, { "16", 483250 },
    { "17", 489250 }, { "18", 495250 }, { "19", 501250 },
    { "20", 507250 }, { "21", 513250 }, { "22", 519250 },
    { "23", 525250 }, { "24", 531250 }, { "25", 537250 },
    { "26", 543250 }, { "27", 549250 }, { "28", 555250 },
    { "29", 561250 }, { "30", 567250 }, { "31", 573250 },
    { "32", 579250 }, { "33", 585250 }, { "34", 591250 },
    { "35", 597250 }, { "36", 603250 }, { "37", 609250 },
    { "38", 615250 }, { "39", 621250 }, { "40", 627250 },
    { "41", 633250 }, { "42", 639250 }, { "43", 645250 },
    { "44", 651250 }, { "45", 657250 }, { "46", 663250 },
    { "47", 669250 }, { "48", 675250 }, { "49", 681250 },
    { "50", 687250 }, { "51", 693250 }, { "52", 699250 },
    { "53", 705250 }, { "54", 711250 }, { "55", 717250 },
    { "56", 723250 }, { "57", 729250 }, { "58", 735250 },
    { "59", 741250 }, { "60", 747250 }, { "61", 753250 },
    { "62", 759250 }, { "63", 765250 }, { "64", 771250 },
    { "65", 777250 }, { "66", 783250 }, { "67", 789250 },
    { "68", 795250 }, { "69", 801250 }, { "70", 807250 },
    { "71", 813250 }, { "72", 819250 }, { "73", 825250 },
    { "74", 831250 }, { "75", 837250 }, { "76", 843250 },
    { "77", 849250 }, { "78", 855250 }, { "79", 861250 },
    { "80", 867250 }, { "81", 873250 }, { "82", 879250 },
    { "83", 885250 }
};

/**
 * China Broadcast bands.
 */
const band_entry_t band_data_cn_bcast[] = {
    { "1",    49750 }, { "2",    57750 }, { "3",    65750 }, 
    { "4",    77250 }, { "5",    85250 }, { "6",   168250 }, 
    { "7",   176250 }, { "8",   184250 }, { "9",   192250 }, 
    { "10",  200250 }, { "11",  208250 }, { "12",  216250 }, 
    { "13",  471250 }, { "14",  479250 }, { "15",  487250 }, 
    { "16",  495250 }, { "17",  503250 }, { "18",  511250 }, 
    { "19",  519250 }, { "20",  527250 }, { "21",  535250 }, 
    { "22",  543250 }, { "23",  551250 }, { "24",  559250 }, 
    { "25",  607250 }, { "26",  615250 }, { "27",  623250 }, 
    { "28",  631250 }, { "29",  639250 }, { "30",  647250 }, 
    { "31",  655250 }, { "32",  663250 }, { "33",  671250 }, 
    { "34",  679250 }, { "35",  687250 }, { "36",  695250 }, 
    { "37",  703250 }, { "38",  711250 }, { "39",  719250 }, 
    { "40",  727250 }, { "41",  735250 }, { "42",  743250 }, 
    { "43",  751250 }, { "44",  759250 }, { "45",  767250 }, 
    { "46",  775250 }, { "47",  783250 }, { "48",  791250 }, 
    { "49",  799250 }, { "50",  807250 }, { "51",  815250 }, 
    { "52",  823250 }, { "53",  831250 }, { "54",  839250 }, 
    { "55",  847250 }, { "56",  855250 }, { "57",  863250 }, 
    { "58",  871250 }, { "59",  879250 }, { "60",  887250 }, 
    { "61",  895250 }, { "62",  903250 }, { "63",  911250 }, 
    { "64",  919250 }, { "65",  927250 }, { "66",  935250 }, 
    { "67",  943250 }, { "68",  951250 }, { "A1",  112250 }, 
    { "A2",  120250 }, { "A3",  128250 }, { "A4",  136250 }, 
    { "A5",  144250 }, { "A6",  152250 }, { "A7",  160250 }, 
    { "B1",  224250 }, { "B2",  232250 }, { "B3",  240250 }, 
    { "B4",  248250 }, { "B5",  256250 }, { "B6",  264250 }, 
    { "B7",  272250 }, { "B8",  280250 }, { "B9",  288250 }, 
    { "B10", 296250 }, { "B11", 304250 }, { "B12", 312250 }, 
    { "B13", 320250 }, { "B14", 328250 }, { "B15", 336250 }, 
    { "B16", 344250 }, { "B17", 352250 }, { "B18", 360250 }, 
    { "B19", 368250 }, { "B20", 376250 }, { "B21", 384250 }, 
    { "B22", 392250 }, { "B23", 400250 }, { "B24", 408250 }, 
    { "B25", 416250 }, { "B26", 424250 }, { "B27", 432250 }, 
    { "B28", 440250 }, { "B29", 448250 }, { "B30", 456250 }, 
    { "B31", 463250 }, { "C1",  567250 }, { "C2",  575250 }, 
    { "C3",  583250 }, { "C4",  591250 }, { "C5",  599250 } 
};

const band_entry_t band_data_japan_bcast[] = {
    { "1",   91250 }, { "2",   97250 }, { "3",  103250 },
    { "4",  171250 }, { "5",  177250 }, { "6",  183250 },
    { "7",  189250 }, { "8",  193250 }, { "9",  199250 },
    { "10", 205250 }, { "11", 211250 }, { "12", 217250 },
    { "13", 471250 }, { "14", 477250 }, { "15", 483250 },
    { "16", 489250 }, { "17", 495250 }, { "18", 501250 },
    { "19", 507250 }, { "20", 513250 }, { "21", 519250 },
    { "22", 525250 }, { "23", 531250 }, { "24", 537250 },
    { "25", 543250 }, { "26", 549250 }, { "27", 555250 },
    { "28", 561250 }, { "29", 567250 }, { "30", 573250 },
    { "31", 579250 }, { "32", 585250 }, { "33", 591250 },
    { "34", 597250 }, { "35", 603250 }, { "36", 609250 },
    { "37", 615250 }, { "38", 621250 }, { "39", 627250 },
    { "40", 633250 }, { "41", 639250 }, { "42", 645250 },
    { "43", 651250 }, { "44", 657250 }, { "45", 663250 },
    { "46", 669250 }, { "47", 675250 }, { "48", 681250 },
    { "49", 687250 }, { "50", 693250 }, { "51", 699250 },
    { "52", 705250 }, { "53", 711250 }, { "54", 717250 },
    { "55", 723250 }, { "56", 729250 }, { "57", 735250 },
    { "58", 741250 }, { "59", 747250 }, { "60", 753250 },
    { "61", 759250 }, { "62", 765250 }
};

const band_entry_t band_data_japan_catv[] = {
    { "13", 109250 }, { "14", 115250 }, { "15", 121250 },
    { "16", 127250 }, { "17", 133250 }, { "18", 139250 },
    { "19", 145250 }, { "20", 151250 }, { "21", 157250 },
    { "22", 165250 }, { "23", 223250 }, { "24", 231250 },
    { "25", 237250 }, { "26", 243250 }, { "27", 249250 },
    { "28", 253250 }, { "29", 259250 }, { "30", 265250 },
    { "31", 271250 }, { "32", 277250 }, { "33", 283250 },
    { "34", 289250 }, { "35", 295250 }, { "36", 301250 },
    { "37", 307250 }, { "38", 313250 }, { "39", 319250 },
    { "40", 325250 }, { "41", 331250 }, { "42", 337250 },
    { "43", 343250 }, { "44", 349250 }, { "45", 355250 },
    { "46", 361250 }, { "47", 367250 }, { "48", 373250 },
    { "49", 379250 }, { "50", 385250 }, { "51", 391250 },
    { "52", 397250 }, { "53", 403250 }, { "54", 409250 },
    { "55", 415250 }, { "56", 421250 }, { "57", 427250 },
    { "58", 433250 }, { "59", 439250 }, { "60", 445250 },
    { "61", 451250 }, { "62", 457250 }, { "63", 463250 }
};

/**
 * Australian Optus Cable.  They provide set-top boxen, but some users want
 * to tune directly.
 */
const band_entry_t band_data_australia_optus[] = {
    { "O1",  224250 }, { "O2",  175250 }, { "O3",  259250 }, { "O4",  266250 },
    { "O5",  280250 }, { "O6",  252250 }, { "O7",  168250 }, { "O8",  154250 },
    { "O9",  287250 }, { "O10", 394250 }, { "O11", 457250 }, { "O12", 338250 },
    { "O13", 216250 }, { "O14", 231250 }, { "O15", 478250 }, { "O16", 464250 },
    { "O17", 443250 }, { "O18", 352250 }, { "O19", 450250 }, { "O20", 499250 },
    { "O21", 534250 }, { "O22", 189250 }, { "O23", 110250 }, { "O24", 117250 },
    { "O25", 408250 }, { "O26", 366250 }, { "O27", 359250 }, { "O28", 294250 },
    { "O29", 147250 }, { "O30", 520250 }, { "O31", 527250 }, { "O32", 506250 },
    { "O33", 492250 }, { "O34", 513250 }, { "O35", 317250 }, { "O36", 245250 },
    { "O37", 380250 }, { "O38", 161250 }, { "O39", 310250 }, { "O40", 485250 },
    { "O41", 345250 }, { "O42", 238250 }, { "O43", 436250 }, { "O44", 401250 },
    { "O45", 138250 }, { "O46", 429250 }, { "O47", 471250 }, { "O48", 303250 },
    { "O49", 387250 }, { "O50", 324250 }, { "O51", 373250 }, { "O52", 182250 },
    { "O53", 422250 }, { "O54", 415250 }, { "O55", 273250 }, { "O56", 196250 },
    { "O57", 209250 }, { "O58", 124250 }
};

const band_t bands[] = {
    { "US Cable", band_data_us_catv, sizeof( band_data_us_catv ) / sizeof( band_entry_t ) },
    { "US Cable 100", band_data_us_catv_100, sizeof( band_data_us_catv_100 ) / sizeof( band_entry_t ) },
    { "US Two-Way", band_data_us_twoway, sizeof( band_data_us_twoway ) / sizeof( band_entry_t ) },
    { "US Broadcast", band_data_us_bcast, sizeof( band_data_us_bcast ) / sizeof( band_entry_t ) },
    { "China Broadcast", band_data_cn_bcast, sizeof( band_data_cn_bcast ) / sizeof( band_entry_t ) },
    { "Japan Broadcast", band_data_japan_bcast, sizeof( band_data_japan_bcast ) / sizeof( band_entry_t ) },
    { "Japan Cable", band_data_japan_catv, sizeof( band_data_japan_catv ) / sizeof( band_entry_t ) },
    { "VHF E2-E12", band_data_vhf_e1_e12, sizeof( band_data_vhf_e1_e12 ) / sizeof( band_entry_t ) },
    { "VHF S1-S41", band_data_vhf_s1_s41, sizeof( band_data_vhf_s1_s41 ) / sizeof( band_entry_t ) },
    { "VHF Misc", band_data_vhf_misc, sizeof( band_data_vhf_misc ) / sizeof( band_entry_t ) },
    { "VHF France", band_data_vhf_france, sizeof( band_data_vhf_france ) / sizeof( band_entry_t ) },
    { "VHF Russia", band_data_vhf_russia, sizeof( band_data_vhf_russia ) / sizeof( band_entry_t ) },
    { "VHF Australia", band_data_vhf_australia, sizeof( band_data_vhf_australia ) / sizeof( band_entry_t ) },
    { "Australia Optus", band_data_australia_optus, sizeof( band_data_australia_optus ) / sizeof( band_entry_t ) },
    { "VHF Italy", band_data_vhf_italy, sizeof( band_data_vhf_italy ) / sizeof( band_entry_t ) },
    { "VHF Ireland", band_data_vhf_ireland, sizeof( band_data_vhf_ireland ) / sizeof( band_entry_t ) },
    { "UHF", band_data_uhf, sizeof( band_data_uhf ) / sizeof( band_entry_t ) },
    { "UHF Australia", band_data_uhf_australia, sizeof( band_data_uhf_australia ) / sizeof( band_entry_t ) }
};
const int numbands = ( sizeof( bands ) / sizeof( band_t ) );
const band_t *us_cable_band = &(bands[ 0 ]);

/*
 UK         VHF Ireland, VHF Misc, UHF
 Europe     VHF Italy, VHF E1-E12, VHF S1-S41, VHF Russia, UHF
 France     VHF France, UHF
 East       VHF Russia, UHF
 Australia  VHF Australia, UHF-AU
 NZ         VHF E1-E12, UHF
 US Cable   US CATV
 US Air     US Bcast
 JP Cable   JP Cable
 JP Air     JP Bcast
*/

#define NTSC_CABLE_HRC(x) ((x == 77250) ? 78000 : ((x == 83250) ? 84000 : (x - 1250)))
#define NTSC_CABLE_IRC(x) ((x == 77250) ? 79250 : ((x == 83250) ? 85250 : x))

#endif /* BANDS_H_INCLUDED */
