/*
 * Copyright (c) 2004 Nokia. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Nokia nor the names of its contributors may be
 * used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <config.h>
#include <assert.h>
#include <gtk/gtkversion.h>
#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>

#include "ImageRendererFactory.h"
#include "ImageRenderer.h"
#ifdef HAVE_LIBRSVG
#include "SVGImageRenderer.h"
#endif
#include "GLibHelpers.h"

void ImageRendererFactory::useAsSharedFactory()
{
    static ImageRendererFactory single;
    m_sharedFactory = &single;
}


ImageRendererFactory::ImageRendererFactory()
    :mimeTypes(0)
{


    GSList *formats = gdk_pixbuf_get_formats();
    GSList *iter = formats;

    while (iter) {
	gchar** gdkTypes = gdk_pixbuf_format_get_mime_types((GdkPixbufFormat*)iter->data);
	int i = 0;
	while (gdkTypes[i]) {
	    mimeTypes = g_list_append(mimeTypes, gdkTypes[i]); // own the strings
	    ++i;
	}

	if (gdkTypes) g_free(gdkTypes); // free just the shell

	iter = g_slist_next(iter);
    }
    g_slist_free(formats);
}

extern "C" {
void mimeTypes_destroy(gpointer data, gpointer user_data)
{
    g_free(data);
}
};

ImageRendererFactory::~ImageRendererFactory()
{
    g_list_foreach(mimeTypes, mimeTypes_destroy, this);
    g_list_free(mimeTypes);
}

WebCoreImageRenderer* ImageRendererFactory::imageRenderer()
{
    WebCoreImageRenderer* r = new ImageRenderer;
    r->retain();
    return r;
}

WebCoreImageRenderer* ImageRendererFactory::imageRendererWithMIMEType(const gchar* MIMEType)
{
#ifdef HAVE_LIBRSVG    
    if (!isEmptyString(MIMEType)) {
	// FIXME: general purpose mechanism for this
	if (strcmp(MIMEType, "text/xml")==0 || strcmp(MIMEType, "image/svg+xml") == 0) {
	    SVGImageRenderer *rnd = new SVGImageRenderer();
	    rnd->retain();
	    return rnd;
	}
    }
#endif

    // fallback to generic 
    WebCoreImageRenderer* r = imageRenderer();
    return r;
}

WebCoreImageRenderer* ImageRendererFactory::imageRendererWithBytes(const gchar* bytes, unsigned length)
{
    WebCoreImageRenderer* r = imageRenderer();
    r->incrementalLoadWithBytes(bytes, length, true);
    return r;
}

WebCoreImageRenderer* ImageRendererFactory::imageRendererWithBytes(const gchar* bytes, unsigned length, const gchar* MIMEType)
{
    WebCoreImageRenderer *r = imageRendererWithMIMEType(MIMEType);
    r->incrementalLoadWithBytes(bytes, length, true);
    return r;
}

WebCoreImageRenderer* ImageRendererFactory::imageRendererWithSize(GdkRectangle* size)
{
    WebCoreImageRenderer *r = imageRenderer();
    r->resize(size);
    return r;
}

extern "C"{
gint compare_strings_func(gconstpointer a, gconstpointer b)
{
    return strcmp((const gchar*)a, (const gchar*)b);
}
};

WebCoreImageRenderer* ImageRendererFactory::imageRendererWithName(const gchar* name)
{
    if (strcmp(name, "missing_image")) {
	WebCoreImageRenderer* r =  imageRenderer();
	//load here the missing image 
	return r;
    }

    WebCoreImageRenderer *r = imageRenderer();
    return r;
}

bool ImageRendererFactory::supports(const gchar* mime)
{
    GList* item = g_list_find_custom(mimeTypes, mime, compare_strings_func);
    return (item!=0);
}
GList* ImageRendererFactory::supportedMIMETypes()
{
    return mimeTypes;
}

int ImageRendererFactory::CGCompositeOperationInContext(CGContextRef context)
{
#if DEBUG
    g_printerr("NotYetImplemented: %s, this:%x, context:%x", __PRETTY_FUNCTION__, (int)this,(int)context);
#endif
    return 0;
}

void ImageRendererFactory::setCGCompositeOperation(CGContextRef context, int op )
{
#if DEBUG
    g_printerr("NotYetImplemented: %s, this:%x, context:%x, op:%d", __PRETTY_FUNCTION__, (int)this,(int)context, op);
#endif
}

void ImageRendererFactory::setCGCompositeOperationFromString(CGContextRef context, const gchar* op)
{
#if DEBUG
    g_printerr("NotYetImplemented: %s, this:%x, context:%x, op:%s", __PRETTY_FUNCTION__, (int)this,(int)context, op);
#endif
}
