/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2002 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rigths reserved.                                                   */
/*                                                                        */
/**************************************************************************/


#include "TransportSSH.h"
#include "Session.h"
#include "Protocol.h"
#include "Process.h"

#include "NXErrors.h"
#include "NXStates.h"
#include "NXParameters.h"

#ifdef WIN32
#include "ProcessWin32.h"
#else
#include "ProcessUnix.h"
#endif



#undef NX_SSH_DEBUG


#if defined( NX_DEBUG ) || defined( NX_SSH_DEBUG )
#include <iostream>
#endif

#include <stdexcept>
using namespace std;



namespace NX
{

TransportSSH::TransportSSH( Session *pSession ) : Transport( pSession )
{
  m_type = "nxssh";
  mp_process = NULL;
}

TransportSSH::~TransportSSH()
{
  DeleteProcess();
}

void TransportSSH::DeleteProcess()
{
  if( mp_process != NULL )
  {
    delete mp_process;
    mp_process = NULL;
  }
}

bool TransportSSH::InitProcess()
{
  if( mp_process != NULL )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): a process already exists." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Cannot create another nxssh process." );
    return false;
  }

  try
  {
#ifdef WIN32
    mp_process = new ProcessWin32();
#else
    mp_process = new ProcessUnix();
#endif
  }
  catch( bad_alloc& )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): unable to find memory to create process." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Cannot create new nxssh process." );
    return false;
  }

  return true;
}

bool TransportSSH::Write( const string &in )
{
  if( mp_process != NULL )
  {
    if( !mp_process->Write( in ) )
    {
      mp_session->SetException( mp_process->GetException() );
      return false;
    }
    else
      return true;
  }
  else
  {
    mp_session->SetException( NX_RuntimeError, "Cannot write on stdin of nxssh process." );
    return false;
  }
}

int TransportSSH::Read( string & out, unsigned int timeout )
{
  int iRet = -1;
  if( mp_process != NULL )
    iRet = mp_process->Read( out, timeout );

  if( iRet == -1 )
  {
    out = "";
    mp_session->SetException( NX_RuntimeError, "Cannot read from stout or stderr of nxssh process." );
  }

  return iRet;
}

bool TransportSSH::EndConnection()
{
  return ( mp_session->GetState() == NX_ConnectionEnd ||
           mp_session->GetState() == NX_SessionAccepted );
}

bool TransportSSH::BreakConnection()
{
#if defined( NX_DEBUG )
  cout << "NXTransport (nxssh): break connection." << endl << flush;
#endif

  if( mp_process == NULL )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): process is NULL." << endl << flush;
#endif
    return false;
  }

  if( mp_process->IsRunning() )
    if( mp_process->StopProcess() )
      mp_process->WaitProcessTerminate();

  DeleteProcess();

  if( mp_session->GetState() != NX_Exception )
    mp_session->SetState( NX_ConnectionEnd );

  return true;
}

bool TransportSSH::InitConnection()
{
#if defined( NX_SSH_DEBUG )
  cout << "NXTransport (nxssh): creating new process." << endl << flush;
#endif

  if( !InitProcess() )
    return false;

#if defined( NX_SSH_DEBUG )
  cout << "NXTransport (nxssh): new process created." << endl << flush;
#endif

  ParametersList& parameters = mp_session->GetParameters();

  string sTmp;

  mp_process->SetRedirectFlags( Process::NX_STDIN | Process::NX_STDERR | Process::NX_STDOUT );

#if defined( NX_SSH_DEBUG )
  cout << "NXTransport (nxssh): set flags for redirection." << endl << flush;
#endif

  sTmp = parameters.GetString( NX_SshPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): parameter 'NX_SshPath' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SshPath' is empty." );
    return false;
  }
  else
    mp_process->SetProcessPath( sTmp );
  

  sTmp = parameters.GetString( NX_SshName, "" );
  if( sTmp.empty() )
  {
#if defined( NX_SSH_DEBUG )
    cout << "NXTransport (nxssh): set default parameter 'NX_SshName'." << endl << flush;
#endif
    mp_process->SetProcessName( "nxssh" );
  }
  else
    mp_process->SetProcessName( sTmp );

  mp_process->AddArgument( "-nx" );

  sTmp = parameters.GetString( NX_SshKeyPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): parameter 'NX_SshKeyPath' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SshKeyPath' is empty." );
    return false;
  }
  else
  {
    mp_process->AddArgument( "-i" );
    mp_process->AddArgument( sTmp );
  }

  if( parameters.GetBool( NX_EnableEncryption, false ) )
  {
    sTmp = parameters.GetString( NX_EncryptionPort, "" );
    if( sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXTransport (nxssh): parameter 'NX_EncryptionPort' is empty." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_EncryptionPort' is empty." );
      return false;
    }
    else
    {
      mp_process->AddArgument( "-D" );
      mp_process->AddArgument( sTmp );
    }
  }

  for( unsigned int i = NX_Ssh_FirstCustomOption; i <= NX_Ssh_LastCustomOption; i++ )
  {
    sTmp = parameters.GetString( i, "" );
    if( !sTmp.empty() )
    {
#if defined( NX_SSH_DEBUG )
      cout << "NXTransport (nxssh): adding custom parameter '" << sTmp << "'." << endl << flush;
#endif
      mp_process->AddArgument( sTmp );
    }
  }

  sTmp = parameters.GetString( NX_SshUsername, "nx" );
  sTmp += string("@");

  string sHost = parameters.GetString( NX_HostName, "" );

  if( sHost.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): parameter 'NX_Hostname' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_HostName' is empty." );
    return false;
  }
  else
  {
    sTmp += sHost;
    mp_process->AddArgument( sTmp );
  }

  sTmp = parameters.GetString( NX_SshLogPath, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): parameter 'NX_SshLogPath' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SshLogPath' is empty." );
    return false;
  }
  else
    mp_process->SetDeviceName( sTmp );

  sTmp = parameters.GetString( NX_Username, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): parameter 'NX_Username' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_Username' is empty." );
    return false;
  }

  sTmp = parameters.GetString( NX_Password, "" );
  if( sTmp.empty() )
  {
    sTmp = parameters.GetString( NX_PasswordMD5, "" );
    if( sTmp.empty() )
    {
#if defined( NX_DEBUG )
      cout << "NXTransport (nxssh): parameter 'NX_Password' is empty." << endl << flush;
#endif
      mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_Password' is empty." );
      return false;
    }
  }

  sTmp = parameters.GetString( NX_SessionCookie, "" );
  if( sTmp.empty() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): parameter 'NX_SessionCookie' is empty." << endl << flush;
#endif
    mp_session->SetException( NX_InvalidParameter, "Parameter 'NX_SessionCookie' is empty." );
    return false;
  }

  m_buffer = "";
  mp_session->SetState( NX_ReadyForConnection );
  return true;
}


bool TransportSSH::StartConnection()
{
  if( mp_process == NULL )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): nxssh process not found." << endl << flush;
#endif
    mp_session->SetException( NX_RuntimeError, "Process nxssh is not correctly initialized." );
    return false;
  }

  if( !mp_process->StartProcess() )
  {
#if defined( NX_DEBUG )
    cout << "NXTransport (nxssh): nxssh cannot start." << endl << flush;
#endif
    mp_session->SetException( mp_process->GetException() );
    return false;
  }

#if defined( NX_DEBUG )
  cout << "NXTransport (nxssh): connecting..." << endl << flush;
#endif

  mp_session->SetState( NX_Connecting );
  return true;
}

bool TransportSSH::AdvanceConnection( unsigned int timeout )
{
  if( mp_session->GetState() == NX_ReadyForConnection )
    return StartConnection();

  if( !CheckCurrentState( timeout )  )
  {
    BreakConnection();
    return false;
  }

  return true;
}

bool TransportSSH::CloseConnection()
{
  if( mp_process->IsRunning() )
    return Protocol::CloseConnection( mp_session, this );
  return true;
}

bool TransportSSH::CheckCurrentState( unsigned int timeout )
{
  string line = "";
  int iSize = mp_process->Read( line, timeout );
  if( iSize == -1 )
  {
    mp_session->SetException( mp_process->GetException() );
    return false;
  }

  m_buffer += line;

  if( mp_process->IsRunning() )
  {
    if( Protocol::CheckConnectionStatus( line, mp_session, this ) == NX_Exception )
      return false;

    if( mp_session->GetState() == NX_SessionAccepted )
    {
      if( !mp_session->GetParameters().GetBool( NX_EnableEncryption, false ) )
        Protocol::CloseConnection( mp_session, this );
    }
  }
  else
  {
    if( !Protocol::CheckConnectionError( mp_session, this ) &&
        mp_session->GetState() != NX_SessionAccepted )
    {
      mp_session->SetException( NX_SSHError, m_buffer );
      return false;
    }
  }

  return true;
}


} /* NX */

