/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Channel_H
#define Channel_H

#include "Transport.h"

#include "WriteBuffer.h"

#include "OpcodeStore.h"

#include "ClientStore.h"
#include "ServerStore.h"

#include "ClientCache.h"
#include "ServerCache.h"

//
// Forward declaration of referenced classes.
//

class EncodeBuffer;
class DecodeBuffer;

class Compressor;
class Decompressor;

//
// Define this to log a line when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// Type of traffic carried by channel.
//

typedef enum
{
  CHANNEL_NONE = -1,
  CHANNEL_X,
  CHANNEL_SYNC,
  CHANNEL_KEYBD,
  CHANNEL_SAMBA,
  CHANNEL_MEDIA,
  CHANNEL_LAST_TAG
} T_channel_type;

//
// Type of notification event to be sent
// by proxy to the X channel.
//

typedef enum
{
  notify_begin_congestion,
  notify_end_congestion,
  notify_begin_reset,
  notify_end_reset
} T_notification_type;

class Channel
{
  public:

  //
  // Maximum number of X connections supported.
  //

  static const int CONNECTIONS_LIMIT = 256;

  Channel(Transport *transport, Compressor *compressor,
              Decompressor *decompressor);

  virtual ~Channel();

  //
  // Read any X message available on the X
  // connection and encode it to the encode
  // buffer.
  //

  virtual int handleRead(EncodeBuffer &encodeBuffer) = 0;

  //
  // Decode any X message encoded in the
  // proxy message and write it to the X
  // connection.
  //

  virtual int handleWrite(const unsigned char *message, unsigned int length) = 0;

  //
  // Other methods to be implemented in
  // client, server and generic channel
  // classes.
  //

  virtual int handleSplit(EncodeBuffer &encodeBuffer, MessageStore *store, 
                              const unsigned char *buffer, const unsigned int size) = 0;

  virtual int handleSplit(EncodeBuffer &encodeBuffer, int packetLimit) = 0;

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                unsigned char *buffer, const unsigned int size) = 0;

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer) = 0;

  virtual int handleAbortSplit(EncodeBuffer &encodeBuffer) = 0;

  virtual int handleAbortSplit(DecodeBuffer &decodeBuffer) = 0;

  virtual int handleMotion(EncodeBuffer &encodeBuffer, int forceMotion = 0) = 0;

  virtual int handleWakeup(EncodeBuffer &encodeBuffer) = 0;

  virtual int handleConfiguration() = 0;

  virtual int handleFinish() = 0;

  virtual int handleNotify(T_notification_type type) = 0;

  //
  // Handle congestion events according to virtual
  // methods provided by channels.
  //

  int handleCongestion();

  //
  // This is not a pure method as derived channels
  // need to call their own handleReset() first.
  // In protocol older than 3, for example, they
  // must reset their not shared encode caches.
  //

  virtual int handleReset();

  //
  // Set pointer to object mapping opcodes
  // of NX specific messages.
  //

  int setOpcodes(OpcodeStore *opcodeStore);

  //
  // At channel creation or once reset is
  // completed, proxy needs to recreate all
  // the message stores. This is called to
  // update pointers in channels.
  //

  int setStores(ClientStore *clientStore, ServerStore *serverStore);

  //
  // The same for channels caches. Note that
  // channel caches are shared only starting
  // from protocol level 3.
  //

  int setCaches(ClientCache *clientCache, ServerCache *serverCache);

  //
  // Check if there are pending split 
  // to send to the remote side.
  //

  virtual int needSplit() const = 0;

  //
  // Check if there are motion events
  // to flush.
  //

  virtual int needMotion() const = 0;

  //
  // Check if there are agents having
  // clients to wakeup.
  //

  virtual int needWakeup() const = 0;

  virtual int updateWakeup() = 0;

  //
  // Check channel's own buffer size.
  //

  virtual int needLimit() = 0;

  //
  // Check if channel's local end needs
  // a timeout to cease the congestion
  // state.
  //

  virtual int needCongestion()
  {
    return (finish_ == 0 && isTimestamp(lastCongestionTs_) == 1);
  }

  virtual int updateCongestion()
  {
    //
    // Force proxy to go through the
    // congestion handling function.
    //

    lastCongestionTs_ = getTimestamp();

    return 1;
  }

  //
  // Return the maximum buffer size.
  //

  virtual int getLimit() = 0;

  //
  // Return the type of traffic carried
  // by this channel.
  //

  virtual T_channel_type getType() = 0;

  //
  // Check if channel must be flushed as
  // socket descriptor has been closed.
  //

  int getFinish() const
  {
    return finish_;
  }

  //
  // Check if we left messages in the 
  // read buffer.
  //

  int getPending() const
  {
    return (transport_ -> pending() != 0 ||
                pending_ != 0);
  }

  int getCongestion()
  {
    return congestion_;
  }

  //
  // Check if agent's clients finished
  // drawing.
  //

  int getFlush()
  {
    return flush_;
  }

  void clearFlush()
  {
    flush_ = 0;
  }

  //
  // Check if channel produced data for
  // the proxy link which has to be sent
  // as soon as possible.
  //

  int getPriority()
  {
    return priority_;
  }

  void clearPriority()
  {
    priority_ = 0;
  }

  protected:

  //
  // Encode and decode X messages based
  // on message stores.
  //

  int handleEncode(EncodeBuffer &encodeBuffer, ChannelCache *channelCache,
                       MessageStore *store, const unsigned char opcode,
                           const unsigned char *buffer, const unsigned int size);

  int handleEncode(EncodeBuffer &encodeBuffer, ChannelCache *channelCache,
                       MessageStore *store, const unsigned char *buffer,
                           const unsigned int size, int bigEndian)
  {
    return (store -> encodeIdentity(encodeBuffer, buffer, size,
                                        bigEndian, channelCache));
  }

  int handleDecode(DecodeBuffer &decodeBuffer, ChannelCache *channelCache,
                       MessageStore *store, unsigned char &opcode,
                           unsigned char *&buffer, unsigned int &size);

  int handleDecode(DecodeBuffer &decodeBuffer, ChannelCache *channelCache,
                       MessageStore *store, unsigned char *&buffer,
                           unsigned int &size, int bigEndian,
                               WriteBuffer *writeBuffer)
  {
    return (store -> decodeIdentity(decodeBuffer, buffer, size, bigEndian,
                                        writeBuffer, channelCache));
  }

  //
  // These offer an implementation valid
  // for both client and server channel.
  //

  int handleEncode(EncodeBuffer &encodeBuffer, ChannelCache *channelCache,
                       MessageStore *store,  const unsigned char *buffer,
                           const unsigned int size);

  int handleUpdate(MessageStore *store, const unsigned int dataSize,
                       const unsigned int compressedDataSize);

  int handleDecode(DecodeBuffer &decodeBuffer, ChannelCache *channelCache,
                       MessageStore *store, unsigned char *&buffer,
                           unsigned int &size);

  int handleSave(MessageStore *store, unsigned char *buffer, unsigned int size,
                     const unsigned char *compressedData = NULL, 
                         const unsigned int compressedDataSize = 0);

  int handleStartup(MessageStore *store, const int position);

  int handleFlush(T_flush type);

  int handleCopy(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                     const unsigned char *buffer, const unsigned int size)
  {
    encodeBuffer.encodeMemory(buffer, size);

    return 1;
  }

  int handleCopy(DecodeBuffer &decodeBuffer, const unsigned char opcode,
                     unsigned char *buffer, const unsigned int size)
  {
    memcpy(buffer, decodeBuffer.decodeMemory(size), size);

    return 1;
  }

  int handleCompress(EncodeBuffer &encodeBuffer, const unsigned char opcode,
                         const unsigned char *buffer, const unsigned int size,
                             unsigned int offset, unsigned char *&compressedData,
                                 unsigned int &compressedDataSize);

  int handleDecompress(DecodeBuffer &decodeBuffer, const unsigned char opcode,
                           unsigned char *buffer, const unsigned int size,
                               unsigned int offset, const unsigned char *&compressedData,
                                   unsigned int &compressedDataSize);

  //
  // Model handling of congestion events.
  //

  virtual int isCongested() = 0;

  virtual int isReliable() = 0;

  //
  // Establish rules on how to handle allocation
  // in message stores.
  //

  int mustCleanStore(MessageStore *store)
  {
    return (store -> getRemoteTotalStorageSize() > control -> RemoteTotalStorageSizeLimit ||
                store -> getLocalTotalStorageSize() > control -> LocalTotalStorageSizeLimit ||
                    (store -> getRemoteStorageSize() >
                         (control -> RemoteTotalStorageSizeLimit / 100 * store -> cacheThreshold)) ||
                              (store -> getLocalStorageSize() >
                                   (control -> LocalTotalStorageSizeLimit / 100 * store -> cacheThreshold)));
  }

  int canCleanStore(MessageStore *store)
  {
    return (store -> getSize() > 0 &&
                (store -> getRemoteStorageSize() >
                     (control -> RemoteTotalStorageSizeLimit / 100 * store -> cacheLowerThreshold)) ||
                          (store -> getLocalStorageSize() >
                               (control -> LocalTotalStorageSizeLimit / 100 *  store -> cacheLowerThreshold)));
  }

  int canSplitMessage(MessageStore *store, unsigned int size)
  {
    if (((int) size) > control -> SplitDataThreshold &&
            store -> lastStatus != is_discarded &&
                clientStore_ -> getSplitStore() -> getTotalStorageSize() <
                    control -> SplitTotalStorageSizeLimit &&
                        clientStore_ -> getSplitStore() -> getSize() < SPLIT_STORE_SIZE_LIMIT)
    {
      return 1;
    }

    return 0;
  }

  int wasSplitMessage(MessageStore *store)
  {
    return (store -> lastStatus == is_added &&
                store -> getLocks(store -> lastAdded) > 0);
  }

  void handleUpdateAgentClients(int client);

  protected:

  int bigEndian() const
  {
    return bigEndian_;
  }

  //
  // This is just used for debug. It checks
  // if any message in store has an invalid
  // lock state.
  //

  int checkLocks(MessageStore *store);

  //
  // Is X server big endian?
  //

  int bigEndian_;

  //
  // Other X server's features
  // saved at session startup.
  //

  unsigned int imageByteOrder_;
  unsigned int bitmapBitOrder_;
  unsigned int scanlineUnit_;
  unsigned int scanlinePad_;

  int clientSequenceReset_;
  int serverSequenceReset_;

  int firstRequest_;
  int firstReply_;

  //
  // Handle delay of congestion notification
  // to proxy.
  //

  T_timestamp lastCongestionTs_;

  //
  // Cache status of last 'keep' operation
  // on message stores.
  //

  int lastKeep_;

  //
  // Use this class for IO operations.
  //

  Transport *transport_;

  //
  // Compressor and decompressor are created
  // by proxy and shared between channels.
  //

  Compressor   *compressor_;
  Decompressor *decompressor_;

  //
  // Map NX operations to opcodes. Propagated
  // by proxy to all channels on the same X
  // server.
  //

  OpcodeStore *opcodeStore_;
 
  //
  // Also stores are shared between channels.
  //

  ClientStore *clientStore_;
  ServerStore *serverStore_;

  //
  // Caches are specific for each channel.
  //

  ClientCache *clientCache_;
  ServerCache *serverCache_;

  //
  // Data going to X connection.
  //

  WriteBuffer writeBuffer_;

  //
  // Other data members.
  //

  int fd_;

  int finish_;
  int pending_;
  int congestion_;

  //
  // Flush of proxy data is mandated in case
  // of priority, suggested in case of flush
  // counter greater than zero.
  //

  int flush_;
  int priority_;

  //
  // Print an info message on standard error
  // at the time the first X channel has been
  // successfully connected.
  //

  static int firstAgent_;

  //
  // Lower and upper limit of managed agent's
  // clients.
  //

  int lowerClient_;
  int upperClient_;

  //
  // Minimum and maximum id of agent's clients.
  //


  //
  // Keep track of object creation and
  // deletion.
  //

  #ifdef REFERENCES

  static int references_;

  #endif
};

#endif /* Channel_H */
