/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2004 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#define NEED_EVENTS

#include "X.h"
#include "Xlib.h"
#include "Xproto.h"
#include "Xatom.h"
#include "selection.h"
#include "windowstr.h"

#include "Window.h"
#include "Atoms.h"

/*
 * Use asyncronous get property replies.
 */

#include "NXlib.h"

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

int             agentClipboardStatus;
int             clientAccum;

Atom            serverCutProperty;
Atom            clientCutProperty;
Window          serverWindow;

Selection*      lastOwnerSelection;
ClientPtr       lastOwnerClient;
Window          lastOwnerWindow;
WindowPtr       lastOwnerWindowPtr;

/*
 * Needed to handle the notify selection event to
 * be sent to client once the selection property
 * has been retrieved from the real X server.
 */

typedef enum
{
  SelectionStageNone,
  SelectionStageQuerySize,
  SelectionStageWaitSize,
  SelectionStageQueryData,
  SelectionStageWaitData
} ClientSelectionStage;

WindowPtr     lastClientWindowPtr;
ClientPtr     lastClientClientPtr;
Window        lastClientRequestor;
Atom          lastClientProperty;
Atom          lastClientSelection;
Atom          lastClientTarget;
Time          lastClientTime;
Time          lastClientReqTime;
unsigned long lastClientPropertySize;

ClientSelectionStage lastClientStage;

Window          lastServerRequestor;
Atom            lastServerProperty;
Atom            lastServerTarget;
Time            lastServerTime;

Atom            serverTARGETS;
Atom            serverTEXT;
Atom            clientTARGETS;
Atom            clientTEXT;
Atom            clientCOMPOUND_TEXT;

char szAgentTARGETS[]       = "TARGETS";
char szAgentTEXT[]          = "TEXT";
char szAgentCOMPOUND_TEXT[] = "COMPOUND_TEXT";
char szAgentNX_CUT_BUFFER_CLIENT[] = "NX_CUT_BUFFER_CLIENT";

extern Display *nxagentDisplay;
extern WindowPtr *WindowTable;

Bool nxagentValidServerTargets(Atom target)
{
  #ifdef DEBUG
  fprintf(stderr, "nxagentValidServerTargets: Got called.\n");
  #endif

  if (target == XA_STRING) return True;
  if (target == serverTEXT) return True;

  return False;
}

void nxagentClearClipboard(ClientPtr pClient, WindowPtr pWindow)
{
  #ifdef DEBUG
  fprintf(stderr, "nxagentClearClipboard: Called with client [%p] window [%p].\n",
              (void *) pClient, (void *) pWindow);
  #endif

  if ((pClient != NULL && lastOwnerClient == pClient) ||
          (pWindow != NULL && lastOwnerWindowPtr == pWindow))
  {
    #ifdef TEST
    fprintf(stderr, "nxagentClearClipboard: Resetting state with client [%p] window [%p].\n",
                (void *) pClient, (void *) pWindow);
    #endif

    lastOwnerSelection = NULL;
    lastOwnerClient    = NULL;
    lastOwnerWindow    = None;
    lastOwnerWindowPtr = NULL;

    lastClientWindowPtr = NULL;
    lastClientStage     = SelectionStageNone;

    lastServerRequestor = None;
  }
}

void nxagentClearSelection(XEvent *X)
{
  xEvent x;
  int i;

  extern int NumCurrentSelections;
  extern Selection* CurrentSelections;

  #ifdef DEBUG
  fprintf(stderr, "nxagentClearSelection: Got called.\n");
  #endif

  if (agentClipboardStatus != 1)
    return;

  #ifdef DEBUG
  fprintf(stderr, "nxagentClearSelection: SelectionClear event.\n");
  #endif

  if (lastOwnerClient != NULL)
  {
    x.u.u.type = SelectionClear;
    x.u.selectionClear.time = GetTimeInMillis();
    x.u.selectionClear.window = lastOwnerWindow;
    x.u.selectionClear.atom = X->xselectionclear.selection;
    (void) TryClientEvents (lastOwnerClient , &x, 1,
                            NoEventMask, NoEventMask /* CantBeFiltered */,
                            NullGrab);
  }
  i = 0;
  while ((i < NumCurrentSelections) &&
           CurrentSelections[i].selection != X->xselectionclear.selection) i++;

  if (i < NumCurrentSelections)
  {
    CurrentSelections[i].window = WindowTable[0]->drawable.id;
    CurrentSelections[i].client = NullClient;
  }

  lastOwnerClient = NULL;
  lastOwnerWindow = None;

  lastClientWindowPtr = NULL;
  lastClientStage     = SelectionStageNone;
}


void nxagentRequestSelection(XEvent *X)
{
  int result;
  XSelectionEvent eventSelection;

  #ifdef DEBUG
  fprintf(stderr, "nxagentRequestSelection: Got called.\n");
  #endif

  if (agentClipboardStatus != 1)
    return;

  if (!nxagentValidServerTargets(X->xselectionrequest.target) ||
      (X->xselectionrequest.selection != XA_PRIMARY) ||
      (lastServerRequestor != None))
  {
/*
   char *strTarget;
     strTarget = XGetAtomName(nxagentDisplay, X->xselectionrequest.target);
     fprintf(stderr, "SelectionRequest event aborting sele=[%s] target=[%s]\n",
               NameForAtom(X->xselectionrequest.selection), NameForAtom(X->xselectionrequest.target));

    fprintf(stderr, "SelectionRequest event aborting sele=[%s] ext target=[%s] Atom size is [%d]\n",
             NameForAtom(X->xselectionrequest.selection), strTarget, sizeof(Atom));

    if (strTarget) XFree(strTarget);
*/
    eventSelection.property = None;

    if (X->xselectionrequest.target == serverTARGETS)
    {
      Atom xa_STRING = XA_STRING;
      result = XChangeProperty (nxagentDisplay,
                           X->xselectionrequest.requestor,
                           X->xselectionrequest.property,
                           X->xselectionrequest.target,
                           sizeof(Atom)*8,
                           PropModeReplace,
                           (unsigned char*)&xa_STRING,
                           1);
      eventSelection.property = X->xselectionrequest.property;
    }

    eventSelection.type = SelectionNotify;
    eventSelection.send_event = True;
    eventSelection.display = nxagentDisplay;
    eventSelection.requestor = X->xselectionrequest.requestor;
    eventSelection.selection = X->xselectionrequest.selection;
    eventSelection.target = X->xselectionrequest.target;
    eventSelection.time = X->xselectionrequest.time;

    result = XSendEvent(nxagentDisplay,
                        eventSelection.requestor,
                        False,
                        0L,
                        (XEvent *) &eventSelection);

    #ifdef DEBUG

    if (result == BadValue || result == BadWindow)
    {
      fprintf(stderr, "nxagentRequestSelection: WARNING! XSendEvent failed.\n");
    }
    else
    {
      fprintf(stderr, "nxagentRequestSelection: XSendEvent sent to window [0x%lx].\n",
                  eventSelection.requestor);
    }

    #endif

    return;
  }

  if ((lastClientWindowPtr != NULL) && (lastOwnerClient != NULL))
  {
    XConvertSelection(nxagentDisplay, X->xselectionrequest.selection,
                          X->xselectionrequest.target, serverCutProperty,
                              serverWindow, lastClientTime);

    #ifdef DEBUG
    fprintf(stderr, "nxagentRequestSelection: Sent XConvertSelection.\n");
    #endif
  }
  else
  {
    if (lastOwnerClient != NULL)
    {
      xEvent x;

      lastServerProperty=X->xselectionrequest.property;
      lastServerRequestor=X->xselectionrequest.requestor;
      lastServerTarget = X->xselectionrequest.target;
      lastServerTime=X->xselectionrequest.time;

      x.u.u.type = SelectionRequest;
      x.u.selectionRequest.time = GetTimeInMillis();
      x.u.selectionRequest.owner = lastOwnerWindow;

      /*
       * Ficticious window.
       */

      x.u.selectionRequest.requestor = WindowTable[0]->drawable.id;

      /*
       * Don't send the same window, some programs are
       * clever and verify cut and paste operations
       * inside the same window and don't Notify at all.
       *
       * x.u.selectionRequest.requestor = lastOwnerWindow;
       */

      x.u.selectionRequest.selection = X->xselectionrequest.selection;

      /*
       * x.u.selectionRequest.target = X->xselectionrequest.target;
       */

      x.u.selectionRequest.target = XA_STRING;
      x.u.selectionRequest.property = clientCutProperty;

      (void) TryClientEvents(lastOwnerClient, &x, 1,
                             NoEventMask, NoEventMask /* CantBeFiltered */,
                             NullGrab);

      #ifdef DEBUG
      fprintf(stderr, "nxagentRequestSelection: Executed TryClientEvents() with clientCutProperty.\n");
      #endif
   }
   else
   {
     /*
      * Probably we must to send a Notify
      * to requestor with property None.
      */

     eventSelection.type = SelectionNotify;
     eventSelection.send_event = True;
     eventSelection.display = nxagentDisplay;
     eventSelection.requestor = X->xselectionrequest.requestor;
     eventSelection.selection = X->xselectionrequest.selection;
     eventSelection.target = X->xselectionrequest.target;
     eventSelection.property = None;
     eventSelection.time = X->xselectionrequest.time;

     result = XSendEvent(nxagentDisplay,
                         eventSelection.requestor,
                         False,
                         0L,
                         (XEvent *) &eventSelection);

      #ifdef DEBUG
      fprintf(stderr, "nxagentRequestSelection: Executed XSendEvent with property None.\n");
      #endif
    }
  }
}

/*
 * The following functions could be used to check
 * if any collected property is available in our
 * event queue.
 */

static Bool nxagentCheckCollectPropertyNotifyPredicate(Display *display, XEvent *event, XPointer ptr)
{
  return (event -> xclient.data.l[0] == NXCollectPropertyNotify &&
              event -> type == ClientMessage && event -> xclient.window == 0 &&
                  event -> xclient.message_type == 0 && event -> xclient.format == 32 &&
                      event -> xclient.data.l[1] == *((unsigned int *) ptr));
}

int nxagentCheckCollectPropertyNotify(unsigned int resource)
{
    XEvent event;

    #ifdef DEBUG
    fprintf(stderr, "nxagentCheckCollectPropertyNotify: Checking collect property notify event with resource [%u].\n",
                resource);
    #endif

    while (XCheckIfEvent(nxagentDisplay, &event, nxagentCheckCollectPropertyNotifyPredicate,
                             (XPointer) &resource))
    {
        #ifdef DEBUG
        fprintf(stderr, "nxagentCheckCollectPropertyNotify: NXCollectPropertyNotify received with client [%ld].\n",
                    event.xclient.data.l[1]);
        #endif

        return 1;
    }

    #ifdef DEBUG
    fprintf(stderr, "nxagentCheckCollectPropertyNotify: NXCongestionNotify event not found in queue.\n");
    #endif

    return 0;
}

void nxagentSendSelectionNotify(Atom property)
{
  xEvent x;

  #ifdef DEBUG
  fprintf (stderr, "nxagentSendSelectionNotify: Sending event to client [%d].\n",
               lastClientClientPtr -> index);
  #endif

  x.u.u.type = SelectionNotify;

  x.u.selectionNotify.time      = lastClientTime;
  x.u.selectionNotify.requestor = lastClientRequestor;
  x.u.selectionNotify.selection = lastClientSelection;
  x.u.selectionNotify.target    = lastClientTarget;

  x.u.selectionNotify.property = property;

  TryClientEvents(lastClientClientPtr, &x, 1, NoEventMask,
                      NoEventMask , NullGrab);

  return;
}

void nxagentTransferSelection(int resource)
{
  int result;

  if (lastClientClientPtr -> index != resource)
  {
    #ifdef DEBUG
    fprintf (stderr, "nxagentTransferSelection: WARNING! Inconsistent resource [%d] with current client [%d].\n",
                 resource, lastClientClientPtr -> index);
    #endif

    nxagentSendSelectionNotify(None);

    lastClientWindowPtr = NULL;
    lastClientStage     = SelectionStageNone;

    return;
  }

  switch (lastClientStage)
  {
    case SelectionStageQuerySize:
    {
      /*
       * Don't get data yet, just get size. We skip
       * this stage in current implementation and
       * go straight to the data.
       */

      result = NXCollectProperty(nxagentDisplay,
                                 lastClientClientPtr -> index,
                                 serverWindow,
                                 serverCutProperty,
                                 0,
                                 0,
                                 False,
                                 AnyPropertyType);

      if (result == 0)
      {
        #ifdef DEBUG
        fprintf (stderr, "nxagentTransferSelection: Aborting selection notify procedure for client [%d].\n",
                     lastClientClientPtr -> index);
        #endif

        nxagentSendSelectionNotify(None);

        lastClientWindowPtr = NULL;
        lastClientStage     = SelectionStageNone;

        return;
      }

      #ifdef DEBUG
      fprintf (stderr, "nxagentTransferSelection: Setting stage to [%d] for client [%d].\n",
                   SelectionStageWaitSize, lastClientClientPtr -> index);
      #endif

      lastClientStage = SelectionStageWaitSize;

      break;
    }
    case SelectionStageQueryData:
    {
      /*
       * Request the selection data now.
       */

      #ifdef DEBUG
      fprintf(stderr, "nxagentTransferSelection: Getting property content from remote server.\n");
      #endif

      result = NXCollectProperty(nxagentDisplay,
                                 lastClientClientPtr -> index,
                                 serverWindow,
                                 serverCutProperty,
                                 0,
                                 lastClientPropertySize,
                                 False,
                                 AnyPropertyType);

      if (result == 0)
      {
        #ifdef DEBUG
        fprintf (stderr, "nxagentTransferSelection: Aborting selection notify procedure for client [%d].\n",
                     lastClientClientPtr -> index);
        #endif

        nxagentSendSelectionNotify(None);

        lastClientWindowPtr = NULL;
        lastClientStage     = SelectionStageNone;

        return;
      }

      #ifdef DEBUG
      fprintf (stderr, "nxagentTransferSelection: Setting stage to [%d] for client [%d].\n",
                   SelectionStageWaitData, lastClientClientPtr -> index);
      #endif

      lastClientStage = SelectionStageWaitData;

      break;
    }
    default:
    {
      #ifdef DEBUG
      fprintf (stderr, "nxagentTransferSelection: WARNING! Inconsistent state [%d] for client [%d].\n",
                   lastClientStage, lastClientClientPtr -> index);
      #endif

      break;
    }
  }
}

void nxagentCollectPropertyEvent(int resource)
{
  Atom                  atomReturnType;
  int                   resultFormat;
  unsigned long         ulReturnItems;
  unsigned long         ulReturnBytesLeft;
  unsigned char         *pszReturnData = NULL;
  int                   result;

  /*
   * We have received the notification so
   * we can safely retrieve data from the
   * client structure.
   */

  result = NXGetCollectedProperty(nxagentDisplay,
                                  lastClientClientPtr -> index,
                                  &atomReturnType,
                                  &resultFormat,
                                  &ulReturnItems,
                                  &ulReturnBytesLeft,
                                  &pszReturnData);

  if (result == 0)
  {
    #ifdef DEBUG
    fprintf (stderr, "nxagentCollectPropertyEvent: Failed to get reply data for client [%d].\n",
                 lastClientClientPtr -> index);
    #endif

    nxagentSendSelectionNotify(None);

    lastClientWindowPtr = NULL;
    lastClientStage     = SelectionStageNone;

    if (pszReturnData != NULL)
    {
      XFree(pszReturnData);
    }

    return;
  }

  if (lastClientClientPtr -> index != resource)
  {
    #ifdef DEBUG
    fprintf (stderr, "nxagentCollectPropertyEvent: WARNING! Inconsistent resource [%d] with current client [%d].\n",
                 resource, lastClientClientPtr -> index);
    #endif

    nxagentSendSelectionNotify(None);

    lastClientWindowPtr = NULL;
    lastClientStage     = SelectionStageNone;

    if (pszReturnData != NULL)
    {
      XFree(pszReturnData);
    }

    return;
  }

  switch (lastClientStage)
  {
    case SelectionStageWaitSize:
    {
      #ifdef DEBUG
      fprintf (stderr, "nxagentCollectPropertyEvent: Got size notify event for client [%d].\n",
                   lastClientClientPtr -> index);
      #endif

      if (ulReturnBytesLeft == 0)
      {
        #ifdef DEBUG
        fprintf (stderr, "nxagentCollectPropertyEvent: Aborting selection notify procedure for client [%d].\n",
                     lastClientClientPtr -> index);
        #endif

        nxagentSendSelectionNotify(None);

        lastClientWindowPtr = NULL;
        lastClientStage     = SelectionStageNone;

        if (pszReturnData != NULL)
        {
          Xfree(pszReturnData);
        }

        return;
      }

      #ifdef DEBUG
      fprintf(stderr, "nxagentCollectPropertyEvent: Got property size from remote server.\n");
      #endif

      /*
       * Request the selection data now.
       */

      lastClientPropertySize = ulReturnBytesLeft;
      lastClientStage        = SelectionStageQueryData;

      nxagentTransferSelection(resource);

      break;
    }
    case SelectionStageWaitData:
    {
      #ifdef DEBUG
      fprintf (stderr, "nxagentCollectPropertyEvent: Got data notify event for client [%d].\n",
                   lastClientClientPtr -> index);
      #endif

      if (ulReturnBytesLeft != 0)
      {
        #ifdef DEBUG
        fprintf (stderr, "nxagentCollectPropertyEvent: Aborting selection notify procedure for client [%d].\n",
                     lastClientClientPtr -> index);
        #endif

        nxagentSendSelectionNotify(None);

        lastClientWindowPtr = NULL;
        lastClientStage     = SelectionStageNone;

        if (pszReturnData != NULL)
        {
          Xfree(pszReturnData);
        }

        return;
      }

      #ifdef DEBUG
      fprintf(stderr, "nxagentCollectPropertyEvent: Got property content from remote server.\n");
      #endif

      ChangeWindowProperty(lastClientWindowPtr,
                           lastClientProperty,
                           lastClientTarget,
                           resultFormat, PropModeReplace,
                           ulReturnItems, pszReturnData, 1);

      #ifdef DEBUG
      fprintf(stderr, "nxagentCollectPropertyEvent: Selection property [%s] changed to [%s]\n",
                   NameForAtom(lastClientProperty), pszReturnData);
      #endif

      nxagentSendSelectionNotify(lastClientProperty);

      /*
       * Enable further requests from clients.
       */

      lastClientWindowPtr = NULL;
      lastClientStage     = SelectionStageNone;

      break;
    }
    default:
    {
      #ifdef DEBUG
      fprintf (stderr, "nxagentCollectPropertyEvent: WARNING! Inconsistent state [%d] for client [%d].\n",
                   lastClientStage, lastClientClientPtr -> index);
      #endif

      break;
    }
  }

  XFree(pszReturnData);
  pszReturnData = NULL;
}

void nxagentNotifySelection(XEvent *X)
{
  int result;

  XSelectionEvent eventSelection;

  #ifdef DEBUG
  fprintf(stderr, "nxagentNotifySelection: Got called.\n");
  #endif

  if (agentClipboardStatus != 1)
  {
    return;
  }

  #ifdef DEBUG
  fprintf(stderr, "nxagentNotifySelection: SelectionNotify event.\n");
  #endif

  if ((lastClientWindowPtr != NULL) && (X->xselection.property == serverCutProperty))
  {
    if (lastClientStage == SelectionStageNone)
    {
      #ifdef DEBUG
      fprintf(stderr, "nxagentNotifySelection: Starting selection transferral for client [%d].\n",
                  lastClientClientPtr -> index);
      #endif

      /*
       * The state machine is able to work in two phases. In the first
       * phase we get the size of property data, in the second we get
       * the actual data. We save a round-trip by requesting a prede-
       * termined amount of data in a single GetProperty and by discar-
       * ding the remaining part. This is not the optimal solution (we
       * could get the remaining part if it doesn't fit in a single
       * reply) but, at least with text, it should work in most situa-
       * tions.
       */

      lastClientStage        = SelectionStageQueryData;
      lastClientPropertySize = 262144;

      nxagentTransferSelection(lastClientClientPtr -> index);
    }
    else
    {
      #ifdef DEBUG
      fprintf(stderr, "nxagentNotifySelection: WARNING! Resetting selection transferral for client [%d].\n",
                  lastClientClientPtr -> index);
      #endif

      nxagentSendSelectionNotify(None);

      lastClientWindowPtr = NULL;
      lastClientStage     = SelectionStageNone;
    }

    return;
  }
  else
  {
/*
    fprintf (stderr, "SelectionNotify from inside property=%s\n",NameForAtom(X.xselection.property));
*/

    if (lastOwnerClient != NULL && lastOwnerWindowPtr != NULL  &&
            X->xselection.property == clientCutProperty)
    {
      Atom            atomReturnType;
      int             resultFormat;
      unsigned long   ulReturnItems;
      unsigned long   ulReturnBytesLeft;
      unsigned char   *pszReturnData = NULL;

      extern int GetWindowProperty(WindowPtr, Atom, long, long, Bool, Atom, Atom*, int*,
                                       unsigned long*, unsigned long*, unsigned char**);

      result = GetWindowProperty(lastOwnerWindowPtr, clientCutProperty, 0, 0, False,
                                      AnyPropertyType, &atomReturnType, &resultFormat,
                                          &ulReturnItems, &ulReturnBytesLeft, &pszReturnData);

      if (result == BadAlloc || result == BadAtom ||
              result == BadMatch || result == BadValue ||
                  result == BadWindow)
      {
        fprintf (stderr, "Client GetProperty failed error =");
        lastServerProperty=None;
        switch (result)
        {
          case BadAtom:
               fprintf (stderr, "BadAtom\n");
               break;
          case BadValue:
               fprintf (stderr, "BadValue\n");
               break;
          case BadWindow:
               fprintf (stderr, "BadWindow\n");
               break;
        }
      }
      else
      {
        result = GetWindowProperty(lastOwnerWindowPtr, clientCutProperty, 0, ulReturnBytesLeft, False,
                                       AnyPropertyType, &atomReturnType, &resultFormat,
                                           &ulReturnItems, &ulReturnBytesLeft, &pszReturnData);

        if (result == BadAlloc || result == BadAtom ||
                result == BadMatch || result == BadValue ||
                    result == BadWindow)
        {
          fprintf (stderr, "SelectionNotify - XChangeProperty failed\n");

          lastServerProperty=None;
        }
        else
        {
          result = XChangeProperty(nxagentDisplay,
                                   lastServerRequestor,
                                   lastServerProperty,
                                   lastServerTarget,
                                   8,
                                   PropModeReplace,
                                   pszReturnData,
                                   ulReturnItems);
        }
/*
        if (pszReturnData)
        {
          free(pszReturnData);
          pszReturnData=NULL;
        }
*/
      }

      eventSelection.type = SelectionNotify;
      eventSelection.send_event = True;
      eventSelection.display = nxagentDisplay;
      eventSelection.requestor = lastServerRequestor;
      eventSelection.selection = X->xselection.selection;;
/*
      eventSelection.target = X->xselection.target;*/
      eventSelection.target = lastServerTarget;
      eventSelection.property = lastServerProperty;
      eventSelection.time = lastServerTime;
/*
      eventSelection.time = CurrentTime;

      eventSelection.time = lastServerTime;
*/

      #ifdef DEBUG
      fprintf(stderr, "nxagentNotifySelection: Sending event to requestor.\n");
      #endif

      result = XSendEvent(nxagentDisplay,
                           eventSelection.requestor,
                           False,
                           0L,
                           (XEvent *) &eventSelection);

      if (result == BadValue || result == BadWindow)
      {
        fprintf (stderr, "SelectionRequest - XSendEvent failed\n");
      }

      lastServerRequestor=None; /* allow further request */
    }
  }
}

/*
 * Acquire selection so we don't get selection
 * requests from real X clients.
 */

void nxagentResetSelectionOwner()
{
  int result;

  if (lastServerRequestor != None)
  {
    #ifdef TEST
    fprintf (stderr, "nxagentResetSelectionOwner: WARNING! Requestor window [0x%lx] already found.\n",
                 lastServerRequestor);
    #endif

    return;
  }

  result = XSetSelectionOwner(nxagentDisplay, XA_PRIMARY, serverWindow, CurrentTime);

  if (result == BadAtom || result == BadWindow)
  {
    fprintf (stderr, "nxagentResetSelectionOwner: WARNING! XSetSelectionOwner() failed.\n");
  }
  else
  {
    fprintf (stderr, "nxagentResetSelectionOwner: Reset clipboard state for XA_PRIMARY.\n");

    lastOwnerSelection = NULL;
    lastOwnerClient    = NULL;
    lastOwnerWindow    = None;
    lastOwnerWindowPtr = NULL;

    lastClientWindowPtr = NULL;
    lastClientStage     = SelectionStageNone;

    lastServerRequestor = None;
  }

  return;
}

void nxagentSetSelectionOwner(Selection *pSelection)
{
  int result;

  #ifdef DEBUG
  fprintf(stderr, "nxagentSetSelectionOwner: Got called.\n");
  #endif

   if (agentClipboardStatus != 1)
   {
     return;
   }

   #ifdef DEBUG
   fprintf(stderr, "nxagentSetSelectionOwner: Setting selection owner to window [0x%lx].\n",
               serverWindow);
   #endif

  #ifdef TEST

  if (lastServerRequestor != None)
  {
    fprintf (stderr, "nxagentSetSelectionOwner: WARNING! Requestor window [0x%lx] already found.\n",
                 lastServerRequestor);
  }

  #endif

  result = XSetSelectionOwner(nxagentDisplay, pSelection->selection, serverWindow, CurrentTime);

  if (result == BadAtom || result == BadWindow)
  {
    fprintf (stderr, "nxagentSetSelectionOwner: WARNING! XSetSelectionOwner() failed.\n");
  }
  else
  {
    lastOwnerSelection=pSelection;
    lastOwnerClient = pSelection->client;
    lastOwnerWindow = pSelection->window;
    lastOwnerWindowPtr = pSelection->pWin;

    lastClientWindowPtr = NULL;
    lastClientStage     = SelectionStageNone;

    lastServerRequestor = None;
  }

/*
FIXME

   if (XGetSelectionOwner(nxagentDisplay,pSelection->selection)==serverWindow)
   {
      fprintf (stderr, "NXdispatch: SetSelectionOwner OK\n");

      lastOwnerSelection=pSelection;
      lastOwnerClient = pSelection->client;
      lastOwnerWindow = pSelection->window;
      lastOwnerWindowPtr = pSelection->pWin;

      lastClientWindowPtr = NULL;
      lastClientStage     = SelectionStageNone;

      lastServerRequestor = None;
   }
   else fprintf (stderr, "nxagentSetSelectionOwner: SetSelectionOwner failed\n");
*/
}

int nxagentConvertSelection(ClientPtr client, WindowPtr pWin, Atom selection,
                                Window requestor, Atom property, Atom target, Time time)
{
   char *strTarget;

  #ifdef DEBUG
  fprintf(stderr, "nxagentConvertSelection: Got called.\n");
  #endif

   if (agentClipboardStatus != 1)
       return 0;
   if (lastOwnerClient != NULL) /* there is a client owner on the agent side, let normal stuff happen */
       return 0;
   if (lastClientWindowPtr != NULL)
         return 0;

  #ifdef DEBUG
  fprintf(stderr, "nxagentConvertSelection: target=[%s] n=%ld, property [%s]\n",
              NameForAtom(target), target, NameForAtom(property));
  #endif

   strTarget = NameForAtom(target);
   if (strTarget == NULL)
      return 0;

   if (target == clientTARGETS)
   {
     Atom xa_STRING[3];
     xEvent x;

     xa_STRING[0] = XA_STRING;
     xa_STRING[1] = clientTEXT;
     xa_STRING[2] = clientCOMPOUND_TEXT;

     ChangeWindowProperty(pWin,
                          property,
                          target,
                          sizeof(Atom)*8,
                          PropModeReplace,
                          3,
                          &xa_STRING, 1);

      x.u.u.type = SelectionNotify;
      x.u.selectionNotify.time = time;
      x.u.selectionNotify.requestor = requestor;
      x.u.selectionNotify.selection = selection;
      x.u.selectionNotify.target = target;
      x.u.selectionNotify.property = property;

      (void) TryClientEvents(lastClientClientPtr, &x, 1, NoEventMask,
                           NoEventMask , NullGrab);

     return 1;
   }

   if (lastClientClientPtr == client /*&& lastClientProperty == property*/
        && (GetTimeInMillis() - lastClientReqTime < 5000))
   {
     /*
      * The same client made consecutive requests
      * of clipboard contents with less than 5
      * seconds time interval between them.
      */

     #ifdef DEBUG
     fprintf(stderr, "nxagentConvertSelection: Consecutives request from client [%p], selection [%ld], elapsed time [%lu] clientAccum [%d]\n",
                 (void *) client, selection, GetTimeInMillis() - lastClientReqTime, clientAccum);
     #endif

     clientAccum++;
/*
FIXME: I have to reset the counter somewhere, but I don't know where.

     if (clientAccum > 5)
        return 0;
*/
   }
   else
   {
     if (lastClientClientPtr != client)
        clientAccum = 0;
   }

   if ((target == clientTEXT) || (target == XA_STRING) || (target == clientCOMPOUND_TEXT))
   {
     lastClientWindowPtr = pWin;
     lastClientStage     = SelectionStageNone;
     lastClientRequestor = requestor;
     lastClientClientPtr = client;
     lastClientTime      = time;
     lastClientProperty  = property;
     lastClientSelection = selection;
     lastClientTarget    = target;

     lastClientReqTime = (lastClientReqTime - GetTimeInMillis()) > 5000?GetTimeInMillis():lastClientReqTime;

     XConvertSelection(nxagentDisplay, selection, XA_STRING,serverCutProperty,serverWindow,CurrentTime);

     #ifdef DEBUG
     fprintf(stderr, "nxagentConvertSelection: Sent XConvertSelection with target=[%s], property [%s]\n",
                 NameForAtom(target), NameForAtom(property));
     #endif

     return 1;
   }
   else
   {
     xEvent x;
     x.u.u.type = SelectionNotify;
     x.u.selectionNotify.time = time;
     x.u.selectionNotify.requestor = requestor;
     x.u.selectionNotify.selection = selection;
     x.u.selectionNotify.target = target;
     x.u.selectionNotify.property = None;
     (void) TryClientEvents(client, &x, 1, NoEventMask, NoEventMask , NullGrab);
     return 1;
   }
   return 0;
}

int nxagentSendNotify(xEvent *event)
{
  #ifdef DEBUG
  fprintf(stderr, "nxagentSendNotify: Got called.\n");
  #endif

   if (agentClipboardStatus != 1)
       return 0;
    if (event->u.selectionNotify.property == clientCutProperty)
    {
      XSelectionEvent x;
      int result;
/*          fprintf(stderr, "received notify from client 1\n");
*/
        /* Setup selection notify event to real server*/

      x.type = SelectionNotify;
      x.send_event = True;
      x.display = nxagentDisplay;
      x.requestor = serverWindow;
      x.selection = event->u.selectionNotify.selection;
      x.target = event->u.selectionNotify.target;
      x.property = event->u.selectionNotify.property;
      x.time = CurrentTime;

/*    fprintf(stderr, "going to send notify to server\n");*/

      #ifdef DEBUG
      fprintf(stderr, "nxagentSendNotify: Propagating clientCutProperty.\n");
      #endif

      result = XSendEvent (nxagentDisplay,
                                x.requestor,
                                False,
                                0L,
                                (XEvent *) &x);

      if (result == BadValue || result == BadWindow)
      {
         fprintf (stderr, "nxagentSendNotify: XSendEvent failed.\n");
      }

      return 1;
    }

    return 0;
}

WindowPtr nxagentGetClipboardWindow(Atom property, WindowPtr pWin)
{
  #ifdef DEBUG
  fprintf(stderr, "nxagentGetClipboardWindow: Got called.\n");
  #endif

  if (property == clientCutProperty && lastOwnerWindowPtr != NULL)
  {
    #ifdef DEBUG
    fprintf(stderr, "nxagentGetClipboardWindow: Returning last clipboard owner window.\n");
    #endif

    return lastOwnerWindowPtr;
  }
  else
  {
    #ifdef DEBUG
    fprintf(stderr, "nxagentGetClipboardWindow: Returning original target window.\n");
    #endif

    return pWin;
  }
}

int nxagentInitClipboard(WindowPtr pWin)
{
  int result;

  Window iWindow = nxagentWindow(pWin);

  #ifdef DEBUG
  fprintf(stderr, "nxagentInitClipboard: Got called.\n");
  #endif

  #ifdef NXAGENT_TIMESTAMP
  {
    extern unsigned long startTime;

    fprintf(stderr, "nxagentInitClipboard: Initializing start [%d] milliseconds.\n",
            GetTimeInMillis() - startTime);
  }
  #endif

  agentClipboardStatus = 0;
  serverWindow = iWindow;
  lastOwnerClient = NULL;
  lastServerRequestor = None;

  lastClientWindowPtr = NULL;
  lastClientStage     = SelectionStageNone;
  lastClientReqTime   = GetTimeInMillis();

  /* Assert ownership of PRIMARY */
  /*
  result = XSetSelectionOwner (nxagentDisplay, XA_PRIMARY,
                                iWindow, CurrentTime);

  if (result == BadAtom || result == BadWindow)
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentInitClipboard: PANIC! Could not set PRIMARY owner\n");
    #endif

    return -1;
  }
  */

  /*
   * Local property to hold pasted data.
   */

  serverCutProperty = nxagentAtoms[5];  /* NX_CUT_BUFFER_SERVER */

  if (serverCutProperty == None)
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentInitClipboard: PANIC! Could not create NX_CUT_BUFFER_SERVER atom\n");
    #endif

    return -1;
  }

  result = XSetSelectionOwner (nxagentDisplay, serverCutProperty,
                                iWindow, CurrentTime);

  if (result == BadAtom || result == BadWindow)
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentInitClipboard: PANIC! Could not set server property owner.\n");
    #endif

    return -1;
  }

  clientCutProperty = MakeAtom(szAgentNX_CUT_BUFFER_CLIENT, 
                                   strlen(szAgentNX_CUT_BUFFER_CLIENT), 1);

  if (clientCutProperty == None)
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentInitClipboard: PANIC! Could not create NX_CUT_BUFFER_CLIENT atom.\n");
    #endif

    return -1;
  }

  serverTARGETS          = nxagentAtoms[6];  /* TARGETS */
  serverTEXT             = nxagentAtoms[7];  /* TEXT */
  clientTARGETS          = MakeAtom(szAgentTARGETS, strlen(szAgentTARGETS), True);
  clientTEXT             = MakeAtom(szAgentTEXT, strlen(szAgentTEXT), True);
  clientCOMPOUND_TEXT    = MakeAtom(szAgentCOMPOUND_TEXT, strlen(szAgentCOMPOUND_TEXT), True);

  agentClipboardStatus = 1;

  #ifdef DEBUG
  fprintf(stderr, "nxagentInitClipboard: Clipboard initialization completed.\n");
  #endif

  #ifdef NXAGENT_TIMESTAMP
  {
    extern unsigned long startTime;

    fprintf(stderr, "nxagentInitClipboard: initializing ends [%d] milliseconds.\n",
                GetTimeInMillis() - startTime);
  }
  #endif

  return 1;
}
