#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <asm/io.h>

#include <asm/arch/gio.h>
#include <asm/arch/msic.h>

static msic_conf_t old_config;
static msic_conf_t act_config;

static int msic_locked;

DECLARE_WAIT_QUEUE_HEAD(msic_wait);

//#define DEBUG

#ifdef DEBUG
#define DBG(f, a...)	printk(f, ## a)
#else
#define DBG(f, a...)	do { } while (0)
#endif

static void do_config_set(msic_conf_t config)
{
	int err = 0;

	/* no change, nothing to do */
	if (config == act_config)
		return;
			
	switch (config) {
	case MSIC_SSFDC:
		/* switch on SSFDC */
		reg_write(DM310_EXTBUS_BUSMD, 0x0002, 0x0003);
				
		/* MSIC to SSFDC-only */
		reg_write(DM310_GIO_GLOCNF1, (2<<6), (7<<6));
		
		/* switch off EXTBUS interrupts on CFRDY */
		reg_write(DM310_EXTBUS_BUSINTEN, 0, 2);
		
		DBG("msic: switching to SSFDC\n");
		break;
		
	case MSIC_CFC:	
		/* switch to CFC */
		reg_write(DM310_EXTBUS_BUSMD, 0x0000, 0x0003);

		/* MSIC to CFC-only */
		reg_write(DM310_GIO_GLOCNF1, (1<<6), (7<<6));
		
		/* enable EXTBUS interrupts on CFRDY */
		reg_write(DM310_EXTBUS_BUSINTEN, 2, 2);

		DBG("msic: switching to CFC\n");
		break;	

	case MSIC_MMCSD:
		/* MSIC to MMC/SD/MS only */
		reg_write(DM310_GIO_GLOCNF1, (7<<6), (7<<6));
		
		/* MSMD to MMC */
		reg_write(DM310_GIO_GLOCNF1, (1<<4), (3<<4));
		DBG("msic: switching to MMCSD\n");
		break;
			
	default:
		printk(KERN_INFO "msic_config_set: unknown config %d\n", config);
		err = 1;
		break;
	}
	
	if (!err) {
		old_config = act_config;
		act_config = config;
	}
}

static int msic_lock(void)
{
	unsigned long flags;
	
	DBG("msic_lock()\n");
	
	save_flags(flags);
	cli();
	
	while (msic_locked) {
		DBG("msic_lock: waiting for lock\n");
		sleep_on(&msic_wait);
	}
	
	msic_locked = 1;
	
	restore_flags(flags);
	
	return 0;
}

static void msic_unlock(void)
{
	unsigned long flags;

	DBG("msic_unlock()\n");
		
	save_flags(flags);
	cli();
	
	if (msic_locked) {
		DBG("msic_unlock: waking up sleepers\n");
		wake_up(&msic_wait);
		msic_locked = 0;
	}
	
	restore_flags(flags);
}

/**
 * set a specific configuration for the MSIC. Access to the MSIC
 * is guarded by a lock, so this function may sleep.
 *
 */
int msic_config_set(msic_conf_t config)
{
	if (msic_lock() < 0) {
		printk(KERN_ERR "msic_config_set: unable to lock MSIC\n");
		return -EBUSY;
	}
	do_config_set(config);
	
	return 0;
}

void msic_config_restore(void)
{
	do_config_set(old_config);
	msic_unlock();
}

/**
 * release the MSIC w it's no longer needed. this function
 * will not restore the old configuration.
 */
void msic_config_release(void)
{
	msic_unlock();
}


/*
 *
 */
int msic_init(void)
{
	printk(KERN_INFO "msic: initializing\n");
	msic_locked = 0;
	do_config_set(MSIC_CFC);		
	return 0;
}

EXPORT_SYMBOL(msic_config_set);
EXPORT_SYMBOL(msic_config_restore);
EXPORT_SYMBOL(msic_config_release);
