/*
 * Glue audio driver for the HP iPAQ H3900 & Philips UDA1380 codec.
 *
 * Copyright (c) 2000 Nicolas Pitre <nico@cam.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * This is the machine specific part of the HP iPAQ (aka Bitsy) support.
 * This driver makes use of the UDA1380 and the pxa-audio modules.
 *
 * History:
 *
 * 2002-12-18   Jamey Hicks     Adapted from h3600-uda1341.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/i2c.h>
#include <linux/kmod.h>
#include "uda1380.h"

#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/dma.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/arch/h3900_asic.h>

#include "pxa-i2s.h"

#undef DEBUG
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif


#define AUDIO_NAME		"Bitsy_UDA1380"

#define AUDIO_RATE_DEFAULT	44100

static int
mixer_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
	/*
	 * We only accept mixer (type 'M') ioctls.
	 */
	if (_IOC_TYPE(cmd) != 'M')
		return -EINVAL;

	return i2c_command(file->private_data, cmd, (void *)arg);
}

static int mixer_open(struct inode *inode, struct file *file)
{
	struct i2c_client *uda1380;

	uda1380 = i2c_get_client(I2C_DRIVERID_UDA1380, I2C_ALGO_PXA, NULL);

	if (uda1380 == NULL) {
		printk("No UDA1380 detected.  Is i2c-adap-pxa loaded?\n");
		return -ENODEV;
	}

	file->private_data = uda1380;
	i2c_inc_use_client(uda1380);

	return 0;
}

static int mixer_release(struct inode *inode, struct file *file)
{
	i2c_dec_use_client(file->private_data);
	return 0;
}

static struct file_operations h3900_mixer_fops = {
	open:		mixer_open,
	release:	mixer_release,
	ioctl:		mixer_ioctl,
	owner:		THIS_MODULE
};


/*
 * Audio interface
 */

static long audio_samplerate = AUDIO_RATE_DEFAULT;

static void h3900_set_samplerate(struct i2c_client *uda1380, long val)
{
	struct uda1380_cfg cfg;

	h3600_audio_clock (val);

	cfg.format = FMT_I2S;
	cfg.mic_connected = 1;
	cfg.line_connected = 0;
	cfg.fs = pxa_i2s_set_samplerate (val);

	i2c_command(uda1380, I2C_UDA1380_CONFIGURE, &cfg);

	audio_samplerate = val;
}

static void h3900_audio_init(void *data)
{
	struct i2c_client *uda1380 = (struct i2c_client *)data;
	unsigned long flags;

	pxa_i2s_init ();

	/* Blip the UDA1380 reset line */
	local_irq_save(flags);
	H3900_ASIC3_GPIO_B_OUT |= GPIO3_AUD_RESET;
 	local_irq_restore(flags);

	/* Enable the audio power */
	h3600_audio_clock(audio_samplerate);
	h3600_audio_power(audio_samplerate);

	mdelay(1);

	local_irq_save(flags);
	H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_AUD_RESET;
 	local_irq_restore(flags);

	/* Wait for the UDA1380 to wake up */
	mdelay(1);

	/* Initialize the UDA1380 internal state */
	i2c_command(uda1380, I2C_UDA1380_OPEN, 0);

	h3600_audio_mute(0);
}

static void h3900_audio_shutdown(void *data)
{
	struct i2c_client *uda1380 = (struct i2c_client *)data;

	/* disable the audio power and all signals leading to the audio chip */
	h3600_audio_mute(1);
	i2c_command(uda1380, I2C_UDA1380_CLOSE, 0);

	pxa_i2s_shutdown ();

	h3600_audio_power(0);
	h3600_audio_clock(0);
}

static int h3900_audio_ioctl(struct inode *inode, struct file *file,
			     uint cmd, ulong arg)
{
	long val;
	int ret = 0;
	audio_state_t *s = (audio_state_t *)file->private_data;
	struct i2c_client *uda1380 = s->data;

	/*
	 * These are platform dependent ioctls which are not handled by the
	 * generic pxa-audio module.
	 */
	switch (cmd) {
	case SNDCTL_DSP_STEREO:
		ret = get_user(val, (int *) arg);
		if (ret)
			return ret;
		/* the UDA1380 is stereo only */
		ret = (val == 0) ? -EINVAL : 1;
		return put_user(ret, (int *) arg);

	case SNDCTL_DSP_CHANNELS:
	case SOUND_PCM_READ_CHANNELS:
		/* the UDA1380 is stereo only */
		return put_user(2, (long *) arg);

	case SNDCTL_DSP_SPEED:
		ret = get_user(val, (long *) arg);
		if (ret) break;
		h3900_set_samplerate(uda1380, val);
		/* fall through */

	case SOUND_PCM_READ_RATE:
		return put_user(audio_samplerate, (long *) arg);

	case SNDCTL_DSP_SETFMT:
	case SNDCTL_DSP_GETFMTS:
		/* we can do 16-bit only */
		return put_user(AFMT_S16_LE, (long *) arg);

	default:
		/* Maybe this is meant for the mixer (As per OSS Docs) */
		return mixer_ioctl(inode, file, cmd, arg);
	}

	return ret;
}



static audio_state_t audio_state = {
	output_stream:		&pxa_i2s_output_stream,
	input_stream:		&pxa_i2s_input_stream,
	hw_init:		h3900_audio_init,
	hw_shutdown:		h3900_audio_shutdown,
	client_ioctl:		h3900_audio_ioctl,
	sem:			__MUTEX_INITIALIZER(audio_state.sem),
};

static int h3900_audio_release(struct inode *inode, struct file *file)
{
	audio_state_t *s = (audio_state_t *)file->private_data;
	struct i2c_client *uda1380 = s->data;

	i2c_dec_use_client(uda1380);
	
	return pxa_audio_release(inode, file);
}

static int h3900_audio_open(struct inode *inode, struct file *file)
{
	int err;
	struct i2c_client *uda1380;

	uda1380 = i2c_get_client(I2C_DRIVERID_UDA1380, I2C_ALGO_PXA, NULL);

	if (uda1380 == NULL) {
		printk("No UDA1380 detected.  Is i2c-adap-pxa loaded?\n");
		return -ENODEV;
	}

	audio_state.data = uda1380;

	err = pxa_audio_attach(inode, file, &audio_state);
	if (err == 0)
		i2c_inc_use_client(uda1380);

	return err;
}

/*
 * Missing fields of this structure will be patched with the call
 * to pxa_audio_attach().
 */
static struct file_operations h3900_audio_fops = {
	open:		h3900_audio_open,
	release:	h3900_audio_release,
	owner:		THIS_MODULE
};


static int audio_dev_id, mixer_dev_id;

static int __init h3900_uda1380_module_init(void)
{
	if (!machine_is_h3900 () && !machine_is_h1900())
		return -ENODEV;

	/* register devices */
	audio_dev_id = register_sound_dsp(&h3900_audio_fops, -1);
	mixer_dev_id = register_sound_mixer(&h3900_mixer_fops, -1);

	/* try to bring in i2c support */
	request_module("i2c-adap-pxa");
	request_module("uda1380");

	printk( KERN_INFO "iPAQ H3900 audio support initialized\n" );
	return 0;
}

static void __exit h3900_uda1380_module_exit(void)
{
	unregister_sound_dsp(audio_dev_id);
	unregister_sound_mixer(mixer_dev_id);
}

module_init(h3900_uda1380_module_init);
module_exit(h3900_uda1380_module_exit);

MODULE_AUTHOR("Nicolas Pitre, George France, Jamey Hicks");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Glue audio driver for the HP iPAQ H3900 & Philips UDA1380 codec.");

EXPORT_NO_SYMBOLS;
