/*
 *  linux/drivers/char/axim_ts.c
 *
 *  Copyright (C) 2003 Martin Demin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Changelog:
 *   3-Jun-2003 Fixed a bug in touchscreen
 *
 *  ToDo: Read battery voltage, move some functions away ...
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/compiler.h>
#include <linux/interrupt.h>
#include <linux/apm_bios.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/axim_ts.h>
#include <asm/hardware/wm9705.h>
#include <asm/hardware/mq1132.h>

/*
 * TSBUF_SIZE must be a power of two
 */
//#define READ_PRESSURE

#define LIGHT_3900_TO_AXIM

#define BATT_V_MAX	0x630
#define BATT_V_MIN	0x5D8

#define AXIM_TS_MINOR	0
#define AXIM_TSRAW_MINOR	1
#define TSBUF_SIZE	256
#define NEXT(index)	(((index) + 1) & (TSBUF_SIZE - 1))
#define IRQ_TOUCHSCREEN (32+22)

#define AXIM_TS_X_MAX	0x0EC1
#define AXIM_TS_Y_MAX	0x0F4F
#define AXIM_TS_X_MIN	0x0109
#define AXIM_TS_Y_MIN	0x00DE

#define PEN(x) ((x%10)<7)
#define MOVEMENT_DELTA_X 3
#define MOVEMENT_DELTA_Y 3

#define CONV_DELAY 9
#define ANTI_MOVE 2


struct pen_status {
unsigned int pressure,x,y;
};

struct pen_status lastraw,lastn;
static unsigned short buffer[TSBUF_SIZE][4],x1,y1;
static int head, tail,pen,pen1;
static DECLARE_WAIT_QUEUE_HEAD(queue);
static DECLARE_MUTEX(open_sem);
static spinlock_t tailptr_lock = SPIN_LOCK_UNLOCKED;
static struct fasync_struct *fasync;
static int gMajor,raw_read;		
static devfs_handle_t devfs_ts, devfs_ts_dir, devfs_tsraw;
static unsigned long last_x,last_y,raw_read_x,raw_read_y,read_x,read_y;
static unsigned anti_x,anti_y;

u16 ac97_digi_read(int);

#ifdef READ_PRESSURE
static unsigned long last_pres;
#endif

/*
 * Interrupt handler and standard file operations
 */
void usleep(int a)
{
    int i;
    for(i=0;i<a*400;i++);
}

inline int abs(int a)
{
    if(a<0) return -a;
    return a;
}

///////////////////////////////////////////////////////////
// This is to be moved to axim.c (or hal ???) !!!
//////////////////////////////////////////////////////////

void axim_get_flite(struct axim_ts_backlight *v)
{
// Todo Turn it off in MediaQ
    v->power=1;
    v->brightness=PWM_PWDUTY0 & 0xff;
}

int axim_set_flite(enum flite_pwr power, int brightness)
{
		struct axim_ts_backlight v;
		axim_get_flite(&v);
#ifdef LIGHT_3900_TO_AXIM
    if(brightness)
	brightness=((brightness+1)*4)-1;
#endif    
    if(brightness) 
	SP08R= (SP08R & ~0x10) + 0x10;
    else
	SP08R= (SP08R & ~0x10);
    
    
    if(brightness>255) brightness=255;
// Todo Turn it on in MediaQ
//    v->power=1;
    PWM_PWDUTY0 = (PWM_PWDUTY0 & 0xffffff00) |  (brightness & 0xff);
    return 0;
}


int axim_get_battery( struct h3600_battery *v )
{
    unsigned int voltage;
    v->ac_status = !(GPLR0 & (1<<17)); // GPIO 17
    v->battery_count=1;
    v->battery[0].chemistry=H3600_BATT_CHEM_LION;
    if(!(GPLR1 & (1<<(59-32))))
        v->battery[0].status=H3600_BATT_STATUS_CHARGING;
    else
        v->battery[0].status=H3600_BATT_STATUS_HIGH|H3600_BATT_STATUS_FULL;
    if(GPLR0 & (1<<9))
        v->battery[0].status=H3600_BATT_STATUS_NOBATT;
    voltage=ac97_digi_read(AC97_DIGICON1_ADR_BMON);
    v->battery[0].voltage=voltage;
//    printk("voltage: %d\t",voltage);
    v->battery[0].percentage=(100*(voltage-BATT_V_MIN))/(BATT_V_MAX-BATT_V_MIN);
    if(v->battery[0].percentage>100) v->battery[0].percentage=100;
    if(v->battery[0].percentage<0) v->battery[0].percentage=0;
    v->battery[0].life=8*60*v->battery[0].percentage/100; // todo
    return 0;
}
EXPORT_SYMBOL(axim_get_battery);

int h3600_apm_get_power_status(u_char *ac_line_status,
			       u_char *battery_status, 
			       u_char *battery_flag, 
			       u_char *battery_percentage, 
			       u_short *battery_life)
{
	struct h3600_battery bstat;
	unsigned char ac    = APM_AC_UNKNOWN;
	unsigned char level = APM_BATTERY_STATUS_UNKNOWN;
	int status, result;

	result = axim_get_battery(&bstat);
	if (result) {
		printk("%s: unable to access battery information\n", __FUNCTION__);
		return 0;
	}

	switch (bstat.ac_status) {
	case H3600_AC_STATUS_AC_OFFLINE:
		ac = APM_AC_OFFLINE;
		break;
	case H3600_AC_STATUS_AC_ONLINE:
		ac = APM_AC_ONLINE;
		break;
	case H3600_AC_STATUS_AC_BACKUP:
		ac = APM_AC_BACKUP;
		break;
	}

	if (ac_line_status != NULL)
		*ac_line_status = ac;

	status = bstat.battery[0].status;
	if (status & (H3600_BATT_STATUS_CHARGING | H3600_BATT_STATUS_CHARGE_MAIN))
		level = APM_BATTERY_STATUS_CHARGING;
	else if (status & (H3600_BATT_STATUS_HIGH | H3600_BATT_STATUS_FULL))
		level = APM_BATTERY_STATUS_HIGH;
	else if (status & H3600_BATT_STATUS_LOW)
		level = APM_BATTERY_STATUS_LOW;
	else if (status & H3600_BATT_STATUS_CRITICAL)
		level = APM_BATTERY_STATUS_CRITICAL;

	if (battery_status != NULL)
		*battery_status = level;

	if (battery_percentage != NULL)
		*battery_percentage = bstat.battery[0].percentage;

	/* assuming C/5 discharge rate */
	if (battery_life != NULL) {
		*battery_life = bstat.battery[0].life;
		*battery_life |= 0x8000;   /* Flag for minutes */
	}
                        
	return 1;
}
EXPORT_SYMBOL(h3600_apm_get_power_status);
//////////////////////////////////////////////////////////////

// reads and decodes nonAC97 codec ADC value
// reg = 1..7

volatile int pen_down;

u16 ac97_digi_read(int reg)
{
    unsigned int old_reg, val, rep=0;
    old_reg = pxa_ac97_read(NULL, AC97_DIGICON1) | AC97_DIGICON1_POLL;
    pxa_ac97_write(NULL, AC97_DIGICON1,
    (old_reg & ~AC97_DIGICON1_ADR) | (reg << AC97_DIGICON1_ADR_SHIFT));
// wait 4 completition
reread:
    rep++;
// Error, we return zero, because that is less harmful to programs than -1
    if(rep>40) return 0;
//    usleep(1);
    val = pxa_ac97_read(NULL, AC97_DIGIDATA);
// check if the right data has been read, if not, repeat    
    if(((val & AC97_DIGIDATA_ADR) >> AC97_DIGIDATA_ADR_SHIFT) != reg) goto reread;
    pen_down=(val & AC97_DIGIDATA_PNDN) >> AC97_DIGIDATA_PNDN_SHIFT;
    return (val & AC97_DIGIDATA_DATA);
}

inline int  ac97_digi_pen_down()
{
 return pen_down;
}

#if 1
static void
axim_ts_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long x,y,q,nx,ny;

	pen_down=GPLR1 & 1;
	if(!pen_down)	anti_x=anti_y=0;
	wake_up_interruptible(&queue);
	kill_fasync(&fasync, SIGIO, POLL_IN);

}
#endif
static ssize_t
axim_ts_read(struct file *filp, char *buf, size_t count, loff_t *l)
{
	unsigned long x,y,q,nx,ny,dx,dy;
#ifdef READ_PRESSURE
	unsigned pres;
#endif
	unsigned short data[4];
	ssize_t written = 0;

	while (count >= sizeof data) {
//	nx=ac97_digi_read(AC97_DIGICON1_ADR_X);
//	ny=ac97_digi_read(AC97_DIGICON1_ADR_Y);

	nx=read_x;
	ny=read_y;
#ifdef READ_PRESSURE
	pres=ac97_digi_read(AC97_DIGICON1_ADR_PRESS);
#endif

	if(ac97_digi_pen_down()) 
	{
	    data[0] = !(pen%3);
	    if(!data[0])
	     pen++;
	}
	else
	{
	    y1=x1=-1;
	    data[0] = 0x0;
	    pen=0;
	}


#if 1
	if(data[0]==0)
	{
	    x=last_x;
	    y=last_y;
	}
	    else
	{
	    last_x=x=nx;
	    last_y=y=ny;
	}
	raw_read=0;
#endif		
	
	if(x<AXIM_TS_X_MIN) x=AXIM_TS_X_MIN;
	if(x>AXIM_TS_X_MAX) x=AXIM_TS_X_MAX;
	x=x-AXIM_TS_X_MIN;
	x*=240;
	x/=AXIM_TS_X_MAX-AXIM_TS_X_MIN;

	if(y<AXIM_TS_Y_MIN) x=AXIM_TS_Y_MIN;
	if(y>AXIM_TS_Y_MAX) x=AXIM_TS_Y_MAX;
	y=y-AXIM_TS_Y_MIN;
	y*=320;
	y/=AXIM_TS_Y_MAX-AXIM_TS_Y_MIN;
	y=320-y;

#if 0
	if(data[0])
	{
	 if(x1==-1) 
	 {
	    x1=x;
	    y1=y;
	 }
	 dx=x-x1;
	 if(dx<0) dx=-dx;
	 dy=y-y1;
	 if(dy<0) dy=-dy;
	 if(dx>MOVEMENT_DELTA_X)
	  if(dy>MOVEMENT_DELTA_Y)
	  {
	   pen++;
	   x1=x;
	   y1=y;
	  } 
	}
#endif
#ifdef READ_PRESSURE
	if(data[0])
	{
	    data[0]=pres;
	    last_pres=pres;
	}
#endif
	lastn.pressure=data[0];
	lastn.x=x;
	lastn.y=y;

	data[1]=x;
	data[2]=y;
	data[3]=0;
//	printk("x=0x%d\ty=0x%d\tpres=%d\n",x,y,data[0]);

		if (copy_to_user(buf, data, sizeof data))
			return -EFAULT;
		count -= sizeof data;
		buf += sizeof data;
		written += sizeof data;
	}
	return written ? written : -EINVAL;
}

static ssize_t
axim_tsraw_read(struct file *filp, char *buf, size_t count, loff_t *l)
{
	unsigned long x,y,q,nx,ny,dx,dy;
//	static unsigned last_x,last_y;
#ifdef READ_PRESSURE
	unsigned pres;
#endif
	unsigned short data[4];
	ssize_t written = 0;

	while (count >= sizeof data) {
//	nx=ac97_digi_read(AC97_DIGICON1_ADR_X);
//	ny=ac97_digi_read(AC97_DIGICON1_ADR_Y);
	nx=raw_read_x;
	ny=raw_read_y;

#ifdef READ_PRESSURE
	pres=ac97_digi_read(AC97_DIGICON1_ADR_PRESS);
#endif
	raw_read=1;
	if(ac97_digi_pen_down()) 
	{
	    data[0] = 1;
	}
	else
	{
//	    y1=x1=-1;
	    data[0] = 0x0;
	}

#if 1
	if(data[0]==0)
	{
	    x=last_x;
	    y=last_y;
	}
	    else
	{
	    last_x=x=nx;
	    last_y=y=ny;
	}
//	raw_read=0;
#else
	x=nx;
	y=ny;
#endif		


#if 0
	if(data[0])
	{
	 if(x1==-1) 
	 {
	    x1=x;
	    y1=y;
	 }
	 dx=x-x1;
	 if(dx<0) dx=-dx;
	 dy=y-y1;
	 if(dy<0) dy=-dy;
	 if(dx>MOVEMENT_DELTA_X)
	  if(dy>MOVEMENT_DELTA_Y)
	  {
	   pen++;
	   x1=x;
	   y1=y;
	  } 
	}
#endif
#ifdef READ_PRESSURE
	if(data[0])
	{
	    data[0]=pres;
	    last_pres=pres;
	}
#endif

	lastraw.pressure=data[0];
	lastraw.x=x;
	lastraw.y=y;
	
	if(x<AXIM_TS_X_MIN) {x=AXIM_TS_X_MIN; data[0]=0;}
	if(x>AXIM_TS_X_MAX) x=AXIM_TS_X_MAX;
	x=x-AXIM_TS_X_MIN;
	x*=240;
	x/=AXIM_TS_X_MAX-AXIM_TS_X_MIN;

	if(y<AXIM_TS_Y_MIN) {y=AXIM_TS_Y_MIN;data[0]=0;}
	if(y>AXIM_TS_Y_MAX) y=AXIM_TS_Y_MAX;
	y=y-AXIM_TS_Y_MIN;
	y*=320;
	y/=AXIM_TS_Y_MAX-AXIM_TS_Y_MIN;
	y=320-y;

	if(abs(x-anti_x)>ANTI_MOVE||abs(y-anti_y)>ANTI_MOVE)
	{
	    anti_x=data[1]=x;
	    anti_y=data[2]=y;
	    data[3]=0;
	}
	    else
	{
	    data[1]=anti_x;
	    data[2]=anti_y;
	    data[3]=0;
	}

//	printk("x=0x%d\ty=0x%d\tpres=%d\n",x,y,data[0]);

		if (copy_to_user(buf, data, sizeof data))
			return -EFAULT;
		count -= sizeof data;
		buf += sizeof data;
		written += sizeof data;
	}
	return written ? written : -EINVAL;

}

int poll,pollraw;

static unsigned int
axim_ts_poll(struct file *filp, poll_table *wait)
{
	int p=0;
	poll_wait(filp, &queue, wait);
	
//	poll++;
//	if(poll%4) return 0;
	
//	ac97_digi_read(0);
	if(ac97_digi_pen_down()!=lastn.pressure) p=1;
        if(ac97_digi_pen_down())
	{
	    if((raw_read_x=ac97_digi_read(AC97_DIGICON1_ADR_X))!=lastraw.x) p=1;
	    if((raw_read_y=ac97_digi_read(AC97_DIGICON1_ADR_Y))!=lastraw.y) p=1;
	}


	return p ? POLLIN | POLLRDNORM : 0;
}

static unsigned int
axim_tsraw_poll(struct file *filp, poll_table *wait)
{
	int p=0;
	poll_wait(filp, &queue, wait);

//	pollraw++;
//	if(pollraw%4) return 0;
	
//	ac97_digi_read(0);
	if(ac97_digi_pen_down()!=lastraw.pressure) {p=1; }
        if(ac97_digi_pen_down())
	{
	    if((raw_read_x=ac97_digi_read(AC97_DIGICON1_ADR_X))!=lastraw.x) p=1;
	    if((raw_read_y=ac97_digi_read(AC97_DIGICON1_ADR_Y))!=lastraw.y) p=1;
	    if(ac97_digi_pen_down()) // throw it away 
	    {
		raw_read_x=ac97_digi_read(AC97_DIGICON1_ADR_X);
		raw_read_y=ac97_digi_read(AC97_DIGICON1_ADR_Y);
	    }
		else
	    p=0;
	    
	}
	return p ? POLLIN | POLLRDNORM : 0;
}

static int
axim_ts_ioctl(struct inode *inode, struct file *filp,
		unsigned int cmd, unsigned long arg)
{
	int retval = 0;

        if (0) printk("%s: cmd=%x\n", __FUNCTION__,cmd);

        switch (cmd) {
	case TS_GET_BACKLIGHT:
	{
		struct axim_ts_backlight v;
		axim_get_flite(&v);
		if ( copy_to_user((void *)arg, &v, sizeof(v)))
			retval = -EFAULT;
		break;
	}
	case TS_SET_BACKLIGHT:
	{
		struct axim_ts_backlight v;
		if (copy_from_user(&v, (void *)arg, sizeof(v)))
			return -EFAULT;
/*		if ( v.power != FLITE_PWR_OFF && v.power != FLITE_PWR_ON )
			return -EINVAL;
*/			
		retval = axim_set_flite(v.power, v.brightness);
		break;
	}

	case FLITE_ON:
	{
		struct h3600_ts_flite v;
		if (copy_from_user(&v, (void *)arg, sizeof(v)))
			return -EFAULT;

		switch (v.mode) {
		case FLITE_AUTO_MODE:
		case FLITE_MANUAL_MODE:
//                        if ( v.pwr != FLITE_PWR_OFF && v.pwr != FLITE_PWR_ON )
//                                return -EINVAL;
			retval = axim_set_flite(v.pwr, v.brightness);
			break;
/*		case FLITE_GET_LIGHT_SENSOR:
			retval = h3600_get_light_sensor(&v.brightness);
			if ( !retval &&  copy_to_user((void *)arg, &v, sizeof(v)))
				retval = -EFAULT;
			break;
*/
		default:
			retval = -EINVAL;
			break;
		}
		break;
	}
	case GET_BATTERY_STATUS:
	{
		struct h3600_battery v;
		unsigned int retval;
		retval = axim_get_battery( &v );
		
		if ( copy_to_user( (void *) arg, &v, sizeof(v)))
			retval = -EFAULT;
		break;
	}
	default:
		retval = -ENOIOCTLCMD;
		break;
	}

        return retval;
}

static int
axim_ts_open(struct inode *inode, struct file *filp)
{
//	printk("Touchscreen opened\n");
/*	if (down_trylock(&open_sem))
		return -EBUSY;*/
	return 0;
}

static int
axim_ts_fasync(int fd, struct file *filp, int on)
{
	return fasync_helper(fd, filp, on, &fasync);
}

static int
axim_ts_release(struct inode *inode, struct file *filp)
{
	axim_ts_fasync(-1, filp, 0);
//	up(&open_sem);
	return 0;
}

static struct file_operations axim_ts_fops = {
	owner:		THIS_MODULE,
	read:		axim_ts_read,
	poll:		axim_ts_poll,
	ioctl:		axim_ts_ioctl,
	open:		axim_ts_open,
	release:	axim_ts_release,
	fasync:		axim_ts_fasync,
};

static struct file_operations axim_tsraw_fops = {
	owner:		THIS_MODULE,
	read:		axim_tsraw_read,
	poll:		axim_tsraw_poll,
	ioctl:		axim_ts_ioctl,
	open:		axim_ts_open,
	release:	axim_ts_release,
	fasync:		axim_ts_fasync,
};


static int h3600_ts_open_generic(struct inode * inode, struct file * filp)
{
        unsigned int minor = MINOR( inode->i_rdev );   /* Extract the minor number */
	int result;

	if ( minor > 2 ) {
		printk("%s: bad minor = %d\n", __FUNCTION__, minor );
		return -ENODEV;
	}

        if (0) printk("%s: minor=%d\n", __FUNCTION__,minor);

	if ( !filp->private_data ) {
		switch (minor) {
		case AXIM_TS_MINOR:
//			filp->private_data = &g_touchscreen.filtered;
			filp->f_op = &axim_ts_fops;
			break;
		case AXIM_TSRAW_MINOR:
//			filp->private_data = &g_touchscreen.filtered;
			filp->f_op = &axim_tsraw_fops;
			break;
		}
	}

	result = filp->f_op->open( inode, filp );
	if ( !result ) 
		return result;

	return 0;
}


struct file_operations generic_fops = {
	open:     h3600_ts_open_generic
};

static struct miscdevice axim_ts_miscdev = {
        AXIM_TS_MINOR,
        "Axim_ts",
        &axim_ts_fops
};


/*
 * Initialization and exit routines
 */
int __init
axim_ts_init(void)
{
	int retval;

// Disable channel 8 enabled from PPC
	DCSR8=0x4;
//
        printk("%s: registering char device\n", __FUNCTION__);
	
//	axim_set_flite(1,1);
	
        gMajor = devfs_register_chrdev(0, "axim_ts", &generic_fops);
        if (gMajor < 0) {
                printk("%s: can't get major number\n", __FUNCTION__);
                return gMajor;
        }

        devfs_ts_dir = devfs_mk_dir(NULL, "touchscreen", NULL);
	if ( !devfs_ts_dir ) return -EBUSY;

        devfs_ts     = devfs_register( devfs_ts_dir, "0", DEVFS_FL_DEFAULT,
				       gMajor, AXIM_TS_MINOR, 
				       S_IFCHR | S_IRUSR | S_IWUSR, 
				       &axim_ts_fops, NULL );

        devfs_tsraw     = devfs_register( devfs_ts_dir, "0raw", DEVFS_FL_DEFAULT,
				       gMajor, AXIM_TSRAW_MINOR, 
				       S_IFCHR | S_IRUSR | S_IWUSR, 
				       &axim_tsraw_fops, NULL );
	
	set_GPIO_IRQ_edge(32,
			  GPIO_BOTH_EDGES);
#if 1
	if ((retval = request_irq(IRQ_TOUCHSCREEN, axim_ts_handler,
			SA_INTERRUPT, "axim_ts", 0))) {
		printk(KERN_WARNING "axim_ts: failed to get IRQ\n");
		return retval;
	}
#endif

	pxa_ac97_write(NULL,AC97_DIGICON1,
	CONV_DELAY << AC97_DIGICON1_DEL_SHIFT);

	pxa_ac97_write(NULL,AC97_DIGICON2,0xF007);
	lastn.pressure=-1;
	lastraw.pressure=-1;
	
	printk(KERN_NOTICE "Axim touchscreen driver initialised\n");
	return 0;
}

void __exit
axim_ts_exit(void)
{
/*	__raw_writel(0, IO_BASE + IO_CONTROLLER + 8);*/
	free_irq(IRQ_TOUCHSCREEN, 0);
        devfs_unregister(devfs_ts);
        devfs_unregister(devfs_tsraw);
//	misc_deregister(&axim_ts_miscdev);
	devfs_unregister(devfs_ts_dir);

	devfs_unregister_chrdev(gMajor, "axim_ts");
}

module_init(axim_ts_init);
module_exit(axim_ts_exit);

MODULE_AUTHOR("Tak-Shing Chan <chan@aleph1.co.uk>");
MODULE_DESCRIPTION("axim touchscreen driver");
MODULE_SUPPORTED_DEVICE("touchscreen/axim");
