/* vi: set sw=4 ts=4 ai: */

// #define MODULE

/**********************************************************************
*  linux/drivers/misc/ssp-lh7x.c
*
*  Provide SSP (synchronous Serial Port) functionality for LH7x EVB boards
*
*  Copyright (C) 2002  Lineo, Inc.
*
*	This program is free software; you can redistribute it and/or modify
*	it under the terms of the GNU General Public License (GPL) version 2
*	as published by the Free Software Foundation.
*
**********************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
//#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/smp_lock.h>
#include <linux/spinlock.h>

#undef DEBUG
#undef VERBOSE
#define DRVNAME	"ssp_lh7x"
#include <linux/verbosedebug.h>

#include <linux/version.h>
#ifdef MODULE
char kernel_version[] = UTS_RELEASE;
#endif /* MODULE */

#include <asm/arch/hardware.h>
#include <asm/arch/gpio.h>
#include <asm/arch/cpld.h>
#include <asm/arch/rcpc.h>
#include <asm/arch/iocon.h>
#include <asm/arch/ssp_lh7x.h>

#include "ssp.h"

unsigned int hclkfreq_get( void);

static gpioARegs_t *gpioa = (gpioARegs_t *)GPIO0_PHYS;
static ioconRegs_t *iocon = (ioconRegs_t *)IOCON_PHYS;
static rcpcRegs_t *rcpc = (rcpcRegs_t *)RCPC_PHYS;
static cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
static sspRegs_t *ssp = (sspRegs_t *)SSP_BASE;

/*
* hclk_freq:
* The frequency of the clock that feeds the SSP clock prescaler in the RCPC
* The frequency is in Hz
*/
static unsigned int hclk_freq = 0;

/**********************************************************************
* Additional RCPC defines
**********************************************************************/
#define rcpc_sspClkControl		spareClkCtrl
#define rcpc_sspClkPrescale		spare1Prescale

#define RCPC_LOCK 1
#define RCPC_LOCKED RCPC_LOCK
#define RCPC_UNLOCK 0
#define RCPC_UNLOCKED RCPC_UNLOCK

/**********************************************************************
* Function: ssp_busy_wait
*
* Purpose:
*	Wait until the state of the SSP busy bit from the status register
*	indicates the SSP is no longer busy.
*
* Returns:
*	N/A
**********************************************************************/
static void ssp_busy_wait(void)
{
	while ( ssp->sr & SSP_SR_BSY ) {
		barrier();
	}
	return;
}

/**********************************************************************
* Function: ssp_flush_tx_fifo
* Function: ssp_flush_rx_fifo
*
* Purpose:
*	Flush the transmit (tx) and receive (rx) fifo buffer
*
* Returns:
*	N/A
**********************************************************************/
static void ssp_flush_tx_fifo(sspContext_t *sspContext)
{
	int i;

	for (i = sspContext->ts_txTimeout; ((i > 0) && (ssp->sr & SSP_SR_TFE)); i--)
	{
		barrier();
	}
	return;
}

static void ssp_flush_rx_fifo(sspContext_t *sspContext)
{
	int i;
	unsigned int junk;

	for (i = sspContext->ts_rxTimeout; ((i > 0) && (ssp->sr & SSP_SR_RNE)); i--)
	{
		barrier();
		junk = ssp->dr;
		//printk("ssp_flush_rx_fifo(0x%04X)\n", junk);
	}
	return;
}

/**********************************************************************
* Function: ssp_chipselect_enable
* Function: ssp_chipselect_disable
* Function: ssp_chipselect_manual
* Function: ssp_chipselect_automatic
*
* Purpose:
*	Controls the chipselect pin associated with the SSP
*
* Returns:
*	N/A
*
**********************************************************************/
static void ssp_chipselect_enable(void)
{
	/* Make the SSPFRM signal (ChipSelect) high (enabled) */
	/* Note: This must have had ssp_chipselect_manual() called first */
	//printk("ssp_chipselect_enable()\n");
	gpioa->dr &= ~(SSPFRM_GPIO_BIT);	/* LOW == Enabled */
	return;
}

static void ssp_chipselect_disable(void)
{
	/* Make the SSPFRM signal (ChipSelect) low (disabled) */
	/* Note: This must have had ssp_chipselect_manual() called first */
	//printk("ssp_chipselect_disable()\n");
	gpioa->dr |= SSPFRM_GPIO_BIT;	/* HIGH == Disabled */
	return;
}

static void ssp_chipselect_manual(void)
{
	/* First, disable the ChipSelect */
	//JMG ssp_chipselect_disable();
	/* Set up muxing so that we manually control the ChipSelect pin */
	/* via GPIO port A bit 2 */
	gpioa->ddr |= SSPFRM_GPIO_BIT;	/* Make GPIO an output */
	iocon->SSIMux &= ~SSIMUX_SSPFRM;
	return;
}

static void ssp_chipselect_automatic(void)
{
	/* First, disable the ChipSelect */
	ssp_chipselect_disable();
	/* Set up muxing so the SSP automatically controls the ChipSelect pin */
	iocon->SSIMux |= SSIMUX_SSPFRM;
	return;
}

/**********************************************************************
* Function: rcpc_lh7x_locked
*
* Purpose:
*	Determine write access to the RCPC
*
* Returns:
*	The lock state of the RCPC
*
**********************************************************************/
static int rcpc_lh7x_locked(void)
{
	int lockState;

	vdprintk("ENTER: rcpc_lh7x_locked()\n");
	if (rcpc->control & RCPC_CTRL_WRTLOCK_ENABLED) {
		lockState = RCPC_LOCKED;
	} else {
		lockState = RCPC_UNLOCKED;
	}
	vdprintk("LEAVE: rcpc_lh7x_locked(%s)\n",
			(lockState==RCPC_LOCKED)?"Locked":"UnLocked");

	return(lockState);
}

/**********************************************************************
* Function: rcpc_lh7x_lock
*
* Purpose:
*	Control write access to the RCPC
*
* Parameters:
*	action:	RCPC_UNLOCK == can    write to RCPC
*			RCPC_LOCK   == cannot write to RCPC
*
* Returns:
*	The previous lock state of the RCPC
*
**********************************************************************/
static int rcpc_lh7x_lock(int action)
{
	int priorState;

	vdprintk("ENTER: rcpc_lh7x_lock(%s)\n",
			(action==RCPC_LOCK)?"Lock":"UnLock");
	priorState = rcpc_lh7x_locked();
	if (action == RCPC_UNLOCK) {
		rcpc->control |= RCPC_CTRL_WRTLOCK_ENABLED;
	} else /* (action == RCPC_LOCK) */ {
		rcpc->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
	}
	vdprintk("LEAVE: rcpc_lh7x_lock(%s)\n",
			(action==RCPC_LOCK)?"Lock":"UnLock");

	return(priorState);
}

#if OLDWAY
/**********************************************************************
* Function: ssp_lh7x_get_hclk_freq
*
* Purpose:
*	Get the HCLK (bus clock) frequency in Hz
**********************************************************************/
static unsigned int ssp_lh7x_get_hclk_freq(void)
{
#define XTAL_IN		14745600		/* 14.7456 MHz crystal */
#define PLL_CLOCK	(XTAL_IN * 21)	/* 309 MHz PLL clock */
	int divider;
	unsigned int _hclk_freq;

	vdprintk("ENTER: ssp_lh7x_get_hclk_freq()\n");
	divider = rcpc->HCLKPrescale * 2;	/* HCLK prescale value */
	if( divider == 0)					/* No prescalar == divide by 1 */
		divider = 1;
	_hclk_freq = PLL_CLOCK / divider;
	vdprintk("LEAVE: ssp_lh7x_get_hclk_freq(%u)\n", _hclk_freq);

	return(_hclk_freq);
}
#endif


/**********************************************************************
* Function: ssp_lh7x_get_speed
*
* Purpose:
*	Get the SSP speed in bits per second
**********************************************************************/
static int ssp_lh7x_get_speed(void)
{
	int bps;
	int rcpc_prescale;
	int ssp_prescale;
	int ssp_divider;

	vdprintk("ENTER: ssp_lh7x_get_speed()\n");
	rcpc_prescale = rcpc->rcpc_sspClkPrescale;
	if (rcpc_prescale == 0) rcpc_prescale = 1;
	else                    rcpc_prescale <<= 1;
	ssp_prescale = ssp->cpsr;
	ssp_divider = (ssp->cr0 & _SBF(8,_BITMASK(8) ) ) >> 8;
	bps = hclk_freq / (rcpc_prescale * (ssp_prescale) * (ssp_divider + 1) );
	vdprintk("LEAVE: ssp_lh7x_get_speed(%d bps)\n", bps);

	return(bps);
}

/**********************************************************************
* Function: ssp_lh7x_set_speed
*
* Purpose:
*	Set the SSP speed in bits per second
*
* Processing:
*	If the requested_bits_per_second is negaitve, return 0
*	If the requested_bits_per_second is too fast, set the bit rate
*	as fast as possible.
*	If the requested_bits_per_second is too slow, set the bit rate as
*	slow as possible.
*	If the requested_bits_per_second is in range, set the RCPC
*	SSP clock prescaler register, SSP prescaler, and SSP divider
*	to obtain the clock as close as possible.
*
* Parameters:
*	bps: The desired bits per second
*
* Returns:
*	The actual bps obtained or 0 if the requested bps is not obtainable.
*
* Notes:
*	The mode (SPI/uWire/TI) must be set first for this function to work!
*
**********************************************************************/
static int ssp_lh7x_set_speed(int bps)
{
	int rcpcLockState;
	int32_t ssp_prescale;
	int32_t ssp_divider;
	int32_t rcpc_prescale;
	int32_t new_prescale;
	int32_t new_divider;
	int32_t quotient;
	int32_t delta1;
	int32_t delta2;
	int32_t min_error;
	int32_t new_error;
#	define MAX_SSP_FREQ (hclk_freq / SSP_PRESCALE_MIN)

	vdprintk("ENTER: ssp_lh7x_set_speed(%d bps)\n", bps);

	/* Ensure we are dealing with a legal BPS */
	if (bps <= 0) {
		printk("%s: requested ssp speed (%d bps) is to slow\n", DRVNAME, bps);
		printk("%s: making ssp speed as slow as possible\n", DRVNAME);
		/* The request bps is slower than the minimum possible */
		/* ... make it as slow as possible */
		/* Don't bother calculating the divider values as we know them */
		rcpc_prescale = RCPC_SSP_PRESCALE_MAX;
		ssp_prescale = SSP_PRESCALE_MAX;
		ssp_divider = SSP_DIVIDER_MAX;
	} else if (bps >= MAX_SSP_FREQ) {
		printk("%s: requested ssp speed (%d bps) is to fast\n", DRVNAME, bps);
		printk("%s: making ssp speed as fast as possible\n", DRVNAME);
		/* Don't bother calculating the divider values as we know them */
		bps = MAX_SSP_FREQ;
		ssp_prescale = SSP_PRESCALE_MIN;
		ssp_divider = 1;
		rcpc_prescale = 1;
	} else {
		/* Calculate the divider values as close as we can */
		quotient = hclk_freq / bps;
		if (quotient <= 0)
			quotient = 1;
		/* round the quotient */
		delta1 = bps - (hclk_freq / quotient );
		if (delta1 < 0)
			delta1 = -delta1;
		delta2 = bps - (hclk_freq / (quotient + 1));
		if (delta2 < 0)
			delta2 = -delta2;
		if (delta1 > delta2)
			quotient++;
		if (quotient >=
				(SSP_PRESCALE_MAX * RCPC_SSP_PRESCALE_MAX * SSP_DIVIDER_MAX))
		{
			printk("%s: requested ssp speed (%d bps) is to slow\n",
					DRVNAME, bps);
			printk("%s: making ssp speed as slow as possible\n", DRVNAME);
			/* The request bps is slower than the minimum possible */
			/* ... make it as slow as possible */
			/* Don't bother calculating the divider values as we know them */
			rcpc_prescale = RCPC_SSP_PRESCALE_MAX;
			ssp_prescale = SSP_PRESCALE_MAX;
			ssp_divider = SSP_DIVIDER_MAX;
		} else {
			/*
			* The computed quotient is in range.
			* Quotient is the target clock divide frequency.
			* Get as close as possible.
			*/
			rcpc_prescale = 1;
			/*
			* Try to reduce power by using RCPC prescaler.
			* Note that the ssp prescaler minimum is two
			* so can only prescale and maintain accuracy
			* if quotient is divisible by 4.
			*/
			while ( ((quotient & 0x3) == 0)
					&& (rcpc_prescale < RCPC_SSP_PRESCALE_MAX) )
			{
				quotient >>= 1;
				rcpc_prescale <<= 1;
			}
			/*
			* Make sure the requested frequency is within range
			* of the SPP's prescaler and divider.
			* Hopefully, this loop never executes.
			* If it does, accuracy suffers.
			*/
			while (quotient > (SSP_PRESCALE_MAX * SSP_DIVIDER_MAX) ) {
				rcpc_prescale <<= 1;
				quotient >>= 1;
			}
			/*
			* Factor the quotient into the divider and prescaler combo
			* that minimizes the error in the quotient by exhaustively
			* searching all legal ssp prescaler values.
			*/
			ssp_prescale = SSP_PRESCALE_MIN;
			ssp_divider = (quotient / ssp_prescale);
			ssp_divider = (ssp_divider > SSP_DIVIDER_MAX)
				? SSP_DIVIDER_MAX : ssp_divider;
			min_error = quotient - (ssp_divider * ssp_prescale);
			min_error = (min_error < 0) ? -min_error : min_error;
			for (new_prescale = SSP_PRESCALE_MIN + 2;
					new_prescale < SSP_PRESCALE_MAX;
					new_prescale += 2)
			{
				new_divider = (quotient / new_prescale);
				new_divider = (new_divider > SSP_DIVIDER_MAX)
					? SSP_DIVIDER_MAX : new_divider;
				new_error = quotient - (new_divider * new_prescale);
				new_error = (new_error < 0) ? -new_error : new_error;
				if (new_error < min_error) {
					min_error = new_error;
					ssp_prescale = new_prescale;
					ssp_divider = new_divider;
				}
			}
		}
	}
	/* Set up the necessary registers to get the desired BSP */
	rcpcLockState = rcpc_lh7x_lock(RCPC_UNLOCK);
	rcpc->rcpc_sspClkPrescale = rcpc_prescale >> 1;
	(void) rcpc_lh7x_lock(rcpcLockState);
	ssp->cpsr = ssp_prescale;
	ssp->cr0 &= 0xff; /* clear old divider value */
	ssp->cr0 |= SSP_CR0_SCR(ssp_divider - 1);

	vdprintk("LEAVE: ssp_lh7x_set_speed(%d bps)\n", bps);

	return(ssp_lh7x_get_speed());
}

/**********************************************************************
* Function: ssp_lh7x_write16
*
* Purpose:
*	Write the LH7x SSP data register
**********************************************************************/
static void ssp_lh7x_write16(sspContext_t *sspContext,
		unsigned int data)
{
	int i;

	//if (sspContext->ssp_dev_sel == SSP_EEPROM) {	//JMG
		//printk("ENTER: ssp_lh7x_write16(0x%04X)\n", (uint16_t)data);
	//}
	for (i=sspContext->ts_txTimeout; ((i>0) && ((ssp->sr&SSP_SR_TNF) == 0)); i--) {
		barrier();
	}
	if (ssp->sr & SSP_SR_TNF) {
		ssp->dr = (uint16_t)data;
	} else {
		printk("%s: write timout\n", DRVNAME);
	}
	//vdprintk("LEAVE: ssp_lh7x_write16(0x%04X)\n", (uint16_t)data);
	return;
}

/**********************************************************************
* Function: ssp_lh7x_read16
*
* Purpose:
*	Read the LH7x SSP data register
**********************************************************************/
static unsigned int ssp_lh7x_read16(sspContext_t *sspContext)
{
	int i;
	unsigned int data = -1;

	//vdprintk("ENTER: ssp_lh7x_read16()\n");
	for (i=sspContext->ts_txTimeout; ((i>0) && ((ssp->sr&SSP_SR_RNE) == 0)); i--) {
		barrier();
	}
	if (ssp->sr & SSP_SR_RNE) {
		if (sspContext->ssp_dev_sel == SSP_EEPROM) {
			data = (unsigned int)ssp->dr;					/* EEPROM */
			//printk("LEAVE: ssp_lh7x_read16(ee: 0x%04X)\n", data);
		} else {
			data = (((unsigned int)ssp->dr) >> 4) & 0x0FFF;	/* TOUCHSCREEN */
			//printk("LEAVE: ssp_lh7x_read16(ts: 0x%04X)\n", data);
		}
	} else {
		//printk("%s: read timout\n", DRVNAME);
	}

	return(data);
}

/**********************************************************************
* Macro: ssp_lh7x_ts_pen_down
**********************************************************************/
static int ssp_lh7x_ts_pen_down(sspContext_t *sspContext)
{
	int pen_down = cpld->misc_stat & CPLD_MISCSTS_TS_IRQ;
	dprintk("ssp_lh7x_ts_pen_down(%d)\n", pen_down);
	return (pen_down);
}

/**********************************************************************
* Macro: ssp_lh7x_ts_pen_down_irq_enable
**********************************************************************/
static int ssp_lh7x_ts_pen_down_irq_enable(sspContext_t *sspContext)
{
	int lastState = (cpld->intr_mask & CPLD_TS_INTR_ENABLE);
	dprintk("ssp_lh7x_ts_pen_down_irq_enable\n");
	cpld->intr_mask |= CPLD_TS_INTR_ENABLE;
	sspContext->irq_state = 1;
	return(lastState);
}

/**********************************************************************
* Macro: ssp_lh7x_ts_pen_down_irq_disable
**********************************************************************/
static int ssp_lh7x_ts_pen_down_irq_disable(
		sspContext_t *sspContext)
{
	int lastState = (cpld->intr_mask & CPLD_TS_INTR_ENABLE);
	dprintk("ssp_lh7x_ts_pen_down_irq_disable\n");
	cpld->intr_mask &= ~CPLD_TS_INTR_ENABLE;
	sspContext->irq_state = 0;
	return(lastState);
}

/**********************************************************************
* Function: ssp_lh7x_ts_pen_down_irq
*
* We only detect touch screen _touches_ (pen down) with this interrupt
* handler, and even then we just schedule our task.
*
* Note: It has already been determined that this is our interrupt
* before we ever get it here so checking is minimal to non-existant.
**********************************************************************/
static void ssp_lh7x_ts_pen_down_irq(int irq, sspContext_t *sspContext,
		struct pt_regs * regs)
{
	/*
	* Disable the touchscreen interrupts
	* by disabling the touchscreen IRQ
	* --- AND ---
	* Enable regular polling of the touchscreen device
	* (The touchscreen IRQ will be re-enabled and polling
	* will be disabled when it is detected that the
	* pen is no longer down.)
	*/
	/* Disable touch screen IRQ */
	dprintk("ENTER: ssp_lh7x_ts_pen_down_irq()\n");
	ssp_lh7x_ts_pen_down_irq_disable(sspContext);
	dprintk("ENTER: wake_up()\n");
	wake_up(sspContext->irq_wait_ptr);
	vdprintk("LEAVE: ssp_lh7x_ts_pen_down_irq()\n");
	return;
}

/**********************************************************************
* Function: ssp_lh7x_irq_handler
*
* This interrupt handler only directs traffic for the interrupts
* by forwarding on the call to the appropriate interrupt handler.
**********************************************************************/
static void ssp_lh7x_irq_handler(int irq, void *_sspContext,
		struct pt_regs * regs)
{
	sspContext_t *sspContext = _sspContext;
	if ((sspContext) && (sspContext->irq_wait_ptr)) {
		if (ssp_lh7x_ts_pen_down(sspContext)) {
			ssp_lh7x_ts_pen_down_irq(irq, sspContext, regs);
		}
#if defined(VERBOSE) && defined(DEBUG)
		else {
			vdprintk("ssp_lh7x_irq_handler() --- Not our interrupt\n");
		}
#endif
	}
#if defined(VERBOSE) && defined(DEBUG)
	else {
		vdprintk("ssp_lh7x_irq_handler( NO ACTION )\n");
	}
#endif
	return;
}

/**********************************************************************
* Function: ssp_lh7x_lock
* Function: ssp_lh7x_unlock
*
* Purpose:
*	Lock/UnLock the SSP for a particular device (ts/ee)
**********************************************************************/
static int ssp_lh7x_lock(sspContext_t *sspContext, int device)
{
	int sts = -1;
	int cr0;

	spin_lock_irq(&sspContext->sspLock);
	if (device == SSP_DEV_TOUCHSCREEN) {
		/* Select the touchscreen */
		sspContext->ssp_dev_sel = SSP_TOUCHSCREEN;
		cr0 = ssp->cr0;
		cr0 &= ~0x00F0;
		/* National Microwire frame format --- SPI Polarity High */
		/* Don't mess with data size or clock rate */
		cr0 |= (SSP_CR0_FRF_NS | SSP_CR0_SPH);
		ssp->cr0 = cr0;
		vdprintk("ssp_lh7x_lock(SSP_DEV_TOUCHSCREEN)\n");
		// sts = 0;
	} else if (device == SSP_DEV_EEPROM) {
		/* Select the eeprom */
		sspContext->ssp_dev_sel = SSP_EEPROM;
		cr0 = ssp->cr0;
		cr0 &= ~0x00F0;
		/* Motorola SPI frame --- w/SPH & w/SPO */
		/* Don't mess with data size or clock rate */
		cr0 |= (SSP_CR0_FRF_MOT | SSP_CR0_SPH | SSP_CR0_SPO);
		ssp->cr0 = cr0;
		vdprintk("ssp_lh7x_lock(SSP_DEV_EEPROM)\n");
		// sts = 0;
	}
	cpld->ssp_dev_sel = sspContext->ssp_dev_sel;

	return(sts);
}

static int ssp_lh7x_unlock(sspContext_t *sspContext, int device)
{
	int sts = -1;
// #define SSP_DEFAULT_DEVICE	SSP_TOUCHSCREEN
#define SSP_DEFAULT_DEVICE	SSP_INVALID_DEVICE

	spin_unlock_irq(&sspContext->sspLock);
	if (device == SSP_DEV_TOUCHSCREEN) {
		/* Select the default device */
		sspContext->ssp_dev_sel = SSP_DEFAULT_DEVICE;
		vdprintk("ssp_lh7x_unlock(SSP_DEV_TOUCHSCREEN)\n");
		// sts = 0;
	} else if (device == SSP_DEV_EEPROM) {
		/* Select the default device */
		sspContext->ssp_dev_sel = SSP_DEFAULT_DEVICE;
		vdprintk("ssp_lh7x_unlock(SSP_DEV_EEPROM)\n");
		// sts = 0;
	}
	//cpld->ssp_dev_sel = sspContext->ssp_dev_sel;

	return(sts);
}

/**********************************************************************
* Function: ssp_lh7x_disable
*
* Purpose:
*	Disconnect I/O pins from the SSP module
*	and disable the SSP peripheral and its clocks.
**********************************************************************/
static void ssp_lh7x_disable(void)
{
	int rcpcLockState;

	vdprintk("ENTER: ssp_lh7x_disable()\n");

	/* Switch all muxed I/O away from the SSP */
	iocon->SSIMux &= ~(
			SSIMUX_SSPIN |
			SSIMUX_SSPOUT |
			SSIMUX_SSPCLK |
			SSIMUX_SSPENB |
			SSIMUX_SSPFRM
			);

	/* Disable ssp clock */
	rcpcLockState = rcpc_lh7x_lock(RCPC_UNLOCK);
	rcpc->rcpc_sspClkControl |= RCPC_SCLKSEL_SSPCLK;
	(void) rcpc_lh7x_lock(rcpcLockState);

	/* Set control register to their reset defaults */
	ssp->cr0 = 0;
	ssp->cr1 = 0;

	/* clear any receive overruns */
	ssp->u.icr = SSP_IIR_RORIS;

	//JMG /* disable the ssp DMA streams */
	//JMG dmac->stream0.max = 0;
	//JMG dmac->stream0.ctrl = 0;
	//JMG dmac->stream1.max = 0;
	//JMG dmac->stream1.ctrl = 0;
	//JMG /* clear any previous SSP DMA completions */
	//JMG dmac->clear = DMAC_EOT0 | DMAC_EOT1;

	vdprintk("LEAVE: ssp_lh7x_disable()\n");

	return;
}

/**********************************************************************
* Function: ssp_lh7x_enable
*
* Purpose:
*	Disconnect I/O pins from the SSP module
*	and disable the SSP peripheral and its clocks.
**********************************************************************/
static void ssp_lh7x_enable(void)
{

	vdprintk("ENTER: ssp_lh7x_enable()\n");

	/* Enable the SSP */
	ssp->cr1 |= SSP_CR1_SSE;		/* Synchronous serial port enable */

	/* Switch all muxed I/O to the SSP */
	/* Note that SSPENB is not required for spi */
	iocon->SSIMux = (
			SSIMUX_SSPIN |
			SSIMUX_SSPOUT |
			SSIMUX_SSPCLK |
			SSIMUX_SSPENB |
			SSIMUX_SSPFRM
			);

	vdprintk("LEAVE: ssp_lh7x_enable()\n");

	return;
}

/**********************************************************************
* Fill in our context structures
**********************************************************************/

static sspContext_t sspContext_l = {
	ts_txTimeout: 10000,
	ts_rxTimeout: 10000,
	ee_txTimeout: 10000,
	ee_rxTimeout: 10000,
	haveIrq: 0,
};

/**********************************************************************
* Function: ssp_request_pointer
* Function: ssp_provide_pointer
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
void *ssp_request_pointer(int device, char *request)
{
	sspContext_t *sspContext = &sspContext_l;
	void *vp = NULL;

	dprintk("ENTER: ssp_request_pointer(\"%s\":\"%s\")\n", device, request);
	if (device == SSP_DEV_TOUCHSCREEN) {
		if        (strcmp(request, "write") == 0) {
			vp = ssp_lh7x_write16;
		} else if (strcmp(request, "read") == 0) {
			vp = ssp_lh7x_read16;
		} else if (strcmp(request, "enable_pen_down_irq") == 0) {
			vp = ssp_lh7x_ts_pen_down_irq_enable;
		} else if (strcmp(request, "disable_pen_down_irq") == 0) {
			vp = ssp_lh7x_ts_pen_down_irq_disable;
		} else if (strcmp(request, "is_pen_down") == 0) {
			vp = ssp_lh7x_ts_pen_down;
		} else if (strcmp(request, "lock") == 0) {
			vp = ssp_lh7x_lock;
		} else if (strcmp(request, "unlock") == 0) {
			vp = ssp_lh7x_unlock;
		} else if (strcmp(request, "sspContext") == 0) {
			vp = sspContext;
		} else if (strcmp(request, "flush_tx_fifo") == 0) {
			vp = ssp_flush_tx_fifo;
		} else if (strcmp(request, "flush_rx_fifo") == 0) {
			vp = ssp_flush_rx_fifo;
		} else if (strcmp(request, "ssp_busy_wait") == 0) {
			vp = ssp_busy_wait;
		}
	} else if (device == SSP_DEV_EEPROM) {
		if        (strcmp(request, "write") == 0) {
			vp = ssp_lh7x_write16;
		} else if (strcmp(request, "read") == 0) {
			vp = ssp_lh7x_read16;
		} else if (strcmp(request, "lock") == 0) {
			vp = ssp_lh7x_lock;
		} else if (strcmp(request, "unlock") == 0) {
			vp = ssp_lh7x_unlock;
		} else if (strcmp(request, "sspContext") == 0) {
			vp = sspContext;
		} else if (strcmp(request, "chipselect_enable") == 0) {
			vp = ssp_chipselect_enable;
		} else if (strcmp(request, "chipselect_disable") == 0) {
			vp = ssp_chipselect_disable;
		} else if (strcmp(request, "chipselect_manual") == 0) {
			vp = ssp_chipselect_manual;
		} else if (strcmp(request, "chipselect_automatic") == 0) {
			vp = ssp_chipselect_automatic;
		} else if (strcmp(request, "flush_tx_fifo") == 0) {
			vp = ssp_flush_tx_fifo;
		} else if (strcmp(request, "flush_rx_fifo") == 0) {
			vp = ssp_flush_rx_fifo;
		} else if (strcmp(request, "ssp_busy_wait") == 0) {
			vp = ssp_busy_wait;
		}
	}
	dprintk("LEAVE: ssp_request_pointer(0x%08X)\n", (unsigned int)vp);

	return(vp);
}

void *ssp_provide_pointer(int device, char *request, void *vp)
{
	sspContext_t *sspContext = &sspContext_l;

	dprintk("ENTER: ssp_provide_pointer(\"%s\":\"%s\":0x%08X)\n",
			device, request, (unsigned int)vp);
	if (device == SSP_DEV_TOUCHSCREEN) {
		if (strcmp(request, "irq_wait_ptr") == 0) {
			sspContext->irq_wait_ptr = vp;
		} else {
			vp = NULL;
		}
	} else if (device == SSP_DEV_EEPROM) {
		vp = NULL;
	} else {
		vp = NULL;
	}
	dprintk("LEAVE: ssp_provide_pointer(0x%08X)\n", (unsigned int)vp);

	return(vp);
}

/**********************************************************************
* Function: ssp_lh7x_init
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
static int __init ssp_lh7x_init(void)
{
	sspContext_t *sspContext = &sspContext_l;
	int sts = 0;
	int rcpcLockState;
	int result;

	vdprintk("ENTER: ssp_lh7x_init()\n");

	vdprintk("ssp = 0x%08X\n", (unsigned int)ssp);

	/* Determine the HCLK (bus clock) frequency in Hz */
#ifdef OLDWAY
	hclk_freq = ssp_lh7x_get_hclk_freq();
#else
	hclk_freq = hclkfreq_get();
#endif

	/*
	* Disconnect I/O pins from the SSP module
	* and disable the SSP peripheral and its clocks.
	*/
	ssp_lh7x_disable();

	/* Initialize the RCPC SSP clock & prescaler */
	rcpcLockState = rcpc_lh7x_lock(RCPC_UNLOCK);
	rcpc->rcpc_sspClkPrescale = 0;					/* As fast as possible */
	rcpc->rcpc_sspClkControl &= ~RCPC_SCLKSEL_SSPCLK;	/* Enable SSP clock */
	(void) rcpc_lh7x_lock(rcpcLockState);

	ssp->cpsr = SSP_CPSR_CPDVSR(SSP_PRESCALE_MIN);

	/* Initialize CR0 */
	ssp->cr0 = (
		  SSP_CR0_DSS(16)		/* 16-bit data */
		| SSP_CR0_FRF_NS		/* National Microwire frame format */
		| SSP_CR0_SPH			/* SPI Polarity */
		| SSP_CR0_SCR(1)		/* Serial clock rate (~922kbps) */
		);

	/* Initialize CR1 */
	ssp->cr1 = (
		  SSP_CR1_SSE				/* Synchronous serial port enable */
		);

	/* Set the SSP speed in bits per second */
	/* Note this MUST be done after the SSP_CR0_FRF_xxx mode is set */
	(void) ssp_lh7x_set_speed(LH7x_TS_BPS);

	/* Select the touchscreen */
	sspContext->ssp_dev_sel = SSP_TOUCHSCREEN;
	cpld->ssp_dev_sel = sspContext->ssp_dev_sel;

	/* Flush the transmit FIFO */
	ssp_flush_tx_fifo(sspContext);

	/* Flush the receive FIFO */
	ssp_flush_rx_fifo(sspContext);

	/* clear any receive overruns */
	ssp->u.icr = SSP_IIR_RORIS;

	//printk("ssp->cr0 = 0x%04X\n", ssp->cr0);
	//printk("ssp->cr1 = 0x%04X\n", ssp->cr1);
	//printk("ssp->sr = 0x%04X\n", ssp->sr);
	//printk("ssp->cpsr = 0x%04X\n", ssp->cpsr);
	//printk("ssp->u.icr | u.iir = 0x%04X\n", ssp->u.icr);

	/*
	* Connect I/O pins from the SSP module
	* and enable the SSP peripheral and its clocks.
	*/
	ssp_lh7x_enable();

	/*
	* Request IRQ2 and attach it to the touchscreen pen_down line and enable it
	*/
	sspContext->haveIrq = 0;
	result = request_irq(2, ssp_lh7x_irq_handler,
			SA_SHIRQ | SA_SAMPLE_RANDOM, DRVNAME, sspContext);
	if (result < 0) {
		printk("%s: cannot get requested IRQ(2)\n", DRVNAME);
	} else {
		sspContext->haveIrq = 1;
		dprintk("%s: got requested IRQ(2)\n", DRVNAME);
	}
	ssp_lh7x_ts_pen_down_irq_enable(sspContext);

	vdprintk("LEAVE: ssp_lh7x_init()\n");
	return(sts);
}

/**********************************************************************
* Function: ssp_lh7x_exit
*
* Purpose:
*	Un-Register & Cleanup the module
**********************************************************************/
static void ssp_lh7x_exit(void)
{
	sspContext_t *sspContext = &sspContext_l;
	int rcpcLockState;

	vdprintk("ENTER: ssp_lh7x_exit()\n");

	//printk("ssp->cr0 = 0x%04X\n", ssp->cr0);
	//printk("ssp->cr1 = 0x%04X\n", ssp->cr1);
	//printk("ssp->sr = 0x%04X\n", ssp->sr);
	//printk("ssp->cpsr = 0x%04X\n", ssp->cpsr);
	//printk("ssp->u.icr | u.iir = 0x%04X\n", ssp->u.icr);

	/*
	* Disable & Return IRQ 2
	*/
	lock_kernel();
	ssp_lh7x_ts_pen_down_irq_disable(sspContext);
	if (sspContext->haveIrq) {
		free_irq(2, sspContext);
		sspContext->haveIrq = 0;
	}
	unlock_kernel();

	ssp->cr0 = 0;
	ssp->cr1 = 0;
	ssp->u.icr = SSP_IIR_RORIS;	/* clear any receive overruns */
	ssp->cpsr = SSP_CPSR_CPDVSR(SSP_PRESCALE_MIN);

	/* Turn off the  RCPC SSP clock */
	rcpcLockState = rcpc_lh7x_lock(RCPC_UNLOCK);
	rcpc->rcpc_sspClkControl |= RCPC_SCLKSEL_SSPCLK;	/* Disable SSP clock */
	(void) rcpc_lh7x_lock(rcpcLockState);

	vdprintk("LEAVE: ssp_lh7x_exit()\n");
	return;
}

module_init(ssp_lh7x_init);
module_exit(ssp_lh7x_exit);

MODULE_AUTHOR("Jim Gleason / Lineo, Inc.");
MODULE_DESCRIPTION("SSP Driver for Sharp LH7x EVB");
MODULE_LICENSE("Copyright (c) 2002 Lineo, Inc.");

