/**********************************************************************
 *  linux/drivers/misc/lpd-cpld.c
 * based on:
 *  linux/drivers/misc/ssp-lh7x.c
 *
 *  Provide SPI bus transactions for lpd's cpld.
 *
 *  Copyright (C) 2003  Logic Product Development
 *  Copyright (C) 2002  Lineo, Inc.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License (GPL) version 2
 *	as published by the Free Software Foundation.
 *
 **********************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/smp_lock.h>
#include <linux/spinlock.h>

//#define PRINTK printk
#define PRINTK(x, ...)

#define DRVNAME	"lpd-cpld"
#include <linux/verbosedebug.h>

#include <linux/version.h>
#ifdef MODULE
char kernel_version[] = UTS_RELEASE;
#endif							/* MODULE */

#include <asm/arch/hardware.h>
#include <asm/arch/platform.h>
#include <asm/arch/gpio.h>
#include <asm/arch/rcpc.h>
#include <asm/arch/iocon.h>
#include "ssp.h"

/*********************************************************************
* Context Structure Definition
*********************************************************************/ 
struct sspContext_t {
	volatile uint16_t *data_reg;
	volatile uint16_t *control_reg;
	volatile uint16_t chip_select;
	int ts_txTimeout;
	int ts_rxTimeout;
	int ee_txTimeout;
	int ee_rxTimeout;
	spinlock_t sspLock;
};
typedef struct sspContext_t sspContext_t;

#if 0
/**********************************************************************
* Function: lpd_cpld_flush_tx_fifo
* Function: lpd_cpld_flush_rx_fifo
*
* Purpose:
*	Flush the transmit (tx) and receive (rx) fifo buffer
*
* Returns:
*	1 for success, 0 for fail 
**********************************************************************/
static int
lpd_cpld_flush_tx_fifo(sspContext_t * sspContext)
{
	int i;

	for (i = sspContext->ts_txTimeout; 
		 (i > 0) && !(*sspContext->control_reg & CPLD_CE_SPI_CNTL_DONE); i--) {
		boink();
	}

	return (*sspContext->control_reg & CPLD_CE_SPI_CNTL_DONE) && 1;
}

static int
lpd_cpld_flush_rx_fifo(sspContext_t * sspContext)
{
	int i;

	for (i = sspContext->ts_txTimeout; 
		 (i > 0) && !(*sspContext->control_reg & CPLD_CE_SPI_CNTL_DONE); i--) {
		boink();
	}

	return (*sspContext->control_reg & CPLD_CE_SPI_CNTL_DONE) && 1;
}
#endif
/**********************************************************************
* Function: lpd_cpld_chipselect_enable
* Function: lpd_cpld_chipselect_disable
* Function: lpd_cpld_chipselect_manual
*
* Purpose:
*	Controls the chipselect pin associated with the SSP
*
* Returns:
*	N/A
*
**********************************************************************/
static void
lpd_cpld_chipselect_enable(sspContext_t *sspContext)
{
	/* Make the SSPFRM signal (ChipSelect) high (enabled) */
	/* Note: This must have had lpd_cpld_chipselect_manual() called first */
	PRINTK("lpd_cpld_chipselect_enable()\n");
	*sspContext->control_reg |= sspContext->chip_select;
}

static void
lpd_cpld_chipselect_disable(sspContext_t *sspContext)
{
	/* Make the SSPFRM signal (ChipSelect) low (disabled) */
	/* Note: This must have had lpd_cpld_chipselect_manual() called first */
	PRINTK("lpd_cpld_chipselect_disable()\n");
	*sspContext->control_reg &= ~sspContext->chip_select;
}

/**********************************************************************
* Function: lpdcpld_spi_write16
*
* Purpose:
*	Write the LH7x SSP data register
**********************************************************************/
static void
lpdcpld_spi_write8(sspContext_t * sspContext, unsigned char data)
{
	int i;

	if (sspContext->control_reg != (uint16_t *)CPLD_CE_REG_SPI_CONTROL) {
		PRINTK("LPD-SPI w ERROR! BAD CONTROL_REG %p\n",
			   sspContext->control_reg);
		return;
	}

	/* load spi data into shift register */
	*sspContext->data_reg = data;

	boink();

	/* write out the command */
	*sspContext->control_reg = (CPLD_CE_SPI_CNTL_LOAD_REG |
								sspContext->chip_select);

	boink();

	/* wait for register to load */
	for (i = sspContext->ts_txTimeout;
		 i > 0 && !(*sspContext->control_reg & CPLD_CE_SPI_CNTL_REG_LOADED); i--)
		boink();

	if (!i)
		PRINTK("lpdcpld_spi_write8 load TIMEDOUT!\n");

	boink();

	/* start the spi TX */
	*sspContext->control_reg = sspContext->chip_select;

	boink();

	/* wait for DONE bit */
	for (i = sspContext->ts_txTimeout;
		 i > 0 && !(*sspContext->control_reg & CPLD_CE_SPI_CNTL_DONE); i--)
		boink();

	if (!i)
		PRINTK("lpdcpld_spi_write8 done TIMEDOUT!\n");
}

/**********************************************************************
* Function: lpdcpld_spi_read16
*
* Purpose:
*	Read the LH7x SSP data register
**********************************************************************/
static unsigned char
lpdcpld_spi_read8(sspContext_t * sspContext)
{
	unsigned char data;
	int i;

	if (sspContext->control_reg != (uint16_t *)CPLD_CE_REG_SPI_CONTROL) {
		PRINTK("LPD-SPI r ERROR! BAD CONTROL_REG %p\n",
			   sspContext->control_reg);
		return 0;
	}

	/* load read command */
	*sspContext->control_reg = (CPLD_CE_SPI_CNTL_LOAD_REG | 
								CPLD_CE_SPI_CNTL_RD | 
								sspContext->chip_select);

	boink();

	/* start the spi RX */
	*sspContext->control_reg = (CPLD_CE_SPI_CNTL_RD | sspContext->chip_select);

	boink();

	/* wait for DONE bit */
	/* wait for DONE bit */
	for (i = sspContext->ts_rxTimeout;
		 i > 0 && !(*sspContext->control_reg & CPLD_CE_SPI_CNTL_DONE); i--)
		boink();

	if (!i)
		PRINTK("lpdcpld_spi_write8 done TIMEDOUT!\n");

	/* read the current byte of the data */
	data = (char)*sspContext->data_reg;

	return data;
}

/**********************************************************************
* Function: lpdcpld_spi_lock
* Function: lpdcpld_spi_unlock
*
* Purpose:
*	Lock/UnLock the SSP for a particular device (ts/ee)
**********************************************************************/
static int
lpdcpld_spi_lock(sspContext_t * sspContext, int device)
{
	int sts = 0;

	spin_lock_irq(&sspContext->sspLock);

	if (device == SSP_DEV_TOUCHSCREEN) {
		/* Select the touchscreen */
		sspContext->data_reg = (uint16_t *)CPLD_CE_REG_SPI_DATA;
		sspContext->control_reg = (uint16_t *)CPLD_CE_REG_SPI_CONTROL;
		sspContext->chip_select = CPLD_CE_SPI_CNTL_TOUCH;

//		PRINTK("lpdcpld_spi_lock(SSP_DEV_TOUCHSCREEN)\n");
	} else if (device == SSP_DEV_EEPROM) {
		/* Select the eeprom */
		sspContext->data_reg = (uint16_t *)CPLD_CE_REG_EEPROMSPI;
		sspContext->control_reg = (uint16_t *)CPLD_CE_REG_EEPROMSPI;
		sspContext->chip_select = CPLD_CE_EEPROMSPI_CS;

		PRINTK("lpdcpld_spi_lock(SSP_DEV_EEPROM)\n");
	}

	if (sspContext->control_reg)
		*sspContext->control_reg |= sspContext->chip_select;

	return sts;
}

static int
lpdcpld_spi_unlock(sspContext_t * sspContext, int device)
{
	int sts = -1;
	
	if (sspContext->control_reg)
		*sspContext->control_reg = 0;

	if (device == SSP_DEV_TOUCHSCREEN) {
		/* Select the default device */
//		PRINTK("lpdcpld_spi_unlock(SSP_DEV_TOUCHSCREEN)\n");
	} else if (device == SSP_DEV_EEPROM) {
		/* Select the default device */
		PRINTK("lpdcpld_spi_unlock(SSP_DEV_EEPROM)\n");
	}

	sspContext->control_reg = NULL;

	spin_unlock_irq(&sspContext->sspLock);

	return sts;
}

/**********************************************************************
* Fill in our context structures
**********************************************************************/

static sspContext_t sspContext_l = {
	ts_txTimeout:	10000,
	ts_rxTimeout:	10000,
	ee_txTimeout:	10000,
	ee_rxTimeout:	10000,
	sspLock: SPIN_LOCK_UNLOCKED
};

/**********************************************************************
* Function: ssp_request_pointer
* Function: ssp_provide_pointer
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
void *
ssp_request_pointer(int device, char *request)
{
	sspContext_t *sspContext = &sspContext_l;
	void *vp = NULL;

	PRINTK("ENTER: ssp_request_pointer(\"%d\":\"%s\")\n", device, request);

	if (device == SSP_DEV_TOUCHSCREEN) {
		if (strcmp(request, "write") == 0) {
			vp = lpdcpld_spi_write8;
		} else if (strcmp(request, "read") == 0) {
			vp = lpdcpld_spi_read8;
		} else if (strcmp(request, "lock") == 0) {
			vp = lpdcpld_spi_lock;
		} else if (strcmp(request, "unlock") == 0) {
			vp = lpdcpld_spi_unlock;
		} else if (strcmp(request, "sspContext") == 0) {
			vp = sspContext;
		}
	} else if (device == SSP_DEV_EEPROM) {
#if 0
		if (strcmp(request, "write") == 0) {
			vp = lpdcpld_spi_write8;
		} else if (strcmp(request, "read") == 0) {
			vp = lpdcpld_spi_read8;
		} else if (strcmp(request, "lock") == 0) {
			vp = lpdcpld_spi_lock;
		} else if (strcmp(request, "unlock") == 0) {
			vp = lpdcpld_spi_unlock;
		} else if (strcmp(request, "sspContext") == 0) {
			vp = sspContext;
		} else if (strcmp(request, "chipselect_enable") == 0) {
			vp = ssp_chipselect_enable;
		} else if (strcmp(request, "chipselect_disable") == 0) {
			vp = ssp_chipselect_disable;
		} else if (strcmp(request, "chipselect_manual") == 0) {
			PRINTK("no manual chipselect");
			vp = NULL;
		} else if (strcmp(request, "chipselect_automatic") == 0) {
			PRINTK("no auto chipselect");
			vp = NULL;
		} else if (strcmp(request, "flush_tx_fifo") == 0) {
			vp = lpd_cpld_flush_tx_fifo;
		} else if (strcmp(request, "flush_rx_fifo") == 0) {
			vp = lpd_cpld_flush_rx_fifo;
		} else if (strcmp(request, "ssp_busy_wait") == 0) {
			vp = lpd_cpld_busy_wait;
		}
#else
		vp = NULL;
#endif
	}
	PRINTK("LEAVE: ssp_request_pointer(0x%p)\n", vp);

	return vp;
}

void *
ssp_provide_pointer(int device, char *request, void *vp)
{
	PRINTK("ENTER: ssp_provide_pointer(\"%d\":\"%s\":0x%p)\n",
		   device, request, vp);

	vp = NULL;

	PRINTK("LEAVE: ssp_provide_pointer(0x%p)\n", vp);

	return vp;
}

/**********************************************************************
* Function: lpdcpld_spi_init
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
static int __init
lpdcpld_spi_init(void)
{
	sspContext_t *sspContext = &sspContext_l;
	int sts = 0;

	PRINTK("ENTER: lpdcpld_spi_init()\n");

	sspContext->control_reg = NULL;

	PRINTK("LEAVE: lpdcpld_spi_init()\n");

	return sts;
}

/**********************************************************************
* Function: lpdcpld_spi_exit
*
* Purpose:
*	Un-Register & Cleanup the module
**********************************************************************/
static void
lpdcpld_spi_exit(void)
{
	PRINTK("ENTER: lpdcpld_spi_exit()\n");

	PRINTK("LEAVE: lpdcpld_spi_exit()\n");
}

module_init(lpdcpld_spi_init);
module_exit(lpdcpld_spi_exit);

MODULE_AUTHOR("Mike Tesch / LPD");
MODULE_DESCRIPTION("SPI Driver for Logic CPLD SPI");
MODULE_LICENSE("Copyright (c) 2002 Lineo, Inc. (c) 2003 Logic Product Development");
