/*
    This file is part of KOrganizer.
    Copyright (c) 2000,2001 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include <qlayout.h>
#include <qheader.h>
#include <qcursor.h>
#include <qwhatsthis.h>

#include <qinputdialog.h>

#include <qvbox.h>
#include <kdebug.h>
#include "koprefs.h"
#include <klocale.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kmessagebox.h>

#include <libkcal/icaldrag.h>
#include <libkcal/vcaldrag.h>
#include <libkcal/calfilter.h>
#include <libkcal/dndfactory.h>
#include <libkcal/calendarresources.h>
#include <libkcal/resourcecalendar.h>
#include <kresources/resourceselectdialog.h>
#include <libkcal/kincidenceformatter.h>
#ifndef DESKTOP_VERSION
#include <qpe/qpeapplication.h>
#else
#include <qapplication.h>
#endif
#ifndef KORG_NOPRINTER
#include "calprinter.h"
#endif
#include "docprefs.h"

#include "kotodoview.h"
using namespace KOrg;


class KOTodoViewWhatsThis :public QWhatsThis
{
public:
  KOTodoViewWhatsThis( QWidget *wid, KOTodoView* view ) : QWhatsThis( wid ), _wid(wid),_view (view) { };

protected:
  virtual QString text( const QPoint& p)
  {
    return _view->getWhatsThisText(p) ;
  }
private:
  QWidget* _wid;
  KOTodoView * _view;
};

KOTodoListView::KOTodoListView(Calendar *calendar,QWidget *parent,
                               const char *name) :
  KListView(parent,name)
{
  mName = QString ( name );
  mCalendar = calendar;
#ifndef DESKTOP_VERSION
  QPEApplication::setStylusOperation(viewport(), QPEApplication::RightOnHold );
#endif
  mOldCurrent = 0;
  mMousePressed = false;

  setAcceptDrops(true);
  viewport()->setAcceptDrops(true);
  int size = 16;
  if (qApp->desktop()->width() < 300 )
      size = 12;
  setTreeStepSize( size + 6 );

}

void KOTodoListView::contentsDragEnterEvent(QDragEnterEvent *e)
{
#ifndef KORG_NODND
//  kdDebug() << "KOTodoListView::contentsDragEnterEvent" << endl;
  if ( !ICalDrag::canDecode( e ) && !VCalDrag::canDecode( e ) &&
       !QTextDrag::canDecode( e ) ) {
    e->ignore();
    return;
  }

  mOldCurrent = currentItem();
#endif
}


void KOTodoListView::contentsDragMoveEvent(QDragMoveEvent *e)
{
#ifndef KORG_NODND
//  kdDebug() << "KOTodoListView::contentsDragMoveEvent" << endl;

  if ( !ICalDrag::canDecode( e ) && !VCalDrag::canDecode( e ) &&
       !QTextDrag::canDecode( e ) ) {
    e->ignore();
    return;
  }

  e->accept();
#endif
}

void KOTodoListView::contentsDragLeaveEvent(QDragLeaveEvent *)
{
#ifndef KORG_NODND
//  kdDebug() << "KOTodoListView::contentsDragLeaveEvent" << endl;

  setCurrentItem(mOldCurrent);
  setSelected(mOldCurrent,true);
#endif
}

void KOTodoListView::contentsDropEvent(QDropEvent *e)
{
#ifndef KORG_NODND
//  kdDebug() << "KOTodoListView::contentsDropEvent" << endl;

  if ( !ICalDrag::canDecode( e ) && !VCalDrag::canDecode( e ) &&
       !QTextDrag::canDecode( e ) ) {
    e->ignore();
    return;
  }

  DndFactory factory( mCalendar );
  Todo *todo = factory.createDropTodo(e);

  if (todo) {
    e->acceptAction();

    KOTodoViewItem *destination =
        (KOTodoViewItem *)itemAt(contentsToViewport(e->pos()));
    Todo *destinationEvent = 0;
    if (destination) destinationEvent = destination->todo();

    Todo *existingTodo = mCalendar->todo(todo->uid());

    if(existingTodo) {
      Incidence *to = destinationEvent;
      while(to) {
        if (to->uid() == todo->uid()) {
          KMessageBox::sorry(this,
              i18n("Cannot move Todo to itself\nor a child of itself"),
              i18n("Drop Todo"));
          delete todo;
          return;
        }
        to = to->relatedTo();
      }
      internalDrop = true;
      if ( destinationEvent )
          reparentTodoSignal( destinationEvent, existingTodo );
      else 
          unparentTodoSignal(existingTodo);
      delete todo;
    } else {
      mCalendar->addTodo(todo);
      emit todoDropped(todo, KOGlobals::EVENTADDED);
      if ( destinationEvent )
          reparentTodoSignal( destinationEvent, todo );
    }
  }
  else {
    QString text;
    if (QTextDrag::decode(e,text)) {
      //QListViewItem *qlvi = itemAt( contentsToViewport(e->pos()) );
      KOTodoViewItem *todoi = static_cast<KOTodoViewItem *>(itemAt( contentsToViewport(e->pos()) ));
      qDebug("Dropped : " + text);
      QStringList emails = QStringList::split(",",text);
      for(QStringList::ConstIterator it = emails.begin();it!=emails.end();++it) {
        int pos = (*it).find("<");
        QString name = (*it).left(pos);
        QString email = (*it).mid(pos);
        if (!email.isEmpty() && todoi) {
          todoi->todo()->addAttendee(new Attendee(name,email));
        }
      }
    }
    else {
        qDebug("KOTodoListView::contentsDropEvent(): Todo from drop not decodable ");
      e->ignore();
    }
  }
#endif
}
void KOTodoListView::wheelEvent (QWheelEvent *e)
{
    QListView::wheelEvent (e);
}

void KOTodoListView::contentsMousePressEvent(QMouseEvent* e)
{

    QPoint p(contentsToViewport(e->pos()));
    QListViewItem *i = itemAt(p);
    bool rootClicked = true;
    if (i) {
        // if the user clicked into the root decoration of the item, don't
        // try to start a drag!
        int X = p.x();
        //qDebug("%d %d %d", X, header()->sectionPos(0), treeStepSize() );
        if (X > header()->sectionPos(0) +
            treeStepSize() * (i->depth() + (rootIsDecorated() ? 1 : 0)) +
            itemMargin() +i->height()||
            X < header()->sectionPos(0)) {
            rootClicked = false;
        }
    } else {
        rootClicked = false;
    }
#ifndef KORG_NODND
    mMousePressed = false; 
    if (! rootClicked && !( e->button() == RightButton)  ) {
        mPressPos = e->pos();
        mMousePressed = true;
    } else {
        mMousePressed = false;
    }
#endif
    //qDebug("KOTodoListView::contentsMousePressEvent %d", rootClicked);
#ifndef DESKTOP_VERSION
    if (!( e->button() == RightButton && rootClicked) )
        QListView::contentsMousePressEvent(e);   
#else
    QListView::contentsMousePressEvent(e);   
#endif
}
void KOTodoListView::paintEvent(QPaintEvent* e)
{
    emit paintNeeded();
    QListView::paintEvent( e);
}
void KOTodoListView::contentsMouseMoveEvent(QMouseEvent* e)
{

#ifndef KORG_NODND
    //QListView::contentsMouseMoveEvent(e);
  if (mMousePressed && (mPressPos - e->pos()).manhattanLength() >
      QApplication::startDragDistance()*3) {
    mMousePressed = false;
    QListViewItem *item = itemAt(contentsToViewport(mPressPos));
    if (item) {
      DndFactory factory( mCalendar );
      ICalDrag *vd = factory.createDrag(
                          ((KOTodoViewItem *)item)->todo(),viewport());
      internalDrop = false;
      // we cannot do any senseful here, because the DnD is still broken in Qt
      if (vd->drag()) {
          if ( !internalDrop ) {
              //emit deleteTodo( ((KOTodoViewItem *)item)->todo() );
              qDebug("Dnd: External move: Delete drag source "); 
          } else
              qDebug("Dnd: Internal move ");
          
      } else {
          if ( !internalDrop ) {
              qDebug("Dnd: External Copy");
          } else
              qDebug("DnD: Internal copy: Copy pending");
      }
    }
  } 
#endif
}
void KOTodoListView::keyReleaseEvent ( QKeyEvent *e ) 
{
    if ( !e->isAutoRepeat() ) {
        mFlagKeyPressed = false;
    }
}


void KOTodoListView::keyPressEvent ( QKeyEvent * e )
{
    qApp->processEvents();
    if ( e->isAutoRepeat() && !mFlagKeyPressed ) {
        e->ignore();
        // qDebug("  ignore  %d",e->isAutoRepeat()  );
        return;
    }
    if (! e->isAutoRepeat() )
        mFlagKeyPressed = true;
    QListViewItem* cn;
    if ( e->key() == Qt::Key_Return || e->key() == Qt::Key_Enter ) {
        cn = currentItem();
        if ( cn ) {
            KOTodoViewItem* ci = (KOTodoViewItem*)( cn ); 
            if ( ci ){
                if ( e->state() == ShiftButton )
                    ci->setOn( false );
                else
                    ci->setOn( true );
                cn = cn->itemBelow();
                if ( cn ) {
                    setCurrentItem ( cn );
                    ensureItemVisible ( cn );
                }
                
            }
        }
        
        return;
    }
   
    if ( e->state() == Qt::ControlButton || e->state() == Qt::ShiftButton || mName != "todolistsmall"  ) {
        switch ( e->key() ) {
        case Qt::Key_Down:
        case Qt::Key_Up:
            QListView::keyPressEvent (  e );
            break; 
        case Qt::Key_Left:
        case Qt::Key_Right:
            QListView::keyPressEvent (  e );
            e->accept();
            return;
            break; 
        default:
            e->ignore();
            break;
        }
        return;
    }
    e->ignore();
}
void KOTodoListView::contentsMouseReleaseEvent(QMouseEvent *e)
{
  QListView::contentsMouseReleaseEvent(e);
  mMousePressed = false;
}

void KOTodoListView::contentsMouseDoubleClickEvent(QMouseEvent *e)
{
  if (!e) return;

  QPoint vp = contentsToViewport(e->pos());

  QListViewItem *item = itemAt(vp);

  emit double_Clicked(item);
  if (!item) return;

  emit doubleClicked(item,vp,0);
}

/////////////////////////////////////////////////////////////////////////////

KOQuickTodo::KOQuickTodo(QWidget *parent) :
  QLineEdit(parent)
{
  setText(i18n("Click to add a new Todo"));
}

void KOQuickTodo::focusInEvent(QFocusEvent *ev)
{
  if ( text()==i18n("Click to add a new Todo") )
    setText("");
  QLineEdit::focusInEvent(ev);
}

void KOQuickTodo::focusOutEvent(QFocusEvent *ev)
{
  setText(i18n("Click to add a new Todo"));
  QLineEdit::focusOutEvent(ev);
}

/////////////////////////////////////////////////////////////////////////////

KOTodoView::KOTodoView(Calendar *calendar,QWidget* parent,const char* name) :
  KOrg::BaseView(calendar,parent,name)
{
    mPendingUpdateBeforeRepaint = false;
    isFlatDisplay = false;
  mNavigator = 0;
  QBoxLayout *topLayout = new QVBoxLayout(this);
  mName = QString ( name );
  mBlockUpdate = false;
  mQuickAdd = new KOQuickTodo(this);
  topLayout->addWidget(mQuickAdd);

  if ( !KOPrefs::instance()->mEnableQuickTodo ) mQuickAdd->hide();

  mTodoListView = new KOTodoListView(calendar,this, name );
  topLayout->addWidget(mTodoListView);
  //mTodoListView->header()->setMaximumHeight(30);
  mTodoListView->setRootIsDecorated(true);
  mTodoListView->setAllColumnsShowFocus(true);

  mTodoListView->setShowSortIndicator(true);

  mTodoListView->addColumn(i18n("Todo"));
  mTodoListView->addColumn(i18n("Prio"));
  mTodoListView->setColumnAlignment(1,AlignHCenter);
  mTodoListView->addColumn(i18n("Complete"));
  mTodoListView->setColumnAlignment(2,AlignCenter);

  mTodoListView->addColumn(i18n("Due Date"));
  mTodoListView->setColumnAlignment(3,AlignLeft);
  mTodoListView->addColumn(i18n("Due Time"));
  mTodoListView->setColumnAlignment(4,AlignHCenter);

  mTodoListView->addColumn(i18n("Start Date"));
  mTodoListView->setColumnAlignment(5,AlignLeft);
  mTodoListView->addColumn(i18n("Start Time"));
  mTodoListView->setColumnAlignment(6,AlignHCenter);

  mTodoListView->addColumn(i18n("Cancelled"));
  mTodoListView->addColumn(i18n("Categories"));
#if 0
  mTodoListView->addColumn(i18n("Sort Id"));
  mTodoListView->setColumnAlignment(4,AlignHCenter);
#endif

  mTodoListView->setMinimumHeight( 60 );
  mTodoListView->setItemsRenameable( true );
  mTodoListView->setRenameable( 0 );
  mTodoListView->setColumnWidth( 0, 120 );
  mTodoListView->setColumnWidthMode(0, QListView::Manual);
  mTodoListView->setColumnWidthMode(1, QListView::Manual);
  mTodoListView->setColumnWidthMode(2, QListView::Manual);
  mTodoListView->setColumnWidthMode(3, QListView::Manual);
  mTodoListView->setColumnWidthMode(4, QListView::Manual);
  mTodoListView->setColumnWidthMode(5, QListView::Manual);
  mTodoListView->setColumnWidthMode(6, QListView::Manual);
  mTodoListView->setColumnWidthMode(7, QListView::Manual);
  mTodoListView->setColumnWidthMode(8, QListView::Manual);
 

  mKOTodoViewWhatsThis = new KOTodoViewWhatsThis(mTodoListView->viewport(),this);

  mPriorityPopupMenu = new QPopupMenu(this);
  for (int i = 1; i <= 5; i++) {
    QString label = QString ("%1").arg (i);
    mPriority[mPriorityPopupMenu->insertItem (label)] = i;
  }
  connect (mPriorityPopupMenu, SIGNAL(activated (int)), SLOT (setNewPriority(int)));

  mPercentageCompletedPopupMenu = new QPopupMenu(this);
  for (int i = 0; i <= 100; i+=20) {
    QString label = QString ("%1 %").arg (i);
    mPercentage[mPercentageCompletedPopupMenu->insertItem (label)] = i;
  }
  connect (mPercentageCompletedPopupMenu, SIGNAL (activated (int)), SLOT (setNewPercentage (int)));


  
  mItemPopupMenu = new QPopupMenu(this); 
  mItemPopupMenu->insertItem(i18n("Show..."), this,
                             SLOT (showTodo()));
  mItemPopupMenu->insertItem(i18n("Edit..."), this,
                             SLOT (editTodo()));
  mItemPopupMenu->insertItem( i18n("Delete"), this,
                             SLOT (deleteTodo()));
  mItemPopupMenu->insertItem( i18n("Clone..."), this,
                             SLOT (cloneTodo()));
  mItemPopupMenu->insertItem( i18n("Move..."), this,
                             SLOT (moveTodo()));
#ifndef DESKTOP_VERSION
  mItemPopupMenu->insertItem( i18n("Beam..."), this,
                             SLOT (beamTodo()));
#endif
  mItemPopupMenu->insertItem( i18n("Toggle Cancel"), this,
                             SLOT (cancelTodo()));
  mItemPopupMenu->insertSeparator();
  mItemPopupMenu->insertItem( i18n("Start/Stop todo..."), this,
                             SLOT (toggleRunningItem()));
  mItemPopupMenu->insertSeparator();
  /*
  mItemPopupMenu->insertItem( i18n("New Todo..."), this,
                             SLOT (newTodo()));
  */
  mItemPopupMenu->insertItem(i18n("New Sub-Todo..."), this,
                             SLOT (newSubTodo())); 
  mItemPopupMenu->insertItem(i18n("Unparent Todo"), this,
                             SLOT (unparentTodo()),0,21);
  mItemPopupMenu->insertItem(i18n("Reparent Todo"), this,
                             SLOT (reparentTodo()),0,22);
  mItemPopupMenu->insertSeparator();
#if 0
  mItemPopupMenu->insertItem(i18n("Delete completed To-Dos","Purge Completed"),
                             this, SLOT( purgeCompleted() ) );
  mItemPopupMenu->insertItem(i18n("toggle completed To-Dos","Show Completed"),
                             this, SLOT( toggleCompleted() ),0, 33 );
  mItemPopupMenu->insertItem(i18n("toggle quick todo","Show Quick Todo"),
                             this, SLOT( toggleQuickTodo() ),0, 34 );
  mItemPopupMenu->insertItem(i18n("toggle running todo","Hide not Running"),
                             this, SLOT( toggleRunning() ),0, 35 );
  
#endif
  mPopupMenu = new QPopupMenu(this);
  mPopupMenu->insertItem(SmallIconSet("todo"), i18n("New Todo..."), this,
                         SLOT (newTodo()),0,1);
  mPopupMenu->insertItem(i18n("delete completed To-Dos","Purge Completed"),
                         this, SLOT(purgeCompleted()),0,2);
  mPopupMenu->insertItem(i18n("Show Completed"),
                             this, SLOT( toggleCompleted() ),0,3 );
  mPopupMenu->insertItem(i18n("toggle running todo","Hide not Running"),
                             this, SLOT( toggleRunning() ),0,5 );
  mPopupMenu->insertItem(i18n(" set all open","Display all opened"),
                             this, SLOT( setAllOpen() ),0,6 );
  mPopupMenu->insertItem(i18n(" set all close","Display all closed"),
                             this, SLOT( setAllClose() ),0,7 );
  mPopupMenu->insertItem(i18n(" set all flat","Display all flat"),
                             this, SLOT( setAllFlat() ),0,8 );
  mPopupMenu->insertSeparator();
  mPopupMenu->insertItem(i18n("toggle quick todo","Show Quick Todo"),
                             this, SLOT( toggleQuickTodo() ),0,4 );
  mDocPrefs = new DocPrefs( name );
 
  mItemPopupMenu->insertItem(i18n("Todo View"),mPopupMenu );
  mPopupMenu->setCheckable( true );
  mItemPopupMenu->setCheckable( true );


  mPopupMenu->setItemChecked( 3,KOPrefs::instance()->mShowCompletedTodo  );
  mItemPopupMenu->setItemChecked( 33 , KOPrefs::instance()->mShowCompletedTodo   );

  mPopupMenu->setItemChecked(4,KOPrefs::instance()->mEnableQuickTodo);
  mItemPopupMenu->setItemChecked( 34 , KOPrefs::instance()->mEnableQuickTodo );

  mPopupMenu->setItemChecked(5,KOPrefs::instance()->mHideNonStartedTodos);
  mItemPopupMenu->setItemChecked( 35 , KOPrefs::instance()->mHideNonStartedTodos );


  // Double clicking conflicts with opening/closing the subtree
  connect( mTodoListView, SIGNAL( doubleClicked( QListViewItem *) ),
           SLOT( editItem( QListViewItem *) ) );
  /*
  connect( mTodoListView, SIGNAL(  rightButtonClicked ( QListViewItem *,
                                                        const QPoint &,int ) ),
           SLOT( popupMenu( QListViewItem *, const QPoint & ,int) ) );
  */ 
  connect( mTodoListView, SIGNAL(  contextRequest ( QListViewItem *,
                                                        const QPoint &,int ) ),
           SLOT( popupMenu( QListViewItem *, const QPoint & ,int) ) );
  connect( mTodoListView, SIGNAL( clicked( QListViewItem * ) ),
           SLOT( itemClicked( QListViewItem * ) ) );
  connect( mTodoListView, SIGNAL( double_Clicked( QListViewItem * ) ),
           SLOT( itemDoubleClicked( QListViewItem * ) ) );
  connect( mTodoListView, SIGNAL( todoDropped( Todo *, int ) ),
           SLOT( updateView() ) );
  connect( mTodoListView, SIGNAL( todoDropped( Todo *, int ) ),
           SLOT( todoModified(Todo *, int) ) );
  connect( mTodoListView, SIGNAL( expanded( QListViewItem * ) ),
           SLOT( itemStateChanged( QListViewItem * ) ) );
  connect( mTodoListView, SIGNAL( collapsed( QListViewItem * ) ),
           SLOT( itemStateChanged( QListViewItem * ) ) );
  connect( mTodoListView, SIGNAL( paintNeeded() ),
           SLOT( paintNeeded()) );

#if 0
  connect(mTodoListView,SIGNAL(selectionChanged(QListViewItem *)),
          SLOT(selectionChanged(QListViewItem *)));
  connect(mTodoListView,SIGNAL(clicked(QListViewItem *)),
          SLOT(selectionChanged(QListViewItem *)));
  connect(mTodoListView,SIGNAL(pressed(QListViewItem *)),
          SLOT(selectionChanged(QListViewItem *)));
#endif

  connect( mTodoListView, SIGNAL(reparentTodoSignal( Todo *,Todo * ) ), SIGNAL(reparentTodoSignal( Todo *,Todo * ) ));
  connect( mTodoListView, SIGNAL(unparentTodoSignal(Todo *)  ), SIGNAL(unparentTodoSignal(Todo *)  ));
  connect( mTodoListView, SIGNAL( deleteTodo(Todo *)  ), SIGNAL(deleteTodoSignal(Todo *)  ));

  connect( mTodoListView, SIGNAL(selectionChanged() ),
           SLOT( processSelectionChange() ) );
  connect( mQuickAdd, SIGNAL( returnPressed () ),
           SLOT( addQuickTodo() ) );
 
}

KOTodoView::~KOTodoView()
{

#if QT_VERSION >= 0x030000

#else
    delete mKOTodoViewWhatsThis;
#endif 

  delete mDocPrefs;
}
QString KOTodoView::getWhatsThisText(QPoint p)
{
    KOTodoViewItem* item = ( KOTodoViewItem* ) mTodoListView->itemAt( p );
    if ( item )
        return KIncidenceFormatter::instance()->getFormattedText( item->todo(),  
                                                                  KOPrefs::instance()->mWTshowDetails,
                                                                  KOPrefs::instance()->mWTshowCreated,
                                                                  KOPrefs::instance()->mWTshowChanged);
    return i18n("That is the todo view" );

}

void KOTodoView::jumpToDate ()
{
  //   if (mActiveItem) {
//     mActiveItem->todo());
//    if ( mActiveItem->todo()->hasDueDate() )
//       emit mActiveItem->todo()jumpToTime( mTodo->dtDue().date() );
}
void  KOTodoView::paintNeeded()
{
    if ( mPendingUpdateBeforeRepaint ) {
        updateView();
       mPendingUpdateBeforeRepaint = false;
    }
}
void  KOTodoView::paintEvent(QPaintEvent * pevent)
{
    if ( mPendingUpdateBeforeRepaint ) {
        updateView();
       mPendingUpdateBeforeRepaint = false;
    }
    KOrg::BaseView::paintEvent( pevent);
}

void KOTodoView::updateView()
{
    pendingSubtodo = 0;
    if ( mBlockUpdate ) {
        return;
    }
    if ( !isVisible() ) {
       mPendingUpdateBeforeRepaint = true;
       return;
    }
    storeCurrentItem();
    //qDebug("KOTodoView::updateView() %x", this);
    if ( isFlatDisplay ) {
        displayAllFlat();
        resetCurrentItem();
        return;
    }
    //qDebug("update ");
//  kdDebug() << "KOTodoView::updateView()" << endl;
    QFont fo = KOPrefs::instance()->mTodoViewFont;
   

    mTodoListView->clear();
    if ( mName == "todolistsmall" ) {
        if ( KOPrefs::instance()->mTodoViewUsesSmallFont ) {
            int ps = fo.pointSize() -2;
            if ( ps > 12 )
                ps -= 2;
            fo.setPointSize( ps );
        } 
    }
    
  mTodoListView->setFont( fo );
  // QFontMetrics fm ( KOPrefs::instance()->mTodoViewFont );
  //mTodoListView->header()->setMaximumHeight(fm.height());
  QPtrList<Todo> todoList = calendar()->todos();

/*
  kdDebug() << "KOTodoView::updateView(): Todo List:" << endl;
  Event *t;
  for(t = todoList.first(); t; t = todoList.next()) {
    kdDebug() << "  " << t->getSummary() << endl;

    if (t->getRelatedTo()) {
      kdDebug() << "      (related to " << t->getRelatedTo()->getSummary() << ")" << endl;
    }

    QPtrList<Event> l = t->getRelations();
    Event *c;
    for(c=l.first();c;c=l.next()) {
      kdDebug() << "    - relation: " << c->getSummary() << endl;
    }
  }
*/

  // Put for each Event a KOTodoViewItem in the list view. Don't rely on a
  // specific order of events. That means that we have to generate parent items
  // recursively for proper hierarchical display of Todos.
  mTodoMap.clear();
  Todo *todo;
  todo = todoList.first();// todo; todo = todoList.next()) {
  while ( todo ) {
      bool next = true;
      // qDebug("todo %s ", todo->summary().latin1());
      Incidence *incidence = todo->relatedTo();
      while  ( incidence ) {
          if ( incidence->type() == "Todo") {
              //qDebug("related %s ",incidence->summary().latin1() );
              if ( !(todoList.contains ( ((Todo* )incidence ) ) )) {
                  //qDebug("related not found ");
                  todoList.remove( );
                  todo = todoList.current();
                  next = false;
                  incidence = 0;

              } else {
                  //qDebug("related found  ");
                  incidence = incidence->relatedTo();
              }
          } else 
              incidence = 0;
      }
      if ( next )
          todo = todoList.next();
  }

  for(todo = todoList.first(); todo; todo = todoList.next()) {
      if (!mTodoMap.contains(todo)  && checkTodo( todo ) )
      {
      insertTodoItem(todo);
    }
  }
  // Restore opened/closed state
  mTodoListView->blockSignals( true );
  if( mDocPrefs ) restoreItemState( mTodoListView->firstChild() );
  mTodoListView->blockSignals( false );
  resetCurrentItem();
  processSelectionChange();
}

void KOTodoView::storeCurrentItem()
{
    mCurItem = 0;
    mCurItemRootParent = 0;
    mCurItemParent = 0;
    mCurItemAbove = 0; 
    mActiveItem = (KOTodoViewItem*)mTodoListView->currentItem();
    if (mActiveItem) {
        mCurItem = mActiveItem->todo();
        KOTodoViewItem* activeItemAbove =  (KOTodoViewItem*)mActiveItem->itemAbove ();
        if ( activeItemAbove )
            mCurItemAbove = activeItemAbove->todo();
        mCurItemRootParent = mCurItem;
        mCurItemParent = mCurItemRootParent->relatedTo();
        while ( mCurItemRootParent->relatedTo() != 0 )
            mCurItemRootParent = mCurItemRootParent->relatedTo();
    }
    mActiveItem = 0;
}

void KOTodoView::resetCurrentItem()
{
    mTodoListView->setFocus();
    KOTodoViewItem* foundItem = 0; 
    KOTodoViewItem* foundItemRoot = 0; 
    KOTodoViewItem* foundItemParent = 0; 
    KOTodoViewItem* foundItemAbove = 0; 
    if ( mTodoListView->firstChild ()  ) {
        if ( mCurItem ) {
            KOTodoViewItem* item = (KOTodoViewItem*)mTodoListView->firstChild ();
            while ( item  ) {
                if ( item->todo() ==  mCurItem ) {
                    foundItem = item; 
                    break;
                } else if ( item->todo() ==  mCurItemAbove ) {
                    foundItemAbove = item;

                } 
                if ( item->todo() ==  mCurItemRootParent ) {
                    foundItemRoot = item;
                }
                if ( item->todo() ==  mCurItemParent ) {
                    foundItemParent = item;
                }
                item = (KOTodoViewItem*)item->itemBelow();
            }
            if ( ! foundItem ) {
                if ( foundItemParent ) {
                    foundItem = foundItemParent;
                } else {
                    if ( foundItemRoot )
                        foundItem = foundItemRoot;
                    else
                        foundItem = foundItemAbove;
                }
            }  
        } 
        if ( foundItem ) {
            mTodoListView->setCurrentItem( foundItem ); 
            mTodoListView->ensureItemVisible(  foundItem );
        } else {
            mTodoListView->setCurrentItem( mTodoListView->firstChild () );
        }
    }
    mTodoListView->setFocus();
}
//Incidence * mCurItem, *mCurItemRootParent,*mCurItemAbove; 
bool KOTodoView::checkTodo( Todo * todo )
{
   
    if (  !KOPrefs::instance()->mShowCompletedTodo &&  todo->isCompleted() )
        return false;
    if ( !todo->isCompleted()  ) {
        if ( todo->hasDueDate() && todo->dtDue().date() <= QDate::currentDate() )
            return true;
    }
    if ( KOPrefs::instance()->mHideNonStartedTodos && mNavigator ) {
        if ( todo->hasStartDate() ) 
            if ( mNavigator->selectedDates().last() < todo->dtStart().date() )
                return false;
        if ( todo->hasDueDate() ) 
            if ( mNavigator->selectedDates().first() > todo->dtDue().date() )
                return false;
    }
    return true;
}

void KOTodoView::restoreItemState( QListViewItem *item )
{
    pendingSubtodo = 0;
  while( item ) {
    KOTodoViewItem *todoItem = (KOTodoViewItem *)item;
    todoItem->setOpen( mDocPrefs->readBoolEntry( todoItem->todo()->uid() ) );
    if( item->childCount() > 0 ) restoreItemState( item->firstChild() );
    item = item->nextSibling();
  }
}


QMap<Todo *,KOTodoViewItem *>::ConstIterator
  KOTodoView::insertTodoItem(Todo *todo)
{

//  kdDebug() << "KOTodoView::insertTodoItem(): " << todo->getSummary() << endl;
  // TODO: Check, if dynmaic cast is necessary

    pendingSubtodo = 0;
  Incidence *incidence = todo->relatedTo();
  if (incidence && incidence->type() == "Todo") {
    Todo *relatedTodo = static_cast<Todo *>(incidence);

//    kdDebug() << "  has Related" << endl;
    QMap<Todo *,KOTodoViewItem *>::ConstIterator itemIterator;
    itemIterator = mTodoMap.find(relatedTodo);
    if (itemIterator == mTodoMap.end()) {
//      kdDebug() << "    related not yet in list" << endl;
      itemIterator = insertTodoItem (relatedTodo);
    }
    // isn't this pretty stupid? We give one Todo  to the KOTodoViewItem
    // and one into the map. Sure finding is more easy but why? -zecke
    KOTodoViewItem *todoItem = new KOTodoViewItem(*itemIterator,todo,this);
    return mTodoMap.insert(todo,todoItem);
  } else {
//    kdDebug() << "  no Related" << endl;
      // see above -zecke
    KOTodoViewItem *todoItem = new KOTodoViewItem(mTodoListView,todo,this);
    return mTodoMap.insert(todo,todoItem);
  }
}


void KOTodoView::updateConfig()
{
    updateView();
  mTodoListView->repaintContents();
}

QPtrList<Incidence> KOTodoView::selectedIncidences()
{
  QPtrList<Incidence> selected;

  KOTodoViewItem *item = (KOTodoViewItem *)(mTodoListView->selectedItem());
//  if (!item) item = mActiveItem;
  if (item) selected.append(item->todo());

  return selected;
}

QPtrList<Todo> KOTodoView::selectedTodos()
{
  QPtrList<Todo> selected;

  KOTodoViewItem *item = (KOTodoViewItem *)(mTodoListView->selectedItem());
//  if (!item) item = mActiveItem;
  if (item) selected.append(item->todo());

  return selected;
}

void KOTodoView::changeEventDisplay(Event *, int)
{
  updateView();
}

void KOTodoView::showDates(const QDate &, const QDate &)
{
}

void KOTodoView::showEvents(QPtrList<Event>)
{
  kdDebug() << "KOTodoView::selectEvents(): not yet implemented" << endl;
}

void KOTodoView::printPreview(CalPrinter *calPrinter, const QDate &fd,
                              const QDate &td)
{
#ifndef KORG_NOPRINTER
  calPrinter->preview(CalPrinter::Todolist, fd, td);
#endif
}

void KOTodoView::editItem(QListViewItem *item )
{
  emit editTodoSignal(((KOTodoViewItem *)item)->todo());
}

void KOTodoView::showItem(QListViewItem *item,const QPoint &,int)
{
  emit showTodoSignal(((KOTodoViewItem *)item)->todo());
}

void KOTodoView::popupMenu(QListViewItem *item,const QPoint &p,int column)
{
    pendingSubtodo = 0;
  mActiveItem = (KOTodoViewItem *)item;
  if (item) {
    switch (column){
    case 1:
      mPriorityPopupMenu->popup(QCursor::pos ()); break;
    case 2:
      mPercentageCompletedPopupMenu->popup(QCursor::pos ()); break;
    case 3:
         moveTodo();
        break;
    case 8:
      getCategoryPopupMenu((KOTodoViewItem *)item)->popup(QCursor::pos ()); break;
    default:
            mItemPopupMenu->popup(QCursor::pos());
    }
 } else mPopupMenu->popup(QCursor::pos());
}
void KOTodoView::newTodo()
{
  emit newTodoSignal();
}

void KOTodoView::newSubTodo()
{
  if (mActiveItem) {
    emit newSubTodoSignal(mActiveItem->todo());
  }
}
void KOTodoView::unparentTodo()
{
 if (mActiveItem) {
    emit unparentTodoSignal(mActiveItem->todo());
  }
}

void KOTodoView::reparentTodo()
{ 
    if (mActiveItem) {
        topLevelWidget()->setCaption(i18n("Click on new parent item"));
        pendingSubtodo = mActiveItem;
    }
}
void KOTodoView::editTodo()
{
  if (mActiveItem) {
    emit editTodoSignal(mActiveItem->todo());
  }
}
void KOTodoView::cloneTodo()
{
  if (mActiveItem) {
    emit cloneTodoSignal((Incidence*)mActiveItem->todo());
  }
}
void KOTodoView::cancelTodo()
{
  if (mActiveItem) {
    emit cancelTodoSignal((Incidence*)mActiveItem->todo());
  }
}
void KOTodoView::moveTodo()
{
  if (mActiveItem) {
    emit moveTodoSignal((Incidence*)mActiveItem->todo());
  }
}
void KOTodoView::beamTodo()
{
  if (mActiveItem) {
    emit beamTodoSignal((Incidence*)mActiveItem->todo());
  }
}


void KOTodoView::showTodo()
{
  if (mActiveItem) {
    emit showTodoSignal(mActiveItem->todo());
  }
}

void KOTodoView::deleteTodo()
{
  if (mActiveItem) {
      emit deleteTodoSignal(mActiveItem->todo());
  }
}

void KOTodoView::setNewPriority(int index)
{
  if (mActiveItem && !mActiveItem->todo()->isReadOnly ()) {
    mActiveItem->todo()->setPriority(mPriority[index]);
    mActiveItem->construct();
   todoModified (mActiveItem->todo(), KOGlobals::PRIORITY_MODIFIED);
   mActiveItem->todo()->setRevision( mActiveItem->todo()->revision()+1  );
  }
}

void KOTodoView::setNewPercentage(int index)
{
    if (mActiveItem && !mActiveItem->todo()->isReadOnly ()) {

        if ( mPercentage[index] == 100 && !mActiveItem->isOn() ) {
            mActiveItem->setOn( true );
            return;
        } else   if ( mPercentage[index] != 100 && mActiveItem->isOn() ) {
            KOTodoViewItem* par = (static_cast<KOTodoViewItem*>(mActiveItem->parent()));
            if ( par && par->isOn()  )
                par->setOn( false );
        }
        if (mPercentage[index] == 100) {
            mActiveItem->todo()->setCompleted(QDateTime::currentDateTime());
        } else {
            mActiveItem->todo()->setCompleted(false);
        }
        mActiveItem->todo()->setPercentComplete(mPercentage[index]);
        mActiveItem->construct();
        todoModified (mActiveItem->todo (), KOGlobals::COMPLETION_MODIFIED);
        mActiveItem->todo()->setRevision( mActiveItem->todo()->revision()+1  );
    }
}


QPopupMenu * KOTodoView::getCategoryPopupMenu (KOTodoViewItem *todoItem)
{
  QPopupMenu* tempMenu = new QPopupMenu (this);
  QStringList checkedCategories = todoItem->todo()->categories ();

  tempMenu->setCheckable (true);
  for (QStringList::Iterator it = KOPrefs::instance()->mCustomCategories.begin ();
       it != KOPrefs::instance()->mCustomCategories.end ();
       ++it) {
    int index = tempMenu->insertItem (*it);
    mCategory[index] = *it;
    if (checkedCategories.find (*it) != checkedCategories.end ()) tempMenu->setItemChecked (index, true);
  }

  connect (tempMenu, SIGNAL (activated (int)), SLOT (changedCategories (int)));
  return tempMenu;


}
void KOTodoView::changedCategories(int index)
{
  if (mActiveItem && !mActiveItem->todo()->isReadOnly ()) {
    QStringList categories = mActiveItem->todo()->categories ();
    QString colcat = categories.first();
    if (categories.find (mCategory[index]) != categories.end ())
      categories.remove (mCategory[index]);
    else
      categories.insert (categories.end(), mCategory[index]);
    categories.sort ();
    if ( !colcat.isEmpty() ) {
        if ( categories.find ( colcat ) != categories.end () ) {
            categories.remove( colcat );
            categories.prepend( colcat );
        }
    }
    mActiveItem->todo()->setCategories (categories);
    mActiveItem->construct();
    mActiveItem->todo()->setRevision( mActiveItem->todo()->revision()+1  );
    todoModified (mActiveItem->todo (), KOGlobals::CATEGORY_MODIFIED);
  }
}
void KOTodoView::itemDoubleClicked(QListViewItem *item)
{
    if ( pendingSubtodo != 0 ) {
        topLevelWidget()->setCaption(i18n("Reparenting aborted!"));
    }
    pendingSubtodo = 0;
    //int row = mTodoListView->header()->sectionAt ( mTodoListView->header()->mapFromGlobal( QCursor::pos()).x() );
    int row = mTodoListView->header()->sectionAt ( mTodoListView->viewportToContents(mTodoListView->viewport()->mapFromGlobal( QCursor::pos())) .x() );
    //qDebug("ROW %d ", row);
    if (!item) {
        newTodo();
        return;
    } else {
        if ( row == 2 || row == 1 ) {
            mActiveItem = (KOTodoViewItem *) item;
            newSubTodo();
            return;
        }
        if ( row == 5 || row == 6 ) {
            mActiveItem = (KOTodoViewItem *) item;
            toggleRunningItem();
            return;
        }
    }
    if ( KOPrefs::instance()->mEditOnDoubleClick )
        editItem( item );
    else
        showItem( item , QPoint(), 0 );
}
void KOTodoView::toggleRunningItem()
{
    // qDebug("KOTodoView::toggleRunning() ");
    if ( ! mActiveItem )
        return;
    Todo * t = mActiveItem->todo();
    if ( t->isRunning() ) {
#if 0
        int result = KMessageBox::warningContinueCancel(this,
                                                        i18n("The todo\n%1\nis started.\nDo you want to set\nthe state to stopped?").arg(mActiveItem->text(0).left( 25 ) ),i18n("Todo is started"),i18n("Stop todo"),i18n("Cancel"), true);
#endif

        int result = KMessageBox::warningYesNoCancel(this,
                                                        i18n("The todo\n%1\nis started.\nDo you want to set\nthe state to stopped?").arg(mActiveItem->text(0).left( 25 ) ),i18n("Todo is started"),i18n("Stop"),i18n("Stop+note"));
        if (result == KMessageBox::Cancel) return;
        if ( result == KMessageBox::No ) {
            QString comment = QInputDialog::getText(mActiveItem->text(0).left( 25 ),i18n("Comment for todo:") );
            t->setRunningFalse( comment );
        } else {
            t->setRunning( false );
        }
        mActiveItem->construct();
    } else {
        int result = KMessageBox::warningContinueCancel(this,
                                                        i18n("The todo\n%1\nis stopped.\nDo you want to set\nthe state to started?").arg(mActiveItem->text(0).left( 25 ) ),i18n("Todo is stopped"),i18n("Start todo"),i18n("Cancel"), true);
        if (result != KMessageBox::Continue) return;
        t->setRunning( true );
        mActiveItem->construct();
    }
}

void KOTodoView::itemClicked(QListViewItem *item)
{
    //qDebug("KOTodoView::itemClicked %d", item);
  if (!item) {
      if ( pendingSubtodo != 0 ) {
          topLevelWidget()->setCaption(i18n("Reparenting aborted!"));
      }
      pendingSubtodo = 0;
      return;
  }
  KOTodoViewItem *todoItem = (KOTodoViewItem *)item;
  if ( pendingSubtodo != 0 ) {
      bool allowReparent = true;
      QListViewItem *par = item;
      while ( par ) {
          if ( par ==  pendingSubtodo ) {
              allowReparent = false;
              break;
          }
          par = par->parent();
      }
      if ( !allowReparent  ) {
          topLevelWidget()->setCaption(i18n("Recursive reparenting not possible!"));
          pendingSubtodo = 0;
      } else {
          Todo* newParent = todoItem->todo();
          Todo* newSub = pendingSubtodo->todo();
          pendingSubtodo = 0;
          emit reparentTodoSignal( newParent,newSub  );
          return;
      }
  }

}

void KOTodoView::setDocumentId( const QString &id )
{

  mDocPrefs->setDoc( id );
}

void KOTodoView::itemStateChanged( QListViewItem *item )
{
  if (!item) return;

  KOTodoViewItem *todoItem = (KOTodoViewItem *)item;

//  kdDebug() << "KOTodoView::itemStateChanged(): " << todoItem->todo()->summary() << endl;

  if( mDocPrefs ) mDocPrefs->writeEntry( todoItem->todo()->uid(), todoItem->isOpen() );
}

void KOTodoView::saveLayout(KConfig *config, const QString &group) const
{
  mTodoListView->saveLayout(config,group);
}

void KOTodoView::restoreLayout(KConfig *config, const QString &group)
{
  mTodoListView->restoreLayout(config,group);
}

void KOTodoView::processSelectionChange()
{
//  kdDebug() << "KOTodoView::processSelectionChange()" << endl;

  KOTodoViewItem *item =
    static_cast<KOTodoViewItem *>( mTodoListView->selectedItem() );

  if ( !item ) {
    emit incidenceSelected( 0 );
  } else {
    emit incidenceSelected( item->todo() );
  }
}

void KOTodoView::modified(bool b)
{
  emit isModified(b);
}
void KOTodoView::setTodoModified( Todo* todo )
{
  todoModified( todo, KOGlobals::UNKNOWN_MODIFIED );
}
void KOTodoView::clearSelection()
{
  mTodoListView->selectAll( false );
}
void  KOTodoView::setAllOpen()
{
    if ( isFlatDisplay ) {
        isFlatDisplay = false;
        mPopupMenu->setItemChecked( 8,false );
        updateView();
    } else {
        storeCurrentItem();
    }
    setOpen(mTodoListView->firstChild(), true);
    resetCurrentItem();
}
void  KOTodoView::setAllClose()
{
    if ( isFlatDisplay ) {
        isFlatDisplay = false;
        mPopupMenu->setItemChecked( 8,false );
        updateView();
    } else {
        storeCurrentItem();
    }
    setOpen(mTodoListView->firstChild(), false);
    resetCurrentItem();
}
void  KOTodoView::setOpen( QListViewItem* item, bool setOpenI)
{

    while ( item ) {
        setOpen( item->firstChild(), setOpenI );
        item->setOpen( setOpenI );
        item = item->nextSibling();
    }
}

void KOTodoView::displayAllFlat()
{
    pendingSubtodo = 0;
    if ( mBlockUpdate ) {
        return;
    }
    mPopupMenu->setItemChecked( 8,true );
    isFlatDisplay = true;
    QPtrList<Todo> todoList = calendar()->todos();
    mTodoMap.clear();
    mTodoListView->clear();
    Todo *todo;   
    for(todo = todoList.first(); todo; todo = todoList.next()) {
        KOTodoViewItem *todoItem = new KOTodoViewItem(mTodoListView,todo,this);
        mTodoMap.insert(todo,todoItem);
    }
    mTodoListView->setFocus();
    processSelectionChange();
}

void  KOTodoView::setAllFlat()
{
    if ( isFlatDisplay ) {
        isFlatDisplay = false;
        mPopupMenu->setItemChecked( 8,false );
        updateView();
        return;
    } 
    storeCurrentItem();
    displayAllFlat();
    resetCurrentItem();
}

void KOTodoView::purgeCompleted()
{
  emit purgeCompletedSignal();
 
}
void KOTodoView::toggleQuickTodo()
{
    if ( mQuickAdd->isVisible() ) {
        mQuickAdd->hide();
        KOPrefs::instance()->mEnableQuickTodo = false;
    }
    else {
        mQuickAdd->show();
        KOPrefs::instance()->mEnableQuickTodo = true;
    }
    mPopupMenu->setItemChecked(4,KOPrefs::instance()->mEnableQuickTodo);
    mItemPopupMenu->setItemChecked( 34 , KOPrefs::instance()->mEnableQuickTodo );
}

void KOTodoView::toggleRunning()
{
    KOPrefs::instance()->mHideNonStartedTodos = !KOPrefs::instance()->mHideNonStartedTodos;
    mPopupMenu->setItemChecked(5,KOPrefs::instance()->mHideNonStartedTodos);
    mItemPopupMenu->setItemChecked( 35 , KOPrefs::instance()->mHideNonStartedTodos );
    updateView();
}

void KOTodoView::toggleCompleted()
{
    KOPrefs::instance()->mShowCompletedTodo = !KOPrefs::instance()->mShowCompletedTodo;
    mPopupMenu->setItemChecked( 3,KOPrefs::instance()->mShowCompletedTodo  );
    mItemPopupMenu->setItemChecked( 33 , KOPrefs::instance()->mShowCompletedTodo   );
    updateView();
}

void KOTodoView::addQuickTodo()
{
  Todo *todo = new Todo();
  todo->setSummary(mQuickAdd->text());
  todo->setOrganizer(KOPrefs::instance()->email());
  CalFilter * cf = mCalendar->filter();
  if ( cf ) {
      if ( cf->isEnabled()&& cf->showCategories()) {
          todo->setCategories(cf->categoryList());
      }
      if ( cf->isEnabled() )
          todo->setSecrecy( cf->getSecrecy());
  }
  mCalendar->addTodo(todo);
  mQuickAdd->setText("");
  todoModified (todo,  KOGlobals::EVENTADDED );
  updateView();
}

void KOTodoView::keyPressEvent ( QKeyEvent * e )
{  
    //    e->ignore();
    //return;
    //qDebug("KOTodoView::keyPressEvent  ");
    switch ( e->key() ) {
    case Qt::Key_Down:
    case Qt::Key_Up:
        QWidget::keyPressEvent (  e );
        break; 
   
    case Qt::Key_Q:
        toggleQuickTodo();
        break; 
    case Qt::Key_U:
        if ( e->state() == Qt::ControlButton||  e->state() == Qt::ShiftButton ) {
            mActiveItem = (KOTodoViewItem*)mTodoListView->currentItem();
            unparentTodo();
            e->accept();
        } else
            e->ignore();
        break; 
    case Qt::Key_S:
        if ( e->state() == Qt::ControlButton ) {
            e->ignore();
            break; 
        }
        if ( e->state() == Qt::ShiftButton ) {
            mActiveItem = (KOTodoViewItem*)mTodoListView->currentItem();
            reparentTodo();
            e->accept();
        } else
            e->ignore();
        break; 
    case Qt::Key_P:
        if ( e->state() == Qt::ControlButton||  e->state() == Qt::ShiftButton ) {
            mActiveItem = (KOTodoViewItem*)mTodoListView->currentItem();
            if ( pendingSubtodo )
                itemClicked(mActiveItem);
            e->accept();
        } else
            e->ignore();
        break;  
    case Qt::Key_Escape:
        if ( pendingSubtodo ) {
            itemClicked(0);
            e->accept();
        } else
            e->ignore();
        break; 
    default:
        e->ignore();
    }
    
    if ( true ) {
        if (  e->key() == Qt::Key_I ) {
            KOTodoViewItem*cn = (KOTodoViewItem*)mTodoListView->currentItem();
            if ( cn ) {
                mActiveItem = cn;
                KOTodoViewItem* ci = (KOTodoViewItem*)( cn ); 
                if ( ci ){
                    showTodo();
                    cn = (KOTodoViewItem*)cn->itemBelow();
                    if ( cn ) {
                        mTodoListView->setCurrentItem ( cn );
                        mTodoListView->ensureItemVisible ( cn );
                    }
                    
                }
            }
            e->accept();
        
        }
        
    }
    
}
void KOTodoView::updateTodo( Todo * t, int type )
{
    if  ( mBlockUpdate)
        return;

    QMap<Todo *,KOTodoViewItem *>::ConstIterator itemIterator;
    itemIterator = mTodoMap.find(t);
    if (itemIterator != mTodoMap.end()) {
        (*itemIterator)->construct();
    } else {
        if ( type == KOGlobals::EVENTADDED ) {
            insertTodoItem( t );
        }
    }

}

void KOTodoView::todoModified(Todo * t , int p )
{
    mBlockUpdate = true;
    emit todoModifiedSignal ( t, p );
    mBlockUpdate = false;
}
