/*
    This file is part of libkresources.

    Copyright (c) 2002 Tobias Koenig <tokoe@kde.org>
    Copyright (c) 2002 Jan-Pascal van Best <janpascal@vanbest.org>
    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifndef KRESOURCES_FACTORY_H
#define KRESOURCES_FACTORY_H

#include <qdict.h>
#include <qstring.h>

#include <kconfig.h>


#include "resource.h"

namespace KRES {

class SyncWidgetContainer;
class ConfigWidget;

//US
struct PluginInfo
{
  QString library;
  QString nameLabel;
  QString descriptionLabel;
};


/**
 * Class for loading resource plugins.
 * Do not use this class directly. Use ResourceManager instead
 *
 * Example:
 *
 * <pre>
 * KABC::Factory<Calendar> *factory = KABC::Factory<Calendar>::self();
 *
 * QStringList list = factory->resources();
 * QStringList::Iterator it;
 * for ( it = list.begin(); it != list.end(); ++it ) {
 *   Resource<Calendar> *resource = factory->resource( (*it),
 *        KABC::StdAddressBook::self(), 0 );
 *   // do something with resource
 * }
 * </pre>
 */
class Factory
{
  public:


    /**
     * Returns the global resource factory.
     */
    static Factory *self( const QString& resourceFamily );

    ~Factory();

    /**
     * Returns the config widget for the given resource type,
     * or a null pointer if resource type doesn't exist.
     *
     * @param type   The type of the resource, returned by @ref resources()
     * @param resource The resource to be editted.
     * @param parent The parent widget
     */
    ConfigWidget *configWidget( const QString& type, QWidget *parent = 0 );

    /**
     * Returns the sync widget for the given resource type,
     * or a null pointer if resource type doesn't exist,
     * or a null pointer if resource does not support syncing.
     *
     * @param type   The type of the resource, returned by @ref resources()
     * @param resource The resource to be editted.
     * @param parent The parent widget
     */
    SyncWidgetContainer *syncWidgetContainer( const QString& type );

    /**
     * Returns a pointer to a resource object or a null pointer
     * if resource type doesn't exist.
     *
     * @param type   The type of the resource, returned by @ref resources()
     * @param ab     The address book, the resource should belong to
     * @param config The config object where the resource get it settings from, or 0 if a new resource should be created.
     * @param syncable If the resource should support syncing capabilities.
     */
    Resource *resource( const QString& type, const KConfig *config, bool syncable );

    /**
     * Returns a list of all available resource types.
     */
    QStringList typeNames() const;

    /**
     * Returns the name for a special type.
     */
    QString typeName( const QString &type ) const;

    /**
     * Returns the description for a special type.
     */
    QString typeDescription( const QString &type ) const;

  protected:
    Factory( const QString& resourceFamily);

  private:
    static QDict<Factory> *mSelves;

    QString mResourceFamily;
//US    QMap<QString, KService::Ptr> mTypeMap;
//US lets store the pluginfo struct as value instead of a KService
    QMap<QString, PluginInfo*> mTypeMap;
};

}
#endif
