/******************************************************************************
//             INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2001 Intel Corporation. All Rights Reserved.
//
//  VSS: 
//		$Workfile: filter.c $
//		$Revision: 1 $
//		$Date: 03-09-10 17:01 $
//
//  Description:
//		FIR and IIR Filtering examples 
//
******************************************************************************/

/* Standard includes */
#include <string.h>

/* Standard IPP definitions */
#include <ippdefs.h>
#include <ippSP.h>

/* External interface */
#include "filter.h"

/* Fixed sinusoidal parameters */
#define		AQ15			1000				/* ~1/16, Q15 */
#define		PHASE			0					/* 0, Q15 */
#define		F				450					/* Desired tone frequency, in Hz */				

/* SNR computation parameters */
#define		LOG10			4716				/* for sf=-11 */
#define		LSF				4096.0				/* Scalefactor for ln() */
#define		LOGLSF			17035				/* ln(LSF)*2^sf (sf=11)

/* Filter parameters */
#define		LPF_TAPS				66
#define		HPF_TAPS				237
#define     LPF_BQ_TAPS				30
#define		LPF_BQ_SECTIONS			5
#define     HPF_BQ_TAPS				30
#define		HPF_BQ_SECTIONS			5
#define		SHAPING_TAPS			30
#define		SHAPING_BQ_SECTIONS		5

/* OSCR timer constants */
#define     FC				3686400.0			/* OSCR clock frequency */
#define		NUM_TIMERS		7					/* Number of timers */
#define		T_FIR_LPF		0					/* FIR LPF timer */
#define		T_FIR_HPF		1					/* FIR HPF timer */
#define		T_IIR_LPF		2					/* IIR LPF timer */
#define		T_IIR_HPF		3					/* IIR HPF timer */
#define		T_SHAPING_HPF	4					/* IIR shaping filter timer */
#define		T_TONE_GEN		5					/* Tone generator timer */
#define		T_NOISE_GEN		6					/* Noise generator timer */

/* Random and tone generator states */
IppToneState_16s		g_ToneState;						/* Tone state */
IppRandGaussState_16s	g_GaussState;						/* Noise generator state */
Ipp16s					g_GaussStdDev;						/* Noise generator sigma */
int						g_SNR;								/* SNR */
int						g_mode, g_type;						/* mode=passthru/LPF/HPF, type=FIR/IIR */
Ipp16s					g_ToneFrq=DEFAULT_TONE_FRQ;			/* Tone frequency */
Ipp16s					g_ToneAmplitude=DEFAULT_TONE_AMP;	/* Tone amplitude */

/* Filter coefficients */
#include "FilterCoefs.h"

/* IIR filter delay lines */
Ipp32s DelayLineBQLPF[2*LPF_BQ_SECTIONS];
Ipp32s DelayLineBQHPF[2*HPF_BQ_SECTIONS];
Ipp32s DelayLineNoiseBQHPF[2*SHAPING_BQ_SECTIONS];

/* LPF and HPF filter memories */
Ipp16s DelayLineLPF[2*LPF_TAPS];
Ipp16s DelayLineHPF[2*HPF_TAPS];
int DelayLineIndexLPF=0;
int DelayLineIndexHPF=0;

/* Adjust random signal parameters */
void SetNoiseSigma(int sigma)
{
	g_GaussStdDev=sigma;
	ippsRandGaussInit_16s(&g_GaussState, 0, g_GaussStdDev, 0);
}

/* Generate signal, apply appropriate filter */
void GenerateSignal(Ipp16s *signal, Ipp16s *noise, int length)
{
	static int initialized=0;
	Ipp32s NoiseEnergy, SignalEnergy;
	Ipp16s tmp;
	int i, t1, t2;
	Ipp64s snr64;
	
	/* Setup timer and signal generators */
	if (!initialized)
	{
		/* Clear filter memories */
		ippsZero_16s(DelayLineLPF,2*LPF_TAPS);
		ippsZero_16s(DelayLineHPF,2*HPF_TAPS);
		for(i=0;i<2*SHAPING_BQ_SECTIONS;i++)
			DelayLineNoiseBQHPF[i]=0;
		for(i=0;i<2*HPF_BQ_SECTIONS;i++)
			DelayLineBQHPF[i]=0;
		for(i=0;i<2*LPF_BQ_SECTIONS;i++)
			DelayLineBQLPF[i]=0;
		initialized=1;
	}

	/* Generate noise */
	ippsRandGauss_16s(noise,length,&g_GaussState);

	/* Apply noise shaping */
	ippsIIR_BiQuadDirect_16s(noise,noise,length,NoiseShapingFilter,SHAPING_BQ_SECTIONS,DelayLineNoiseBQHPF);

	/* Generate tone */
	ippsToneQ15_16s(signal,length,&g_ToneState);

	/* Update SNR */
	ippsNorm_L2_16s32s(signal,length,&SignalEnergy);
	ippsNorm_L2_16s32s(noise,length,&NoiseEnergy);

	/* Mix: noise = signal + noise */
	ippsAdd_16s_I(signal,noise,length);
		
	/* Update SNR */
	if (NoiseEnergy!=0)
	{
		if (SignalEnergy>=NoiseEnergy)
		{
			g_SNR=SignalEnergy/NoiseEnergy;
			tmp=g_SNR&0xffff;
			ippsLn_16s_ISfs(&tmp,1,-11);
			g_SNR=((20*tmp)/LOG10);
		}
		/* Handle SigEng/NoiEng < 1 using log(ab)=log(a)+log(b)=>log(a)=log(ab)-log(b) */
		else
		{
			snr64=(Ipp64s)SignalEnergy*(Ipp64s)LSF/(Ipp64s)NoiseEnergy;
			g_SNR=snr64&0xffffffff;
			tmp=g_SNR&0xffff;
			ippsLn_16s_ISfs(&tmp,1,-11);
			tmp-=LOGLSF;
			g_SNR=((20*tmp)/LOG10);
		}
	}
	else
		g_SNR=100000;

	/* Apply filters as specified by mode */
	switch(g_mode)
	{
		case PASS_THRU:
			/* Pass input through unfiltered */
			ippsCopy_16s(noise,signal,length);
			break;
		case LPF:
			/* Apply FIR LPF */	
			if (g_type==FIR)
			{
				ippsFIR_Direct_16s(noise,signal,length,lpf,LPF_TAPS,DelayLineLPF,&DelayLineIndexLPF);
			}
			/* Apply IIR LPF */	
			else
			{
				ippsIIR_BiQuadDirect_16s(noise,signal,length,lpf_bq,LPF_BQ_SECTIONS,DelayLineBQLPF);
			}
			break;
		case HPF:
			/* Apply FIR HPF */
			if (g_type==FIR)
			{
				ippsFIR_Direct_16s(noise,signal,length,hpf,HPF_TAPS,DelayLineHPF,&DelayLineIndexHPF);
			}
			/* Apply IIR HPF */
			else
			{
				ippsIIR_BiQuadDirect_16s(noise,signal,length,hpf_bq,HPF_BQ_SECTIONS,DelayLineBQHPF);
			}
			break;
	}
}

/* Return SNR */
void GetSNR(int *snr)
{
	*snr=g_SNR;
}

/* Mode select */
void SelectOutputMode(int mode)
{
	g_mode=mode;
}

/* Filter type select */
void SelectFilterType(int type)
{
	g_type=type;
}

/* Adjust tone frequency */
void SetToneFrequency(Ipp16s frequency, int SampleRate)
{
	Ipp32s tmp;
	tmp=(frequency<<14)/(SampleRate/2);
	g_ToneFrq=tmp&0xffff;
	ippsToneInitQ15_16s(&g_ToneState, g_ToneAmplitude, g_ToneFrq, PHASE);
}

/* Adjust tone amplitude */
void SetToneAmplitude(Ipp16s a)
{
	g_ToneAmplitude=a;
	ippsToneInitQ15_16s(&g_ToneState,g_ToneAmplitude,g_ToneFrq,PHASE);
}

