import os
import rox
from rox.options import OptionGroup, Option
from rox import OptionsBox, g

try:
	import dbus
except ImportError:
	rox.croak("Failed to import dbus module. You probably need "
		"to install a package with a name like 'python2.3-dbus'.\n"
		"D-BUS can also be downloaded from http://freedesktop.org.")

try:
	bus = dbus.Bus(dbus.Bus.TYPE_SESSION)
except:
	rox.croak('Failed to connect to D-BUS session bus. This probably '
		"means that you're running an old version of ROX-Session "
		'(or not using ROX-Session at all). Settings cannot be set '
		"using this program until a newer version of ROX-Session is "
		"running.")
rox_session = bus.get_service('net.sf.rox.Session')
rox_settings = rox_session.get_object('/Settings',
		'net.sf.rox.Session.Settings')

dbus_service = bus.get_service('org.freedesktop.DBus')
dbus_object = dbus_service.get_object('/org/freedesktop/DBus',
                                           'org.freedesktop.DBus')
if 'net.sf.rox.Session' not in dbus_object.ListServices():
	rox.alert("ROX-Session doesn't appear to be running (or "
		"you are running an old version). Changing many of "
		"these settings will have no effect.")

class Settings(OptionGroup):
	program = os.path.basename(rox.app_dir)
	def __init__(self):
		self.options = {}
		self.callbacks = []
	
	def notify(self):
		map(apply, self.callbacks)
		for option in self:
			option.has_changed = False
	
	def save(self):
		pass

class Setting(Option):
	def __init__(self, name, default):
		self.name = name
		self.default = default
		settings.options[name] = self
		self.value = None
		try:
			type, value = rox_settings.GetSetting(name)
		except: #XXX: dbus.DBusException:
			self._set(default)
		else:
			self._set(value, notify = False)
	
	def _set(self, value, notify = True):
		Option._set(self, value)
		if not notify: return
		
		def set():
			# Changing the theme will make GTK try to rebuild all
			# its top-levels. Included destroyed ones that Python's
			# GC holds a reference to still. Drop them now...
			import gc; gc.collect()
			if type(self.default) is str:
				rox_settings.SetString(self.name, str(value))
			else:
				rox_settings.SetInt(self.name, self.int_value)
		g.idle_add(set)

def build_slider(box, node, label, option):
	minv = int(node.getAttribute('min'))
	maxv = int(node.getAttribute('max'))
	fixed = int(node.getAttribute('fixed') or "0")
	showvalue = int(node.getAttribute('showvalue') or "0")
	end = node.getAttribute('end')

	hbox = g.HBox(False, 4)
	if label:
		widget = g.Label(label)
		widget.set_alignment(0, 0.5)
		hbox.pack_start(widget, False, True, 0)
	
	if end:
		hbox.pack_end(g.Label(_(end)), False, True, 0)
	
	adj = g.Adjustment(minv, minv, maxv, 1, 10, 0)
	slide = g.HScale(adj)
	if fixed:
		slide.set_size_request(adj.upper, 24)
	if showvalue:
		slide.draw_value(True)
		slide.set_value_pos(g.POS_LEFT)
		slide.set_digits(0)
	else:
		slide.set_draw_value(False)

	box.may_add_tip(slide, node)
	hbox.pack_start(slide, not fixed, True, 0)

	box.handlers[option] = (
		lambda: str(adj.get_value()),
		lambda: adj.set_value(option.int_value))

	slide.connect('value-changed', lambda w: box.check_widget(option))

	return [hbox]

def build_label(box, node, label):
	text = OptionsBox.data(node)
	widget = g.Label(text)
	help = int(node.getAttribute("help") or "0")
	if help:
		widget.set_alignment(0, 0.5)
	else:
		widget.set_alignment(0, 1)
	widget.set_justify(g.JUSTIFY_LEFT)
	widget.set_line_wrap(True)

	if help:
		hbox = g.HBox(False, 4)
		image = g.Image()
		image.set_from_stock(g.STOCK_DIALOG_INFO, g.ICON_SIZE_BUTTON)
		align = g.Alignment(0, 0, 0, 0)

		align.add(image)
		hbox.pack_start(align, False, True, 0)
		hbox.pack_start(widget, False, True, 0)

		spacer = g.EventBox()
		spacer.set_size_request(6, 6)

		return [hbox, spacer]
	return [widget]

OptionsBox.widget_registry['slider'] = build_slider
OptionsBox.widget_registry['label'] = build_label

settings = Settings()
