#include "mvpwindow.h"

#if 0
#define DEBUG(xxx...) fprintf(stderr,xxx)
#else
#define DEBUG(xxx,...)
#endif

struct _label {
	char *text;
	uint32_t fgcol;
	uint32_t bgcol;
	font_t *font;
	char align;
	char valign;
};

static void     label_print(window_t *label, int x, int y, int maxx,  char *text);
static void     label_printchar(window_t *label, int x, int y, glyph_t *glyph);
static void		delete_label_data(window_t *label);

extern glyph_t nbsp;

static int label_calc_x(window_t *label, char *text)
{
	label_t *data;
	int x = 0;
	int label_width, text_width;

	if(window_get_type(label) != WINDOW_LABEL) {
		return x;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return x;
	}

	label_width = window_get_width(label);
	text_width = font_get_textwidth(data->font, text);

	switch(data->align) {
		case LABEL_ALIGN_CENTER:
			x = (label_width / 2) - (text_width / 2);
			break;
		case LABEL_ALIGN_RIGHT:
			fprintf(stderr, "LABEL_ALIGN_RIGHT: %d %d %d\n", label_width, text_width, window_get_xoffset(label));
			x = label_width - text_width - window_get_xoffset(label);
			break;
		case LABEL_ALIGN_LEFT:
		default:
			x = window_get_xoffset(label);
			break;
	}

	if(x < 0) {
		x = 0;
	}

	fprintf(stderr, "Calc X: %d\n", x);
	return x;
}

static void draw_label(window_t *label)
{
	label_t *data;
	int height;
	int x,y;
	char *label_text, *ptr, *current_line;
	int num_lines = 1;

	fprintf(stderr, "draw_label\n");
	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

	if(! data->text ) {
		return;
	}

	/* Count the number of lines */
	ptr = strchr(data->text, '\n');
	while(ptr++) {
		num_lines++;
		ptr = strchr(ptr, '\n');
	}
	fprintf(stderr, "Number of lines: %d\n", num_lines);

	surface_clear(window_get_contents(label), data->bgcol);
	
	height = window_get_height(label);

	switch(data->valign) {
		case LABEL_VALIGN_BOTTOM:
			y = height - (num_lines * font_get_height(data->font));
			break;
		case LABEL_VALIGN_MIDDLE:
			y = (height / 2) - ( num_lines * font_get_height(data->font) / 2) ;
			break;
		case LABEL_VALIGN_TOP:
		default:
			y = 0;
			break;
	}

	if(y < 0) {
		y = 0;
	}

	label_text = strdup(data->text);
	current_line = label_text;
	while(num_lines > 0) {
		ptr = strchr(current_line, '\n');
		if(ptr) *(ptr++)='\0';
		x = label_calc_x(label, current_line);
		label_print(label, x, y, window_get_width(label), current_line);
		num_lines--;
		current_line = ptr;
		y += font_get_height(data->font);

		if(y >= window_get_height(label) - window_get_yoffset(label)) {
			break;
		}
	}
	free(label_text);
}

window_t *new_label(window_t *parent, int x, int y, int width, int height, char *text)
{
	window_t *label;
	label_t *data;

	Dprintf(DEBUG, "new_label\n");
	label = new_window(parent, x, y, width, height);
	data = (label_t *)malloc(sizeof(label_t));

	data->text = (char *)strdup(text);
	data->fgcol = 0xFFFFFF;
	data->bgcol = 0x000000;
	data->font = font_read(default_font);
	data->align = 0;
	data->valign = 0;

	window_set_type(label, WINDOW_LABEL);
	window_set_data(label, data);
	window_set_draw_func(label, draw_label);
	window_set_delete_func(label, delete_label_data);

	window_set_xoffset(label, 1);
	window_set_yoffset(label, 1);

	window_mark_dirty(label);

	return label;
}

void label_set_font(window_t *label, font_t *font)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

    data->font = font;
	window_mark_dirty(label);
}

font_t *label_get_font(window_t *label)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return NULL;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return NULL;
	}

    return data->font;
}

void label_set_fgcol(window_t *label, uint32_t rgb)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

    data->fgcol = rgb;
	window_mark_dirty(label);
}

void label_set_bgcol(window_t *label, uint32_t rgb)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

    data->bgcol = rgb;
	window_mark_dirty(label);
}

uint32_t label_get_fgcol(window_t *label)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return 0;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return 0;
	}

	return data->fgcol;
}

static void label_print(window_t *label, int x, int y, int maxwidth, char *text)
{
	label_t *data;
    glyph_t  *glyph;
    int   tabwidth = ( (maxwidth - x )  ) / 6;
    int width  = 0;

	if(! label) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

	Dprintf(DEBUG, "label_print(%s)\n", text);
    /* Check to see if a font is defined */
    if(! data->font) {
        Dprintf(ERROR, "No font defined for label\n");
        return;
    }

    while ( *text ) {
        switch ( *text ) {
        case '\t':        /* Do some nasty stuff to calculate tab stops,
                             only really works for a full display */
            Dprintf(DEBUG, "label_print(tabstop)\n");
            glyph_set_height(&nbsp, font_get_height(data->font));
            while ( x + 1 < ( ( x / tabwidth ) + 1 ) * tabwidth && x < maxwidth ) {
                label_printchar(label,x,y, &nbsp);
                x++;
            }
            break;
			/*
        case '\n':
            Dprintf(DEBUG, "label_print(cr)\n");
            glyph_set_height(&nbsp, font_get_height(data->font));
            width = glyph_get_width(&nbsp);
            y += font_get_height(data->font);
			x = label_calc_x(label);
            if ( y >= window_get_height(label) - window_get_yoffset(label)) {
                return;
            }
            break;
			*/
        default:
            Dprintf(DEBUG,  "label_print(%c)\n", *(unsigned char *)text);
            if(font_is_defined(data->font, *(unsigned char *)text)) {
                glyph = font_get_glyph(data->font, *(unsigned char *)text);
                width = glyph_get_width(glyph);
                if ( x + width < maxwidth &&
                     ( y < window_get_height(label) - window_get_yoffset(label)) ) {
                    label_printchar(label,x,y,glyph);
                    x += width;
                }
            }
            break;
        }
        text++;
    }
}

static void label_printchar(window_t *label,int x, int y, glyph_t *glyph)
{
	label_t *data;
    int *bitmap;
    int i,j, mask;
    int width, height;

	if(! label) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

	Dprintf(DEBUG, "label_printchar\n");

    bitmap = glyph_get_bitmap(glyph);

    width = glyph_get_width(glyph);
    height = glyph_get_height(glyph);

    for(j = 0 ; j < height; j++) {
        for ( i = 0; i < width; i++ ) {
            mask = 1 << i;
            if ( bitmap && (bitmap[j] & mask ) ) {
                surface_set_pixel(window_get_contents(label),x + (width - i),y,data->fgcol);
            } else {
                surface_set_pixel(window_get_contents(label),x + (width - i),y,data->bgcol);
            }
        }
        y++;
    }
}

void label_set_align(window_t *label, int align)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

	fprintf(stderr, "label_set_align(%d)\n", align);

	data->align = align;

	window_mark_dirty(label);
}

void label_set_valign(window_t *label, int valign)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

	fprintf(stderr, "label_set_valign(%d)\n", valign);
	data->valign = valign;
	window_mark_dirty(label);

	return;
}


char label_get_align(window_t *label)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return 0;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return 0;
	}

	return data->align;
}

char label_get_valign(window_t *label)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return 0;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return 0;
	}

	return data->valign;
}

static void delete_label_data(window_t *label)
{
	label_t *data;

	if(window_get_type(label) != WINDOW_LABEL) {
		return;
	}

	if(! (data = (label_t *)window_get_data(label)) ) {
		return;
	}

	free(data->text);
	free(data);

	window_set_data(label, NULL);
}

/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
