/*
 *   MediaMVP Server Library
 *
 *   (C) 2004 Dominic Morris
 *
 *   $Id: surface.c,v 1.9 2004/04/25 19:56:59 dom Exp $
 *   $Date: 2004/04/25 19:56:59 $
 *
 *
 *   Split up the display handling into a low level surface and a higher level text display engine
 *
 *   This bit deals with the font engine
 */

#include "libmvp_internal.h"


typedef struct _colortbl {
    uint32_t rgb;
    uint8_t y, v, u;
} colortbl_t;

typedef struct {
    uint8_t  *image;
    colortbl_t colortbl[256];
    int       next_free_color;
} surface8_t;

typedef struct {
    uint8_t  *layers[3];
} surface24_t;

struct _surface {
    int       width;
    int       depth;
    int       yuvsize;
    bool_t    dirty;
    union {
        surface8_t   s8;
        surface24_t  s24;
    } sfc;
    void     (*dealloc)(surface_t *);
    void     (*clear)(surface_t *, int);
    void    *(*rgb2yvuy)(surface_t *);
    void     (*setpixel)(surface_t *, int, int, uint32_t);
    uint32_t (*getpixel)(surface_t *, int, int);
};


static void          rgb2yvu(uint8_t *py, uint8_t *pv, uint8_t *pu, int red, int green, int blue);
static int           surface8_alloc(surface_t *surface);
static void          surface8_dealloc(surface_t *surface);
static uint8_t       get_color(surface_t *surface, uint32_t rgb);
static void          surface8_clear(surface_t *surface, int rgb);
static void         *surface8_rgb2yvuy(surface_t *surface);
static void          surface8_set_pixel(surface_t *surface, int x, int y, uint32_t rgb);
static uint32_t      surface8_get_pixel(surface_t *surface,int x, int y);
static int           surface24_alloc(surface_t *surface);
static void          surface24_dealloc(surface_t *surface);
static void          surface24_clear(surface_t *surface, int rgb);
static void         *surface24_rgb2yvuy(surface_t *surface);
static void          surface24_set_pixel(surface_t *surface, int x, int y, uint32_t rgb);
static uint32_t      surface24_get_pixel(surface_t *surface,int x, int y);


#if 0
static void rgb2yvu(uint8_t *py, uint8_t *pv, uint8_t *pu, int red, int green, int blue)
{
    int y,v,u;

    y = 0.30 * (double)red +
        0.59 * (double)green +
        0.11 * (double)blue;

    if ( y > 240 )
        y = 240;
    else if ( y < 0 )
        y = 0;

    v = -(0.15 * (double)red) -
        (0.29 * (double)green) +
        (0.44 * (double)blue) +
        128.0;

    u = (0.62 * (double)red) -
        (0.52 * (double)green) +
        (0.10 * (double)blue) +
        128.0;

    if ( v > 240 )
        v = 240;
    else if ( v < 0 )
        v = 0;

    if ( u > 240 )
        u = 240;
    else if ( u < 0 )
        u = 0;

    *py = y;
    *pv = v;
    *pu = u;
}
#endif

#if 1
static void rgb2yvu(uint8_t *py, uint8_t *pu, uint8_t *pv, int red, int green, int blue)
{
	double r1, g1, b1, r2, g2, b2, y, u, v;

	/*
	 * convert 0..255 RGB to floating point range 0.0->1.0
	 */
	r1 = (double)red / 255.0;
	g1 = (double)green / 255.0;
	b1 = (double)blue / 255.0;

	/*
	 * apply gamma correction
	 */
	if (r1 < 0.018)
		r2 = r1 * 4.5;
	else
		r2 = (1.099 * pow(r1, 0.45)) - 0.099;
		
	if (g1 < 0.018)
		g2 = g1 * 4.5;
	else
		g2 = (1.099 * pow(g1, 0.45)) - 0.099;
		
	if (b1 < 0.018)
		b2 = b1 * 4.5;
	else
		b2 = (1.099 * pow(b1, 0.45)) - 0.099;
		

	/*
	 * convert gamma corrected RGB to YUV space
	 */
	y = (0.2989 * r2) + (0.5866 * g2) + (0.1145 * b2);
	u = ( (-0.147 * r2)  - (0.289 * g2) + (0.436 * b2) ) + 0.5;
	v = ( (0.615 * r2) - (0.515 * g2) + (0.100 * b2) ) + 0.5;

	/*
	 * range clamp for errors (shouldn't be necessary)
	 */
	if (y < 0.0)
		y = 0.0;
	else if (y > 1.0)
		y = 1.0;
		
	if (u < 0.0)
		u = 0.0;
	else if (u > 1.0)
		u = 1.0;
		
	if (v < 0.0)
		v = 0.0;
	else if (v > 1.0)
		v = 1.0;

	/*
	 * convert back to byte values
	 */
	*py = (uint8_t)(y * 255.0);
	*pu = (uint8_t)(u * 255.0);
	*pv = (uint8_t)(v * 255.0);
}


#endif

#if 0

/*
 * rgb2yuv() -- convert a from RGB colour space to YUV colour space
 *
 * This version does it as per the Colour Space FAQ found at:
 *
 *	http://www.neuro.sfc.keio.ac.jp/~aly/polygon/info/color-space-faq.html
 *
 * and:
 *
 *	http://www.video-demystified.com/mm/tutor/ch03.pdf
 *
 * coded in a hurry by Mike Tubby, mike@tubby.org, 18-ARP-2004
 *
 * I think the RGB to YUV conversion only works properly if you take
 * gamma correction into account before doing the transform...
 *
 */

#define GAMMA_CORRECTION	(1.0/2.8)

static void rgb2yvu(uint8_t *py, uint8_t *pu, uint8_t *pv, int red, int green, int blue)
{
	double r1, g1, b1, r2, g2, b2, y, u, v;

	/*
	 * convert 0..255 RGB to floating point range 0.0->1.0
	 */
	r1 = (double)red / 255.0;
	g1 = (double)green / 255.0;
	b1 = (double)blue / 255.0;

	/*
	 * apply gamma correction
	 */
	r2 = pow(r1, GAMMA_CORRECTION);
	g2 = pow(g1, GAMMA_CORRECTION);
	b2 = pow(b1, GAMMA_CORRECTION);

	/*
	 * convert gamma corrected RGB to YUV space
	 */
	y = (0.2989 * r2) + (0.5866 * g2) + (0.1145 * b2);
	u = ( (-0.147 * r2)  - (0.289 * g2) + (0.436 * b2) ) + 0.5;
	v = ( (0.615 * r2) - (0.515 * g2) + (0.100 * b2) ) + 0.5;

	/*
	 * range clamp for errors - shouldn't be necessary
	 */
	if (y < 0.0)
		y = 0.0;
	else if (y > 1.0)
		y = 1.0;
		
	if (u < 0.0)
		u = 0.0;
	else if (u > 1.0)
		u = 1.0;
		
	if (v < 0.0)
		v = 0.0;
	else if (v > 1.0)
		v = 1.0;

	/*
	 * convert back to byte values
	 */
	*py = (uint8_t)(y * 255.0);
	*pu = (uint8_t)(u * 255.0);
	*pv = (uint8_t)(v * 255.0);
}

#endif

#if 0

#define SATURATION 0.70


static void rgb2yvu(uint8_t *py, uint8_t *pu, uint8_t *pv, int red, int green, int blue)
{
	double r1, g1, b1, y, u, v;

	/*
	 * convert 0..255 RGB to floating point range 0.0->1.0
	 */
	r1 = (double)red / 255.0;
	g1 = (double)green / 255.0;
	b1 = (double)blue / 255.0;

	/*
	 * convert gamma corrected RGB to YUV space
	 */
	y = (0.2989 * r1) + (0.5866 * g1) + (0.1145 * b1);
	u = ( ( (-0.147 * r1)  - (0.289 * g1) + (0.436 * b1) ) + 0.5) * SATURATION ;
	v = ( ( (0.615 * r1) - (0.515 * g1) + (0.100 * b1) ) + 0.5) * SATURATION;

	/*
	 * range clamp for errors - shouldn't be necessary
	 */
	if (y < 0.0)
		y = 0.0;
	else if (y > 1.0)
		y = 1.0;
		
	if (u < 0.0)
		u = 0.0;
	else if (u > 1.0)
		u = 1.0;
		
	if (v < 0.0)
		v = 0.0;
	else if (v > 1.0)
		v = 1.0;

	/*
	 * convert back to byte values
	 */
	*py = (uint8_t)(y * 255.0);
	*pu = (uint8_t)(u * 255.0);
	*pv = (uint8_t)(v * 255.0);
}

#endif

surface_t *surface_alloc(int width, int depth, int bpp)
{
    surface_t  *surface = (surface_t*)malloc(sizeof(*surface));

    if ( surface == NULL ) {
        return NULL;
    }

    /* Set up sizes */
    surface->width = width;
    surface->depth  = depth;
    surface->yuvsize = width * depth * 2;
    surface->dirty = TRUE;

    switch ( bpp ) {
    case 8:
        surface8_alloc(surface);
        surface->clear = surface8_clear;
        surface->rgb2yvuy = surface8_rgb2yvuy;
        surface->setpixel = surface8_set_pixel;
        surface->getpixel = surface8_get_pixel;
        surface->dealloc = surface8_dealloc;
        break;
    case 24:
        surface24_alloc(surface);
        surface->clear = surface24_clear;
        surface->rgb2yvuy = surface24_rgb2yvuy;
        surface->setpixel = surface24_set_pixel;
        surface->getpixel = surface24_get_pixel;
        surface->dealloc = surface24_dealloc;
        break;
    default:
        free(surface);
        surface = NULL;
        break;
    }

    return surface;
}


void surface_dealloc(surface_t *surface)
{
    if ( surface == NULL ) {
        return;
    }

    surface->dealloc(surface);


    free(surface);
}

int surface_get_yuvsize(surface_t *surface)
{
    return surface->yuvsize;
}



void surface_set_pixel(surface_t *surface, int x, int y, uint32_t rgb)
{
    if ( x >= 0 && x < surface->width && y >= 0 && y < surface->depth ) {
        surface->setpixel(surface,x,y,rgb);
        surface->dirty = TRUE;
    }
}

uint32_t surface_get_pixel(surface_t *surface, int x, int y)
{
    if ( x >= 0 && x < surface->width && y >= 0 && y < surface->depth ) {
        return surface->getpixel(surface,x,y);
    }
    return 0;
}

void surface_clear(surface_t *surface, uint32_t rgb)
{
    surface->clear(surface,rgb);
}

void *surface_rgb2yvuy(surface_t *surface)
{
    return surface->rgb2yvuy(surface);
}

int surface_get_dirty(surface_t *surface)
{
    return surface->dirty;
}

void surface_set_dirty(surface_t *surface, int state)
{
    surface->dirty = state;
}


/* Routines for dealing with 8 bit surface */
static int surface8_alloc(surface_t *surface)
{
    surface->sfc.s8.next_free_color = 0;

    /* Allocate the RGB surfaces */
    surface->sfc.s8.image = (uint8_t*)calloc(surface->width * surface->depth,1);

    return 1;
}

static void surface8_dealloc(surface_t *surface)
{
    if ( surface->sfc.s8.image ) {
        free(surface->sfc.s8.image);
    }
}


static uint8_t get_color(surface_t *surface, uint32_t rgb)
{
    int i;
    colortbl_t* ct;
    for (i=0; i<surface->sfc.s8.next_free_color; i++) {
            if (surface->sfc.s8.colortbl[i].rgb == rgb) {
                return i;
            }
    }

    if (surface->sfc.s8.next_free_color >= 256) {
         Dprintf(DEBUG, "I don't like too many colors. Bye.\n");
         exit(1);
    }

    Dprintf(DEBUG, "allocating color %d\n", surface->sfc.s8.next_free_color);
    ct = &surface->sfc.s8.colortbl[surface->sfc.s8.next_free_color];
    ct->rgb = rgb;
    rgb2yvu(&ct->y, &ct->v, &ct->u, (rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

    return surface->sfc.s8.next_free_color++;
}



static void surface8_clear(surface_t *surface, int rgb)
{
    uint8_t index = get_color(surface,rgb);

    memset(&surface->sfc.s8.image[0],index,surface->width * surface->depth);
}

static void *surface8_rgb2yvuy(surface_t *surface)
{
    uint8_t    *sfc;
    int         i,j;


    sfc = (uint8_t*)malloc(surface->yuvsize);

    for ( j = 0, i = 0; i < surface->width * surface->depth; i+=2 ) {

        sfc[j++] = surface->sfc.s8.colortbl[ surface->sfc.s8.image[i+0] ].y;
        sfc[j++] = surface->sfc.s8.colortbl[ surface->sfc.s8.image[i+0] ].v;
        sfc[j++] = surface->sfc.s8.colortbl[ surface->sfc.s8.image[i+0] ].u;
        sfc[j++] = surface->sfc.s8.colortbl[ surface->sfc.s8.image[i+1] ].y;
    }


    return sfc;
}


static void surface8_set_pixel(surface_t *surface, int x, int y, uint32_t rgb)
{
    int offset = y * surface->width + x;
    uint8_t  index = get_color(surface,rgb);

    surface->sfc.s8.image[offset] = index;
}

static uint32_t surface8_get_pixel(surface_t *surface,int x, int y)
{
    int offset = y * surface->width + x;

    return surface->sfc.s8.colortbl[ surface->sfc.s8.image[offset] ].rgb;
}


/* Routines for dealing with the 24 bit surface */
static int surface24_alloc(surface_t *surface)
{
    int   i;

    /* Allocate the RGB surfaces */
    for ( i = 0; i < 3; i++ ) {
        surface->sfc.s24.layers[i] = (uint8_t*)calloc(surface->width * surface->depth,1);
    }
    return 0;
}

static void surface24_dealloc(surface_t *surface)
{
   int   i;

    /* Deallocate the RGB surfaces */
    for ( i = 0; i < 3; i++ ) {
        if ( surface->sfc.s24.layers[i] ) {
            free(surface->sfc.s24.layers[i]);
        }
    }
}



static void surface24_clear(surface_t *surface, int rgb)
{
    memset(&surface->sfc.s24.layers[0][0],(rgb >> 16) & 0xFF,surface->width * surface->depth);
    memset(&surface->sfc.s24.layers[1][0],(rgb >> 8) & 0xFF,surface->width * surface->depth);
    memset(&surface->sfc.s24.layers[2][0],(rgb >> 0) & 0xFF,surface->width * surface->depth);
}

static void *surface24_rgb2yvuy(surface_t *surface)
{
    uint8_t    *sfc;
    int         i,j;


    sfc = (uint8_t*)malloc(surface->yuvsize);

    for ( j = 0, i = 0; i < surface->width * surface->depth; i+=2 ) {
        uint8_t   y1,u1,v1,y2,u2,v2;

        rgb2yvu(&y1,&v1,&u1,surface->sfc.s24.layers[0][i],surface->sfc.s24.layers[1][i],surface->sfc.s24.layers[2][i]);
        rgb2yvu(&y2,&v2,&u2,surface->sfc.s24.layers[0][i+1],surface->sfc.s24.layers[1][i+1],surface->sfc.s24.layers[2][i+1]);

        sfc[j++] = y1;
        sfc[j++] = v1;
        sfc[j++] = u1;
        sfc[j++] = y2;
    }

    return sfc;
}

static void surface24_set_pixel(surface_t *surface, int x, int y, uint32_t rgb)
{
    int offset = y * surface->width + x;

    surface->sfc.s24.layers[0][offset] = (rgb >> 16 ) & 0xFF;
    surface->sfc.s24.layers[1][offset] = (rgb >> 8 ) & 0xFF;
    surface->sfc.s24.layers[2][offset] = (rgb >> 0 ) & 0xFF;
}

static uint32_t surface24_get_pixel(surface_t *surface,int x, int y)
{
    int offset = y * surface->width + x;

    return ( (  surface->sfc.s24.layers[0][offset] << 16 ) | ( surface->sfc.s24.layers[1][offset] << 8) | ( surface->sfc.s24.layers[2][offset] ) );
}

/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
