#ifdef TESTING
#warning "TESTING MACRO ENABLED"
#endif

#include "libmvp_internal.h"
#include "mvpwindow.h"
#include "mvpfont.h"

struct _pos {
    int x,y,z;
};

struct _dimension {
    int width, height;
    pos_t offset;
};

struct _window {
    char *name;
    int type; // If we need it
	int flags;
    pos_t pos; // Position within parent
    dimension_t dimension; // Size of window
    surface_t *draw_surface; // Surface to draw on
    surface_t *contents; // Surface to draw on
    window_t *parent; 
    window_t *first_child;
	window_t *child_focus;
    window_t *next_sibling;
    window_t *prev_sibling;
    void (*draw_func)(window_t *window); // User-defined function to draw window
	void (*focus_func)(window_t *window); // User-defined focus handler
	void (*delete_func)(window_t *window); // User defined data deletion
    void *data; // Window-specific data
};

window_t *new_window(window_t *parent, int x, int y, int width, int height)
{
    window_t *window = (window_t *)malloc(sizeof(window_t));

    window->name = NULL;
    window->type = 0;
	window->flags = 0;
    window->pos.x = x;
    window->pos.y = y;
    window->pos.z = 0;
    window->dimension.width = width;
    window->dimension.height = height;
    window->dimension.offset.x = 0;
    window->dimension.offset.y = 0;
    window->draw_surface = NULL;
    window->contents = surface_alloc(width, height, 24);
    window->parent = parent;
    window->first_child = NULL;
	window->child_focus = NULL;
    window->next_sibling = NULL;
    window->prev_sibling = NULL;
    window->draw_func = NULL;
	window->focus_func = NULL;
	window->delete_func = NULL;
    window->data = NULL;

    if(parent) {
	window_add_child(parent, window);
    }

    return window;
}

void window_set_name(window_t *window, char *name)
{
    if(! window) {
	return;
    }

    window->name = (char *)strdup(name);
}

char *window_get_name(window_t *window)
{
    if(! window) {
	return NULL;
    }

    return window->name;
}

void window_set_data(window_t *window, void *data)
{
    if(! window) {
	return;
    }

    window->data = data;
	window_mark_dirty(window);
}

void *window_get_data(window_t *window)
{
    if(! window) {
	return NULL;
    }

    return window->data;
}

void window_set_draw_func(window_t *window, void (*draw_func)(window_t *this))
{
    if(! window) {
	return;
    }

    window->draw_func = draw_func;
	window_mark_dirty(window);
}

void window_set_focus_func(window_t *window, void (*focus_func)(window_t *this))
{
	if(! window) {
		return;
	}

	window->focus_func = focus_func;
	window_mark_dirty(window);
}

void window_set_delete_func(window_t *window, void (*delete_func)(window_t *this))
{
	if(! window) {
		return;
	}

	window->delete_func = delete_func;
}

surface_t *window_get_draw_surface(window_t *window)
{
    if(! window) {
	return NULL;
    }

    return window->draw_surface;
}

void window_set_draw_surface(window_t *window, surface_t *draw_surface)
{
    if(! window) {
	return;
    }

    window->draw_surface = draw_surface;
}

void window_set_type(window_t *window, int type)
{
    if(! window) {
	return;
    }

    window->type = type;
}

int window_get_type(window_t *window)
{
    if(! window) {
	return WINDOW_NULL;
    }

    return window->type;
}

int window_get_xpos(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->pos.x;
}

void window_set_xpos(window_t *window, int x)
{
    if(! window) {
	return;
    }

    window->pos.x = x;
	window_mark_dirty(window);
}

int window_get_ypos(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->pos.y;
}

void window_set_ypos(window_t *window, int y)
{
    if(! window) {
	return;
    }

    window->pos.y = y;
	window_mark_dirty(window);

	window_mark_dirty(window->parent);
}

void window_set_pos(window_t *window, int x, int y)
{
    if(! window) {
	return;
    }

    window->pos.x = x;
    window->pos.y = y;
	window_mark_dirty(window);

	window_mark_dirty(window->parent);
}

int window_get_height(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->dimension.height;
}

void window_set_height(window_t *window, int height)
{
    if(! window) {
	return;
    }

    window->dimension.height = height;
	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

int window_get_width(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->dimension.width;
}

void window_set_width(window_t *window, int width)
{
    if(! window) {
	return;
    }

    window->dimension.width = width;
	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

void window_set_dimension(window_t *window, int width, int height)
{
    if(! window) {
	return;
    }

    window->dimension.width = width;
    window->dimension.height = height;
	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

int window_get_xoffset(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->dimension.offset.x;
}

void window_set_xoffset(window_t *window, int xoffset)
{
    if(! window) {
	return;
    }

    window->dimension.offset.x = xoffset;
	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

int window_get_yoffset(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->dimension.offset.y;
}

void window_set_yoffset(window_t *window, int yoffset)
{
    if(! window) {
	return;
    }

    window->dimension.offset.y = yoffset;
	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

void window_set_offset(window_t *window, int xoffset, int yoffset)
{
    if(! window) {
	return;
    }

    window->dimension.offset.x = xoffset;
    window->dimension.offset.y = yoffset;

	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

int window_get_zorder(window_t *window)
{
    if(! window) {
	return 0;
    }

    return window->pos.z;
}

void window_set_zorder(window_t *window, int z)
{
    window_t *current_window;

    if(! window) {
	return;
    }

    /* If the zorder is the same */
    if(window->pos.z == z) {
	return;
    }

    if(z < window->pos.z) {
	if(! window->prev_sibling ) {
	    window->pos.z = z;
		window_mark_dirty(window);
		window_mark_dirty(window->parent);
	    return;
	}

	if(window->prev_sibling->pos.z <= z) {
	    window->pos.z = z;
		window_mark_dirty(window);
		window_mark_dirty(window->parent);
	    return;
	}

	/* Detach this window from the sibling list */
	current_window = window->prev_sibling;

	if(window->next_sibling) {
	    window->next_sibling->prev_sibling = window->prev_sibling;
	}

	window->prev_sibling->next_sibling = window->next_sibling;

	window->next_sibling = NULL;
	window->prev_sibling = NULL;

	/* Work back through the list until we find a place to insert */
	/* our window                                                 */
	while(current_window->prev_sibling && current_window->pos.z > z) {
	    current_window = current_window->prev_sibling;
	}

	/* Reset the parent's child pointer */
	if(! current_window->prev_sibling) {
	    window->parent->first_child = window;
	}

	/* Re-insert the window into the list */
	window->next_sibling = current_window;
	window->prev_sibling = current_window->prev_sibling;
	current_window->prev_sibling = window;
    } else {
	if(! window->next_sibling ) {
	    window->pos.z = z;
		window_mark_dirty(window);
		window_mark_dirty(window->parent);
	    return;
	}

	if(window->next_sibling->pos.z >= z) {
	    window->pos.z = z;
		window_mark_dirty(window);
		window_mark_dirty(window->parent);
	    return;
	}

	/* Detach this window from the sibling list */
	current_window = window->next_sibling;

	if(window->prev_sibling) {
	    window->prev_sibling->next_sibling = window->next_sibling;
	} else {
	    /* Reset the parent's pointer */
	    window->parent->first_child = window->next_sibling;
	}

	window->next_sibling->prev_sibling = window->prev_sibling;

	window->next_sibling = NULL;
	window->prev_sibling = NULL;

	/* Work forward through the list until we find a place to insert */
	/* our window                                                    */
	while(current_window->next_sibling && current_window->pos.z < z) {
	    current_window = current_window->next_sibling;
	}

	window->prev_sibling = current_window;
	window->next_sibling = current_window->next_sibling;
	current_window->next_sibling = window;
    }

    window->pos.z = z;
	window_mark_dirty(window);
	window_mark_dirty(window->parent);
}

void window_add_child(window_t *parent, window_t *child)
{
    window_t *current_window;

    if( ! parent || ! child ) {
	return;
    }

    child->parent = parent;

    /* If there are no children */
    if(! parent->first_child ) {
		parent->first_child = child;
		window_mark_dirty(parent);
		return;
    }


    current_window = parent->first_child;

    /* If the first child is higher in the zorder */
    if(current_window->pos.z >= child->pos.z) {
	parent->first_child = child;
	current_window->prev_sibling = child;
	child->next_sibling = current_window;
	window_mark_dirty(parent);
	return;
    }

    while(current_window->next_sibling) {
		current_window = current_window->next_sibling;
		if(current_window->pos.z >= child->pos.z) {
			current_window->prev_sibling->next_sibling = child;
			child->next_sibling = current_window;
			current_window->prev_sibling = child;
			window_mark_dirty(parent);
			return;
		}
	}

    current_window->next_sibling = child;
    child->prev_sibling = current_window;

	window_mark_dirty(parent);
}

/* Internal function for deleting windows */
/* includes deletion of siblings */
static void delete_window_internal(window_t *window)
{
    if(! window) {
	return;
    }

    /* Mark the parent as dirty */
	window_mark_dirty(window->parent);

    /* Recursively delete all children */
    if(window->first_child) {
		delete_window_internal(window->first_child);
    }

    /* Recursively delete all siblings */
    if(window->next_sibling) {
		delete_window_internal(window->next_sibling);
    }

    /* Delete the surface */
    if(window->draw_surface) {
		surface_dealloc(window->draw_surface);
    }

    /* Delete the contents */
    if(window->contents) {
		surface_dealloc(window->contents);
    }

    /* Delete any window-specific data */
    if(window->data) {
		if(window->delete_func) {
			window->delete_func(window);
		} else {
			free(window->data);
		}
    }

    if(window->name) {
		free(window->name);
    }

    /* Delete the window memory */
    free(window);
}

/* External function for deleting windows */
/* Does not delete siblings */
void delete_window(window_t *window)
{
    /* Mark the parent as dirty */
	window_mark_dirty(window->parent);

    if(window->first_child) {
		delete_window_internal(window->first_child);
    }

    /* Remove this window from the sibling list */
    if(! window->prev_sibling) {
		if(window->parent) {
			window->parent->first_child = window->next_sibling;
		}
	} else {
		window->prev_sibling->next_sibling = window->next_sibling;
    }

    /* Delete the surface */
    if(window->draw_surface) {
		surface_dealloc(window->draw_surface);
    }

    /* Delete the contents surface */
    if(window->contents) {
		surface_dealloc(window->contents);
    }

    /* Delete any window-specific data */
    if(window->data) {
		free(window->data);
    }

    if(window->name) {
		free(window->name);
    }

    /* Delete the window memory */
    free(window);
}

/* Internal function for drawing windows */
/* Draws siblings */
static void window_draw_internal(window_t *window, int *dirty)
{
    if(! window) {
	return;
    }

    /* Call the draw function for this window */
    if(window_is_dirty(window) && window->draw_func) {
		window->draw_func(window);
		window_mark_clean(window);
		*dirty +=1 ;
    } 

    window_blit(window);

    /* Draw the next sibling */
    if(window->next_sibling) {
		window_draw_internal(window->next_sibling, dirty);
    }

    /* Draw any children */
    if(window->first_child) {
		window_draw_internal(window->first_child, dirty);
    }
}

/* External function for drawing windows */
/* Does not draw siblings */
int window_draw(window_t *window)
{
	int dirty = 0;

    if(! window) {
		return 0;
    }

    /* Call the draw function for this window */
    if(window_is_dirty(window) && window->draw_func) {
		window->draw_func(window);
		window_mark_clean(window);
		dirty++;
    }

    window_blit(window);

    /* Draw any children */
    if(window->first_child) {
		window_draw_internal(window->first_child, &dirty);
    }

	return dirty;
}

window_t *new_root(int width, int height, int bpp)
{
    window_t *root = new_window(NULL, 0,0, width, height);

    root->draw_surface = surface_alloc(width, height, bpp);

    return root;
}

#ifdef TESTING
static void window_internal_dump_window(window_t *window, int depth)
{
    int i;
    if(! window) {
	return;
    }

    fprintf(stderr, "|_%s (%d)\n", window->name, window->pos.z);

    /* Dump children */
    if(window->first_child) {
	for(i = 0 ; i < depth; i++)
	{
	    fprintf(stderr, "\t");
	}

	window_internal_dump_window(window->first_child, depth+1);
    }

    /* Dump siblings */
    if(window->next_sibling) {
	for(i = 0 ; i < depth-1; i++)
	{
	    fprintf(stderr, "\t");
	}

	window_internal_dump_window(window->next_sibling, depth);
    }
}

void window_dump_window(window_t *window)
{
    if(! window) {
	return;
    }

    fprintf(stderr, "--- WINDOW DUMP ---\n");
    fprintf(stderr, "%s (%d)\n", window->name, window->pos.z);

    /* Dump children */
    if(window->first_child) {
	window_internal_dump_window(window->first_child, 1);
    }
    fprintf(stderr, "-------------------\n");
}
#endif

void window_blit(window_t *window)
{
    surface_t *surface;
    window_t *current;
    int delta_x = 0, delta_y = 0;
    int max_x = 0, max_y = 0;
    int x,y;

    if(! window) {
        return;
    }


	Dprintf(DEBUG, "window_blit(%s)\n", window_get_name(window));

    /* Determine which surface to use */
    current = window;
    max_x = window->dimension.width;
    max_y = window->dimension.height;
    while( current->draw_surface == NULL && current->parent) {
        delta_x += current->pos.x;
        delta_y += current->pos.y;

		/*
        if ( current != window ) {
            if(max_x + delta_x > current->dimension.width ) {
                max_x = current->dimension.width - delta_x;
            }
            
            if(max_y + delta_y > current->dimension.height) {
                max_y = current->dimension.height - delta_y;
            }
        }
		*/

        current = current->parent;

    }

	Dprintf(DEBUG, "Surface: %s\n", window_get_name(current));
	Dprintf(DEBUG, "max_x: %d\n", max_x);
	Dprintf(DEBUG, "max_y: %d\n", max_y);
	Dprintf(DEBUG, "delta_x: %d\n", delta_x);
	Dprintf(DEBUG, "delta_y: %d\n", delta_y);

    surface = current->draw_surface;

    if(! surface) {
	    return;
    }
    for( y = 0; y < max_y ; y++) {
	    for( x = 0; x < max_x; x++) {
		    uint32_t c = surface_get_pixel(window->contents, x, y);
		    surface_set_pixel(surface, x + delta_x, y + delta_y, c);
	    }
    }
}

surface_t *window_get_contents(window_t *window)
{
	if(! window) {
		return NULL;
	}

	return window->contents;
}

void window_set_pixel(window_t *window, int x, int y, uint32_t col)
{
	if(! window) {
		return;
	}

	if(! window->contents) {
		return;
	}

	surface_set_pixel(window->contents, x, y, col);
}

void window_mark_dirty(window_t *window)
{
	if(! window) {
		return;
	}

	window->flags |= FLAG_DIRTY;
}

void window_mark_clean(window_t *window)
{
	if(! window) {
		return;
	}

	window->flags &= ~FLAG_DIRTY;
}

int window_is_dirty(window_t *window)
{
	if(! window) {
		return 0;
	}

	return window->flags & FLAG_DIRTY;
}

int window_has_focus(window_t *window)
{
	if(! window) {
		return 0;
	}

	return window->flags & FLAG_FOCUS;
}

void window_lose_focus(window_t *window)
{
	if(! window) {
		return;
	}

	window->flags &= ~FLAG_FOCUS;
	window_mark_dirty(window);

	if(window->parent) {
		window->parent->child_focus = NULL;
	}
}

void window_set_focus(window_t *window)
{
	if(! window) {
		return;
	}

	if(window->parent) {
		window->parent->child_focus = window;
	}

	window->flags |= FLAG_FOCUS;
	window_mark_dirty(window);
}

void window_toggle_modal(window_t *window)
{
	if(! window) {
		return;
	}

	if(window->flags & FLAG_MODAL) {
		window->flags &= ~ FLAG_MODAL;
	} else {
		window->flags |= FLAG_MODAL;
	}

}

int window_is_modal(window_t *window)
{
	if(! window) {
		return 0;
	}

	return window->flags & FLAG_MODAL;
}

window_t *window_get_root(window_t *window)
{
	if(! window) {
		return NULL;
	}

	while(window->parent) {
		window = window->parent;
	}

	return window;
}
