/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: menuapp.c,v 1.17 2004/02/22 15:45:59 dom Exp $
 *   $Date: 2004/02/22 15:45:59 $
 *
 *
 *   Implements the application
 */


/* Include the definition of the plugin - we need to use some inherited functions */
#include "mediamvp.h"

#include "app.h"
#include "menu.h"
#include "menuapp.h"
#include "misc.h"

#include <dirent.h>
#include <sys/stat.h>
#include <ctype.h>



typedef struct {
    char   *name;
    int     flags;
    int     vdrrecdir ;
} filelist_t;

typedef struct {
    char     *name;
    char     *url;
} channel_t;


struct _menuapp {
    dongle_t  *dongle;
    render_t  *render;
    app_t     *app;
    menu_t    *menu;
    char      *current_media;
    int        num_next_media;
    char     **next_media;
    char       paused;

    /* Information about media stuff - this is shared... */
    bool_t        Now;
    int           channels_num;
    channel_t    *channels; 

 
};


typedef struct {
    menuapp_t      *menuapp;
    char       *path;
    /* For handling scanning of directories */
    int         filelist_num;
    filelist_t *filelist;
} dirlist_t;




/* These need to be configured somehow... */
static char         *c_mp3_dir = "/home/mp3";
static char         *c_mpeg_dir = "/";
static char         *c_vdrrecs_dir = "/video0";


static int           menuapp_main(menu_t *menu,void *imenuapp, int sel);

static int           menuapp_channels(menu_t *menu,void *imenuapp, int sel);
static int           channels_process(menu_t *menu,void *imenuapp, int sel);
static int           channels_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);


static int           schedule_menuapp(menu_t *menu,void *imenuapp, int sel);
static int           schedule_process(menu_t *menu,void *imenuapp, int sel);
static int           schedule_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);



static int           menuapp_httpradio(menu_t *menu,void *imenuapp, int sel);
static int           httpradio_process(menu_t *menu,void *imenuapp, int sel);

static int           menuapp_mpeg(menu_t *menu,void *imenuapp, int sel);
static int           mpeg_process(menu_t *menu,void *imenuapp, int sel);
static int           mpeg_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);

static int           menuapp_vdrrecs(menu_t *menu,void *imenuapp, int sel);
static int           vdrrecs_process(menu_t *menu,void *imenuapp, int sel);
static int           vdrrecs_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);
static int	     vdrnamemangle(char *in, char *out);

static int           menuapp_music(menu_t *menu,void *imenuapp, int sel);
static int           music_process(menu_t *menu,void *imenuapp, int sel);
static int           music_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);


static int           menuapp_settings(menu_t *menu,void *imenuapp, int sel);
static int           settings_process(menu_t *menu,void *imenuapp, int sel);

static int           media_keys(void *ptr, int code);
static int           media_ack(int acktype, void *param, unsigned char *buf, int len);


/* A whole bunch of supporting routines */
static filelist_t   *directory_scan(char *path, int *scanned);
static filelist_t   *directory_scan_vdr(char *path, int *scanned);
static int is_vdr_recording(const struct dirent * pde);
static void vdr_push(char *path,menuapp_t *app);

static void          directory_sort(int num,filelist_t *list);
static dirlist_t    *new_dirlist(menuapp_t *menuapp);
static void          delete_dirlist(void *dir);    /* Ahem */
static void          next_media_clear(menuapp_t *menuapp);
static void          next_media_pop(menuapp_t *menuapp);
static void          next_media_push(menuapp_t *menuapp, char *url);
static void          channel_list_clear(channel_t **list, int *num);
static void          channel_list_add(channel_t **list, int *num, char *name, char *url);
static int           read_channel_file(char *filename, int *num, channel_t **list);
static void          playlist_parse(menuapp_t *menuapp, char *path, char *name);


menuapp_t *new_menuapp(dongle_t *dongle, render_t *render, app_t *app)
{
    menuapp_t      *menuapp = (menuapp_t *)malloc(sizeof(*menuapp));

    menuapp->render = render;
    menuapp->app    = app;
    menuapp->dongle = dongle;
    menuapp->current_media = NULL;
    menuapp->num_next_media = 0;
    menuapp->next_media = NULL;
    menuapp->paused = FALSE;
    menuapp->channels_num = 0;
    menuapp->channels = NULL;
    menuapp->Now = TRUE;

    menuapp_main(NULL,menuapp,0);    

    return menuapp;
}


void delete_menuapp(menuapp_t *menuapp)
{

    delete_menu_tree(menuapp->menu);

    if ( menuapp->current_media ) {
        free(menuapp->current_media);
        menuapp->current_media = NULL;
    }

    if ( menuapp->next_media ) {
        next_media_clear(menuapp);
    }

    channel_list_clear(&menuapp->channels,&menuapp->channels_num);

    free(menuapp);
}




static int menuapp_main(menu_t *menu,void *imenuapp, int sel)
{   
    menuapp_t   *menuapp = (menuapp_t *)imenuapp;

    menuapp->menu = new_menu(menuapp->dongle,menuapp->render,menuapp->app,NULL,menuapp,MENU_NUMBERS);
    menu_set_title(menuapp->menu,tr("MediaMVP"));
    menu_add(menuapp->menu,tr("Channels"),menuapp_channels);
    menu_add(menuapp->menu,tr("Schedules"),schedule_menuapp);
    menu_add(menuapp->menu,tr("Internet Radio"),menuapp_httpradio);
    menu_add(menuapp->menu,tr("VDR Recordings"),menuapp_vdrrecs);
    menu_add(menuapp->menu,tr("Other Recordings"),menuapp_mpeg);
    menu_add(menuapp->menu,tr("Music"),menuapp_music);
    menu_add(menuapp->menu,tr("Settings"),menuapp_settings);
    menu_display(menuapp->menu);
    return 1;
}


/***********************
 *
 *   Handling of the Channel Menus
 *
 ***********************/
static int menuapp_channels(menu_t *menu,void *imenuapp, int sel)
{
    char       buf[512]; 
    char       buf2[512]; 
    menuapp_t *menuapp = (menuapp_t *)imenuapp;
    menu_t    *child;
    cChannel  *chan;

    channel_list_clear(&menuapp->channels,&menuapp->channels_num);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,tr("Channels")); 

    for (chan = Channels.First(); chan != NULL; chan = Channels.Next(chan) ) {
        if ( chan->GroupSep() && !*chan->Name())
            continue;

        if ( chan->GroupSep() ) {

        } else {
            snprintf(buf,sizeof(buf),"% 3d %s",chan->Number(),chan->Name());
            snprintf(buf2,sizeof(buf2),"%s",chan->GetChannelID().ToString());
            menu_add(child,buf,channels_process);
            channel_list_add(&menuapp->channels,&menuapp->channels_num,buf,buf2);            
        }
    }
    menu_set_colour_actions(child,"Back",NULL,NULL,tr("Switch"),channels_colour_keys);

    menu_display(child);
    return 1;
}

static int channels_process(menu_t *menu,void *imenuapp, int sel)
{
    char           buf[1024];
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    channel_t *chan = &menuapp->channels[sel];

    snprintf(buf,sizeof(buf),"%s%s",VDRURL,chan->url);
    if ( menuapp->current_media == NULL ) {        
        menuapp->current_media = strdup(buf);
        menuapp->paused = FALSE;
        dongle_send_play(menuapp->dongle,buf);
    } else {
        next_media_clear(menuapp);
        next_media_push(menuapp,buf);       
        dongle_send_message(menuapp->dongle,RDC_STOP);
    }
  
    app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
    app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
    return 1;
}

static int channels_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    switch ( key ) {
    case keyRed:
	return 0 ;
    case keyBlue:
        dongle_send_play(menuapp->dongle,menuapp->channels[sel].url);
        app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
        app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
        return 1;   
    default:
        break;
    }

    return 1;     /* Carry on displaying this menu */
}

/***********************
 *
 *   Handling of the Schedule Menus
 *
 ***********************/
static int CompareEventChannel(const void *p1, const void *p2)
{
#if VDRVERSNUM < 10300
  return (int)( (*(const cEventInfo **)p1)->GetChannelNumber() - (*(const cEventInfo **)p2)->GetChannelNumber());
#else
  return (int)( (*(const cEvent **)p1)->ChannelNumber() - (*(const cEvent **)p2)->ChannelNumber());
#endif
}

static int schedule_menuapp(menu_t *menu, void *imenuapp, int sel)
{
#if VDRVERSNUM < 10300
	cMutexLock MutexLock;
    const cSchedules *Schedules = cSIProcessor::Schedules(MutexLock);
    const cEventInfo **pArray = NULL;
#else
    cSchedulesLock MutexLock;
    const cSchedules *Schedules = cSchedules::Schedules(MutexLock);
    const cEvent **pArray = NULL;
#endif
    menuapp_t *menuapp = (menuapp_t *)imenuapp;
    menu_t    *child;
    const cSchedule *Schedule;
    int num = 0;

    if ( Schedules != NULL ) {
        Schedule = Schedules->First();

        /* If it ain't broke, why fix it? - Stolen from vdr */
        while ( Schedule ) {
#if VDRVERSNUM < 10300
            pArray = (const cEventInfo **)realloc(pArray, (num + 1) * sizeof(cEventInfo *));
#else
            pArray = (const cEvent **)realloc(pArray, (num + 1) * sizeof(cEvent *));
#endif
            
            pArray[num] = menuapp->Now ? Schedule->GetPresentEvent() : Schedule->GetFollowingEvent();
            
            if (pArray[num]) {
#if VDRVERSNUM < 10300
                cChannel *channel = Channels.GetByChannelID(pArray[num]->GetChannelID(), true);
#else
                cChannel *channel = Channels.GetByChannelID(pArray[num]->ChannelID(), true);
#endif
                if (channel) {
                    pArray[num]->SetChannelNumber(channel->Number());
                    num++;
                }
            }
            Schedule = (const cSchedule *)Schedules->Next(Schedule);
        }
#if VDRVERSNUM < 10300
        qsort(pArray, num, sizeof(cEventInfo *), CompareEventChannel);
#else
        qsort(pArray, num, sizeof(cEvent *), CompareEventChannel);
#endif
    }
        
    /* Now, create a new menu for us */
    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,menuapp->Now ? tr("What's on now?") : tr("What's on next?"));

    /* We hijack the channel list since both can't be displayed at the same time */
    channel_list_clear(&menuapp->channels,&menuapp->channels_num);

    for ( int i = 0; i < num; i++ ) {
        char       buf[1024];
        char       buf2[1024];
#if VDRVERSNUM < 10300
        const cEventInfo *eventInfo = pArray[i];
        cChannel *channel = Channels.GetByNumber(eventInfo->GetChannelNumber());
#else
        const cEvent *eventInfo = pArray[i];
        cChannel *channel = Channels.GetByNumber(eventInfo->ChannelNumber());
#endif


#if VDRVERSNUM < 10300
        snprintf(buf,sizeof(buf),"%d\t%.*s\t%.*s\t%s", 
                 pArray[i]->GetChannelNumber(), 6,  channel ? channel->Name() : "???",
                 5, eventInfo->GetTimeString(), eventInfo->GetTitle());
#else
        snprintf(buf,sizeof(buf),"%d\t%.*s\t%.*s\t%s", 
                 pArray[i]->ChannelNumber(), 6,  channel ? channel->Name() : "???",
                 5, eventInfo->GetTimeString(), eventInfo->Title());
#endif

        menu_add(child,buf,schedule_process);
#if VDRVERSNUM < 10300
        snprintf(buf,sizeof(buf),"%d",eventInfo->GetEventID());
#else
        snprintf(buf,sizeof(buf),"%d",eventInfo->EventID());
#endif
        snprintf(buf2,sizeof(buf2),"%s",channel ? channel->GetChannelID().ToString() : "???");
        /* We place the event id in the name and the name in the url */
        channel_list_add(&menuapp->channels,&menuapp->channels_num,buf,buf2);
    }
    if ( pArray != NULL ) {
        free(pArray);
    }

    menu_set_colour_actions(child,tr("Back") /* tr("Record") */, menuapp->Now ? tr("Next") : tr("Now"), NULL /* tr("Schedule") */, tr("Switch"),schedule_colour_keys);

    menu_display(child);

    return 1;
}

static int schedule_process(menu_t *menu,void *imenuapp, int sel)
{
    char          buf[1024];
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    channel_t *chan = &menuapp->channels[sel];

    snprintf(buf,sizeof(buf),"%s%s",VDRURL,chan->url);

    if ( menuapp->current_media == NULL ) {        
        menuapp->current_media = strdup(buf);
        menuapp->paused = FALSE;
        dongle_send_play(menuapp->dongle,buf);
    } else {
        next_media_clear(menuapp);
        next_media_push(menuapp,buf);       
        dongle_send_message(menuapp->dongle,RDC_STOP);
    }
    app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
    app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
    return 1;
}

static int schedule_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    char         buf[1024];
    menuapp_t   *menuapp = (menuapp_t *)imenuapp;
    channel_t   *chan = &menuapp->channels[sel];
    menu_t      *parent;

    switch ( key ) {
    case keyRed:
        return 0 ;
    case keyGreen:
        menuapp->Now ^= 1;
        parent = menu_parent(menu);
        schedule_menuapp(parent,menuapp,0);
        return 0;
    case keyYellow:
        /* Display the schedule for a channel */   
        break;   
    case keyBlue:     /* Switch to channel */
        if ( chan == NULL ) {   /* Nothing there.. */
            return 1;
        }
        snprintf(buf,sizeof(buf),"%s%s",VDRURL,chan->url);
        if ( menuapp->current_media == NULL ) {        
            menuapp->current_media = strdup(buf);
            menuapp->paused = FALSE;
            dongle_send_play(menuapp->dongle,buf);
        } else {
            next_media_clear(menuapp);
            next_media_push(menuapp,buf);       
            dongle_send_message(menuapp->dongle,RDC_STOP);
        }
        app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
        app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
        break;
    }   
    return 1;
}


/***********************
 *
 *   Handling of the HTTP Radio Menus
 *
 ***********************/
static int menuapp_httpradio(menu_t *menu,void *imenuapp, int sel)
{
 
    char       buf[FILENAME_MAX];
    menuapp_t *menuapp = (menuapp_t *)imenuapp;
    channel_t *chan;
    menu_t    *child;
    int        i;
 
    snprintf(buf,sizeof(buf),"%s/httpradio.conf",cPluginMediamvp::ConfigDirectory());

    /* Force a re-read of the channel file */
    read_channel_file(buf,&menuapp->channels_num,&menuapp->channels);
    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,tr("Live Radio Stations"));  
    for ( i = 0; i < menuapp->channels_num; i++ ) {
        chan = &menuapp->channels[i];
        menu_add(child,chan->name,httpradio_process);
    } 
    menu_display(child);
    return 1;
}

static int httpradio_process(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    channel_t     *chan = &menuapp->channels[sel];

    if ( menuapp->current_media == NULL ) {        
        menuapp->current_media = strdup(chan->url);
        menuapp->paused = FALSE;
        dongle_send_play(menuapp->dongle,chan->url);
    } else {
        next_media_clear(menuapp);
        next_media_push(menuapp,chan->url);       
        dongle_send_message(menuapp->dongle,RDC_STOP);
    }
    app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
    app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
    return 1;
}


/***********************
 *
 *   Handling of the vdrrecs Menus
 *
 ***********************/
static int menuapp_vdrrecs(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t      *menuapp = (menuapp_t *)imenuapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[1024];
    int         i;
    char newdirname[1024] ;

    dirlist = new_dirlist(menuapp);

    dirlist->path = strdup(c_vdrrecs_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan_vdr(c_vdrrecs_dir,&dirlist->filelist_num);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,dirlist,0);
    menu_set_title(child,"Video Recordings");   
    menu_set_param_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
	    vdrnamemangle(dirlist->filelist[i].name,newdirname);
	    if (dirlist->filelist[i].vdrrecdir)
	    {
            	snprintf(buf,sizeof(buf),"<%s>",newdirname);
	    } else
	    {
            	snprintf(buf,sizeof(buf),"[%s]",newdirname);
	    }
            //snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,vdrrecs_process);
    }
    menu_set_colour_actions(child,"Back",NULL,NULL,NULL,vdrrecs_colour_keys);

    menu_display(child);
    return 1;  
}

static int vdrrecs_process(menu_t *menu,void *imenuapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = (dirlist_t *)imenuapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;
    char newdirname[1024];

    entry = &dirlist->filelist[sel];

    if (entry->vdrrecdir) {


        if ( dirlist->menuapp->current_media == NULL ) {        
//            dirlist->menuapp->current_media = strdup(buf);
            dirlist->menuapp->paused = FALSE;
//            next_media_push(dirlist->menuapp,buf);
//            dongle_send_play(dirlist->menuapp->dongle,buf);
        } else {
            next_media_clear(dirlist->menuapp);
        }

        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        vdr_push(buf,dirlist->menuapp);
        dongle_send_message(dirlist->menuapp->dongle,RDC_STOP);    

        next_media_pop(dirlist->menuapp);
        dongle_send_play(dirlist->menuapp->dongle,dirlist->menuapp->current_media);

        app_register_keys(dirlist->menuapp->app,REGISTER_MEDIA,media_keys,dirlist->menuapp);
        app_register_ack(dirlist->menuapp->app,REGISTER_MEDIA,media_ack,dirlist->menuapp);
    }
    else if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->menuapp);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan_vdr(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->menuapp->dongle,dirlist2->menuapp->render,dirlist2->menuapp->app,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Video - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_param_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
	    	vdrnamemangle(dirlist2->filelist[i].name,newdirname);
	    	if (dirlist2->filelist[i].vdrrecdir)
	  	{
            		snprintf(buf,sizeof(buf),"<%s>",newdirname);
	    	} else
	    	{
            		snprintf(buf,sizeof(buf),"[%s]",newdirname);
	    	}
                //snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,vdrrecs_process);
        }
        menu_set_colour_actions(child,"Back",NULL,NULL,NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        snprintf(buf,sizeof(buf),"file://%s/%s",dirlist->path,entry->name);
        if ( dirlist->menuapp->current_media == NULL ) {        
            dirlist->menuapp->current_media = strdup(buf);
            dirlist->menuapp->paused = FALSE;
//            next_media_push(dirlist->menuapp,buf);
            dongle_send_play(dirlist->menuapp->dongle,buf);
        } else {
            next_media_clear(dirlist->menuapp);
            next_media_push(dirlist->menuapp,buf);           
            dirlist->menuapp->paused = FALSE;
            dongle_send_message(dirlist->menuapp->dongle,RDC_STOP);    
        }
        app_register_keys(dirlist->menuapp->app,REGISTER_MEDIA,media_keys,dirlist->menuapp);
        app_register_ack(dirlist->menuapp->app,REGISTER_MEDIA,media_ack,dirlist->menuapp);
    }  

    return 1;     /* Carry on displaying this menu */
}

static int vdrrecs_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    switch ( key ) {
    case keyRed:
        return 0;   
    default:
        break;
    }

    return 1;     /* Carry on displaying this menu */
}

// mangles names of .rec directories
static int vdrnamemangle(char *oldname, char *newname)
{
  int year,mon,day,hour,min,sec,prio ;
  int i ;

  if ((i=sscanf(oldname,"%d-%d-%d.%d:%d.%d.%d.rec",
		&year,&mon,&day,&hour,&min,&sec,&prio)) == 7)
  
{
	sprintf(newname,"%2.2d.%2.2d. %2.2d:%2.2d",day,mon,hour,min);
	return 1 ;
  }
  else
  {
	strcpy(newname,oldname);
	return 0 ;
  }


}


/***********************
 *
 *   Handling of the Mpegs Menus
 *
 ***********************/
static int menuapp_mpeg(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t      *menuapp = (menuapp_t *)imenuapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[1024];
    int         i;

    dirlist = new_dirlist(menuapp);

    dirlist->path = strdup(c_mpeg_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan(c_mpeg_dir,&dirlist->filelist_num);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,dirlist,0);
    menu_set_title(child,tr("Video Recordings"));   
    menu_set_param_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
            snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,mpeg_process);
    }
    menu_set_colour_actions(child,tr("Back"),NULL,NULL,NULL,mpeg_colour_keys);

    menu_display(child);
    return 1;  
}

static int mpeg_process(menu_t *menu,void *imenuapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = (dirlist_t *)imenuapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;


    entry = &dirlist->filelist[sel];

    if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->menuapp);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->menuapp->dongle,dirlist2->menuapp->render,dirlist2->menuapp->app,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Video - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_param_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
                snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,mpeg_process);
        }
        menu_set_colour_actions(child,tr("Back"),NULL,NULL,NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        snprintf(buf,sizeof(buf),"file://%s/%s",dirlist->path,entry->name);
        if ( dirlist->menuapp->current_media == NULL ) {        
            dirlist->menuapp->current_media = strdup(buf);
            dirlist->menuapp->paused = FALSE;
//            next_media_push(dirlist->menuapp,buf);
            dongle_send_play(dirlist->menuapp->dongle,buf);
        } else {
            next_media_clear(dirlist->menuapp);
            next_media_push(dirlist->menuapp,buf);           
            dirlist->menuapp->paused = FALSE;
            dongle_send_message(dirlist->menuapp->dongle,RDC_STOP);    
        }
        app_register_keys(dirlist->menuapp->app,REGISTER_MEDIA,media_keys,dirlist->menuapp);
        app_register_ack(dirlist->menuapp->app,REGISTER_MEDIA,media_ack,dirlist->menuapp);
    }  

    return 1;     /* Carry on displaying this menu */
}

static int mpeg_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    switch ( key ) {
    case keyRed:
        return 0;   
    default:
        break;
    }

    return 1;     /* Carry on displaying this menu */
}






/***********************
 *
 *   Handling of the Music Menus
 *
 ***********************/
static int menuapp_music(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t      *menuapp = (menuapp_t *)imenuapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[1024];
    int         i;

    dirlist = new_dirlist(menuapp);

    dirlist->path = strdup(c_mp3_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan(c_mp3_dir,&dirlist->filelist_num);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,dirlist,0);
    menu_set_title(child,tr("Music"));   
    menu_set_param_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
            snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,music_process);
    }
    menu_set_colour_actions(child,tr("Back"),tr("Play dir"),tr("Play list"),NULL,music_colour_keys);

    menu_display(child);
    return 1;
}


static int music_process(menu_t *menu,void *imenuapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = (dirlist_t *)imenuapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;


    entry = &dirlist->filelist[sel];

    if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->menuapp);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->menuapp->dongle,dirlist2->menuapp->render,dirlist2->menuapp->app,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Music - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_param_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
                snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,music_process);
        }
        menu_set_colour_actions(child,tr("Back"),tr("Play dir"),tr("Play list"),NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        int filetype;
        /* It was a file, quick get the file type */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        filetype = file_get_type(buf);
        Dprintf(MOO,"Filename %s file type %d\n",buf,filetype);
        if ( filetype == MEDIA_MPEG ||
             filetype == MEDIA_MP3 ) {
            snprintf(buf,sizeof(buf),"file://%s/%s",dirlist->path,entry->name);
            if ( dirlist->menuapp->current_media == NULL ) {        
                dirlist->menuapp->current_media = strdup(buf);
                dirlist->menuapp->paused = FALSE;
                dongle_send_play(dirlist->menuapp->dongle,buf);
            } else {
                next_media_clear(dirlist->menuapp);
                next_media_push(dirlist->menuapp,buf);           
                dirlist->menuapp->paused = FALSE;
                dongle_send_message(dirlist->menuapp->dongle,RDC_STOP);    
            }
        } else if ( filetype == 0 ) {
            playlist_parse(dirlist->menuapp,dirlist->path,entry->name);
        } else if ( filetype == -1 ) {  /* Totally unknown/bad file */
            Dprintf(INFO,"Can't understand file %s\n",entry->name);
            return 1;     /* Just carry on displaying the menu */
        } 
    }  
    app_register_keys(dirlist->menuapp->app,REGISTER_MEDIA,media_keys,dirlist->menuapp);
    app_register_ack(dirlist->menuapp->app,REGISTER_MEDIA,media_ack,dirlist->menuapp);
    return 1;     /* Carry on displaying this menu */
}

static int music_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    char           buf[FILENAME_MAX+1];
    dirlist_t     *dir = (dirlist_t *)imenuapp;
    int            i;
    char           play = TRUE;

    switch ( key ) {
    case keyRed:
        return 0;
    case keyGreen:       /* Add all contents of the directory */
        if ( dir->menuapp->current_media ) {
            play = FALSE;
        }
        next_media_clear(dir->menuapp);

        for ( i = 0; i < dir->filelist_num; i++ ) {
            if ( dir->filelist[i].flags == 0 ) {
                snprintf(buf,sizeof(buf),"file://%s/%s",dir->path,dir->filelist[i].name);
                next_media_push(dir->menuapp,buf);
            }
        }
        if ( dir->menuapp->num_next_media ) {
            if ( play ) {
                next_media_pop(dir->menuapp);
                dongle_send_play(dir->menuapp->dongle,dir->menuapp->current_media);
                app_register_keys(dir->menuapp->app,REGISTER_MEDIA,media_keys,dir->menuapp);
                app_register_ack(dir->menuapp->app,REGISTER_MEDIA,media_ack,dir->menuapp);
            } else {
                dongle_send_message(dir->menuapp->dongle,RDC_STOP);
            }
        }
        break;
    case keyYellow:
        playlist_parse(dir->menuapp,dir->path,dir->filelist[sel].name);
        break;
    default:
        break;
    }

    return 1;     /* Carry on displaying this menu */
}





/***********************
 *
 *   Handling of the settings Menu
 *
 ***********************/
static int menuapp_settings(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t       *menuapp = (menuapp_t *)imenuapp;
    menu_t          *child;
    static char const    *tvmode[2];
    static char const    *flicker[4];
    static char const    *output[4];
    static char const    *aspect[]  = { "4:3", "16:9" };

    tvmode[0] = tr("NTSC");
    tvmode[1] = tr("PAL");

    flicker[0] = tr("None");
    flicker[1] = tr("Low");
    flicker[2] = tr("Medium");
    flicker[3] = tr("High");

    output[0] = tr("RGB Detected");
    output[1] = tr("SVideo");
    output[2] = tr("Composite");
    output[3] = tr("RGB/Composite");


    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,tr("Settings"));

 

    /* Add in all possible options - we'll validate things on the callback */
    menu_add_option(child,tr("TV Mode:"),settings_process,
                    MENUOPT_INT,2,tvmode,&menuapp->dongle->tvmode);

    menu_add_option(child,tr("Output:"),settings_process,
                    MENUOPT_INT,4,output,&menuapp->dongle->videooutput);

    menu_add_option(child,tr("Flicker Control:"),settings_process,
                    MENUOPT_INT,4,flicker,&menuapp->dongle->flickermode);

    menu_add_option(child,tr("Aspect Ratio:"),settings_process,
                    MENUOPT_INT,2,aspect,&menuapp->dongle->aspectratio);

    menu_add(child,tr("Cancel changes"),settings_process);
    menu_add(child,tr("Save settings (may take a while)"),settings_process);

    menu_display(child);
    return 1;
}

static int settings_process(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t   *menuapp = (menuapp_t *)imenuapp;
    /* Some rules preened from the configuration.html 

    1). For NTSC unit -- in response to GUI query, video output value 
    is always '3', so you show "Video Output" as "Composite/Svideo" no changes
    allowed !!

    2). For SCART Unit --
    a). '0' : then you show "Video Output" as "RGB Detected", No changes 
    allowed !!!
    (b). '1' : then you show "Video Output" as "SVIDEO", it can only be
    changed to RGB/COMPOSITE (value == 2)
    (c). '2': you show "Video Output" as "RGB/COMPOSITE", it can only be 
    changed to SVIDEO (value == 1)

    */

    /* If we come through with -1 then we should test the settings */
    switch ( sel ) {
    case -1:  /* If we come through with -1 then we should test the settings */
        dongle_send_settings(menuapp->dongle,RDC_SETTINGS_TEST);
        return 1;  /* Carry on displaying current menu */
    case 4:
        dongle_send_settings(menuapp->dongle,RDC_SETTINGS_CANCEL);
        return 0;
    case 5:
        dongle_send_settings(menuapp->dongle,RDC_SETTINGS_SAVE);
        return 0;
    default:
        return 0;   /* Carry on displaying */
    } 
}


/***********************
 *
 *   Generic media key and ack handling (at long last!)
 *
 ***********************/
static int media_keys(void *ptr, int code)
{
    menuapp_t     *menuapp = (menuapp_t *)ptr;
    int            streamtype = dongle_get_type(menuapp->dongle);

    switch ( code ) { 
    case keyChanUp:
    case keyChanDn:
        if (  ( (streamtype & (MEDIA_LIVE | MEDIA_MPEG) ) == (MEDIA_LIVE | MEDIA_MPEG) ) ) {
            if ( menuapp->current_media && strncmp(menuapp->current_media,VDRURL,strlen(VDRURL)) == 0  ) {
                char           buf[2048];
                const char    *newchannel = ChannelSkip(menuapp->current_media + strlen(VDRURL),
                                                        code == keyChanUp ? 1 : -1 );
                if ( newchannel ) {
                    next_media_clear(menuapp);
                    snprintf(buf,sizeof(buf),"%s%s",VDRURL,newchannel);
                    next_media_push(menuapp,buf);
                    dongle_send_message(menuapp->dongle,RDC_STOP);
                }               
            }
        } else {
            return 0;   /* Carry on with menu */
        }
        break;      
    case keyMenu:
        if ( streamtype & MEDIA_MASK == MEDIA_MPEG ) {
            next_media_clear(menuapp);
            dongle_send_message(menuapp->dongle,RDC_STOP);
            menu_display(menuapp->menu);
            break;
        }
        break;
    case keyStop:
        next_media_clear(menuapp);
        dongle_send_message(menuapp->dongle,RDC_STOP);
        menu_display(menuapp->menu);
        break;
    case keyPlay:
        /* Live media can't be paused.. */
        if  ( ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_play(menuapp->dongle,menuapp->current_media);
            menuapp->paused = FALSE;
        }
        break;
    case keyPause:
        /* Shouldn't allow pausing of live media */
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            if ( menuapp->paused ) {
                menuapp->paused = FALSE;
                dongle_send_play(menuapp->dongle,menuapp->current_media);
            } else {
                menuapp->paused = TRUE;
                dongle_send_message(menuapp->dongle,RDC_PAUSE);
            }
        }
        break;
    case keyVolUp:
        dongle_send_message(menuapp->dongle,RDC_VOLUP);
        break;
    case keyVolDn:
        dongle_send_message(menuapp->dongle,RDC_VOLDOWN);
        break;
    case keyMute:
        dongle_send_message(menuapp->dongle,RDC_MUTE);
        break;
    case keyFastFwd:
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_message(menuapp->dongle,RDC_FORWARD);
        }
        break;
    case keyFastRew:
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_message(menuapp->dongle,RDC_REWIND);
        }
        break;
    case keySkip:
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_message(menuapp->dongle,RDC_STOP);
        }
        break;
    default:
        return 0;      /* Carry on processing menu keys */
    }   
    return 1;
}

static int media_ack(int acktype, void *param, unsigned char *buf, int len)
{
    menuapp_t         *menuapp = (menuapp_t *)param;

    switch ( acktype ) {
    case RDC_PLAY:
        if ( (dongle_get_type(menuapp->dongle) & MEDIA_MASK) == MEDIA_MPEG ) {
            dongle_send_message(menuapp->dongle,RDC_DISPLAY_OFF);
        } else if ( dongle_get_type(menuapp->dongle) == 0 ) {
            if ( menuapp->num_next_media ) {
                next_media_pop(menuapp);
                menuapp->paused = FALSE;
                dongle_send_play(menuapp->dongle,menuapp->current_media);
            } else {
                /* Remove registered handlers */
                if ( menuapp->current_media ) {
                    free(menuapp->current_media);
                    menuapp->current_media = NULL;
                }
                menuapp->paused = FALSE;
                app_register_keys(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
                app_register_ack(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
            }
        }
        break;
    case RDC_STOP:     /* The client has sent a stop message */  
        if ( ( dongle_get_lasttype(menuapp->dongle) & MEDIA_MASK ) == MEDIA_MPEG ) {
            dongle_send_message(menuapp->dongle,RDC_DISPLAY_ON);
        }
        if ( menuapp->num_next_media ) {
            next_media_pop(menuapp);
            menuapp->paused = FALSE;
            dongle_send_play(menuapp->dongle,menuapp->current_media);          
        } else {
            /* Remove registered handlers */
            if ( menuapp->current_media ) {
                free(menuapp->current_media);
                menuapp->current_media = NULL;
            }
            menuapp->paused = FALSE;
            app_register_keys(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
            app_register_ack(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
        }
        break;
    case RDC_FORWARD:
    case RDC_REWIND:
        dongle_send_play(menuapp->dongle,menuapp->current_media);
        break;
    default:
        return 0;    /* Kick it up a level */
    }

    return 1;        /* Don't process anymore */
}


/* Utility functions below this point */

static dirlist_t    *new_dirlist(menuapp_t *menuapp)
{
    dirlist_t   *dir = (dirlist_t *)malloc(sizeof(*dir));

    dir->filelist_num = 0;
    dir->filelist = NULL;
    dir->menuapp = menuapp;

    return dir;
}

static void          delete_dirlist(void *idir)
{
    dirlist_t *dir = (dirlist_t *)idir;
    int        i;

    for ( i = 0; i < dir->filelist_num; i++ ) {
        free(dir->filelist[i].name);
    }
    if ( dir->filelist ) {
        free(dir->filelist);
    }

    free(dir->path);

    free(dir);
}

static filelist_t *directory_scan_vdr(char *path, int *scanned)
{
    DIR    *dir;
    struct dirent *dirent;
    struct stat sb;
    char   filename[FILENAME_MAX+1];
    char   dummy[FILENAME_MAX+1];
    filelist_t *list = NULL;
    int         num  = 0;
    int         flags;
    int 	recdir ;
    int         i;

    char *ptr;

    *scanned = 0;

    if ( ( dir = opendir(path) ) == NULL ) {
        perror("opendir");
        return NULL;
    }

    while ( (dirent = readdir(dir)) != NULL ) {
	recdir = 0 ;
        if ( dirent->d_name[0] == '.' ) {
            continue;
        }

	// files I do not want to see
	if (( !strcmp(dirent->d_name,"lost+found"))
		|| (!strcmp(dirent->d_name,"index.vdr"))
		|| (!strcmp(dirent->d_name,"resume.vdr"))
		|| (!strcmp(dirent->d_name,"summary.vdr"))
	   )
	{
		continue ;
	}

        snprintf(filename,sizeof(filename),"%s/%s",path,dirent->d_name);
        if ( stat(filename,&sb) < 0 ) {
            continue;
        }


        if ( S_ISDIR(sb.st_mode) ) {
            flags = 1;
	    if (vdrnamemangle(dirent->d_name,dummy))
	    {
		recdir = 1 ;
	    }	    
        } else {
	   //WWWWW 
           if ( (ptr = strrchr(filename,'.') ) == NULL ) {
          
            continue;
           }
 
        if ( strcasecmp(ptr,".vdr") ) {
	    continue;
            }        
            flags = 0;
	    recdir = 0 ;
        }

        i = num++;
        list = (filelist_t *)realloc(list, num * sizeof(filelist_t));

        list[i].name = strdup(dirent->d_name);
        list[i].flags = flags;
        list[i].vdrrecdir = recdir;
            
    }

    closedir(dir);

    directory_sort(num,list);

    *scanned = num;
    return list;
}

static int is_vdr_recording(const struct dirent * pde)
{
   int rnumber ;

   if (sscanf(pde->d_name,"%3d.vdr",&rnumber)==1)
	return 1 ;
   else
	return 0 ;

}

static void vdr_push(char *path,menuapp_t *app)
{
  int i,n ;
   struct dirent **list ;
   char buf[1024];

   n = scandir(path,&list, is_vdr_recording, alphasort);

   if (n == 0 ) return ;

   i = n ;
   for (i=0;i<n;i++)
   {
     snprintf(buf,sizeof(buf),"file://%s/%s",path,list[i]->d_name);
     next_media_push(app,buf);
   }

   while(n--) {
     free(list[n]);
     }
   free(list);    

}


static filelist_t *directory_scan(char *path, int *scanned)
{
    DIR    *dir;
    struct dirent *dirent;
    struct stat sb;
    char   filename[FILENAME_MAX+1];
    filelist_t *list = NULL;
    int         num  = 0;
    int         flags;
    int         i;


    *scanned = 0;

    if ( ( dir = opendir(path) ) == NULL ) {
        perror("opendir");
        return NULL;
    }

    while ( (dirent = readdir(dir)) != NULL ) {
        if ( dirent->d_name[0] == '.' ) {
            continue;
        }

        snprintf(filename,sizeof(filename),"%s/%s",path,dirent->d_name);
        if ( stat(filename,&sb) < 0 ) {
            continue;
        }

        if ( S_ISDIR(sb.st_mode) ) {
            flags = 1;
        } else {
            flags = 0;
        }

        i = num++;
        list = (filelist_t *)realloc(list, num * sizeof(filelist_t));

        list[i].name = strdup(dirent->d_name);
        list[i].flags = flags;
        list[i].vdrrecdir = 0;
            
    }

    closedir(dir);

    directory_sort(num,list);

    *scanned = num;
    return list;
}

static void directory_sort(int num,filelist_t *list)
{
    int    i,j;
    int    t_flags;
    int	   t_vdrrec ;
    char  *t_name;

    for ( i = (num-1) ; i >= 0; i-- ) {
        t_name = list[i].name;
        t_flags = list[i].flags;
	t_vdrrec = list[i].vdrrecdir ;
        j = i + 1;

        while ( j < num && strcasecmp(list[j].name,t_name) < 0 ) {
            list[j-1].name = list[j].name;
            list[j-1].flags = list[j].flags;
            list[j-1].vdrrecdir = list[j].vdrrecdir;

            j++;
        }

        list[j-1].name = t_name;
        list[j-1].flags = t_flags;
        list[j-1].vdrrecdir = t_vdrrec;

    }
}


static void channel_list_clear(channel_t **list, int *num)
{
    channel_t   *chan = *list;
    int          i;
    for ( i = 0; i < *num; i++ ) {
        free(chan[i].name);
        free(chan[i].url);
    }
    *list = NULL;
    *num = 0;
}

static void channel_list_add(channel_t **list, int *num, char *name, char *url)
{
    channel_t  *chan = *list;
    int         i;

    i = *num;
    *num = *num + 1;

    chan = (channel_t *)realloc(chan,*num * sizeof(channel_t));

    chan[i].name = strdup(name);
    chan[i].url  = strdup(url);

    *list = chan;
}



static int read_channel_file(char *filename, int *num, channel_t **list)
{
    char         buf[500];
    int          channum;
    channel_t   *chan = *list;
    char        *ptr;
    char        *endline;
    FILE        *fp;
    struct stat  sb;
    int          i;

    channel_list_clear(list,num);

    if ( stat(filename,&sb) < 0 ) {
        return 0;
    }

    
   

    if ( ( fp = fopen(filename,"r") ) == NULL ) {
        return -1;
    }

    if ( chan != NULL ) {
        free(chan);
    }
    chan = NULL;
    channum = 0;

    *list = NULL;
    *num = 0;

    while ( !feof(fp) && fgets(buf,sizeof(buf),fp) ) {
        if ( (ptr = strchr(buf,';') ) == NULL ) {
            continue;
        }
        *ptr = 0;
        ptr++;

        i = channum++;

        while (  ( endline = strrchr(ptr,'\n')  ) != NULL || ( endline = strrchr(ptr,'\r') ) != NULL ) {
            *endline = 0;
        }

        chan = (channel_t *)realloc(chan,channum * sizeof(channel_t));
        chan[i].name = strdup(buf);
        chan[i].url = strdup(ptr);            
    }
    *list = chan;
    *num = channum;

    return 0;
}

static void next_media_clear(menuapp_t *menuapp)
{
    int         i;

    if ( menuapp->num_next_media ) {
        for ( i = 0; i > menuapp->num_next_media; i++ ) {
            free(menuapp->next_media[i]);
        }
        free(menuapp->next_media);
        menuapp->next_media = NULL;
        menuapp->num_next_media = 0;
    }
}

static void next_media_pop(menuapp_t *menuapp)
{
    int       i;

    if ( menuapp->num_next_media ) {     /* Which it already is... */
        if ( menuapp->current_media ) {
            free(menuapp->current_media);
        }

        menuapp->current_media = menuapp->next_media[0];

        for ( i = 1; i < menuapp->num_next_media; i++ ) {
            menuapp->next_media[i-1] = menuapp->next_media[i];
        }
        if ( --menuapp->num_next_media == 0 ) {
            free(menuapp->next_media);
            menuapp->next_media = NULL;
        }
    } 
}

static void next_media_push(menuapp_t *menuapp, char *url)
{
    int       i = menuapp->num_next_media++;

    menuapp->next_media = (char **)realloc(menuapp->next_media,menuapp->num_next_media * sizeof(char *));
    menuapp->next_media[i] = strdup(url);
}


/* Try to parse a playlist here */
static void playlist_parse(menuapp_t *menuapp, char *path, char *name)
{
    char    buf[FILENAME_MAX+1];
    char    filename[FILENAME_MAX+1];
    char   *ptr;
    FILE   *fp;
    bool_t  firsttime = TRUE;
    bool_t  ispls = FALSE;
    bool_t  isfile = FALSE;

    snprintf(buf,sizeof(buf),"%s/%s",path,name);

    /* Try and open our "filelist" file */
    if ( ( fp = fopen(buf,"r") ) == NULL ) {
        return;
    }

    while ( fgets(filename,sizeof(filename),fp) != NULL ) {
        /* Kill any EOL characters */
        while ( ( ptr = strchr(filename,'\n') ) || (ptr = strchr(filename,'\r') ) ) {
            *ptr = 0;
        }
            
        if ( filename[0] == '[' ) {
            if ( firsttime && strncasecmp(filename+1,"playlist]",strlen("playlist]")) == 0 ) {
                ispls = TRUE;
            }
            continue;
        }
            
        if ( filename[0] == ';' ||
             filename[0] == '#' || strlen(filename) == 0 ) {
            continue;
        }
            
        /* Skip over .pls related things */
        if ( ispls ) {
            if ( strncasecmp(filename,"file",strlen("file")) ) {
                continue;
            } else {                
                int      num;
                char    *endptr;
                char    *tempptr;
                    
                num = strtol(filename+4,&endptr,10);
                    
                if ( endptr != NULL && *endptr == '=' ) {
                    endptr++; /* Now points to filename, so get it */
                    while ( isspace(*endptr) )
                        endptr++;
                    /* Shove things along - can't use strcpy because
                       the strings overlap */
                    tempptr = filename;
                    while ( (*tempptr++ = *endptr++) != 0 )
                        ;
                    /* Additional sanity check */ 
                    if ( strlen(filename) == 0 ) {
                        continue;
                    }
                } else {        /* Not a number, ignore line? */
                    continue;
                }
            }        
        }            
            
        /* Try and be almost intelligent about adding the file */
        isfile = TRUE;
        if ( filename[0] != '/' ) {
            /* Add the line in the buffer onto the playlist */
            if ( strncmp(filename,"http://",strlen("http://")) == 0 ) {
                snprintf(buf,sizeof(buf),"%s",filename);
                isfile = FALSE;
            } else {
                snprintf(buf,sizeof(buf),"file://%s/%s",path,filename);
            }
        } else {  /* We assume the complete path is there.. */
            snprintf(buf,sizeof(buf),"file://%s",filename);
        }
            
        /* Test to see if the file exists now */
        if ( isfile && file_get_type(buf + strlen("file://")) <= 0 ) {
            continue;
        }
            
        Dprintf(DEBUG,"Adding file %s to playlist\n",buf);
        if ( firsttime ) {
            next_media_clear(menuapp);
            firsttime = FALSE;
        }
        next_media_push(menuapp,buf);
    }
    /* Close playlist file */
    fclose(fp);
    /* Now start playing files */
    if ( menuapp->current_media == NULL ) {        
        next_media_pop(menuapp);
        menuapp->paused = FALSE;
        if ( menuapp->current_media != NULL ) {
            dongle_send_play(menuapp->dongle,menuapp->current_media);      
            app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
            app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
        }
    } else {
        menuapp->paused = FALSE;
        dongle_send_message(menuapp->dongle,RDC_STOP);
    }     
}

/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
