/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: menuapp.c,v 1.20 2004/02/22 19:21:41 dom Exp $
 *   $Date: 2004/02/22 19:21:41 $
 *
 *
 *   Implements the application
 */

#ifdef __cplusplus
extern "C" {
#endif


#include "app.h"
#include "menuapp.h"
#include "libini.h"

#include <ctype.h>
#include <dirent.h>
#include <sys/stat.h>



typedef struct {
    char   *name;
    int     flags;
} filelist_t;


struct _menuapp {
    dongle_t  *dongle;
    render_t  *render;
    app_t     *app;
    menu_t    *menu;
    char      *current_media;
    int        num_next_media;
    char     **next_media;
    char       paused;
};


typedef struct {
    menuapp_t      *menuapp;
    char       *path;
    /* For handling scanning of directories */
    int         filelist_num;
    filelist_t *filelist;
} dirlist_t;



typedef struct {
    char     *name;
    char     *url;
} channel_t;

#ifdef VDR_PLUGIN
static char         *c_livetv_file = "/video/livetv";
static char         *c_liveradio_file = "/video/liveradio";
static char         *c_mp3_dir     = "/home/mp3";
static char         *c_mpeg_dir    = "/video";
#else
static char         *c_livetv_file = NULL;
static char         *c_liveradio_file = NULL;
static char         *c_mp3_dir     = ".";
static char         *c_mpeg_dir    = ".";
#endif

static int           livetv_num    = 0;
static channel_t    *livetv        = NULL;
static time_t        livetv_time   = 0;

static int           liveradio_num    = 0;
static channel_t    *liveradio        = NULL;
static time_t        liveradio_time   = 0;

static int           menuapp_main(menu_t *menu,void *imenuapp, int sel);
static int           menuapp_live_tv(menu_t *menu,void *imenuapp, int sel);
static int           process_livetv(menu_t *menu,void *imenuapp, int sel);

static int           menuapp_live_radio(menu_t *menu,void *imenuapp, int sel);
static int           process_liveradio(menu_t *menu,void *imenuapp, int sel);


static int           menuapp_mpeg(menu_t *menu,void *imenuapp, int sel);
static int           process_mpeg(menu_t *menu,void *imenuapp, int sel);
static int           mpeg_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);

static int           menuapp_music(menu_t *menu,void *imenuapp, int sel);
static int           process_music(menu_t *menu,void *imenuapp, int sel);
static int           music_colour_keys(menu_t *menu, void *imenuapp, int key, int sel);


static int           menuapp_settings(menu_t *menu,void *imenuapp, int sel);
static int           process_settings(menu_t *menu,void *imenuapp, int sel);

static int           media_keys(void *ptr, int code);
static int           media_ack(int acktype, void *param, unsigned char *buf, int len);

static filelist_t   *directory_scan(char *path, int *scanned);
static void          directory_sort(int num,filelist_t *list);

static dirlist_t    *new_dirlist(menuapp_t *menuapp);
static void          delete_dirlist(void *dir);    /* Ahem */


static void          next_media_clear(menuapp_t *menuapp);
static void          next_media_pop(menuapp_t *menuapp);
static void          next_media_push(menuapp_t *menuapp, char *url);



static int           read_channel_file(char *filename, int *num, channel_t **list, time_t *update_time);
static void          playlist_parse(menuapp_t *menuapp, char *path, char *name);

#ifndef VDR_PLUGIN
void menuapp_config()
{
    iniparse_add("media:livetv",OPT_STR,&c_livetv_file);
    iniparse_add("media:liveradio",OPT_STR,&c_liveradio_file);
    iniparse_add("media:mp3",OPT_STR,&c_mp3_dir);
    iniparse_add("media:mpeg",OPT_STR,&c_mpeg_dir);
}
#endif


void menuapp_init()
{
    if ( c_livetv_file != NULL ) {
        if ( read_channel_file(c_livetv_file,&livetv_num,&livetv,&livetv_time)  < 0 ) {
            printf("Malformed file %s\n",c_livetv_file);
            exit(1);
        }
    }   
    if ( c_liveradio_file != NULL ) {
        if ( read_channel_file(c_liveradio_file,&liveradio_num,&liveradio,&liveradio_time)  < 0 ) {
            printf("Malformed file %s\n",c_liveradio_file);
            exit(1);
        }
    }   

}

menuapp_t *new_menuapp(dongle_t *dongle, render_t *render, app_t *app)
{
    menuapp_t      *menuapp = (menuapp_t *)malloc(sizeof(*menuapp));

    menuapp->render = render;
    menuapp->app    = app;
    menuapp->dongle = dongle;
    menuapp->current_media = NULL;
    menuapp->num_next_media = 0;
    menuapp->next_media = NULL;
    menuapp->paused = FALSE;


    menuapp_main(NULL,menuapp,0);    

    return menuapp;
}


void delete_menuapp(menuapp_t *menuapp)
{

    delete_menu_tree(menuapp->menu);

    if ( menuapp->current_media ) {
        free(menuapp->current_media);
        menuapp->current_media = NULL;
    }

    if ( menuapp->next_media ) {
        next_media_clear(menuapp);
    }

    free(menuapp);
}




static int menuapp_main(menu_t *menu,void *imenuapp, int sel)
{   
    menuapp_t   *menuapp = (menuapp_t *)imenuapp;

    menuapp->menu = new_menu(menuapp->dongle,menuapp->render,menuapp->app,NULL,menuapp,MENU_NUMBERS);
    menu_set_title(menuapp->menu,"MediaMVP");
    menu_add(menuapp->menu,"Live TV",menuapp_live_tv);
    menu_add(menuapp->menu,"Live Radio",menuapp_live_radio);
    menu_add(menuapp->menu,"Recordings",menuapp_mpeg);
    menu_add(menuapp->menu,"Music",menuapp_music);
    menu_add(menuapp->menu,"Settings",menuapp_settings);
    menu_display(menuapp->menu);
    return 1;
}


/***********************
 *
 *   Handling of the Live TV Menus
 *
 ***********************/

static int menuapp_live_tv(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    channel_t *chan;
    menu_t    *child;
    int        i;

    /* Force a re-read of the channel file */
    read_channel_file(c_livetv_file,&livetv_num,&livetv,&livetv_time);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,"Live TV"); 

    for ( i = 0; i < livetv_num; i++ ) {
        chan = &livetv[i];
        menu_add(child,chan->name,process_livetv);
    }
    menu_display(child);
    return 1;
}

static int process_livetv(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;

    channel_t *chan = &livetv[sel];
    dongle_send_play(menuapp->dongle,chan->url);
    app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
    app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
    return 1;
}




/***********************
 *
 *   Handling of the Live Radio Menus
 *
 ***********************/

static int menuapp_live_radio(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    channel_t *chan;
    menu_t    *child;
    int        i;

    /* Force a re-read of the channel file */
    read_channel_file(c_liveradio_file,&liveradio_num,&liveradio,&liveradio_time);
    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,"Live Radio Stations");  
    for ( i = 0; i < liveradio_num; i++ ) {
        chan = &liveradio[i];
        menu_add(child,chan->name,process_liveradio);
    } 
    menu_display(child);
    return 1;
}

static int process_liveradio(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t     *menuapp = (menuapp_t *)imenuapp;
    channel_t *chan = &liveradio[sel];

    if ( menuapp->current_media == NULL ) {        
        menuapp->current_media = strdup(chan->url);
        menuapp->paused = FALSE;
        dongle_send_play(menuapp->dongle,chan->url);
    } else {
        next_media_clear(menuapp);
        next_media_push(menuapp,chan->url);       
        dongle_send_message(menuapp->dongle,RDC_STOP);
    }
    app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
    app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
    return 1;
}



/*
 * Handle menus for mpeg files 
 */
static int menuapp_mpeg(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t      *menuapp = (menuapp_t *)imenuapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[256];
    int         i;

    dirlist = new_dirlist(menuapp);

    dirlist->path = strdup(c_mpeg_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan(c_mpeg_dir,&dirlist->filelist_num);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,dirlist,0);
    menu_set_title(child,"Video Recordings");   
    menu_set_param_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
            snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,process_mpeg);
    }
    menu_set_colour_actions(child,"Back",NULL,NULL,NULL,mpeg_colour_keys);

    menu_display(child);
    return 1;  
}

static int process_mpeg(menu_t *menu,void *imenuapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = (dirlist_t *)imenuapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;


    entry = &dirlist->filelist[sel];

    if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->menuapp);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->menuapp->dongle,dirlist2->menuapp->render,dirlist2->menuapp->app,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Video - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_param_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
                snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,process_mpeg);
        }
        menu_set_colour_actions(child,"Back",NULL,NULL,NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        snprintf(buf,sizeof(buf),"file://%s/%s",dirlist->path,entry->name);
        if ( dirlist->menuapp->current_media == NULL || dongle_get_type(dirlist->menuapp->dongle) == 0 ) {        
            if ( dirlist->menuapp->current_media )
                free(dirlist->menuapp->current_media);
            dirlist->menuapp->current_media = strdup(buf);
            dirlist->menuapp->paused = FALSE;
            dongle_send_play(dirlist->menuapp->dongle,buf);
        } else {
            next_media_clear(dirlist->menuapp);
            next_media_push(dirlist->menuapp,buf);           
            dirlist->menuapp->paused = FALSE;
            dongle_send_message(dirlist->menuapp->dongle,RDC_STOP);    
        }
        app_register_keys(dirlist->menuapp->app,REGISTER_MEDIA,media_keys,dirlist->menuapp);
        app_register_ack(dirlist->menuapp->app,REGISTER_MEDIA,media_ack,dirlist->menuapp);
    }  

    return 1;     /* Carry on displaying this menu */
}

static int mpeg_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    switch ( key ) {
    case keyRed:
        return 0;   
    default:
        break;
    }

    return 1;     /* Carry on displaying this menu */
}





/*
 *   Handle the root menu for Music menu
 */
static int menuapp_music(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t      *menuapp = (menuapp_t *)imenuapp;
    dirlist_t  *dirlist;
    menu_t     *child;
    char        buf[256];
    int         i;

    dirlist = new_dirlist(menuapp);

    dirlist->path = strdup(c_mp3_dir);

    /* Scan the dirlist */
    dirlist->filelist = directory_scan(c_mp3_dir,&dirlist->filelist_num);

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,dirlist,0);
    menu_set_title(child,"Music");   
    menu_set_param_clearup(child,delete_dirlist);

    for ( i = 0; i < dirlist->filelist_num; i++ ) {
        if ( dirlist->filelist[i].flags ) {
            snprintf(buf,sizeof(buf),"[%s]",dirlist->filelist[i].name);
        } else {
            snprintf(buf,sizeof(buf),"%s",dirlist->filelist[i].name);
        }       
        menu_add(child,buf,process_music);
    }
    menu_set_colour_actions(child,"Back","Play dir","Play list",NULL,music_colour_keys);

    menu_display(child);
    return 1;
}


static int process_music(menu_t *menu,void *imenuapp, int sel)
{
    char         buf[FILENAME_MAX+1];
    dirlist_t    *dirlist = (dirlist_t *)imenuapp;
    dirlist_t    *dirlist2;
    filelist_t   *entry;
    menu_t       *child;
    int           i;


    entry = &dirlist->filelist[sel];

    if ( entry->flags ) {
        /* The selection was a directory - we need to create a new menu - hah! */
        dirlist2 = new_dirlist(dirlist->menuapp);

        /* Scan the dirlist */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        dirlist2->path = strdup(buf);
        dirlist2->filelist = directory_scan(buf,&dirlist2->filelist_num);

        child = new_menu(dirlist2->menuapp->dongle,dirlist2->menuapp->render,dirlist2->menuapp->app,menu,dirlist2,0);
        snprintf(buf,sizeof(buf),"Music - %s",entry->name);
        menu_set_title(child,buf);   
        menu_set_param_clearup(child,delete_dirlist);
            
        for ( i = 0; i < dirlist2->filelist_num; i++ ) {
            if ( dirlist2->filelist[i].flags ) {
                snprintf(buf,sizeof(buf),"[%s]",dirlist2->filelist[i].name);
            } else {
                snprintf(buf,sizeof(buf),"%s",dirlist2->filelist[i].name);
            }       
            menu_add(child,buf,process_music);
        }
        menu_set_colour_actions(child,"Back","Play dir","Play list",NULL,music_colour_keys);

        menu_display(child);
        return 1;        
    } else {
        int filetype;
        /* It was a file, quick get the file type */
        snprintf(buf,sizeof(buf),"%s/%s",dirlist->path,entry->name);
        filetype = file_get_type(buf);
        printf("%s %d\n",buf,filetype);
        if ( filetype == MEDIA_MPEG ||
             filetype == MEDIA_MP3 ) {
            snprintf(buf,sizeof(buf),"file://%s/%s",dirlist->path,entry->name);
            if ( dirlist->menuapp->current_media == NULL || dongle_get_type(dirlist->menuapp->dongle) == 0 ) {        
                if ( dirlist->menuapp->current_media )
                    free(dirlist->menuapp->current_media);
                dirlist->menuapp->current_media = strdup(buf);
                dirlist->menuapp->paused = FALSE;
                dongle_send_play(dirlist->menuapp->dongle,buf);
            } else {
                next_media_clear(dirlist->menuapp);
                next_media_push(dirlist->menuapp,buf);           
                dirlist->menuapp->paused = FALSE;
                dongle_send_message(dirlist->menuapp->dongle,RDC_STOP);    
            }
        } else if ( filetype == 0 ) {   /* Try and grok a placelist */
            playlist_parse(dirlist->menuapp,dirlist->path,entry->name);
        } else if ( filetype == -1 ) {  /* Totally unknown/bad file */
            printf("Can't understand file %s\n",entry->name);
            return 1;     /* Just carry on displaying the menu */
        }
        app_register_keys(dirlist->menuapp->app,REGISTER_MEDIA,media_keys,dirlist->menuapp);
        app_register_ack(dirlist->menuapp->app,REGISTER_MEDIA,media_ack,dirlist->menuapp);
    }

    return 1;     /* Carry on displaying this menu */
}

static int music_colour_keys(menu_t *menu, void *imenuapp, int key, int sel)
{
    char           buf[FILENAME_MAX+1];
    dirlist_t     *dir = (dirlist_t *)imenuapp;
    int            i;
    char           play = TRUE;

    switch ( key ) {
    case keyRed:
        return 0;
    case keyGreen:       /* Add all contents of the directory */
        if ( dir->menuapp->current_media ) {
            play = FALSE;
        }
        next_media_clear(dir->menuapp);

        for ( i = 0; i < dir->filelist_num; i++ ) {
            if ( dir->filelist[i].flags == 0 ) {
                snprintf(buf,sizeof(buf),"file://%s/%s",dir->path,dir->filelist[i].name);
                next_media_push(dir->menuapp,buf);
            }
        }
        if ( dir->menuapp->num_next_media ) {
            if ( play ) {
                next_media_pop(dir->menuapp);
                dongle_send_play(dir->menuapp->dongle,dir->menuapp->current_media);
                app_register_keys(dir->menuapp->app,REGISTER_MEDIA,media_keys,dir->menuapp);
                app_register_ack(dir->menuapp->app,REGISTER_MEDIA,media_ack,dir->menuapp);
            } else {
                dongle_send_message(dir->menuapp->dongle,RDC_STOP);
            }
        }
        break;
   case keyYellow:
    {
       
        playlist_parse(dir->menuapp,dir->path,dir->filelist[sel].name);
        break;


  
        break;
    }
    default:
        break;
    }

    return 1;     /* Carry on displaying this menu */
}





/***********************
 *
 *   Handling of the settings Menu
 *
 ***********************/
static int menuapp_settings(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t       *menuapp = (menuapp_t *)imenuapp;
    menu_t          *child;
    static char     *tvmode[]  = { "NTSC", "PAL" };
    static char     *flicker[] = { "None", "Low", "Medium", "High" };
    static char     *output[]  = { "RGB Detected", "Svideo", "Composite", "RGB/Composite" };
    static char     *aspect[]  = { "4:3", "16:9" };

    child = new_menu(menuapp->dongle,menuapp->render,menuapp->app,menu,menuapp,0);
    menu_set_title(child,"Settings");

 

    /* Add in all possible options - we'll validate things on the callback */
    menu_add_option(child,"TV Mode:",process_settings,
                    MENU_INT,2,tvmode,&menuapp->dongle->tvmode);

    /* A little bit of fiddling - see rules in proces_settings */
    menu_add_option(child,"Output:",process_settings,
                    MENU_INT,4,output,&menuapp->dongle->videooutput);

    menu_add_option(child,"Flicker Control:",process_settings,
                    MENU_INT,4,flicker,&menuapp->dongle->flickermode);

    menu_add_option(child,"Aspect Ratio:",process_settings,
                    MENU_INT,2,aspect,&menuapp->dongle->aspectratio);

    menu_add(child,"Cancel changes",process_settings);
    menu_add(child,"Save settings (may take a while)",process_settings);

    menu_display(child);
    return 1;
}

static int process_settings(menu_t *menu,void *imenuapp, int sel)
{
    menuapp_t   *menuapp = (menuapp_t *)imenuapp;
    /* Some rules preened from the configuration.html:

    1). For NTSC unit -- in response to GUI query, video output value 
    is always '3', so you show "Video Output" as "Composite/Svideo" no changes
    allowed !!

    2). For SCART Unit --
    a). '0' : then you show "Video Output" as "RGB Detected", No changes 
    allowed !!!
    (b). '1' : then you show "Video Output" as "SVIDEO", it can only be
    changed to RGB/COMPOSITE (value == 2)
    (c). '2': you show "Video Output" as "RGB/COMPOSITE", it can only be 
    changed to SVIDEO (value == 1)

    */

    /* If we come through with -1 then we should test the settings */
    switch ( sel ) {
    case -1:  /* If we come through with -1 then we should test the settings */
        dongle_send_settings(menuapp->dongle,RDC_SETTINGS_TEST);
        return 1;  /* Carry on displaying current menu */
    case 4:
        dongle_send_settings(menuapp->dongle,RDC_SETTINGS_CANCEL);
        return 0;
    case 5:
        dongle_send_settings(menuapp->dongle,RDC_SETTINGS_SAVE);
        return 0;
    default:
        return 0;   /* Carry on displaying */
    } 
}



/***********************
 *
 *   Generic media key and ack handling (at long last!)
 *
 ***********************/
static int media_keys(void *ptr, int code)
{
    menuapp_t     *menuapp = (menuapp_t *)ptr;
    int            streamtype = dongle_get_type(menuapp->dongle);

    switch ( code ) {    
    case keyMenu:
        if ( (streamtype & MEDIA_MASK) == MEDIA_MPEG ) {
            next_media_clear(menuapp);
            dongle_send_message(menuapp->dongle,RDC_STOP);
            menu_display(menuapp->menu);
            break;
        }
        break;
    case keyStop:
        next_media_clear(menuapp);
        dongle_send_message(menuapp->dongle,RDC_STOP);
        menu_display(menuapp->menu);
        break;
    case keyPlay:
        /* Live media can't be paused.. */
        if  ( ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_play(menuapp->dongle,menuapp->current_media);
            menuapp->paused = FALSE;
        }
        break;
    case keyPause:
        /* Shouldn't allow pausing of live media */
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            if ( menuapp->paused ) {
                menuapp->paused = FALSE;
                dongle_send_play(menuapp->dongle,menuapp->current_media);
            } else {
                menuapp->paused = TRUE;
                dongle_send_message(menuapp->dongle,RDC_PAUSE);
            }
        }
        break;
    case keyVolUp:
        dongle_send_message(menuapp->dongle,RDC_VOLUP);
        break;
    case keyVolDn:
        dongle_send_message(menuapp->dongle,RDC_VOLDOWN);
        break;
    case keyMute:
        dongle_send_message(menuapp->dongle,RDC_MUTE);
        break;
    case keyFastFwd:
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_message(menuapp->dongle,RDC_FORWARD);
        }
        break;
    case keyFastRew:
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_message(menuapp->dongle,RDC_REWIND);
        }
        break;
    case keySkip:
        if (  ( streamtype & MEDIA_LIVE ) == 0 ) {
            dongle_send_message(menuapp->dongle,RDC_STOP);
        }
        break;
    default:
        return 0;      /* Carry on processing menu keys */
    }   
    return 1;
}

static int media_ack(int acktype, void *param, unsigned char *buf, int len)
{
    menuapp_t         *menuapp = (menuapp_t *)param;

    switch ( acktype ) {
    case RDC_PLAY:
        if ( (dongle_get_type(menuapp->dongle) & MEDIA_MASK) == MEDIA_MPEG ) {
            dongle_send_message(menuapp->dongle,RDC_DISPLAY_OFF);
        } else if ( dongle_get_type(menuapp->dongle) == 0 ) {
            if ( menuapp->num_next_media ) {
                next_media_pop(menuapp);
                menuapp->paused = FALSE;
                dongle_send_play(menuapp->dongle,menuapp->current_media);
            } else {
                /* Remove registered handlers */
                if ( menuapp->current_media ) {
                    free(menuapp->current_media);
                    menuapp->current_media = NULL;
                }
                menuapp->paused = FALSE;
                app_register_keys(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
                app_register_ack(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
            }
        }
        break;
    case RDC_STOP:     /* The client has sent a stop message */  
        if ( ( dongle_get_lasttype(menuapp->dongle) & MEDIA_MASK ) == MEDIA_MPEG ) {
            dongle_send_message(menuapp->dongle,RDC_DISPLAY_ON);
        }
        if ( menuapp->num_next_media ) {
            next_media_pop(menuapp);
            menuapp->paused = FALSE;
            dongle_send_play(menuapp->dongle,menuapp->current_media);          
        } else {
            /* Remove registered handlers */
            if ( menuapp->current_media ) {
                free(menuapp->current_media);
                menuapp->current_media = NULL;
            }
            menuapp->paused = FALSE;
            app_register_keys(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
            app_register_ack(menuapp->app,REGISTER_MEDIA,NULL,menuapp);
        }
        break;
    case RDC_FORWARD:
    case RDC_REWIND:
        dongle_send_play(menuapp->dongle,menuapp->current_media);
        break;
    default:
        return 0;    /* Kick it up a level */
    }

    return 1;        /* Don't process anymore */
}

/* Utility functions below here */


static dirlist_t    *new_dirlist(menuapp_t *menuapp)
{
    dirlist_t   *dir = (dirlist_t *)malloc(sizeof(*dir));

    dir->filelist_num = 0;
    dir->filelist = NULL;
    dir->menuapp = menuapp;

    return dir;
}

static void          delete_dirlist(void *idir)
{
    dirlist_t *dir = (dirlist_t *)idir;
    int        i;

    for ( i = 0; i < dir->filelist_num; i++ ) {
        free(dir->filelist[i].name);
    }
    if ( dir->filelist ) {
        free(dir->filelist);
    }

    free(dir->path);

    free(dir);
}

static filelist_t *directory_scan(char *path, int *scanned)
{
    DIR    *dir;
    struct dirent *dirent;
    struct stat sb;
    char   filename[FILENAME_MAX+1];
    filelist_t *list = NULL;
    int         num  = 0;
    int         flags;
    int         i;


    *scanned = 0;

    if ( ( dir = opendir(path) ) == NULL ) {
        perror("opendir");
        return NULL;
    }

    while ( (dirent = readdir(dir)) != NULL ) {
        if ( dirent->d_name[0] == '.' ) {
            continue;
        }

        snprintf(filename,sizeof(filename),"%s/%s",path,dirent->d_name);
        if ( stat(filename,&sb) < 0 ) {
            continue;
        }

        if ( S_ISDIR(sb.st_mode) ) {
            flags = 1;
        } else {
            flags = 0;
        }

        i = num++;
        list = (filelist_t *)realloc(list, num * sizeof(filelist_t));

        list[i].name = strdup(dirent->d_name);
        list[i].flags = flags;
            
    }

    closedir(dir);

    directory_sort(num,list);

    *scanned = num;
    return list;
}

static void directory_sort(int num,filelist_t *list)
{
    int    i,j;
    int    t_flags;
    char  *t_name;

    for ( i = (num-1) ; i >= 0; i-- ) {
        t_name = list[i].name;
        t_flags = list[i].flags;
        j = i + 1;

        while ( j < num && strcasecmp(list[j].name,t_name) < 0 ) {
            list[j-1].name = list[j].name;
            list[j-1].flags = list[j].flags;

            j++;
        }

        list[j-1].name = t_name;
        list[j-1].flags = t_flags;

    }
}



static int read_channel_file(char *filename, int *num, channel_t **list, 
                             time_t *update_time)
{
    char         buf[500];
    int          channum;
    channel_t   *chan = *list;
    char        *ptr;
    char	*endline;
    FILE        *fp;
    struct stat  sb;
    int          i;

    if ( stat(filename,&sb) < 0 ) {
        return 0;
    }

    if ( *update_time < sb.st_mtime ) {
        *update_time = sb.st_mtime;
        for ( i = 0; i < *num; i++ ) {
            free(chan[i].name);
            free(chan[i].url);
        }


        if ( ( fp = fopen(filename,"r") ) == NULL ) {
            return -1;
        }

        if ( chan != NULL ) {
            free(chan);
        }
        chan = NULL;
        channum = 0;

        *list = NULL;
        *num = 0;

        while ( !feof(fp) && fgets(buf,sizeof(buf),fp) ) {
            if ( (ptr = strchr(buf,';') ) == NULL ) {
                continue;
            }
            *ptr = 0;
            ptr++;

            i = channum++;

	    while (  ( endline = strrchr(ptr,'\n')  ) != NULL || ( endline = strrchr(ptr,'\r') ) != NULL ) {
		    *endline = 0;
	    }

            chan = (channel_t *)realloc(chan,channum * sizeof(channel_t));
            chan[i].name = strdup(buf);
            chan[i].url = strdup(ptr);            
        }
        *list = chan;
        *num = channum;
    }

    return 0;
}

static void next_media_clear(menuapp_t *menuapp)
{
    int         i;

    if ( menuapp->num_next_media ) {
        for ( i = 0; i > menuapp->num_next_media; i++ ) {
            free(menuapp->next_media[i]);
        }
        free(menuapp->next_media);
        menuapp->next_media = NULL;
        menuapp->num_next_media = 0;
    }
}

static void next_media_pop(menuapp_t *menuapp)
{
    int       i;

    if ( menuapp->num_next_media ) {     /* Which it already is... */
        if ( menuapp->current_media ) {
            free(menuapp->current_media);
        }

        menuapp->current_media = menuapp->next_media[0];

        for ( i = 1; i < menuapp->num_next_media; i++ ) {
            menuapp->next_media[i-1] = menuapp->next_media[i];
        }
        if ( --menuapp->num_next_media == 0 ) {
            free(menuapp->next_media);
            menuapp->next_media = NULL;
        }
    } else {
        menuapp->current_media = NULL;
    }
}

static void next_media_push(menuapp_t *menuapp, char *url)
{
    int       i = menuapp->num_next_media++;

    menuapp->next_media = (char **)realloc(menuapp->next_media,menuapp->num_next_media * sizeof(char *));
    menuapp->next_media[i] = strdup(url);
}


/* Try to parse a playlist here */
static void playlist_parse(menuapp_t *menuapp, char *path, char *name)
{
    char    buf[FILENAME_MAX+1];
    char    filename[FILENAME_MAX+1];
    char   *ptr;
    FILE   *fp;
    bool_t  firsttime = TRUE;
    bool_t  ispls = FALSE;
    bool_t  isfile = FALSE;

    snprintf(buf,sizeof(buf),"%s/%s",path,name);

    /* Try and open our "filelist" file */
    if ( ( fp = fopen(buf,"r") ) == NULL ) {
        return;
    }

    while ( fgets(filename,sizeof(filename),fp) != NULL ) {
        /* Kill any EOL characters */
        while ( ( ptr = strchr(filename,'\n') ) || (ptr = strchr(filename,'\r') ) ) {
            *ptr = 0;
        }

            
        if ( filename[0] == '[' ) {
            if ( firsttime && strncasecmp(filename+1,"playlist]",strlen("playlist]")) == 0 ) {
                ispls = TRUE;
            }
            continue;
        }
            
        if ( filename[0] == ';' ||
             filename[0] == '#' || strlen(filename) == 0 ) {
            continue;
        }
            
        /* Skip over .pls related things */
        if ( ispls ) {
            if ( strncasecmp(filename,"file",strlen("file")) ) {
                continue;
            } else {                
                int      num;
                char    *endptr;
                char    *tempptr;
                    
                num = strtol(filename+4,&endptr,10);
                    
                if ( endptr != NULL && *endptr == '=' ) {
                    endptr++; /* Now points to filename, so get it */
                    while ( isspace(*endptr) )
                        endptr++;
                    /* Shove things along - can't use strcpy because
                       the strings overlap */
                    tempptr = filename;
                    while ( (*tempptr++ = *endptr++) != 0 )
                        ;
                    /* Additional sanity check */ 
                    if ( strlen(filename) == 0 ) {
                        continue;
                    }
                } else {        /* Not a number, ignore line? */
                    continue;
                }
            }        
        }            
            
        /* Try and be almost intelligent about adding the file */
        isfile = TRUE;
        if ( filename[0] != '/' ) {
            /* Add the line in the buffer onto the playlist */
            if ( strncmp(filename,"http://",strlen("http://")) == 0 ) {
                snprintf(buf,sizeof(buf),"%s",filename);
                isfile = FALSE;
            } else {
                snprintf(buf,sizeof(buf),"file://%s/%s",path,filename);
            }
        } else {  /* We assume the complete path is there.. */
            snprintf(buf,sizeof(buf),"file://%s",filename);
        }
            
        /* Test to see if the file exists now */
        if ( isfile && file_get_type(buf + strlen("file://")) <= 0 ) {
            continue;
        }
            
        Dprintf(DEBUG,"Adding file %s to playlist\n",buf);
        if ( firsttime ) {
            next_media_clear(menuapp);
            firsttime = FALSE;
        }
        next_media_push(menuapp,buf);
    }
    /* Close playlist file */
    fclose(fp);
    /* Now start playing files */
    if ( menuapp->current_media == NULL ) {        
        next_media_pop(menuapp);
        menuapp->paused = FALSE;
        if ( menuapp->current_media != NULL ) {
            dongle_send_play(menuapp->dongle,menuapp->current_media);      
            app_register_keys(menuapp->app,REGISTER_MEDIA,media_keys,menuapp);
            app_register_ack(menuapp->app,REGISTER_MEDIA,media_ack,menuapp);
        }
    } else {
        menuapp->paused = FALSE;
        dongle_send_message(menuapp->dongle,RDC_STOP);
    }     
}

#ifdef __cplusplus
}
#endif

/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
