/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: app.c,v 1.5 2003/12/04 19:43:33 dom Exp $
 *   $Date: 2003/12/04 19:43:33 $
 *
 *
 *   Implements the skeleton application
 * 
 *   This abstracts part of the old interface from gui.c into a separate file for libraryification
 */

#ifdef __cplusplus
extern "C" {
#endif

#include "app.h"

static void      *new_app(dongle_t *dongle, render_t *render);
static void       delete_app(void *iapp);
static void       app_keypress(void *iapp, int code);
static void       app_ack(void *iapp, int ackcode, uint8_t *buf, int buflen);


typedef struct {
    void    *param;
    int    (*ack_func)(int acktype, void *param, uint8_t *buf, int len);
} ackhandle_t;

typedef struct {
    void    *param;
    int    (*key_func)(void *menu,int keycode); /* Key press callback */
} keyhandle_t;


struct _app {
    dongle_t     *dongle;
    render_t     *render;
    keyhandle_t   menus;
    keyhandle_t   media;
    ackhandle_t   menus_ack;
    ackhandle_t   media_ack;
    menuapp_t    *menuapp;
};

#ifndef VDR_PLUGIN
void app_config()
{
    menuapp_config();
}
#endif


void app_init(mvpinit_t *init)
{
    menuapp_init();

    /* Register our handlers */
    init->new_app = new_app;
    init->delete_app = delete_app;
    init->app_keypress = app_keypress;
    init->app_handle_ack = app_ack;

}



static void *new_app(dongle_t *dongle, render_t *render)
{
    app_t  *app = (app_t *) malloc(sizeof(*app));

    app->dongle = dongle;
    app->render = render;
    app->media.key_func = NULL;
    app->menus.key_func = NULL;
    app->media_ack.ack_func = NULL;
    app->menus_ack.ack_func = NULL;

    app->menuapp = new_menuapp(dongle,render,app);

    return app;
}


static void delete_app(void *iapp)
{
    app_t      *app = (app_t *) iapp;

    delete_menuapp(app->menuapp);

    free(app);
}







static void app_keypress(void *iapp, int code)
{
    app_t   *app = (app_t *) iapp;

    if ( app->media.key_func ) {
        /* Now pass the key onto the menu stuff if necessary */
        if ( (app->media.key_func)(app->media.param,code) == 0 ) {
            if ( app->menus.key_func ) {
                (app->menus.key_func)(app->menus.param,code);
            }
        }
    } else if ( app->menus.key_func ) {
        (app->menus.key_func)(app->menus.param,code);
    }
}

static void app_ack(void *iapp, int ackcode, uint8_t *buf, int buflen)
{
    app_t   *app = (app_t *) iapp;

    if ( app->media_ack.ack_func ) {
        /* Now pass the key onto the menu stuff if necessary */
        if ( (app->media_ack.ack_func)(buf[1],app->media_ack.param,buf,buflen) == 0 ) {
            if ( app->menus_ack.ack_func ) {
                (app->menus_ack.ack_func)(buf[1],app->menus_ack.param,buf,buflen);
            }
        }
    } else if ( app->menus_ack.ack_func ) {
        (app->menus_ack.ack_func)(buf[1],app->menus_ack.param,buf,buflen);
    } else {
        printf("Can't deal with MEDIA_ACK type %d\n",buf[1]);
    }
}

/** \brief Register key handling functions
 *
 *  \param app Handle the the app channel
 *  \param types REGISTER_MENU or REGISTER_MEDIA
 *  \param key_func Function to call back with key code
 *  \param param Generic pointer to pass back to the callback function
 *
 *  We have two sets of functions for handling keys - one whilst in the
 *  menu, and a second for playing media
 */
void app_register_keys(app_t *app,int type, 
                       int (*key_func)(void *param,int keycode), void *param)
{
    if ( type == REGISTER_MENU ) {
        app->menus.key_func = key_func;
        app->menus.param = param;
    } else {
        app->media.key_func = key_func;
        app->media.param = param;
    }
}


/** \brief Register ack handling functions
 *
 *  \param app Handle the the app channel
 *  \param types REGISTER_MENU or REGISTER_MEDIA
 *  \param key_func Function to call back with the ack
 *  \param param Generic pointer to pass back to the callback function
 *
 *  We have two sets of functions for handling acks - one whilst in the
 *  menu, and a second for playing media
 */
void app_register_ack(app_t *app, int type,
                      int    (*ack_func)(int acktype, void *param, unsigned char
 *buf, int len),
                      void  *param)
{
    if ( type == REGISTER_MENU ) {
        app->menus_ack.ack_func = ack_func;
        app->menus_ack.param = param;
    } else {
        app->media_ack.ack_func = ack_func;
        app->media_ack.param = param;
    }
}

#ifdef __cplusplus
}
#endif





