/*
 *   MediaMVP Server Library
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: url.c,v 1.1.1.1 2003/12/04 00:10:13 dom Exp $
 *   $Date: 2003/12/04 00:10:13 $
 *
 *
 *   Deals with handling URLs and passing the buck around the place, not
 *   entirely certain where stream detection should be, but place it here
 *   for the moment...
 */

#include "mvp_internal.h"

typedef struct {
    char    *prefix;
    int      (*open)(char *name, int *type);
    int      (*close)(char *name, int fd);
    off_t    (*seek)(char *name, int fd, off_t offset, int whence);
} url_t;


static url_t          *url_list = NULL;
static int             url_num  = 0;

static int             http_open(char *name, int *type);
static int             http_close(char *name, int fd);

static int             file_open(char *name, int *type);
static off_t           file_seek(char *name, int fd, off_t offset, int whence);
static int             file_close(char *name, int fd);


int urlhandler_init()
{
    /* Add some inbuilt directors */
    urlhandler_add("mpeg;http://",http_open,http_close,NULL);
    urlhandler_add("mp3;http://",http_open,http_close,NULL);
    urlhandler_add("file://",file_open,file_close,file_seek);

    return 0;
}

static url_t *urlhandler_find(char *name)
{
    url_t       *url;
    int          i;

    for ( i = 0; i < url_num; i++ ) {
        url = &url_list[i];
        if ( strncmp(url->prefix,name,strlen(url->prefix)) == 0 ) {
            return url;
        }
    }
    return NULL;
}


void urlhandler_add(char *prefix,
                    int      (*open)(char *name, int *type),
                    int      (*close)(char *name, int fd),
                    off_t    (*seek)(char *name, int fd, off_t offset, int whence))
{
    url_t   *url = urlhandler_find(prefix);
    int      i;



    if ( url == NULL ) {
        i = url_num++;
        url_list = realloc(url_list,url_num * sizeof(url_t));
        url = &url_list[i];
        url->prefix = strdup(prefix);
    }

    url->open  = open;
    url->close = close;
    url->seek  = seek;
}

int urlhandler_open(char *name, int *type)
{
    url_t  *url = urlhandler_find(name);

    if ( url && url->open ) {
        return url->open(name,type);       
    }
    return -1;
}

off_t urlhandler_seek(char *name, int fd, off_t offset, int whence)
{
    url_t   *url = urlhandler_find(name);

    if ( url && url->seek ) {
        return url->seek(name,fd,offset,whence);
    }

    return (off_t) -1;
}


int urlhandler_close(char *name, int fd)
{
    url_t   *url = urlhandler_find(name);

    if ( url && url->close ) {
        return url->close(name,fd);
    }

    return -1;
}



/* HTTP stuff is stolen from mpegtools/ctools.c */
static void write_all (int fd, uint8_t *data, int length)
{
        int r;

        while (length) {
                if ((r = write(fd, data, length)) > 0) {
                        data += r;
                        length -= r;
                }
        }
}

static int read_all (int fd, uint8_t *data, int length)
{
        int c = 0;

        while(c<length) {
                if( read(fd, data+c, 1) == 1) {
                        c++;
                        if(data[c-1] == '\n') {
                                data[c] = 0;
                                break;
                        }
                }
                else {
                        return -1;
                }
        }
        return 0;
}

static char *url2host (uint8_t *url, char **name, uint32_t *ip, uint32_t *port)
{
        uint8_t *murl;
        struct hostent *hoste;
        struct in_addr haddr;
        int found_ip = 1;
        
        if (!(strncmp(url, "http://", 7)))
                url += 7;

        *name = strdup(url);
        if (!(*name)) {
                *name = NULL;
                return (NULL);
        }

        murl = url;
        while (*murl && *murl != ':' && *murl != '/') {
                if ((*murl < '0' || *murl > '9') && *murl != '.')
                        found_ip = 0;
                murl++;
        }
        (*name)[murl - url] = 0;
        if (found_ip) {
                if ((*ip = inet_addr(*name)) == INADDR_NONE)
                        return (NULL);
        } else {
                if (!(hoste = gethostbyname(*name)))
                        return (NULL);
                memcpy (&haddr, hoste->h_addr, sizeof(haddr));
                *ip = haddr.s_addr;
        }

        if (!*murl || *murl == '/') {
                *port = 80;
                return (murl);
        }
        *port = atoi(++murl);

        while (*murl && *murl != '/')
                murl++;
        return (murl);
}

#define ACCEPT "Accept: video/mpeg, video/x-mpegurl, */*\r\n"
static int http_open(char *url, int *type)
{
    char purl[1024], *host, req[1024], *sptr;
    uint32_t ip;
    uint32_t port;
    int sock;
    int reloc, relocnum = 0;
    struct sockaddr_in server;
    int mfd;

    /* This is a bit of splendiferous hack <grin> */
    if ( strncmp(url,"mpeg;",5) == 0 ) {
        *type = MEDIA_MPEG | MEDIA_SOCKET;
        url += 5;
    } else if ( strncmp(url,"mp3;",3) == 0 ) {
        *type = MEDIA_MP3 | MEDIA_SOCKET;
        url += 4;
    } else {       /* Unknown type */
        return -1;
    }

    strncpy (purl, url, 1023);
    purl[1023] = '\0';

    do {
        host = NULL;
        strcpy (req, "GET ");
        if (!(sptr = url2host(purl, &host, &ip, &port))) {
            DPRINTF("Unknown host %s\n",purl);
            return -1;
        }
        strcat (req, sptr);
        sprintf (req + strlen(req),
                 " HTTP/1.0\r\nUser-Agent: %s/%s\r\n",
                 SERVER_NAME, SERVER_VERSION);
        if (host) {
            sprintf(req + strlen(req),
                    "Host: %s:%u\r\n", host, port);
            free (host);
        }

        strcat (req, ACCEPT);
        strcat (req, "\r\n");
                
        server.sin_port = htons(port);
        server.sin_family = AF_INET;
        server.sin_addr.s_addr = ip;

        if ((sock = socket(PF_INET, SOCK_STREAM, 6)) < 0) {
            return -1;  
        }

        if (connect(sock, (struct sockaddr *)&server, 
                    sizeof(server))) {
            close(mfd);
            return -1;                
        }
                
        write_all (sock, req, strlen(req));
        if (!(mfd = fileno(fdopen(sock, "rb")))) {
            close(mfd);
            shutdown(mfd,SHUT_RDWR);
            return -1;    
        }
        reloc = 0;
        purl[0] = '\0';
        if ( read_all (mfd, req, 1023) == -1 ) {
            return -1;
        }
        if ((sptr = strchr(req, ' '))) {
            switch (sptr[1]) {
            case '2':
                break;
            case '3':
                reloc = 1;
            default:
                DPRINTF (stderr, "HTTP req failed:%s",sptr+1);
                close(mfd);
                shutdown(mfd,SHUT_RDWR);
                return -1;             
            }
        }
        do {
            if ( read_all (mfd,req, 1023) == -1 ) {
                return -1;
            }
            if (!strncmp(req, "Location:", 9))
                strncpy (purl, req+10, 1023);
        } while (req[0] != '\r' && req[0] != '\n');
    } while (reloc && purl[0] && relocnum++ < 3);
    if (reloc) {
        DPRINTF("Too many HTTP relocations.\n");
        close(mfd);
        shutdown(mfd,SHUT_RDWR);
        return -1;
    }

    return sock;
}

static int http_close(char *filename, int fd)
{
    close(fd);
    shutdown(fd,SHUT_RDWR);
    return 0;
}


                
static int file_close(char *name, int fd)
{
    close(fd);
    return 0;
}

static off_t file_seek(char *name, int fd, off_t off, int whence)
{
    return lseek(fd,off,whence);
}

static int file_open(char *name, int *type)
{
    char   *ptr;
    char   *filename;
    int     fd;

    /* This shouldn't happen... */
    if ( strncmp(name,"file://",7) ) {
        return -1;
    }

    filename = name + 7;


    if ( (ptr = strrchr(filename,'.') ) == NULL ) {
        /* No extension, we're not smart so return no such file */
        return -1;
    }


    if ( strcasecmp(ptr,".mp3") == 0 ) {
        *type = MEDIA_MP3;
    } else if ( strcasecmp(ptr,".mpeg") == 0 || strcasecmp(ptr,".vdr") == 0 ||
                strcasecmp(ptr,".mpg") == 0 ) {
        *type = MEDIA_MPEG;
    } else {
        return -1;
    }


    if ( (fd = open(filename,O_RDONLY) ) < 0 ) {
        return -1;
    }

    return fd;
}
