/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: render.c,v 1.1.1.1 2003/12/04 00:10:13 dom Exp $
 *   $Date: 2003/12/04 00:10:13 $
 *
 *
 *   Handle rendering onto the RGB buffer stuff we've got
 *
 *   And yup, some form of OO would be kinda handy here <grin>
 */

#include "mvp_internal.h"

#include "font.h"


#define OFFSET_Y          32
#define OFFSET_X          32


struct _render {
    int       width;
    int       depth;
    int       yuvsize;
    uint8_t  *layers[3];
    int       fgred;
    int       fggreen;
    int       fgblue;
    int       bgred;
    int       bggreen;
    int       bgblue;
    uint8_t   updated;
};

static void     render_print(render_t *render, int x, int y, int maxx, char *text);
static void     printchar(render_t *render,int x, int y, int *letter);

render_t *new_render(int width, int depth)
{
    render_t  *render = malloc(sizeof(*render));
    int        i;

    if ( render == NULL ) {
        return NULL;
    }

    /* Set up sizes */
    render->width = width;
    render->depth  = depth;
    render->yuvsize = width * depth * 2;

    /* Background colour is black */
    render->bgred = 0;
    render->bggreen = 0;
    render->bgblue = 0;

    /* Foreground is white */
    render->fgred = 0xFF;
    render->fggreen = 0xFF;
    render->fgblue = 0xFF;

    /* Allocate the RGB surfaces */
    for ( i = 0; i < 3; i++ ) {
        render->layers[i] = calloc(width * depth,1);
    }

    /* Set up some handy functions.. */
    return render;
}


void delete_render(render_t *render)
{
    int      i;

    if ( render == NULL ) {
        return;
    }

    for ( i = 0; i < 3; i++ ) {
        free(render->layers[i]);
    }

    free(render);
}

void render_clear(render_t *render)
{
    memset(&render->layers[0][0],render->bgred,render->width * render->depth);
    memset(&render->layers[1][0],render->bggreen,render->width * render->depth);
    memset(&render->layers[2][0],render->bgblue,render->width * render->depth);
    render->updated = TRUE;
}

void *render_rgb2yvuy(render_t *render)
{
    uint8_t    *surface;
    int         i,j;
    int         y,u,v,y2;

    /* If it's not been updated, don't bother doing it.. */
    if ( !render->updated ) {
        return NULL;
    }

    surface = malloc(render->yuvsize);

    for ( j = 0, i = 0; i < render->width * render->depth; i+=2 ) {

        y = 0.30 * (double)render->layers[0][i] +
            0.59 * (double)render->layers[1][i] +
            0.11 * (double)render->layers[2][i];

        if ( y > 255 )
            y = 255;
        else if ( y < 0 )
            y = 0;

        v = -(0.15 * render->layers[0][i]) -
            (0.29 * render->layers[1][i]) +
            (0.44 * render->layers[2][i]) +
            128.0;

       

        u = (0.62 * render->layers[0][i]) -
            (0.52 * render->layers[1][i]) +
            (0.10 * render->layers[2][i]) +
            128.0;

        if ( v > 255 )
            v = 255;
        else if ( v < 0 )
            v = 0;

        if ( u > 255 )
            u = 255;
        else if ( u < 0 )
            u = 0;

        y2 = 0.30 * (double)render->layers[0][i+1] +
            0.59 * (double)render->layers[1][i+1] +
            0.11 * (double)render->layers[2][i+1];
             
        if ( y2 > 255 )
            y2 = 255;
        else if ( y2 < 0 )
            y2 = 0;
            
        surface[j++] = y;
        surface[j++] = v;
        surface[j++] = u;
        surface[j++] = y2;
    }

    render->updated = FALSE;

    return surface;
}

void render_set_fgcol(render_t *render, int red, int green, int blue)
{
    render->fgred = red;
    render->fggreen = green;
    render->fgblue = blue;
}

void render_set_bgcol(render_t *render, int red, int green, int blue)
{
    render->bgred = red;
    render->bggreen = green;
    render->bgblue = blue;
}

void render_printrow(render_t *render, int row, char *text)
{
    int  x,y;

    x = OFFSET_X;
    y = row * FONT_HEIGHT + OFFSET_Y;    

    render_print(render,x,y,render->width - OFFSET_X,text);
}

void render_printrow_width(render_t *render, int row, int x, int width,char *text)
{
    int      y;

    y = row * FONT_HEIGHT + OFFSET_Y;   

    x += OFFSET_X;

    width += x;

    render_print(render,x,y,width,text);
}


void render_print(render_t *render, int x, int y, int maxwidth,char *text)
{
    int   nbsp[FONT_HEIGHT+2] = { 1, FONT_HEIGHT, 0 };
    int  *letter;
    int   width;

    while ( *text ) {
        switch ( *text ) {
        case '\n':
            width = 1;          
            while ( x + width < maxwidth ) {
                printchar(render,x,y,nbsp);
                x += width;
            }
            y += FONT_HEIGHT;
            if ( y >= render->depth - OFFSET_Y ) {
                return;
            }
            break;           
        default:
            if ( *text >= 32 ) {
                letter = Font[*text - 32];
                width = letter[0];

                if ( x + width < maxwidth &&
                     ( y < render->depth - OFFSET_Y) ) {
                    printchar(render,x,y,letter);
                    x += width;
                } 
            }
            break;
        }
        text++;
    }
}

static void printchar(render_t *render,int x, int y, int *letter)
{
    int offset = y * render->width + x;
    int width = letter[0];
    int depth = letter[1];
    int row;
    int i,mask;

    render->updated = TRUE;

    letter += 2;

    while ( depth-- ) {
        row = *letter;
        offset += width;

        for ( i = 0; i < width; i++ ) {
            mask = 1 << i;
            if ( row & mask ) {
                render->layers[0][offset] = render->fgred;
                render->layers[1][offset] = render->fggreen;
                render->layers[2][offset] = render->fgblue;                
            } else {
                render->layers[0][offset] = render->bgred;
                render->layers[1][offset] = render->bggreen;
                render->layers[2][offset] = render->bgblue;
            }
            offset--;
        }
        letter++;
        offset += render->width;
    }
}

int render_get_yuvsize(render_t *render)
{
    return render->yuvsize;
}

int render_get_maxrows(render_t *render)
{
    return ((render->depth - ( OFFSET_Y * 2) ) / FONT_HEIGHT);
}

int render_get_depth(render_t *render)
{
    return render->depth;
}

int render_get_width(render_t *render)
{
    return render->width;
}

int render_get_textspace(render_t *render, char *text)
{
    int  *letter;
    int   width = 0;

    while ( *text ) {
        if ( *text >= 32 ) {
            letter = Font[*text -32];
            width += letter[0];
        }
        text++;
    }

    return width;
}

int render_get_xoffset(render_t *render)
{
    return OFFSET_X;
}

int render_get_yoffset(render_t *render)
{
    return OFFSET_Y;
}



/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
