/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: menu.c,v 1.1.1.1 2003/12/04 00:10:12 dom Exp $
 *   $Date: 2003/12/04 00:10:12 $
 *
 *
 *   Constructs a menu and all such things
 *
 */

#include "menuapp.h"


/* Number of rows we can display on the menu */

#define BLACK   0x00, 0x00, 0x00
#define RED     0xFC, 0x14, 0x14
#define GREEN   0x24, 0xFC, 0x24
#define YELLOW  0xFC, 0xC0, 0x24
#define BLUE    0x00, 0x00, 0xFC
#define CYAN    0x00, 0xFC, 0xFC
#define MAGENTA 0xB0, 0x00, 0xFC
#define WHITE   0xFC, 0xFC, 0xFC


typedef struct {
    char    *name;
    int    (*callback_func)(menu_t *menu,void *app, int sel);
    int      num_options;
    char   **options;
    int      type;
    void    *value_ptr;
} option_t;


struct _menu {
    dongle_t     *dongle;
    render_t     *render;
    app_t        *app;
    void         *param;
    int           numoptions;
    option_t     *options;
    char         *title;
    int           top_offset;
    int           current_selection;
    int           maxrows;                /*!< Number of lines that can be displayed on screen */
    int           flags;
    void         (*param_clearup)(void *ptr);
    menu_t        *parent;
    menu_t        *child;
    char          *colours[4];   
    int          (*colour_func)(menu_t *menu,void *param, int key, int sel);
};

#define MENU_ROWS(menu)  ((menu)->maxrows -  3)

static int        menu_keys(void *ptr, int code);



menu_t *new_menu(dongle_t *dongle,render_t *render, app_t *app,menu_t *parent, void *param, int flags)
{
    menu_t       *menu = malloc(sizeof(*menu));

    menu->dongle = dongle;
    menu->render = render;
    menu->app    = app;
    menu->param  = param;
    menu->title  = NULL;
    menu->top_offset = 0;
    menu->numoptions = 0;
    menu->options = NULL;
    menu->current_selection = 0;
    menu->flags   = flags;
    menu->param_clearup = NULL;
    menu->maxrows = render_get_maxrows(render);
    menu->colours[0] = menu->colours[1] = menu->colours[2] = menu->colours[3] = NULL;

    if ( parent ) {
        menu->parent = parent;
        parent->child = menu;
    } else {
        menu->parent = NULL;
    }
    menu->child = NULL;

    return menu;
}

void delete_menu_tree(menu_t *menu)
{
    menu_t   *temp;

    while ( menu->child ) {
        menu = menu->child;
    }

    while ( menu->parent ) {
        temp = menu->parent;
        delete_menu(temp);
        menu = temp;
    }

}

void delete_menu(menu_t *menu)
{
    int     i;

    if ( menu->title ) {
        free(menu->title);
        menu->title = NULL;
    }

    for ( i = 0; i < menu->numoptions; i++ ) {
        free(menu->options[i].name);
    }
    if ( menu->options ) {
        free(menu->options);
    }

    if ( menu->param_clearup ) {
        (menu->param_clearup)(menu->param);
    }

    for ( i = 0; i < 3; i++ ) {
        if ( menu->colours[i] ) {
            free(menu->colours[i]);
        }
        menu->colours[i] = NULL;
    }

    free(menu);
}

void menu_set_title(menu_t *menu, char *text)
{
    if ( menu->title ) {
        free(menu->title);
    }

    menu->title = strdup(text);
}


void menu_add_option(menu_t *menu, char *name, int (callback_func)(menu_t *menu,void *param, int sel),
                     int type, int num_options, char **options, void *value_ptr)
{
    int     i = menu->numoptions++;

    menu->options = realloc(menu->options,menu->numoptions * sizeof(option_t));

    menu->options[i].name = strdup(name);
    menu->options[i].callback_func = callback_func;
    menu->options[i].type = type;
    menu->options[i].num_options = num_options;
    menu->options[i].options = options;
    menu->options[i].value_ptr = value_ptr;
}

void menu_add(menu_t *menu, char *name, int (*callback_func)(menu_t *menu,void *param, int sel))
{
    int     i = menu->numoptions++;

    menu->options = realloc(menu->options,menu->numoptions * sizeof(option_t));

    menu->options[i].name = strdup(name);
    menu->options[i].callback_func = callback_func;
    menu->options[i].type          = MENU_NONE;
    menu->options[i].num_options = 0;
    menu->options[i].value_ptr = NULL;
}




void menu_display(menu_t *menu)
{
    char          buf[256];
    char         *extra;
    render_t     *render = menu->render;
    int           row;
    int           i;
    int           last;
    int           width;
    int           colwidth;

    while ( menu->child ) {
        menu = menu->child;
    }

    render_set_bgcol(render,BLACK);
    render_clear(render);

    snprintf(buf,sizeof(buf)," %s\n",menu->title ? menu->title : "");
    render_set_bgcol(render,CYAN);
    render_set_fgcol(render,BLACK);
    render_printrow(render,0,buf);


    /* Calculate how many rows to print */
    last = (menu->top_offset + MENU_ROWS(menu) < menu->numoptions) 
        ? menu->top_offset + MENU_ROWS(menu) : menu->numoptions;
    for ( row = 2, i = menu->top_offset; i < last; i++, row++ ) {
        if ( menu->options[i].type == MENU_INT ) {
            printf("option %d val %d\n",i,*((int *)menu->options[i].value_ptr));
            extra = menu->options[i].options[*((int *)menu->options[i].value_ptr)];
        } else {
            extra = "";
        }
        if ( i < 9 && menu->flags & MENU_NUMBERS ) {
            snprintf(buf,sizeof(buf)," %d. %s %s\n",i+1,menu->options[i].name,extra);
        } else {
            snprintf(buf,sizeof(buf),"   %s %s\n",menu->options[i].name,extra);
        }      

        if ( i == menu->current_selection ) {
            render_set_bgcol(render,CYAN);
            render_set_fgcol(render,BLACK);
        } else {
            render_set_bgcol(render,BLACK);
            render_set_fgcol(render,WHITE);
        }
        render_printrow(render,row,buf);
    }

    /* FIXME: Hardcoded hack... */
    colwidth = (render_get_width(render) - render_get_xoffset(render) * 2) / 4;
    for ( i = 0; i < 4; i++ ) {
        int    offset; 

        if ( menu->colours[i] ) {
            width = render_get_textspace(render,menu->colours[i]);

            if ( width > colwidth ) {
                offset = 0;
            } else {
                offset = (colwidth - width) / 2;
            }

            offset += colwidth * i;

            render_set_fgcol(render,BLACK);
            switch ( i ) {
            case 0:
                render_set_bgcol(render,RED);
                break;
            case 1:
                render_set_bgcol(render,GREEN);
                break;
            case 2:
                render_set_bgcol(render,YELLOW);
                break;
            case 3:
                render_set_fgcol(render,WHITE);
                render_set_bgcol(render,BLUE);
                break;
            }
            if ( menu->colours[i] ) {
                render_printrow_width(render,menu->maxrows-1,colwidth * i,colwidth,"\n");
                render_printrow_width(render,menu->maxrows-1,offset,colwidth,menu->colours[i]);
            }
        }
    }
        
    app_register_keys(menu->app,REGISTER_MENU,menu_keys,menu);
}


static int menu_keys(void *ptr, int code)
{
    menu_t    *menu = ptr;
    int        pos;
    int      (*func)(menu_t *,void *param, int sel);
    int      (*cfunc)(menu_t *,void *param, int key,int sel);
    option_t  *option;
    unsigned int       *num;
    int      streamtype = 0;


    if ( menu->dongle->stream ) {
        streamtype = stream_get_type(menu->dongle->stream);
    }


    /* If we're in streaming mpeg mode, then we'd like to change channels
       without going back to the menu 
    */
    if ( streamtype == ( MEDIA_MPEG | MEDIA_SOCKET ) ) {
        switch ( code ) {
        case kChanUp:
            if ( menu->current_selection < menu->numoptions - 1 ) {
                menu->current_selection++;
                if ( menu->top_offset + MENU_ROWS(menu) <= menu->current_selection ) {
                    menu->top_offset++;
                }
            } else {
                menu->current_selection = 0;
                menu->top_offset = 0;
            }

            func = menu->options[menu->current_selection].callback_func;
            if ( func && func(menu,menu->param,menu->current_selection) == 0 ) {
                menu_t  *parent;                
                parent = menu->parent;
                delete_menu(menu);
                parent->child = NULL;        
            }  
            break;
        case kChanDn:
            if ( menu->current_selection > 0 ) {
                menu->current_selection--;
                if ( menu->top_offset > menu->current_selection ) {
                    menu->top_offset = menu->current_selection;
                }
            } else {
                menu->current_selection = menu->numoptions - 1;          
                if ( menu->numoptions > MENU_ROWS(menu) ) {
                    menu->top_offset = menu->numoptions - MENU_ROWS(menu);
                }
            }
            func = menu->options[menu->current_selection].callback_func;
            if ( func && func(menu,menu->param,menu->current_selection) == 0 ) {
                menu_t  *parent;                
                parent = menu->parent;
                delete_menu(menu);
                parent->child = NULL;        
            }  
            break;
        }
        return 0;
    }



    switch ( code ) {
    case kChanUp:
        if ( streamtype != MEDIA_MPEG ) {
            if ( menu->current_selection > 0 ) {
                menu->current_selection--;
                if ( menu->top_offset > menu->current_selection ) {
                    menu->top_offset = menu->current_selection;
                }
            } else {
                menu->current_selection = menu->numoptions - 1;          
                if ( menu->numoptions > MENU_ROWS(menu) ) {
                    menu->top_offset = menu->numoptions - MENU_ROWS(menu);
                }
            }
            
            menu_display(menu);
        }
        break;
    case kChanDn:
        if ( streamtype != MEDIA_MPEG ) {
            if ( menu->current_selection < menu->numoptions - 1 ) {
                menu->current_selection++;
                if ( menu->top_offset + MENU_ROWS(menu) <= menu->current_selection ) {
                    menu->top_offset++;
                }
            } else {
                menu->current_selection = 0;
                menu->top_offset = 0;
            }

            if ( streamtype == ( MEDIA_MPEG | MEDIA_SOCKET ) ) {
                menu_t  *parent;
                func = menu->options[menu->current_selection].callback_func;
                if ( func && func(menu,menu->param,menu->current_selection) == 0 ) {
                    parent = menu->parent;
                    delete_menu(menu);
                    parent->child = NULL;        
                    menu_display(parent);
                }  
            } else if ( (streamtype & MEDIA_MASK) != MEDIA_MPEG ) {
                menu_display(menu);
            }
        }
        break;
    case kVolUp:
        option = &menu->options[menu->current_selection];
        if ( option->type == MENU_INT ) {
            menu_t *parent = menu->parent;
            num = option->value_ptr;           
            *num = ( *num + 1 ) % option->num_options;
            func = menu->options[menu->current_selection].callback_func;
            if ( func && func(menu,menu->param,-1) == 0 ) {
                delete_menu(menu);
                parent->child = NULL;        
            }    
            menu_display(parent);
  
        }
        break;
    case kVolDn:
        option = &menu->options[menu->current_selection];
        if ( option->type == MENU_INT ) {
            menu_t *parent = menu->parent;
            num = option->value_ptr;
            if ( *num == 0 ) {
                *num = option->num_options - 1;
            } else {
                *num = ( *num - 1 ) % option->num_options;
            }
            func = menu->options[menu->current_selection].callback_func;
            if ( func && func(menu,menu->param,-1) == 0 ) {
                delete_menu(menu);
                parent->child = NULL;        
            }      
            menu_display(parent);

        }
        break;
    case kBack:
        if ( menu->parent ) {
            menu_t *parent = menu->parent;
            delete_menu(menu);
            parent->child = NULL;
            menu_display(parent);
        }
        break;
    case kSkip:        /* Go down a page */
	if ( menu->current_selection + MENU_ROWS(menu) > ( menu->numoptions - MENU_ROWS(menu) ) ) {
	    menu->current_selection = menu->top_offset = menu->numoptions - MENU_ROWS(menu);
	    if ( menu->current_selection < 0 ) {
		menu->current_selection = menu->top_offset = 0;
	    }
	} else {
	    menu->current_selection = menu->top_offset =  menu->current_selection + MENU_ROWS(menu);
	}
	menu_display(menu);
	break;
    case kReplay:       /* Go up a page */
	if ( menu->current_selection - MENU_ROWS(menu) < 0 ) {
	    menu->current_selection = menu->top_offset = 0;
	} else {
	    menu->current_selection = menu->top_offset =  menu->current_selection - MENU_ROWS(menu);
	}
	menu_display(menu);
	break;
    case kGo:
        while ( menu->parent ) {
            menu_t *parent = menu->parent;
            delete_menu(menu);
            menu = parent;
            menu->child = NULL;
        }
        menu_display(menu);
        break;
    case kOk:
        if ( menu->options != NULL ) {
            menu_t  *parent;
            func = menu->options[menu->current_selection].callback_func;
            if ( func && func(menu,menu->param,menu->current_selection) == 0 ) {
                parent = menu->parent;
                delete_menu(menu);
                parent->child = NULL;        
                menu_display(parent);
            }        
        }
        break;
    case k1 ... k9:
        if ( menu->flags & MENU_NUMBERS ) {
            pos = code - k1;
            if ( pos < menu->numoptions && menu->options ) {
                func = menu->options[pos].callback_func;
                if ( func && func(menu,menu->param,pos) == 0 ) {
                    delete_menu(menu);
                }
            }
        }
        break;
    case kRed ... kBlue:
        cfunc = menu->colour_func;        
        if ( cfunc && cfunc(menu,menu->param,code,menu->current_selection) == 0 ) {
	    menu_t *parent;
            parent = menu->parent;
            delete_menu(menu);
            parent->child = NULL;        
            menu_display(parent);
        }     
        break;
    default:
        printf("Received IR code %d - not used here\n",code);
    }
    return 0;
}


/** \brief Set the clearup pointer for the param datatype 
 * 
 *  \param menu Menu handle
 *  \param clearup Function to call to clearup
 *
 *  Used for those situations when we recurse down through directory structures
 */
void menu_set_param_clearup(menu_t *menu, void (*clearup)(void *ptr))
{
    menu->param_clearup = clearup;
}


/** \brief Set what the colour keys do 
 *
 *  \param menu    Menu handle
 *  \param red     Red button text
 *  \param green   Green button text
 *  \param yellow  Yellow button text
 *  \param blue    Blue button text
 *  \param func    Callback function - called back with 0 - 3 for the colour button that has been
 *                 pressed
 */
void menu_set_colour_actions(menu_t *menu,
                             char   *red,
                             char   *green,
                             char   *yellow,
                             char   *blue,
                             int   (*func)(menu_t *menu,void *param, int key, int sel))
{
    int     i;

    for ( i = 0; i < 4; i++ ) {
        if ( menu->colours[i] ) {
            free(menu->colours[i]);
        }
        menu->colours[i] = NULL;
    }

    if ( red ) {
        menu->colours[0] = strdup(red);
    }
    if ( green ) {
        menu->colours[1] = strdup(green);
    }
    if ( yellow ) {
        menu->colours[2] = strdup(yellow);
    }
    if ( blue ) {
        menu->colours[3] = strdup(blue);
    }

    menu->colour_func = func;
}




 
/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
