/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: stream.c,v 1.23 2003/11/30 23:06:20 dom Exp $
 *   $Date: 2003/11/30 23:06:20 $
 *
 *
 *   Handles the media stream socket..all a bit of mess at the moment..
 *
 */

#include "mediamvp.h"
#include "ringbuffy.h"

typedef struct {
    int           fd;
    int           id;
    char          filename[FILENAME_MAX+1];
    off_t         filelength;
    ringbuffy     buffy;
    struct event *buffy_event;
    uint32_t      offset;              /* Should be 64 bit - check protocol */
    uint32_t      blocklen;
} media_t;

struct _stream {
    dongle_t      *dongle; 
    int            fd;
    struct event   event;
    char           inbuf[2048];
    int            inbufstart;
    media_t        media;
    int            mediatype;
    int            last_mediatype;
    bool_t         pending_data;
};




/* Definitions for the commands the MVP gives us */
#define MEDIA_REQUEST    2
#define MEDIA_STOP       3
#define MEDIA_BLOCK      4
#define MEDIA_STEP       5
#define MEDIA_SEEK       7
#define MEDIA_8          8

static void       stream_accept(int fd, short event, void *arg);
static void       stream_read(int fd, short event, void *arg);
static int        media_request(stream_t *stream, unsigned char *buf, int len);
static int        media_stop(stream_t *stream, unsigned char *buf, int len);
static int        media_block(stream_t *stream, unsigned char *buf, int len);
static int        media_seek(stream_t *stream, unsigned char *buf, int len);
static int        media_step(stream_t *stream, unsigned char *buf, int len);
static int        media_8(stream_t *stream, unsigned char *buf, int len);

static int        filename_parse(stream_t *stream,media_t *media);
static void       buffer_read(int fd, short event, void *arg);
static void       stream_write_data(stream_t *stream, bool_t flush);


struct event      listen_event;

static int        c_mpeg_bufsize = 200000;
static int        c_mp3_bufsize  = 4000;

int               c_stream_port = 6337;
static char      *c_stream_host = NULL;
uint32_t          c_stream_hostip;

void stream_config()
{
    iniparse_add("stream:port",OPT_INT,&c_stream_port);
    iniparse_add("stream:host",OPT_STR,&c_stream_host);
}


void stream_init()
{
    int   sock;

    if ( c_stream_host ) {
        c_stream_hostip = host_resolve(c_stream_host);
        return;
    } else {
        c_stream_hostip = main_interfaceip;
    }
        
    /* Yes, quite, we really ought to do something with the supplied
       host name, but <shrug>
    */
    if ( (sock = tcp_listen(NULL,c_stream_port,0) ) == -1 ) {
        perror("stream");
        exit(1);
    }

    event_set(&listen_event,sock,EV_READ,stream_accept,&listen_event);

    event_add(&listen_event,NULL);

}

static stream_t *new_stream(dongle_t *dongle)
{
    stream_t   *stream = malloc(sizeof(*stream));

    stream->dongle     = dongle;
    stream->inbufstart = 0;
    stream->mediatype  = 0; 

    return stream;
}

void delete_media(media_t *media)
{
    ring_destroy(&media->buffy);

    if ( media->buffy_event ) {
        event_del(media->buffy_event);
        media->buffy_event = NULL;
    }
    close(media->fd);
    shutdown(media->fd,SHUT_RDWR);
}

void delete_stream(stream_t *stream)
{
    /* Cleanup the media file stuff */
    if ( stream->mediatype ) {
        delete_media(&stream->media);
        stream->mediatype = 0;
    }
    

    /* Now clean up the stream input channel */
    event_del(&stream->event);
    close(stream->fd);
    shutdown(stream->fd,SHUT_RDWR);
    free(stream);
}

static void stream_accept(int fd, short event, void *arg)
{
    struct event       *ev = arg;
    dongle_t           *dongle;
    stream_t           *stream;
    char               *hostname;
    int                 cfd;

    /* Reschedule event */
    event_add(ev,NULL);

    if ( ( cfd = tcp_accept(fd,&hostname) ) == -1 ) {
        return;
    }
    printf("Accept STREAM\n");
    dongle = dongle_return(hostname);

    dongle->stream = stream = new_stream(dongle);

    event_set(&stream->event,cfd,EV_READ,stream_read,dongle);
    stream->fd = cfd;
    event_add(&stream->event,NULL);
}



static void stream_read(int fd, short event, void *arg)
{
    dongle_t    *dongle = arg;
    stream_t    *stream = dongle->stream;
    char         buf[500];
    int          r,diff;

    /* Reschedule event */
    event_add(&stream->event,NULL);
  

    /* We should do some buffering here - we occasionally don't get the
       filename with the initial request packet
    */
    r = read(fd,stream->inbuf + stream->inbufstart,sizeof(stream->inbuf)-stream->inbufstart);

    /* Socket closed/error */
    if ( r == 0 || r == -1 ) {
        dongle_close(dongle);
        return;
    }
  
    /* We only expect to get one message a time */

    stream->inbufstart += r;

    while ( 1 ) {
        switch ( stream->inbuf[0] ) {
        case MEDIA_REQUEST:
            diff = media_request(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_STOP:
            diff = media_stop(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_BLOCK:
            diff = media_block(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_SEEK:
            diff = media_seek(stream,stream->inbuf,stream->inbufstart);
            break;     
        case MEDIA_STEP:
            diff = media_step(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_8:
            diff = media_8(stream,stream->inbuf,stream->inbufstart);
            break;
        default:
            printf("Unknown Stream Message\n");
            dump_hex(buf,r);
            diff = stream->inbufstart;

        }
        if ( diff == 0 ) {
            break;
        } else if ( diff > 0 ) {
            int  rem = stream->inbufstart - diff;
            if ( rem >= 0 ) {
                memmove(stream->inbuf,stream->inbuf+diff,rem);
                stream->inbufstart = rem;
            } else {
                stream->inbufstart = 0;
            }
        }           
    } 
}

static int media_request(stream_t *stream, unsigned char *buf, int len)
{
    int            filename_len;
    unsigned char *ptr;
    media_t       *media = &stream->media;

    if ( len < 24 ) {
        return 0;
    }


    ptr = buf + 20;

    BUF_TO_INT16(filename_len,ptr);
    ptr += 2;

    if ( len < filename_len + 24  ) {
        return 0; /* Wait fo more data */
    }

    strcpy(media->filename,ptr);    /* Filename is NUL terminated */

    printf("Client has requested %s\n",media->filename);

    /* Now set up the reply string */
    /* Do a random stream id */
    ++media->id;

    /* Setup a media id thing */
    ptr = buf + 18;   
    INT16_TO_BUF(media->id,ptr);

    if ( stream->mediatype ) {
        delete_media(&stream->media);
    }  

    /* Open the filename now - need to handle no file existing.. */
    filename_parse(stream,media);

    if ( (stream->mediatype & MEDIA_MASK) == MEDIA_MPEG ) {
        ring_init(&media->buffy, c_mpeg_bufsize * 4);
        buf[4] = 0x01;
        buf[6] = 0xFF;

        /* These parameters don't actually seem to matter... */
        buf[8] =  0x02;    /* FIXME: 720 */
        buf[9] =  0xc0; 
        buf[10] = 0x02;    /* FIXME: 576 */
        buf[11] = 0x40;
        buf[14] = 0x09; // 0x19;    /* 6500 - like it matters... */
        buf[15] = 0xf0; // 0x64;
    } else {
        ring_init(&media->buffy, c_mp3_bufsize * 4);
        buf[4] = 0x02;
    }
   
    /* Set up a read event for streaming media */
    if ( stream->mediatype & MEDIA_SOCKET ) {        
        media->buffy_event = malloc(sizeof(*media->buffy_event));
        event_set(media->buffy_event,media->fd,EV_READ,buffer_read,stream);
        event_add(media->buffy_event,NULL);
    } 
   
    write(stream->fd,buf,24);

    return 24 + filename_len;
}

static int media_stop(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;
    media_t       *media = &stream->media;

    if ( len < 24 ) {
        return 0;
    }
    printf("Stop\n");

    if ( stream->mediatype ) {
        delete_media(media);
    }

    stream->last_mediatype = stream->mediatype;
    stream->mediatype = 0;

    ptr = buf + 18;   
    INT16_TO_BUF(media->id,ptr);
    write(stream->fd,buf,len);   

    return 24;
}

/* We've had a request for a block of data */
static int media_block(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr = buf + 8;
    int            mlen;
    media_t       *media = &stream->media;

    if ( len < 24 ) {
        return 0;
    }


    BUF_TO_INT32(media->blocklen,ptr);

    /* Move this somewhere.. */
       

    if ( stream->mediatype & MEDIA_SOCKET ) {
        if ( media->fd == -1 ) {
            /* The file has come to an end */
            stream_write_data(stream,TRUE);            
        } else if ( ring_rest(&media->buffy) >= media->blocklen ) {
            stream_write_data(stream,FALSE);
            stream->pending_data = FALSE;
        } else {
            stream->pending_data = TRUE;
        }

        return 24;
    }

    /* It's just a file on disc, no buffering needed */
    mlen = ring_write_file(&media->buffy,media->fd,media->blocklen);
   

    media->offset += mlen;

    
    ptr = buf + 8;
    INT32_TO_BUF(mlen,ptr);
    INT32_TO_BUF(media->offset,ptr);
 
    write(stream->fd,buf,24);
    ring_read_file(&media->buffy,stream->fd,mlen);
    return 24;
}

static int media_seek(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;
    media_t       *media = &stream->media;


    /* Needs extending slightly... */
    if ( len < 24 ) {
        return 0;
    }
    printf("Seek received\n");
    dump_hex(buf,24);

    if ( (stream->mediatype & MEDIA_SOCKET ) == 0 ) {
        ptr = buf + 8;
        BUF_TO_INT32(media->offset,ptr);
    }

    /* Just acknowledge it back */
    write(stream->fd,buf,24);

    return 24;
}

static int media_8(stream_t *stream, unsigned char *buf, int len)
{
    /* Needs extending slightly... */
    if ( len < 24 ) {
        return 0;
    }
    printf("Media 8\n");
    dump_hex(buf,24);
    write(stream->fd,buf,24);

    return 24;
}


static int media_step(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;
    media_t       *media = &stream->media;

    /* Needs extending slightly... */
    if ( len < 24 ) {
        return 0;
    } 

    printf("Step received\n");
    dump_hex(buf,24);

    ptr = buf + 16;
    INT32_TO_BUF(media->offset,ptr);


    if ( buf[9] == 1 ) {  /* Forward */
        media->offset += media->blocklen;
    } else {              /* And back - but it doesn't work at the moment */
        media->offset -= media->blocklen;
    }

    if ( media->offset < 0 ) {
        media->offset = 0;
    } else if ( media->offset > media->filelength ) {
        media->offset -= media->blocklen;
    }

    /* Nasty, possibly non portable bit... */
    lseek(media->fd,media->offset,SEEK_SET);

    write(stream->fd,buf,24);

    return 24;  
}



static int filename_parse(stream_t *stream, media_t *media)
{
    struct    stat sb;
    char      buffer[FILENAME_MAX+1];
    char     *url;
    int       len;

    media->filelength = 0;
    /* Just a quick and dirty hack.. */
    if ( strncmp(media->filename,"mpeg;http://",7) == 0 ) {
        stream->mediatype = MEDIA_MPEG|MEDIA_SOCKET;
        media->fd = http_open(media->filename+5);
        return media->fd;
    } else if ( strncmp(media->filename,"mp3;http://",7) == 0 ) {
        stream->mediatype = MEDIA_MP3|MEDIA_SOCKET;
        media->fd = http_open(media->filename+4);
        return media->fd;
    } 

    if ( stat(media->filename,&sb) >= 0 ) {
        if ( (sb.st_mode & S_IFIFO)  ) {
            stream->mediatype = MEDIA_MPEG;
            if ( (media->fd = open(media->filename,O_RDONLY|O_NONBLOCK) ) == -1 ) {
                printf("Can't open file\n");
                return -1;
            }
            return stream->fd;
        }
    }


    if ( (media->fd = open(media->filename,O_RDONLY) ) == -1 ) {
        printf("Can't open file\n");
        return -1;
    }

    media->filelength = sb.st_size;

    len = read(media->fd,buffer,sizeof(buffer) - 1 );
    buffer[len] = 0;

    /* Yes, these are meant to be assigns for MEDIA_MPEG and MEDIA_MP3 */
    if ( (strncasecmp(buffer,"mpeg",4) == 0 && (stream->mediatype = MEDIA_MPEG|MEDIA_SOCKET) ) ||
         (strncasecmp(buffer,"mp3",3) == 0 && (stream->mediatype = MEDIA_MP3|MEDIA_SOCKET) )) {
        close(media->fd);
        if ( (url = strchr(buffer,' ') ) == NULL )
            url = strchr(buffer,'\t');
        if ( url == NULL )
            return -1;
        url++;

        /* Now, we should parse the url */
        media->fd = http_open(url);
    } else if ( strncasecmp(buffer,"mplayer",7) == 0 ) {
#if 0
        close(smedia->fd);
        media->fd = pipe_open(stream.buffer);
#endif
    } else { /* It's a normal file, rewind to the start */
        /* Should have some file interrogation logic here */
        if ( strstr(media->filename,".mpeg") || strstr(media->filename,".mpg") || strstr(media->filename,".MPEG") || strstr(media->filename,".MPG") || strstr(media->filename,".vdr") || strstr(media->filename,".VDR") ) {
            stream->mediatype = MEDIA_MPEG;
        } else {
            stream->mediatype = MEDIA_MP3;
        }

        
        lseek(media->fd,0L,SEEK_SET);
    }
    return media->fd;
}



/* This event is used for reading into the event buffer */
static void buffer_read(int fd, short event, void *arg)
{
    stream_t   *stream = arg;
    media_t    *media  = &stream->media;
    int         r;

    r = ring_write_file(&media->buffy,fd,media->blocklen);

    /* Check for socket shutdown */
    if ( r == 0 || r == -1 ) {
        close(fd);
        shutdown(fd,SHUT_RDWR);
        media->fd = -1;
        free(media->buffy_event);
        media->buffy_event = NULL;
        return;
    }


    if ( stream->pending_data && ring_rest(&media->buffy) >= media->blocklen ) {
        stream->pending_data = FALSE;
        stream_write_data(stream,FALSE);

    }

    event_add(media->buffy_event,NULL);

}


static void stream_write_data(stream_t *stream, bool_t flush)
{
    unsigned char      msg[24];
    unsigned char     *ptr;
    int                len;
    media_t           *media = &stream->media;

    if ( flush ) {
        if ( ring_rest(&media->buffy) >= media->blocklen ) {
            len = media->blocklen;
        } else {
            len = ring_rest(&media->buffy);
        }
    } else {
        if ( ring_rest(&media->buffy) >= media->blocklen ) {
            len = media->blocklen;
        } else {
            return;  /* And wait for more data */
        }
    }

    /* Set up the media message */
    memset(msg,0,24);
    msg[0] = 0x04;
    ptr = msg+6;
    INT16_TO_BUF(media->id,ptr);

    INT32_TO_BUF(len,ptr);
    media->offset += len;
    INT32_TO_BUF(media->offset,ptr);


    write(stream->fd,msg,24);
    if ( len ) {
        ring_read_file(&media->buffy,stream->fd,len);       
    }

    return;
}



/** \brief Return the streamtype
 *
 *  \param stream Handle to the stream channel
 *
 *  \return The current streamtype
 */
int stream_get_type(stream_t *stream)
{
    return ( stream->mediatype );
}

/** \brief Return the last streamtype
 *
 *  \param stream Handle to the stream channel
 *
 *  \return The current streamtype
 */
int stream_get_lasttype(stream_t *stream)
{
    return ( stream->last_mediatype );
}


/** \brief Remux the recording so it only has one audio stream
 *
 *  
 */

void remux_init(media_t *media)
{
    /* We should read a bit of this in to find out parameters etc */

}



/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
