/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: stream.c,v 1.21 2003/11/27 22:47:59 dom Exp $
 *   $Date: 2003/11/27 22:47:59 $
 *
 *
 *   Handles the media stream socket..all a bit of mess at the moment..
 *
 */

#include "mediamvp.h"

struct _stream {
    dongle_t      *dongle;
    char           filename[FILENAME_MAX+1];
    int            filename_len;
    struct event   event;
    int            fd;
    uint32_t       offset;
    uint32_t       blocklen;
    int            mediaid;
    int            mediatype;
    int            last_mediatype;
    char           inbuf[2048];
    int            inbufstart;
    int            mediafd;
    off_t          media_filelength;
    char          *buffer;
    int            buffer_len;
    int            buffer_start;
    struct event  *buffer_event;
    bool_t         pending_data;
};



/* Definitions for the commands the MVP gives us */
#define MEDIA_REQUEST    2
#define MEDIA_STOP       3
#define MEDIA_BLOCK      4
#define MEDIA_STEP       5
#define MEDIA_SEEK       7
#define MEDIA_8          8

static void       stream_accept(int fd, short event, void *arg);
static void       stream_read(int fd, short event, void *arg);
static int        media_request(stream_t *stream, unsigned char *buf, int len);
static int        media_stop(stream_t *stream, unsigned char *buf, int len);
static int        media_block(stream_t *stream, unsigned char *buf, int len);
static int        media_seek(stream_t *stream, unsigned char *buf, int len);
static int        media_step(stream_t *stream, unsigned char *buf, int len);
static int        media_8(stream_t *stream, unsigned char *buf, int len);

static int        filename_parse(stream_t *stream);
static void       buffer_read(int fd, short event, void *arg);
static void       stream_write_data(stream_t *stream, bool_t flush);


struct event      listen_event;


int               c_stream_port = 6337;
static char      *c_stream_host = NULL;
uint32_t          c_stream_hostip;

void stream_config()
{
    iniparse_add("stream:port",OPT_INT,&c_stream_port);
    iniparse_add("stream:host",OPT_STR,&c_stream_host);
}


void stream_init()
{
    int   sock;

    if ( c_stream_host ) {
        c_stream_hostip = host_resolve(c_stream_host);
        return;
    } else {
        c_stream_hostip = main_interfaceip;
    }
        
    /* Yes, quite, we really ought to do something with the supplied
       host name, but <shrug>
    */
    if ( (sock = tcp_listen(NULL,c_stream_port,0) ) == -1 ) {
        perror("stream");
        exit(1);
    }

    event_set(&listen_event,sock,EV_READ,stream_accept,&listen_event);

    event_add(&listen_event,NULL);

}

static stream_t *new_stream(dongle_t *dongle)
{
    stream_t   *stream = malloc(sizeof(*stream));

    stream->dongle     = dongle;
    stream->inbufstart = 0;
    stream->mediafd    = -1;
    stream->buffer     = NULL;
    stream->buffer_start = 0;
    stream->buffer_len   = 0;
    stream->buffer_event = NULL;

    return stream;
}

void delete_stream(stream_t *stream)
{
    event_del(&stream->event);

    if ( stream->buffer_event ) {
        event_del(stream->buffer_event);
    }

    if ( stream->buffer ) {
        free(stream->buffer);
    }

    close(stream->fd);
    shutdown(stream->fd,SHUT_RDWR);
    free(stream);
}

static void stream_accept(int fd, short event, void *arg)
{
    struct event       *ev = arg;
    dongle_t           *dongle;
    stream_t           *stream;
    char               *hostname;
    int                 cfd;

    /* Reschedule event */
    event_add(ev,NULL);

    if ( ( cfd = tcp_accept(fd,&hostname) ) == -1 ) {
        return;
    }
    printf("Accept STREAM\n");
    dongle = dongle_return(hostname);

    dongle->stream = stream = new_stream(dongle);

    event_set(&stream->event,cfd,EV_READ,stream_read,dongle);
    stream->fd = cfd;
    event_add(&stream->event,NULL);
}



static void stream_read(int fd, short event, void *arg)
{
    dongle_t    *dongle = arg;
    stream_t    *stream = dongle->stream;
    char         buf[500];
    int          r,diff;

    /* Reschedule event */
    event_add(&stream->event,NULL);
  

    /* We should do some buffering here - we occasionally don't get the
       filename with the initial request packet
    */
    r = read(fd,stream->inbuf + stream->inbufstart,sizeof(stream->inbuf)-stream->inbufstart);

    /* Socket closed/error */
    if ( r == 0 || r == -1 ) {
        dongle_close(dongle);
        return;
    }
  
    /* We only expect to get one message a time */

    stream->inbufstart += r;

    while ( 1 ) {
        switch ( stream->inbuf[0] ) {
        case MEDIA_REQUEST:
            diff = media_request(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_STOP:
            diff = media_stop(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_BLOCK:
            diff = media_block(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_SEEK:
            diff = media_seek(stream,stream->inbuf,stream->inbufstart);
            break;     
        case MEDIA_STEP:
            diff = media_step(stream,stream->inbuf,stream->inbufstart);
            break;
        case MEDIA_8:
            diff = media_8(stream,stream->inbuf,stream->inbufstart);
            break;
        default:
            printf("Unknown Stream Message\n");
            dump_hex(buf,r);
            diff = stream->inbufstart;

        }
        if ( diff == 0 ) {
            break;
        } else if ( diff > 0 ) {
            int  rem = stream->inbufstart - diff;
            if ( rem >= 0 ) {
                memmove(stream->inbuf,stream->inbuf+diff,rem);
                stream->inbufstart = rem;
            } else {
                stream->inbufstart = 0;
            }
        }           
    } 
}

static int media_request(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;

    if ( len < 24 ) {
        return 0;
    }


    ptr = buf + 20;

    BUF_TO_INT16(stream->filename_len,ptr);
    ptr += 2;

    if ( len < stream->filename_len + 24  ) {
        return 0; /* Wait fo more data */
    }

    strcpy(stream->filename,ptr);    /* Filename is NUL terminated */

    printf("Client has requested %s\n",stream->filename);

    /* Now set up the reply string */
    /* Do a random stream id */
    ++stream->mediaid;

    /* Setup a media id thing */
    ptr = buf + 18;   
    INT16_TO_BUF(stream->mediaid,ptr);

    if (stream->mediafd != 0 ) {
        close(stream->mediafd);
        shutdown(stream->mediafd,SHUT_RDWR);
        stream->mediafd = 0;
        /* Free up buffers */
        if ( stream->buffer != NULL ) {
            free(stream->buffer);
            if ( stream->buffer_event ) {
                event_del(stream->buffer_event);
                free(stream->buffer_event);
                stream->buffer_event = NULL;
            }
            stream->buffer = NULL;
            stream->buffer_start = 0;
            stream->buffer_len = 0;
        }
    }


    /* Open the filename now - need to handle no file existing.. */
    filename_parse(stream);

    if ( (stream->mediatype & MEDIA_MASK) == MEDIA_MPEG ) {
        buf[4] = 0x01;
        buf[6] = 0xFF;

        buf[8] =  0x02;    /* FIXME: 720 */
        buf[9] =  0xd0; 
        buf[10] = 0x02;    /* FIXME: 576 */
        buf[11] = 0x40;
        buf[14] = 0x09; // 0x19;    /* 6500 - like it matters... */
        buf[15] = 0xf0; // 0x64;
    } else {
        buf[4] = 0x02;
    }
   
   
    write(stream->fd,buf,24);

    return 24 + stream->filename_len;
}

static int media_stop(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;

    if ( len < 24 ) {
        return 0;
    }
    printf("Stop\n");

    if ( stream->mediafd ) {
        close(stream->mediafd);
        shutdown(stream->mediafd,SHUT_RDWR);
        stream->mediafd = 0;
        if ( stream->buffer != NULL ) {
            free(stream->buffer);
            if ( stream->buffer_event ) {
                event_del(stream->buffer_event);
                free(stream->buffer_event);
                stream->buffer_event = NULL;
            }
            stream->buffer = NULL;
            stream->buffer_start = 0;
            stream->buffer_len = 0;
        }
    }

    stream->last_mediatype = stream->mediatype;
    stream->mediatype = 0;

    ptr = buf + 18;   
    INT16_TO_BUF(stream->mediaid,ptr);
    write(stream->fd,buf,len);   

    return 24;
}

/* We've had a request for a block of data */
static int media_block(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr = buf + 8;
    int           mlen;

    if ( len < 24 ) {
        return 0;
    }


    BUF_TO_INT32(stream->blocklen,ptr);

    if (stream->buffer == NULL ) {
        if ( stream->mediatype & MEDIA_SOCKET ) {
            stream->buffer = malloc(stream->blocklen * 4);
            stream->buffer_len = stream->blocklen * 4;
            stream->buffer_start = 0;
            stream->buffer_event = malloc(sizeof(struct event));
            event_set(stream->buffer_event,stream->mediafd,EV_READ,buffer_read,stream);
            event_add(stream->buffer_event,NULL);

        } else {
            stream->buffer = malloc(stream->blocklen+1);
            stream->buffer_event = NULL;
        }
    }

    if ( stream->mediatype & MEDIA_SOCKET ) {
        if ( stream->mediafd == -1 ) {
            /* The file has come to an end */
            stream_write_data(stream,TRUE);            
        } else if ( stream->buffer_start >= stream->blocklen ) {
            stream_write_data(stream,FALSE);
            stream->pending_data = FALSE;
        } else {
            stream->pending_data = TRUE;
        }

        return 24;
    }

    /* It's just a file on disc, no buffering needed */
    mlen = read(stream->mediafd,stream->buffer,stream->blocklen);
   
    stream->offset += mlen;

    
    ptr = buf + 8;
    INT32_TO_BUF(mlen,ptr);
    INT32_TO_BUF(stream->offset,ptr);
 
    write(stream->fd,buf,24);
    write(stream->fd,stream->buffer,mlen);
    return 24;
}

static int media_seek(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;


    /* Needs extending slightly... */
    if ( len < 24 ) {
        return 0;
    }
    printf("Seek received\n");
    dump_hex(buf,24);

    if ( (stream->mediatype & MEDIA_SOCKET ) == 0 ) {
	ptr = buf + 8;
	BUF_TO_INT32(stream->offset,ptr);
    }

    /* Just acknowledge it back */
    write(stream->fd,buf,24);

    return 24;
}

static int media_8(stream_t *stream, unsigned char *buf, int len)
{
    /* Needs extending slightly... */
    if ( len < 24 ) {
        return 0;
    }
    printf("Media 8\n");
    dump_hex(buf,24);
    write(stream->fd,buf,24);

    return 24;
}


static int media_step(stream_t *stream, unsigned char *buf, int len)
{
    unsigned char *ptr;

    /* Needs extending slightly... */
    if ( len < 24 ) {
        return 0;
    } 

    printf("Step received\n");
    dump_hex(buf,24);

    ptr = buf + 16;
    INT32_TO_BUF(stream->offset,ptr);


    if ( buf[9] == 1 ) {  /* Forward */
	stream->offset += stream->blocklen;
    } else {              /* And back - but it doesn't work at the moment */
	stream->offset -= stream->blocklen;
    }

    if ( stream->offset < 0 ) {
	stream->offset = 0;
    } else if ( stream->offset > stream->media_filelength ) {
	stream->offset -= stream->blocklen;
    }

    /* Nasty, possibly non portable bit... */
    lseek(stream->mediafd,stream->offset,SEEK_SET);

    write(stream->fd,buf,24);

    return 24;  
}



static int filename_parse(stream_t *stream)
{
    struct    stat sb;
    char      buffer[FILENAME_MAX+1];
    char     *url;
    int       len;

    stream->media_filelength = 0;
    /* Just a quick and dirty hack.. */
    if ( strncmp(stream->filename,"mpeg;http://",7) == 0 ) {
        stream->mediatype = MEDIA_MPEG|MEDIA_SOCKET;
        stream->mediafd = http_open(stream->filename+5);
        return stream->mediafd;
    } else if ( strncmp(stream->filename,"mp3;http://",7) == 0 ) {
        stream->mediatype = MEDIA_MP3|MEDIA_SOCKET;
        stream->mediafd = http_open(stream->filename+4);
        return stream->mediafd;
    }
      
    if ( stat(stream->filename,&sb) >= 0 ) {
        if ( (sb.st_mode & S_IFIFO)  ) {
            stream->mediatype = MEDIA_MPEG;
            if ( (stream->mediafd = open(stream->filename,O_RDONLY|O_NONBLOCK) ) == -1 ) {
                printf("Can't open file\n");
                return -1;
            }
            return stream->mediafd;
        }
    }


    if ( (stream->mediafd = open(stream->filename,O_RDONLY) ) == -1 ) {
        printf("Can't open file\n");
        return -1;
    }

    stream->media_filelength = sb.st_size;

    len = read(stream->mediafd,buffer,sizeof(buffer) - 1 );
    buffer[len] = 0;

    /* Yes, these are meant to be assigns for MEDIA_MPEG and MEDIA_MP3 */
    if ( (strncasecmp(buffer,"mpeg",4) == 0 && (stream->mediatype = MEDIA_MPEG|MEDIA_SOCKET) ) ||
         (strncasecmp(buffer,"mp3",3) == 0 && (stream->mediatype = MEDIA_MP3|MEDIA_SOCKET) )) {
        close(stream->mediafd);
        if ( (url = strchr(buffer,' ') ) == NULL )
            url = strchr(buffer,'\t');
        if ( url == NULL )
            return -1;
        url++;

        /* Now, we should parse the url */
        stream->mediafd = http_open(url);
    } else if ( strncasecmp(buffer,"mplayer",7) == 0 ) {
#if 0
        close(stream->mediafd);
        stream->fd = pipe_open(stream.buffer);
#endif
    } else { /* It's a normal file, rewind to the start */
        if ( strstr(stream->filename,".mpeg") || strstr(stream->filename,".mpg") || strstr(stream->filename,".MPEG") || strstr(stream->filename,".MPG") ) {
            stream->mediatype = MEDIA_MPEG;
        } else {
            stream->mediatype = MEDIA_MP3;
        }

        
        lseek(stream->mediafd,0L,SEEK_SET);
    }
    return stream->mediafd;
}



/* This event is used for reading into the event buffer */
static void buffer_read(int fd, short event, void *arg)
{
    stream_t   *stream = arg;
    int         r;

    if ( stream->buffer_start == stream->buffer_len ) {
        /* Buffer overflow... */
        stream->buffer_start = stream->buffer_len - stream->blocklen;
        memmove(stream->buffer,stream->buffer + stream->blocklen,stream->buffer_start);
        printf("Buffer overflow, discard %d bytes\n",stream->blocklen);
    }

    r = read(fd,stream->buffer + stream->buffer_start,stream->buffer_len - stream->buffer_start );

    /* Check for socket shutdown */
    if ( r == 0 || r == -1 ) {
        close(fd);
        shutdown(fd,SHUT_RDWR);
        stream->fd = -1;
        free(stream->buffer_event);
        stream->buffer_event = NULL;
        return;
    }

    stream->buffer_start += r;

    if ( stream->pending_data && stream->buffer_start >= stream->blocklen ) {
        stream->pending_data = FALSE;
        stream_write_data(stream,FALSE);

    }

    event_add(stream->buffer_event,NULL);

}


static void stream_write_data(stream_t *stream, bool_t flush)
{
    unsigned char      msg[24];
    unsigned char     *ptr;
    int                len;

    if ( flush ) {
        if ( stream->buffer_start >= stream->blocklen ) {
            len = stream->blocklen;
        } else {
            len = stream->buffer_start;
        }
    } else {
        if ( stream->buffer_start >= stream->blocklen ) {
            len = stream->blocklen;
        } else {
            return;  /* And wait for more data */
        }
    }

    /* Set up the media message */
    memset(msg,0,24);
    msg[0] = 0x04;
    ptr = msg+6;
    INT16_TO_BUF(stream->mediaid,ptr);

    INT32_TO_BUF(len,ptr);
    stream->offset += len;
    INT32_TO_BUF(stream->offset,ptr);


    write(stream->fd,msg,24);
    if ( len ) {
        write(stream->fd,stream->buffer,len);
        memmove(stream->buffer,stream->buffer+len,stream->buffer_start - len);
        stream->buffer_start -= len;
        if ( stream->buffer_start < 0 )
            stream->buffer_start = 0;
    }

    return;
}



/** \brief Return the streamtype
 *
 *  \param stream Handle to the stream channel
 *
 *  \return The current streamtype
 */
int stream_get_type(stream_t *stream)
{
    return ( stream->mediatype );
}

/** \brief Return the last streamtype
 *
 *  \param stream Handle to the stream channel
 *
 *  \return The current streamtype
 */
int stream_get_lasttype(stream_t *stream)
{
    return ( stream->last_mediatype );
}

/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
