#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/signal.h>
#include <sys/utsname.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>



/* libvbi */
#include "vt.h"
#include "misc.h"
#include "fdset.h"
#include "vbi.h"
#include "lang.h"
#include "dllist.h"
#include "export.h"


/* TODO: 
   - Changing channels by ourselves without v4lctl - this will stop
   weird things happening if channels are changed on us
   - Proper daemon behaviour (uid/gid/forking)
   - Interface configuration for the listen port
   - Proper control language for the listen port?
   - Improve the php so it's not quite so kludgey
*/




static int         socket_port         = 5655;       /* config */
char              *channel             = NULL;
static time_t      channel_time        = 0;
static int         channel_switchtime  = 10;         /* config */
static char       *vbi_name            = "/dev/vbi"; /* config */
static int         bttv                = -1;         /* config */
int                debug               = 0;          /* config */       


static struct vbi *vbi;
struct export     *pngfmt;
struct export     *txtfmt;
static int         listen_sock;


static void        unix_read(int sock);
static void        page_decode(struct vt_page *vtp);
static int         sock_create();
static int         set_channel(char *newchan);


static void sighandler(int sig)
{
    vbi_close(vbi);
    shutdown(listen_sock,0);
    close(listen_sock);
    exit(1);
}

static void event_client(struct dl_head *reqs, struct vt_event *ev)
{
    struct vt_page *vtp;

    if ( !debug )
	return;

    switch (ev->type) {
    case EV_PAGE:
        vtp = ev->p1;
        fprintf(stderr,"page (%s) %x.%02x\n", channel, vtp->pgno, vtp->subno);
        break;
    }
}

static void usage(char *name)
{
    printf("alevt PHP Teletext Daemon\n"
	   "\n"
	   "usage: %s [options]\n"
	   "\n"
	   "Options:\n"
	   "  -h    print this text\n"
	   "  -d    vbi device\n"
	   "  -6    16 vbi lines\n"
	   "  -9    19 vbi lines\n"
	   "  -d    debug output\n"
	   "  -p    listen port for messages\n"
	   "  -t    Minimum time between channel switches\n" ,
	   name);
    exit(1);
}

int main(int argc, char *argv[])
{
    fd_set   rd;
    struct timeval      tv;
    int                 max;
    int                 i = 1;

    while ( i < argc ) {
	if ( strcmp(argv[i],"-h") == 0 ) {
	    usage(argv[0]);
	} else if ( strcmp(argv[i],"-d") == 0 ) {  /* device */
	    if ( i + 1 < argc ) {
		vbi_name = argv[i+1];
		i++;
	    }
	} else if ( strcmp(argv[i],"-p") == 0 ) {  /* port */
	    if ( i + 1 < argc ) {
		socket_port = atoi(argv[i+1]);
		i++;
	    }
	} else if ( strcmp(argv[i],"-t") == 0 ) {  /* switch time */
	    if ( i + 1 < argc ) {
		channel_switchtime = atoi(argv[i+1]);
		i++;
	    }
	} else if ( strcmp(argv[i],"-6") == 0 ) {  /* vbi mode */
	    bttv = 0;
	} else if ( strcmp(argv[i],"-9") == 0 ) {  /* vbi mode */
	    bttv = 1;
	} else if ( strcmp(argv[i],"-d") == 0 ) {  /* debug */
	    debug++;
#if 0
	} else if ( strcmp(argv[i],"-i") == 0 ) {  /* listen ip */
	    if ( i + 1 < argc ) {
	        listen_ip = argv[i+1];
		i++;
	    }  
#endif
	} else {
	    usage(argv[0]);
	}
	i++;
    }


	    

    channel = strdup("unknown");
    listen_sock = sock_create();

    signal(SIGHUP,sighandler);
    signal(SIGINT,sighandler);
    signal(SIGKILL,sighandler);


    fdset_init(fds);

    vbi = vbi_open(vbi_name,cache_open(),0,bttv);

    if ( vbi == NULL ) {
	perror("Cannot open vbi device");
	exit(1);
    }
    pngfmt = export_open("png");
    txtfmt = export_open("ascii");
    vbi_add_handler(vbi, event_client, NULL);


    while ( 1 ) {
	FD_ZERO(&rd);
        FD_SET(vbi->fd,&rd);
	FD_SET(listen_sock,&rd);
	max = vbi->fd > listen_sock ? (vbi->fd) : (listen_sock);
	tv.tv_sec = 1;
	tv.tv_usec = 0;
	if ( -1 == select(max+1,&rd,NULL,NULL,&tv) ) {
		perror("select");
		break;
	}
        if (FD_ISSET(vbi->fd,&rd)) {
            vbi_handler(vbi,vbi->fd);
	}

	if (FD_ISSET(listen_sock,&rd)) {
	    unix_read(listen_sock);
	}

    }
    vbi_close(vbi);
    shutdown(listen_sock,0);
    close(listen_sock);
    exit(1);
}

void unix_read(int sock)
{
    int  s,t,len;
    struct sockaddr_un remote;
    char   buf[100];
    char   *subptr,*channel;
    int    page,subpage;
    struct vt_page *vtp;

    t = sizeof(remote);
    if ((s = accept(sock, (struct sockaddr *)&remote, &t)) == -1) {
	perror("accept");
	return;
    }
    len = recv(s,buf,sizeof(buf),0);
    if ( len > 0 ) {
	buf[len] = 0;
	if ( debug )
	    fprintf(stderr,"Received <%s>\n",buf);
    }
    close(s);

    if ( (subptr = strchr(buf,'-') ) ) {
	*subptr = 0;
	subptr++;
	if ( channel = strchr(subptr,'-') ) {
	    *channel = 0;
	    channel++;
	    sscanf(buf,"%03x",&page);
	    sscanf(subptr,"%02x",&subpage);
	    if ( debug )
		fprintf(stderr,"We want channel <%s> page <%d> subpage <%d>\n",channel,page,subpage);
	    /* Change channel here... */
	    set_channel(channel);
	    if ( subpage == 0 )
		subpage = ANY_SUB;
	    if ( vtp = vbi->cache->op->get(vbi->cache,page,subpage) ) {
		page_decode(vtp);
	    } else {
		/* Well, the user will request, so not much point.. */
	    }
	}
    }

    return;
}


/* fontsize is 9 * 16 */
static void page_decode(struct vt_page *vtp)
{
    char     filename[FILENAME_MAX+1];
    struct fmt_page pg[1];
    struct fmt_char l[W+2];
    int size,len,x,y,col,x2,link;
    FILE  *fp;

#define L(z,m) (pg->data[y][z].m)
    /* Export the png file */
    sprintf(filename,"%s/%x-%d.png",channel, vtp->pgno, vtp->subno);
    fmt_page(pngfmt, pg, vtp);
    pngfmt->mod->output(pngfmt,filename,pg);

    sprintf(filename,"%s/%x-%d.map",channel,vtp->pgno, vtp->subno);

    if ( ( fp = fopen(filename,"w") ) == NULL ) {
	if ( debug )
	    fprintf(stderr,"Cannot open file %s\n",filename);
	return;
    }
    
#ifndef SERVER
    fprintf(fp,"<MAP NAME=\"vtpage\">\n");
#endif
    /* Now work on the image map */
    fmt_page(txtfmt,pg,vtp);
    for (y = 0; y < H; y++) {
	for (  x = 0; x < W; ++x) {	    
	    /* References to other pages */
	    if (y > 0 && x < W-2 &&
		isdigit(L(x,ch)) &&
		isdigit(L(x+1,ch)) &&
		isdigit(L(x+2,ch)) &&
		!isdigit(L(x+3,ch)) &&
		!isdigit(L(x-1,ch))) {
#ifndef SERVER
		fprintf(fp,"<AREA SHAPE=\"rect\" COORDS=\"%d,%d,%d,%d\" HREF=\"index.php?CHANNEL=%s&PAGE=%c%c%c\">\n",
			x * 9, y * 16, 
			 (( x + 3 ) * 9 ) -1,  (( y + 1 + ( L(x,attr) & EA_DOUBLE ) ) * 16) - 1,
			channel,L(x,ch),L(x+1,ch),L(x+2,ch) );
#else
		fprintf(fp,"rect ../index.php?CHANNEL=%s&PAGE=%c%c%c %d,%d %d,%d\n",channel,
			L(x,ch),L(x+1,ch),L(x+2,ch),
			x * 9, y * 16, 
			 (( x + 3 ) * 9 ) -1,  (( y + 1 + ( L(x,attr) & EA_DOUBLE ) ) * 16) - 1 );
#endif
	    }

	    /* Subpages - only handles single ones - need to handle multiple digits me thinks.. */
	    if ( y > 0 && x < W-2 && vtp->subno &&
		 isdigit(L(x,ch)) &&
		 '/' == L(x+1,ch) &&
		 isdigit(L(x+2,ch)) &&
		 !isdigit(L(x+3,ch)) &&
		 !isdigit(L(x+1,ch)) ) {
		if ( L(x,ch) == L(x+2,ch) ) {
#ifndef SERVER
		    fprintf(fp,"<AREA SHAPE=\"rect\" COORDS=\"%d,%d,%d,%d\" HREF=\"index.php?CHANNEL=%s&PAGE=%03x&SUBPAGE=1\">\n",
			    x * 9, y * 16, 
			    (( x + 2 ) * 9 ) -1,  (( y + 1 + ( L(x,attr) & EA_DOUBLE ) ) * 16) - 1,
			    channel,vtp->pgno);
#else
		    fprintf(fp,"rect ../index.php?CHANNEL=%s&PAGE=%03x&SUBPAGE=1 %d,%d %d,%d\n",channel,
			    vtp->pgno,
			    x * 9, y * 16, 
			    (( x + 2 ) * 9 ) -1,  (( y + 1 + ( L(x,attr) & EA_DOUBLE ) ) * 16) - 1 );
#endif
		} else {
#ifndef SERVER
		    fprintf(fp,"<AREA SHAPE=\"rect\" COORDS=\"%d,%d,%d,%d\" HREF=\"index.php?CHANNEL=%s&PAGE=%03x&SUBPAGE=%d\">\n",
			    x * 9, y * 16, 
			    (( x + 2 ) * 9 ) -1,  (( y + 1 + ( L(x,attr) & EA_DOUBLE ) ) * 16) - 1,
			    channel,vtp->pgno,L(x,ch)+1-'0');
#else
		    fprintf(fp,"rect ../index.php?CHANNEL=%s&PAGE=%03x&SUBPAGE=%d %d,%d %d,%d\n",channel,
			    vtp->pgno,L(x,ch)+1-'0',
			    x * 9, y * 16, 
			    (( x + 2 ) * 9 ) -1,  (( y + 1 + ( L(x,attr) & EA_DOUBLE ) ) * 16) - 1 );
#endif

		}
	    }
	    /* Fasttext links */
	    if ( vtp->flof && x < W-2 && 24 == y && 
		 L(x,fg) > 0 && L(x,fg) < 8 &&
		 x > 0 &&
		 !isspace(L(x,ch))  ) {
		col = L(x,fg);
		x2 = x+1;
		link = L(x,fg) == 6 ? 3 : L(x,fg)-1;

		if ( vtp->link[link].pgno ) {
		    while ( x2 < W && L(x2,fg) == col )
			++x2;
		    if ( debug )
			fprintf(stderr,"Fasttext link %d is %03x\n",link,vtp->link[link].pgno);
		    if( vtp->link[link].subno == ANY_SUB ) {
#ifndef SERVER
			fprintf(fp,"<AREA SHAPE=\"RECT\" COORDS=\"%d,%d,%d,%d\" HREF=\"index.php?CHANNEL=%s&PAGE=%03x\">\n",x  * 9, y * 16, (x2 * 9)  - 1, (y + 1) * 16 - 1, channel,vtp->link[link].pgno);
#else
			fprintf(fp,"rect ../index.php?CHANNEL=%s&PAGE=%03x %d,%d %d,%d\n",channel,
				vtp->link[link].pgno,			
				x  * 9, y * 16, 
				(x2 * 9)  - 1, (y + 1) * 16 - 1);
#endif
		    } else {
#ifndef SERVER
			fprintf(fp,"<AREA SHAPE=\"RECT\" COORDS=\"%d,%d,%d,%d\" HREF=\"index.php?CHANNEL=%s&PAGE=%03x&SUBPAGE=%02x\">\n",x  * 9, y * 16, (x2 * 9)  - 1, (y + 1) * 16 - 1, 
				channel,vtp->link[link].pgno,vtp->link[link].subno);
#else
			fprintf(fp,"rect ../index.php?CHANNEL=%s&PAGE=%03x&SUBPAGE=%02x %d,%d %d,%d\n",channel,
				vtp->link[link].pgno,			
				vtp->link[link].subno,
				x * 9, y * 16, 
				(x2 * 9 ) - 1, (y + 1 ) * 16 - 1);
#endif
		    }
		    vtp->link[link].pgno = 0;
		}

	    }

		
	    
	}
    }
#ifndef SERVER
    fprintf(fp,"</MAP>\n");
#endif
    
    fclose(fp);
}


int  sock_create()
{
    struct sockaddr_in     saddr;
    int    sock;

    sock = socket(AF_INET,SOCK_STREAM,0);

    if ( sock < 0) {
	perror("Socket");
	exit(1);
    }

    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(socket_port);
    saddr.sin_addr.s_addr = htonl(INADDR_ANY);

    if (bind(sock,(struct sockaddr *)&saddr,sizeof(saddr)) ) {
	perror("Bind");
	exit(1);
    }
    listen(sock,2);
    return (sock);
}



int set_channel(char *newchan)
{
    char    buf[512];
    time_t  tim;

    /* Check to see if we're already on that channel */
    if ( channel ) {
	if ( strcmp(channel,newchan) == 0 ) {
	    return 0;
	}
    }

    /* Check to see if we're switching too quickly */
    tim = time(NULL);
    if ( time(NULL) < channel_time + channel_switchtime ) {
	return;
    }
    channel_time = tim;

    /* Now call v4lctl */
    snprintf(buf,sizeof(buf),"v4lctl setstation \"%s\"",newchan);
    system(buf);

    /* And replace the channel */
    if ( channel )
	free(channel);
    channel = strdup(newchan);
}
