package com.ibm.ulc.util;

/*
 * Copyright (c) 1999 Object Technology International Inc.
 */
import java.io.*;
import java.util.*;

/**
 * A convenience implementation of FileFilter that filters out
 * all files except for those type extensions that it knows about.
 * Case is ignored.
 * Extensions are of the type ".java".
 * @see  com.ibm.ulc.application.ULCFile#list
 * @see  com.ibm.ulc.application.ULCFileChooser#setFileFilter
 * @see  com.ibm.ulc.application.ULCFileChooser#addChoosableFileFilter
 */
public class UlcFileFilter implements Serializable, java.io.FilenameFilter {
	private Hashtable fExtensions = null;
	private boolean fIncludeDirectories = true;
	private boolean fAcceptAll = false;
	private String fDescription = null;
	private static UlcFileFilter fgAcceptAllFileFilter = null;
	static {
		fgAcceptAllFileFilter = new UlcFileFilter(null);
		fgAcceptAllFileFilter.fAcceptAll = true;
	}
/**
 * Creates a file filter from the given string array.
 * Example: <code>new ULCFileFilter(new String {"gif", "jpg"});</code>
 *
 * Note that the "." before the extension is not needed.
 * @param      ext the extensions to be accepted. Case is ignored.
 *
 */
public UlcFileFilter(String ext[]) {
	this(ext, true);
}
/**
 * Creates a file filter from the given string array and description.
 * Example: <code>new ULCFileFilter(new String {"gif", "jpg"}, "Gif and JPG Images");</code>
 *
 * Note that the "." before the extension is not needed.
 * @param      ext[] the extensions to be accepted. Case is ignored.
 * @param      description a string decribing the filter.
 *
 */
public UlcFileFilter(String ext[], String description) {
	this(ext, true);
	fDescription = description;
}
/**
 * Creates a file filter from the given string array and description. Directories can be excluded
 * Example: <code>new ULCFileFilter(String {"gif", "jpg"}, "Gif and JPG Images", false);</code>
 *
 * Note that the "." before the extension is not needed.
 * @param      ext[] the extensions to be accepted. Case is ignored.
 * @param      description a string decribing the filter.
 * @param      includeDirectories specifies if directories should be included.
 *
 */
public UlcFileFilter(String ext[], boolean includeDirectories) {
	fIncludeDirectories = includeDirectories;
	fExtensions = new Hashtable(5);
	if (ext != null) {
		for (int i = 0; i < ext.length; i++) {
			fExtensions.put(ext[i].toLowerCase(), this);
		}
	}
}
public boolean accept(File f, String name) {
	if (fAcceptAll)
		return true;
	String extension = getExtension(name);
	if (extension != null && fExtensions.get(extension) != null) {
		return true;
	} else
		if (fIncludeDirectories) {
			File file = null;
			if (f.getAbsolutePath().endsWith(File.separator)) {
				file = new File(f.getAbsolutePath() + name);
			} else {
				file = new File(f.getAbsolutePath() + File.separator + name);
			}
			if (file.isDirectory())
				return true;
		}
	return false;
}
public boolean equals(Object obj) {
	if (fAcceptAll && (obj instanceof UlcFileFilter) && (((UlcFileFilter) obj).fAcceptAll == true)) {
		return true;
	} else {
		return super.equals(obj);
	}
}
public static UlcFileFilter getAcceptAllFileFilter() {
	return fgAcceptAllFileFilter;
}
public String getDescription() {
	if (fDescription != null) {
		return fDescription;
	}
	return "no description";
}
private String getExtension(String filename) {
	int i = filename.lastIndexOf('.');
	if (i > 0 && i < filename.length() - 1) {
		return filename.substring(i + 1).toLowerCase();
	}
	return null;
}
}
