package com.ibm.ulc.util;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.io.*;

/**
 * Writes an Anything in a pure ASCII format. Since the format is nicely formatted
 * and quite readable it can be considered a "Pretty Printing" format.
 * <p>
 * The (unverified) format is:
 * <pre>
 * anything:=         &lt;simple&gt; | &lt;structure&gt;
 * simple:=           &lt;null&gt; | &lt;string&gt; | &lt;bytes&gt; | &lt;long&gt; | &lt;double&gt; | &lt;boolean&gt; | &lt;serializable&gt;
 * structure:=        '{' { &lt;slot&gt; } '}'
 * slot:=             [ &lt;label&gt; ] &lt;anything&gt;
 * label:=            '/' &lt;ident&gt;
 * ident:=            &lt;long&gt; | &lt;string&gt;
 * string:=           &lt;quoted_string&gt; | &lt;unquoted_string&gt;
 * quoted_string:=    '"' { &lt;character&gt; } '"'
 * unquoted_string:=  &lt;character&gt; { &lt;non_blank_character&gt; }
 * bytes:=            '&lt;' { &lt;hexdigit&gt; &lt;hexdigit&gt; } '&gt;'
 * serializable:=		'[' &lt;serializedJavaObject&gt; ']'
 * long:=             [ '+' | '-' ] &lt;digits&gt; { &lt;digit&gt; }
 * double:=           [ '+' | '-' ] { &lt;digit&gt; } '.' { &lt;digit&gt; }
 * digit:=            '0' | '1' | '2' | '3' | '4' | '5' | '6' | '7' | '8' | '9'
 * hexdigit:=         &lt;digit&gt; | 'a' | 'b' | 'c' | 'd' | 'e' | 'f'
 * boolean:=          't' | 'f'
 * null:=             '*'
 */

public
class AnythingWriter extends UlcObject implements IAnythingWriter {
	private boolean fSkipBytes= false;
	private boolean fPretty= true;
	private OutputStream fOut= null;
	private int fCol= 0;

	private static char[] HEX= { '0', '1', '2', '3', '4', '5', '6', '7',
								 '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
public AnythingWriter() {
}
public AnythingWriter(boolean compact, boolean skipbytes) {
	fSkipBytes = skipbytes;
	fPretty = !compact;
}
/**
 * Creates the corresponding reader for this format
 */
public IAnythingReader createReader() {
	return new AnythingReader();
}
private void intPrint(int c) throws InterruptedIOException, IOException {
	fOut.write(c);
}
private void intPrint(String s) throws InterruptedIOException, IOException {
	int len = s.length();
	for (int i = 0; i < len; i++) {
		char c = s.charAt(i);
		switch (c) {
			case '\t' :
				fOut.write('\\');
				fOut.write('t');
				break;
			case '\n' :
				fOut.write('\\');
				fOut.write('n');
				break;
			case '\b' :
				fOut.write('\\');
				fOut.write('b');
				break;
			case '\\' :
			case '"' :
				fOut.write('\\');
				fOut.write(c);
				break;
			default :
				if (c >= ' ' && c < 128)
					fOut.write(c);
				else {
					fOut.write('\\');
					if (c > 255) { // 2 bytes
						fOut.write('u');
						String ss = Integer.toHexString(c);
						if (ss.length() < 4) {
							fOut.write('0');
							if (ss.length() < 3)
								fOut.write('0');
							if (ss.length() < 2)
								fOut.write('0');
						}
						intPrint(ss);
					}
					else { // 1 byte
						String ss = Integer.toOctalString(c);
						if (ss.length() < 3) {
							fOut.write('0');
							if (ss.length() < 2)
								fOut.write('0');
						}
						intPrint(ss);
					}
				}
				break;
		}
	}
}
private void newline() throws InterruptedIOException, IOException {
	if (fPretty)
		intPrint('\n');
}
private void print(byte[] k) throws InterruptedIOException, IOException {
	if (fSkipBytes) {
		intPrint("<...>");
	}
	else {
		intPrint('<');
		for (int i = 0; i < k.length; i++) {
			int v = k[i];
			if (v < 0)
				v += 256;
			if (v >= 0 && v <= 255) {
				char c1 = HEX[v / 16], c2 = HEX[v % 16];
				//System.out.print("" + c1 + "" + c2 + "");
				intPrint(c1);
				intPrint(c2);
			}
			else {
				trouble("print", "value out range");
			}
		}
		intPrint('>');
	}
}
public void print(OutputStream os, Anything a) {
	try {
		printEx(os, a);
	}
	catch (InterruptedIOException e1) {
		// We've been interrupted.  Make sure we're still interrupted.
		Thread.currentThread().interrupt();
	}
	catch (IOException e2) {
	}
}
private void print(Serializable ob) {
	try {
		intPrint("[");
		ObjectOutputStream oos = new ObjectOutputStream(fOut);
		oos.writeObject(ob);
		intPrint(']');
	}
	catch (IOException ex) {
	}
}
private void print(String k) throws InterruptedIOException, IOException {
	boolean needqoutes = false;
	if (k.equals("t") || k.equals("f"))
		needqoutes = true;
	else {
		int l = k.length();
		if (l <= 0 || Character.isDigit(k.charAt(0)))
			needqoutes = true;
		else {
			for (int i = 0; i < l; i++) {
				if (!AnythingReader.isWord(k.charAt(i))) {
					needqoutes = true;
					break;
				}
			}
		}
	}
	if (needqoutes) {
		intPrint('\"');
		intPrint(k);
		intPrint('\"');
	}
	else {
		intPrint(k);
	}
}
public synchronized void printEx(OutputStream os, Anything a) throws InterruptedIOException, IOException {
	fOut = os;
	switch (a.getType()) {
		case Anything.NULL :
			intPrint('*');
			newline();
			break;
		case Anything.LONG :
		case Anything.DOUBLE :
			intPrint(a.toString());
			newline();
			break;
		case Anything.BOOLEAN :
			intPrint(a.asBoolean(false) ? 't' : 'f');
			newline();
			break;
		case Anything.STRING :
			print(a.toString());
			newline();
			break;
		case Anything.BYTES :
			print(a.asBytes());
			newline();
			break;
		case Anything.SERIALIZABLE :
			print(a.asSerializable(null));
			newline();
			break;
		case Anything.VECTOR :
			intPrint('{');
			newline();
			int s = a.size();
			for (int i = 0; i < s; i++) {
				fCol++;
				tab();
				String key = a.slotName(i);
				if (key != null) {
					intPrint('/');
					print(key);
				}
				intPrint(' ');
				print(os, a.get(i));
				fCol--;
			}
			tab();
			intPrint('}');
			newline();
			break;
		default :
			break;
	}
}
private void tab() throws InterruptedIOException, IOException {
	if (fPretty)
		for (int i = 0; i < fCol; i++)
			intPrint("  ");
}
}
