package com.ibm.ulc.examples.CustomerInformation;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.net.URL;
import java.util.*;
import com.ibm.ulc.base.IDefaults;
import com.ibm.ulc.application.*;

/**
 * This class is responsible for the editing of the Customer Information
 * application. Editing includes building up the whole UI using the ULC
 * widgets, and setting up all the adapters, callbacks etc. which are 
 * required for the actual editing of the application.
 *
 * @see CustomerInformationEditor#edit(ULCApplication)
 * @see CustomerInformationEditor#build
 */
public class CustomerInformationEditor {
	
	/**
	 * The CustomerInformation, which is the 'model' for this viewer.
	 */
	protected CustomerInformation fCustInfo;
	
	/**
	 * The selector class which can be used for selecting a customer
	 * from a pre-defined list
	 */
	protected CustomerSelector fCustomerSelector= null;
	
	/*
	 * The adapter which will map the columns in the CustomerInformation
	 * name and address fields, to the 'real' values (i.e. actual values for
	 * these attributes as held by the customer)
	 */
	protected NameAddressAdapter fNameAddressAdapter= null;

	/**
	 * The adapter which will map the columns in the CustomerInformation accounts
	 * table, to the 'real' model (i.e. the accounts held on to by the customer)
	 */
	protected AccountsAdapter fAccountsAdapter= null;

	/**
	 * The adapter which will map the columns in the CustomerSelector
	 * list, to the 'real' model (i.e. the (test) customers). This adapter is held
	 * here, rather than in the CustomerSelector, since any changes made to the
	 * customer name-address have to be saved here.
	 */
	protected CustomersAdapter fCustomersAdapter= null;

	/** 
	 * The adapter which will map the (one) selected account,
	 * to the 'real' model (i.e. the account in the list held on to by the customer)
	 */
	protected AccountAdapter fAccountAdapter= null;

	/** 
	 * The same AccountAdapter above can be implemented in the
	 * form of a RowModel. By doing this, the update of the account
	 * data will happen without any traffic on the wire.
	 */
	protected ULCRowModel fAccountRowModel= null;
		
	/** 
	 * The adapter which will map the (one) selected credit,
	 * to the 'real' model (i.e. the credit in the list held on to by the customer)
	 */
	protected CreditAdapter fCreditAdapter= null;

	private ULCShell	fShell; 				// The shell into which all the widgets are built
	private ULCPagebook	fAccountSpecificFields;	// Display depending on type of account
	private ULCPagebook	fAccountActions; 		// Account to be added / modified
	private ULCTable 	fAccountsTable; 		// Acts as an 'enabler' for other buttons...
	protected ULCButton	fNewAccountButton;
	protected ULCButton	fDeleteAccountButton;
	protected ULCField	fAccountNumberField; 	// required for setting focus to it.
	protected ULCField	fUrlField;				//Holds the url entered in the documents page
	
	// The index of a collection (of Accounts, in this case), 
	// when no selection has been made as yet.
	protected static final int UNDEFINED_SELECTION_INDEX = -1; 
	
	protected static final int BOX_GAP= 5; 

	protected static final int SHELL_WIDTH= 800; 
	protected static final int SHELL_HEIGHT= 675; 

	protected static final int NOTEBOOK_WIDTH= 120; 
	protected static final int NOTEBOOK_HEIGHT= 100;
	protected static final int NOTEBOOK_COLUMN_WIDTH= 30;
	protected static final int NUMBER_OF_TABLE_ROWS= 8;	
public CustomerInformationEditor() {
}
public CustomerInformationEditor(CustomerInformation custInfo) {
	fCustInfo= custInfo;
	fAccountsAdapter= new AccountsAdapter(this);
	fCustomersAdapter= new CustomersAdapter(CustomerInformation.fTestCustomers);
	fNameAddressAdapter= new NameAddressAdapter(fCustInfo.getCustomer());
	fAccountAdapter= new AccountAdapter(this);
	fCreditAdapter= new CreditAdapter();
	fAccountsTable= new ULCTable(fAccountsAdapter, NOTEBOOK_WIDTH, NUMBER_OF_TABLE_ROWS);
	fUrlField= new ULCField(getDefaultUrlString(), 80);
}
/** 
 * The dynamic menus work based on the RowModel (fAccountRowModel) defined earlier. 
 * You need to define separate ULCMenu objects upfront, each of which tracks a particular 
 * attributeName present in the RowModel.  And you need to provide one attributeValue for 
 * this attributeName in each menu object, so that when the RowModel has that value for 
 * that attribute, theappropriate menu object will be displayed (dynamically).
 *
 * In our example the attribute to track is the 'AccountType', which can have the values :
 * "Savings Account", "Private Account" or "Loan Account".
 */
public void addDynamicMenusTo(ULCMenu popupMenu) {

	//Menu to be displayed if the accountType is "Savings Account"
	ULCMenu m = new ULCMenu("Savings Account Menu");
	m.add(new ULCMenuItem("Savings Account MenuItem"));
	m.setFormModel(fAccountRowModel);
	m.setFormAttributeName("Type");
	m.setFormAttributeValue("Savings Account");
	popupMenu.add(m);

	//Menu to be displayed if the accountType is "Private Account"
	m = new ULCMenu("Private Account Menu");
	m.add(new ULCMenuItem("Private Account MenuItem"));
	m.setFormModel(fAccountRowModel);
	m.setFormAttributeName("Type");
	m.setFormAttributeValue("Private Account");
	popupMenu.add(m);

	//Menu to be displayed if the accountType is "Loan Account"	
	m = new ULCMenu("Loan Account Menu");
	ULCMenu m1 = new ULCMenu("Loan Account SubMenu");
	m1.add(new ULCMenuItem("Loan Account SubMenuItem"));
	m.add(m1);
	m.setFormModel(fAccountRowModel);
	m.setFormAttributeName("Type");
	m.setFormAttributeValue("Loan Account");
	popupMenu.add(m);
}
/**
 * This is a critical step here. Note that the accountAdapter
 * at this point contains one of the accounts selected from the list.
 * So we need to save any changes made by the user for the new account,
 * into a new account. Also, if the Validation process therein results 
 * in an invalid result, the newAccount gets set to null, so skip the
 * further steps.
 * <p>
 * Also, after doing this, set the new account as the selected account,
 * so that all the enabling-disabling happens correctly.
 */
public void addNewAccount () {
	Account newAccount= fAccountAdapter.copyChangesIntoNewAccount();
	if (newAccount != null) {	
		fAccountsAdapter.addAccount(newAccount);
		setAccountSelectionIrrespective(newAccount);

		//Now for the new account just inserted, 'force' the table
		//to select this account.
		setAccountsTableSelectedAccount(newAccount);
	}	
}
/**
 * Add Static menus (which will always be shown) on the given
 * popupMenu.
 */
public void addStaticMenusTo(ULCMenu popupMenu) {
	popupMenu.add(createNewAccountMenuItem());
	popupMenu.add(createDeleteAccountMenuItem());
}
/**
 * Build the application widgets into the receiver's shell.
 * Use split panes for building the lower layer of
 * components.
 */
public void build() {
	fShell.setSize(SHELL_WIDTH, SHELL_HEIGHT);
	fShell.addWindowClosingListener(new WindowCloseAction(this));
	ULCBox box= new ULCBox(2,3);

	buildMenus();
	box.span(3,"et", buildNameAddressPart());
	
	ULCComponent accountsTableButtons= buildAccountsTableButtons();
	ULCComponent accountsPart= buildAccountsPart();
	ULCComponent notebookPart= buildNotebookPart();
	
	//A split pane to combine the buttons and the notbook horizontally
	ULCSplitPane nestedSplitPane = new ULCSplitPane(false, accountsTableButtons, notebookPart);
	nestedSplitPane.setDividerLocation(0.2);

	//A split pane to combine the table with the above splitPane, horizontally	
	ULCSplitPane lowerSplitPane = new ULCSplitPane(false, accountsPart, nestedSplitPane);
	lowerSplitPane.setDividerLocation(0.47);
	
	box.add("ee", lowerSplitPane);
	
	fShell.add(box);
}
/**
 * The buttons pertaining to the account, consist of a pagebook
 * (which will show either a 'Save' or an 'Add' button (depending
 * upon the selections), and a 'Cancel' button.
 */
 
public ULCBox buildAccountActionChoiceBox() {
	ULCBox choices= new ULCVBox(2);
	choices.add("eb", buildAddOrModifyActionsPagebook());	// Save or Add

	ULCButton accountCancelButton= new ULCButton(getResource("ButCancel"));
	accountCancelButton.addActionListener(new RollbackAccountModificationAction(this));
	accountCancelButton.setEnabled(false);
	accountCancelButton.setEnabler(fAccountAdapter);
	choices.add("eb", accountCancelButton);
	
	return choices;
}
/**
 * There are two ways of displaying the details of an account which is also part of a tableModel.
 * One way is the (usual) adapter-based approach, where the AccountAdapter is set with the right
 * account based on the selectionChanged() event coming to the ULC side.
 *
 * The other (new) way of doing this is by having a ULCRowModel which is associated with the
 * table. The ULCRowModel is essentially a FormModel which can directly track the selection
 * on the UI side, *without* requiring calls to the ULC side.
 *
 * @see #buildAccountDetailsFor(IForm)
 */
public ULCComponent buildAccountDetails() {
	ULCBox accountDetailsBox= new ULCVBox(2);

	ULCBorder adapterBasedDetailsBorder= new ULCBorder(getResource("BordrAdapterBasedAccountDetails"));	
	adapterBasedDetailsBorder.add(buildAccountDetailsFor(fAccountAdapter));
	accountDetailsBox.add("ee", adapterBasedDetailsBorder);
	
	ULCBorder rowModelBasedDetailsBorder= new ULCBorder(getResource("BordrRowModelBasedAccountDetails"));		
	rowModelBasedDetailsBorder.add(buildAccountDetailsFor(fAccountRowModel));	
	accountDetailsBox.add("ee", rowModelBasedDetailsBorder);

	return accountDetailsBox;
}
/**
 * Use the given IForm to display the attributes for the account. This IForm can be the 'old' (i.e. pre-R1.6)
 * way of handling formModels, or else a ULCRowModel,
 */
public ULCComponent buildAccountDetailsFor(IForm formModel) {
	String left = "lt";
	// Will not work when a formModel is attached to a widget (refer FormModel-widgets documentation):
	// accountTypesComboBox.setNotificationPolicy(ULCComboBox.IMMEDIATE);
	// 	accountTypesComboBox.addSelectionChangedListener(new AccountTypeChangedAction(this));

	ULCBox accountDetailsBox = new ULCBox(8, 2);
	//
	accountDetailsBox.add(left, new ULCLabel(getResource("LblAccountNumber")));
	accountDetailsBox.add("et", fAccountNumberField = new ULCField(formModel, "Number", 10));
	//
	accountDetailsBox.add(left, new ULCLabel(getResource("LblAccountType")));
	ULCStringTable accountTypesTable = new ULCStringTable(Account.possibleAccountTypes());
	ULCComboBox accountTypesComboBox = new ULCComboBox(formModel, "Type", accountTypesTable);
	accountDetailsBox.add("et", accountTypesComboBox);
	//
	accountDetailsBox.add(left, new ULCLabel(getResource("LblAccountCurrency")));
	accountDetailsBox.add(left, new ULCComboBox(formModel, "Currency", new ULCStringTable(Account.possibleCurrencies())));
	//
	accountDetailsBox.add(left, new ULCLabel(getResource("LblAccountBalance")));
	accountDetailsBox.add(left, new ULCField(formModel, "Balance", getAccountBalanceValidator(), 10));
	//
	accountDetailsBox.add(left, new ULCLabel(getResource("LblAccountLimit")));
	accountDetailsBox.add(left, new ULCField(formModel, "Limit", getAccountLimitValidator(), 10));
	//
	accountDetailsBox.add(left, new ULCLabel(getResource("LblAccountInfo")));
	accountDetailsBox.add("et", buildAccountSpecificPagebook(formModel));
	//
	return accountDetailsBox;
}
public ULCComponent buildAccountPage() {
	ULCBox accountsBox= new ULCHBox(BOX_GAP);
	accountsBox.add("ee", buildAccountDetails());
	return accountsBox;
}
public ULCComponent buildAccountsPart() {
	ULCBorder accountsPartBorder= new ULCBorder(getResource("BordrAccountList"));
	ULCBox accountsPartBox= new ULCHBox(BOX_GAP);
	accountsPartBox.add("ee", buildAccountsTable());
	accountsPartBorder.add(accountsPartBox);
	return accountsPartBorder;
}
/**
 * This pagebook contains the various fields which need to be added
 * to the account details, depending upon the type of account selected.
 */
 
public ULCComponent buildAccountSpecificPagebook(IForm formModel) {

	ULCBorder savingsBorder= new ULCBorder(getResource("BordrSavingsAccount"));	
	ULCHBox savingsAccountBox= new ULCHBox(2);		
	savingsAccountBox.add("lt", new ULCLabel(getResource("LblSavingsAccountRates")));
	savingsAccountBox.skip(1);
	savingsBorder.add(savingsAccountBox);

	ULCBorder privateBorder= new ULCBorder(getResource("BordrPrivateAccount"));		
	ULCHBox privateAccountBox= new ULCHBox(2);		
	privateAccountBox.add("lt", new ULCLabel(getResource("LblPrivateAccountCharges")));
	privateAccountBox.add("lt", new ULCField(formModel, "Charges", getAccountChargesValidator(), 4));
	privateBorder.add(privateAccountBox);

	ULCBorder loanBorder= new ULCBorder(getResource("BordrLoanAccount"));		
	ULCHBox loanAccountBox= new ULCHBox(2);		
	loanAccountBox.add("lt", new ULCLabel(getResource("LblLoanAccountCredits"))); 
	ULCField numOfCreditsField= new ULCField(formModel, "NumberOfCredits", 4);
	numOfCreditsField.setEditable(false);		// Not editable, and
	numOfCreditsField.setEnabled(false);		// not enabled. 
	loanAccountBox.add("lt", numOfCreditsField);
	loanBorder.add(loanAccountBox);
	
	fAccountSpecificFields= new ULCPagebook(formModel, "Type");	// Hooked onto the adapter !
	fAccountSpecificFields.addPage("Savings Account", savingsBorder, true);
	fAccountSpecificFields.addPage("Private Account", privateBorder, true);
	fAccountSpecificFields.addPage("Loan Account", loanBorder, true);
	
	return fAccountSpecificFields;
}
/**
 * Build the table which will show the list of accounts.
 * One of the columns is editable, and also has a validator associated
 * with it.
 * The table also has a listener which will get the callback, when an element
 * (i.e. one account from the list) is selected.
 */

public ULCComponent buildAccountsTable() {
	fAccountsTable.setAutoResize(IDefaults.TABLE_AUTO_RESIZE_ALL_COLUMNS);
	fAccountsTable.addColumn("Icon", getResource("ColAccountIcon"), NOTEBOOK_COLUMN_WIDTH / 4); //Smaller column for icon
	fAccountsTable.addColumn("Number", getResource("ColAccountNumber"), NOTEBOOK_COLUMN_WIDTH / 2);
	fAccountsTable.addColumn("Type", getResource("ColAccountType"), NOTEBOOK_COLUMN_WIDTH);

	//Define the 'currency' column using a ComboBox as renderer.
	ULCComboBox comboBox = new ULCComboBox(new ULCStringTable(Account.possibleCurrencies()));
	ULCColumn ccyColumn = new ULCColumn("Currency", getResource("ColAccountCurrency"), comboBox);
	ccyColumn.setEditable(true);
	ccyColumn.setWidth(NOTEBOOK_COLUMN_WIDTH / 2);
	fAccountsTable.addColumn(ccyColumn);

	//Define the 'balance' column as editable, with input acceptable in the given range.
	ULCColumn balanceColumn = new ULCColumn("Balance", getResource("ColAccountBalance"), NOTEBOOK_COLUMN_WIDTH / 2);
	balanceColumn.setEditable(true);
	balanceColumn.setDataType(getAccountBalanceValidator());
	fAccountsTable.addColumn(balanceColumn);
	//
	fAccountsTable.addSelectionChangedListener(new SelectAccountAction(this));
	createAccountRowModel();
	createPopupMenuFor(fAccountsTable);
	//
	ULCBox accountsBox = new ULCVBox(BOX_GAP);
	accountsBox.add("ee", fAccountsTable);
	return accountsBox;
}
public ULCBox buildAccountsTableButtons() {
	ULCBox choices= new ULCVBox(4);
	
	choices.add("et", createNewAccountButton());
	choices.add("et", createDeleteAccountButton());
	choices.add("eb", buildAccountActionChoiceBox());

	return choices;
}
public ULCComponent buildAddOrModifyActionsPagebook() {
	ULCButton addNewAccountButton= new ULCButton(getResource("ButAddAccount"));
	addNewAccountButton.setEnabler(fAccountAdapter);
	addNewAccountButton.setToolTipText(getResource("TTTxtButAddAccount"));
	addNewAccountButton.addActionListener(new AddAccountAction(this));

	ULCButton modifyAccountButton= new ULCButton(getResource("ButSave"));
	modifyAccountButton.setEnabler(fAccountAdapter);
	modifyAccountButton.setToolTipText(getResource("TTTxtButSaveAccount"));
	modifyAccountButton.addActionListener(new CommitAccountModificationsAction(this));
		
	fAccountActions= new ULCPagebook();
	fAccountActions.addPage("Add", addNewAccountButton, true);
	fAccountActions.addPage("Save", modifyAccountButton, true);
	
	return fAccountActions;
}
/**
 * The various fields describing a credit, are built here. Note that the date field 
 * (to store the startDate of the credit), we use a DateValidator. This is a 'smart' 
 * object which will validate the input to this field in the UI, as soon as the user 
 * 'tabs out' of that field.
 */

public ULCComponent buildCreditDetails () {
	String left= "lt";
	String right= "et";
	ULCBox creditDetailsBox= new ULCBox(9,2);

	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditReference")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "Number", 5));
	
	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditType")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "Type", 10));
	
	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditAccount")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "AccountNumber", 5));
	
	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditInterestRate")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "InterestRate", new ULCPercentValidator(), 10));

	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditStartDate")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "StartDate", new ULCDateValidator()));
	
	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditDuration")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "Duration", 5));
	
	creditDetailsBox.add(left, new ULCLabel(getResource("LblCreditInterest")));
		creditDetailsBox.add(right, new ULCField(fCreditAdapter, "InterestTillDate", 5));
		
	return creditDetailsBox;
}
public ULCComponent buildCreditPage() {
	ULCBox creditsBox= new ULCHBox(BOX_GAP);
	creditsBox.add("ee", buildCreditDetails());
	return creditsBox;
}
/**
 * Build the 'header' for the application. This includes the icon, header text
 * and the menus.
 */
public ULCBox buildCustomerSelector() {
	ULCBox selectorBox= new ULCHBox();

	ULCButton previousCustomerButton= new ULCButton("<");
	previousCustomerButton.addActionListener(new SetPreviousCustomerAction(this));
	previousCustomerButton.setToolTipText(getResource("TTTxtButPreviousCustomer"));
	selectorBox.add("lt", previousCustomerButton);

	//Factored out this button creation, since the same button is used in the
	//toolbar.
	selectorBox.add("lt", createCustomerSelectorButton());
	
	ULCButton nextCustomerButton= new ULCButton(">");
	nextCustomerButton.addActionListener(new SetNextCustomerAction(this));
	nextCustomerButton.setToolTipText(getResource("TTTxtButNextCustomer"));
	selectorBox.add("lt", nextCustomerButton);
		
	return selectorBox;
}
public ULCComponent buildDocumentsPage() {
	ULCBox documentsBox= new ULCVBox();
	documentsBox.add("ee", createHtmlPane());
	return documentsBox;
}
/**
 * Build the 'header' for the application. This includes the icon, header text
 * and the menus.
 */
public ULCBox buildHeader() {
	ULCBox header= new ULCVBox();
	
	ULCIcon customerPicture= new ULCIcon(getClass().getResource("dukeMagnify.gif"));
	ULCLabel headerText= new ULCLabel(customerPicture);
	headerText.setToolTipText(getResource("TTTxtLabelHeader"));
	header.add("ct", headerText);

	header.add("lt", buildCustomerSelector());
		
	return header;
}
public ULCComponent buildMenus() {
	ULCMenuBar menuBar= new ULCMenuBar();
	
	//The "File" Menu
	ULCMenu file= new ULCMenu(getResource("MnuFile"));
	file.setMnemonic('F');
	ULCMenuItem menuItem= new ULCMenuItem(getResource("MnuQuit"), new ApplicationCloseAction(this));
	file.add(menuItem);
	menuBar.add(file);
	
	//The "Customer" Menu
	ULCMenu customerMenu= new ULCMenu(getResource("MnuCustomer"));
	ULCMenuItem customerMenuItem= new ULCMenuItem(getResource("MnuSelect"),
		getSearchIcon(),
		new SelectCustomerAction(this));
	customerMenuItem.setAccelerator('S',8,true);
	customerMenu.add(customerMenuItem);
	menuBar.add(customerMenu);
	
	//The "Account" Menu
	ULCMenu accountMenu= new ULCMenu(getResource("MnuAccount"));
	accountMenu.add(createNewAccountMenuItem());
	accountMenu.add(createDeleteAccountMenuItem());
	menuBar.add(accountMenu);

	//The "Look&Feel" Menu	
	menuBar.add(createLookAndFeelMenu());

	fShell.setMenuBar(menuBar);
	return menuBar;
}
/**
 * This method creates the 'Save' and 'Cancel' buttons for the name-addrss part.
 * The Save-button is attached to it's enabler, which is the nameAddress (FormModel). 
 * This means that whenever any changes occur in any of the name or address fields, 
 * the 'save' button will be enabled.
 */

public ULCComponent buildNameAddressButtons() {
	ULCBox buttonsBox = new ULCVBox(BOX_GAP);
	
	ULCButton applyButton= new ULCButton(getResource("ButSave"));
	applyButton.setEnabler(fNameAddressAdapter);
	applyButton.setEnabled(false);
	applyButton.addActionListener(new ApplyAction(this));
	buttonsBox.add("ec", applyButton);
	
	ULCButton cancelButton= new ULCButton(getResource("ButCancel"));
	cancelButton.setEnabler(fNameAddressAdapter);
	cancelButton.setEnabled(false);
	cancelButton.addActionListener(new RollbackNameAddressModificationAction(this));
	buttonsBox.add("ec", cancelButton);

	return buttonsBox;
}
/**
 * This part is made of a Box, which will in turn contain three VBox(es), 
 * one for the name-address, , one for the buttons and one for the 'header' (i.e. the
 * title, icon etc.)
 */
public ULCComponent buildNameAddressPart () {
	ULCBorder overallBorder= new ULCBorder(getResource("BordrCustomer"));	
	ULCBox overallBox= new ULCBox(2,3);

	overallBox.add("lt", buildToolBar());
	overallBox.skip(2);
	
	ULCBox nameAddressBox= new ULCBox(3,5);
	
	// First row: Name-label + FirstName + LastName
	nameAddressBox.add("lt", new ULCLabel(getResource("LblCustomerName")));
	nameAddressBox.add("lt", new ULCField(fNameAddressAdapter, "FirstName", 10));
	nameAddressBox.span(2, "et", new ULCField(fNameAddressAdapter, "LastName", 10));
	nameAddressBox.add("et", new ULCFiller());
	
	// Second row: Address-label + Street + ZipCode + City
	nameAddressBox.add("lt", new ULCLabel(getResource("LblCustomerAddress")));
	nameAddressBox.add("lt", new ULCField(fNameAddressAdapter, "Street", 10));
	nameAddressBox.add("lt", new ULCField(fNameAddressAdapter, "ZipCode", 4));
	nameAddressBox.add("et", new ULCField(fNameAddressAdapter, "City", 6));
	nameAddressBox.add("et", new ULCFiller());
	
	// Last row: Country-label + Country
	nameAddressBox.add("lt", new ULCLabel(getResource("LblCustomerCountry")));
	nameAddressBox.span(3, "et", new ULCField(fNameAddressAdapter, "Country", 10));
	nameAddressBox.add("et", new ULCFiller());

	overallBox.add("ee", buildHeader());	
	overallBox.add("ee", nameAddressBox);	
	overallBox.add("re", buildNameAddressButtons());	
	overallBorder.add(overallBox);
	return overallBorder;
}
public ULCComponent buildNotebookPart() {
	ULCNotebook notebook= new ULCNotebook(
		false, NOTEBOOK_WIDTH, NOTEBOOK_HEIGHT);
	notebook.addTab("Accounts", buildAccountPage());
	notebook.addTab("Credits", buildCreditPage());
	notebook.addTab("Documents", buildDocumentsPage());
	return notebook;
}
/**
 * Build the toolbar for the application. This will allow for adding and deleting
 * accounts for the customer.
 */
public ULCToolBar buildToolBar() {
	ULCToolBar toolBar = new ULCToolBar();

	//Use the existing buttons, but make them presentable inside a toolbar.

	ULCButton selectCustomerButton= createCustomerSelectorButton();
	selectCustomerButton.setLabel("");
	selectCustomerButton.setIcon(getSearchIcon());
	toolBar.add(selectCustomerButton);

	toolBar.add(new ULCSeparator());
	
	ULCButton newAccountButton= createNewAccountButton();
	newAccountButton.setLabel("");
	newAccountButton.setIcon(getPasteIcon());
	toolBar.add(newAccountButton);
	
	ULCButton deleteAccountButton= createDeleteAccountButton();
	deleteAccountButton.setLabel("");
	deleteAccountButton.setIcon(getCutIcon());
	toolBar.add(deleteAccountButton);
	
	return toolBar;
	
}
/**
 * Associate the table and it's model with a 'RowModel', so that the RowModel will
 * automatically track the account selections. It is essentially a FormModel, so it can then be
 * used for updating fields associated with it, without traffic going to ULC.
 *
 */
protected void createAccountRowModel() {
	fAccountRowModel = new ULCRowModel(fAccountsAdapter, fAccountsTable);
}
/**
 * Create and answer a button which, when pressed, results
 * in the list of possible customers to select from. This
 * is factored out so that this button can be used directly
 * as well as in the toolbar,
 */
protected ULCButton createCustomerSelectorButton() {
	ULCButton selectButton= new ULCButton(getResource("ButSelectCustomer"));
	selectButton.addActionListener(new SelectCustomerAction(this));
	createMultiLineToolTipFor(selectButton);
	return selectButton;
}
/**
 * Create and answer a button which, when pressed, results
 * in the deletion of the selected account. This is factored
 * out so that this button can be used directly as well as
 * in the toolbar.
 */
protected ULCButton createDeleteAccountButton() {
	ULCButton deleteButton= new ULCButton(getResource("ButDeleteAccount"));
	deleteButton.addActionListener(new DeleteAccountAction(this));
	deleteButton.setEnabled(false);
	deleteButton.setEnabler(fAccountsTable);
	deleteButton.setToolTipText(getResource("TTTxtButDeleteAccount"));	

	return deleteButton;
}
/**
 * Create and answer a menuItem which, when selected, results
 * in the deletion of the selected account.
 * This is factored out so that this menuItem can be used directly 
 * as well as in the popup menu.
 */
protected ULCMenuItem createDeleteAccountMenuItem() {
	ULCMenuItem menuItem= 
		new ULCMenuItem(
			getResource("MnuDeleteAccount"), 
			getCutIcon(), 
			new DeleteAccountAction(this));
	menuItem.setEnabler(fAccountsTable);
	return menuItem;
}
/**
 * Create an Html page in which to display the documents
 * for the receiver
 */
protected ULCComponent createHtmlPane() {
	ULCBox htmlBox = new ULCVBox();
	ULCHtmlPane htmlPane = new ULCHtmlPane("Html Pane");	

	//Create the part where the Url is entered / displayed
	htmlBox.add(createUrlInputPart(htmlPane));

	//Create the button to be able to launch the document in a separate browser
	ULCButton button = new ULCButton(getResource("ButLaunchNewBrowser"));
	button.addActionListener(new SetDocumentPageAction(getBrowserContext(), fUrlField));
	htmlBox.add("ec", button);
	
	htmlPane.setVeto(true);
	htmlPane.addLinkActivatedListener(new SetDocumentPageAction(htmlPane, fUrlField)); 
	try {
		URL u = new URL((String) fUrlField.getValue());
		htmlPane.setUrl(u);
	}
	catch (Exception e) {
		e.printStackTrace();
	}
	htmlBox.add("ee", htmlPane);

	return htmlBox;
}
/**
 * Create a menu that will allow the user to switch the Look and feel of the UI.
 * 
 */
public ULCMenu createLookAndFeelMenu() {
	ULCMenu lookAndFeelMenu = new ULCMenu(getResource("MnuLook"));
	lookAndFeelMenu.setMnemonic('L');
	//Obtain the existing LookAndFeel objects
	Vector looks = fCustInfo.getInstalledLookAndFeels();
	if (looks != null) {
		Enumeration allLooks = looks.elements();
		while (allLooks.hasMoreElements()) {
			//Add each one to the menu, alon with it's callback.
			final Look aLook = (Look) allLooks.nextElement();
			ULCMenuItem mi = new ULCMenuItem(aLook.getName());
			mi.addActionListener(new IActionListener() {
				public void actionPerformed(ULCActionEvent e) {
					fCustInfo.setLook(aLook.getClassName());
				}
			});
			lookAndFeelMenu.add(mi);
		}
	}
	return lookAndFeelMenu;
}
/**
 * A Multi-line tooltip is made by defining the end-of-line character inside the string
 * itself. For instance,
 *	TTTxtButSelectCustomer= Select a Customer\nfrom the pre-defined list
 * (as defined in the resource bundle).
 */
protected void createMultiLineToolTipFor(ULCButton selectButton) {
	selectButton.setToolTipText(getResource("TTTxtButSelectCustomer"));
}
/**
 * Create and answer a button which, when pressed, results
 * in the creation of a new account for the selected customer.
 * This is factored out so that this button can be used directly 
 * as well as in the toolbar.
 */
protected ULCButton createNewAccountButton() {
	ULCButton newButton= new ULCButton(getResource("ButNewAccount"));
	newButton.addActionListener(new NewAccountToBeAddedAction(this));
	newButton.setEnabled(true);
	newButton.setToolTipText(getResource("TTTxtButNewAccount"));

	return newButton;
}
/**
 * Create and answer a menuItem which, when selected, results
 * in the creation of a new account for the selected customer.
 * This is factored out so that this menuItem can be used directly 
 * as well as in the popup menu.
 */
protected ULCMenuItem createNewAccountMenuItem() {
	return new ULCMenuItem(
		getResource("MnuNewAccount"), 
		getPasteIcon(), 
		new NewAccountToBeAddedAction(this));
}
/** 
 * The popup menu can include 'static' as well as 'dynamic' parts. 
 */
public void createPopupMenuFor(ULCComponent component) {
	ULCMenu popupMenu = new ULCMenu("Dynamic Popup Menu");
	addStaticMenusTo(popupMenu);
	addDynamicMenusTo(popupMenu);
	component.setPopupMenu(popupMenu);
}
/**
 * Create the label and field for URL input (& display)
 */
protected ULCComponent createUrlInputPart(ULCHtmlPane htmlPane) {
	ULCBox urlBox = new ULCHBox();
	urlBox.add(new ULCLabel("URL: "));
	fUrlField.addActionListener(new SetDocumentPageAction(htmlPane, fUrlField));
	urlBox.add("ec", fUrlField);
	return urlBox;
}
/**
 * Deletion involves three steps : First the deletion of the selected
 * account from the list; and secondly, to update the (single) account
 * fields with fresh (dummy) values, and thirdly, to do the same for
 * the credit fields.
 */
public void deleteSelectedAccount () {
	if (reconfirmDeletion().equals("Yes")) {
		fAccountsAdapter.deleteAccount(fAccountAdapter.getAccount());
		fAccountAdapter.setNewDummyAccount();
		fCreditAdapter.setNewDummyCredit();
	}	
}
/**
 * Discarding the modifications means putting back the
 * original values of the selected account. So tell the
 * account adapter to do the needful...
 */
public void discardAccountModifications () {
	fAccountAdapter.accountModificationCancelled();
}
/**
 * Discarding the modifications means putting back the
 * original values of the name-address. So tell the
 * nameAddress adapter to do the needful...
 */
public void discardNameAddressModifications () {
	fNameAddressAdapter.nameAddressModificationCancelled();
}
/**
 * Build and display the widgets for the given application
 */
public void edit(ULCApplication app) {
	fShell= new ULCShell(getResource("LblCustomerInformation"), true); // Create the shell
	app.add(fShell);				// This shell is part of the overall application
	build();						// Build all the widgets into this shell
	fShell.setVisible(true);				// Now show the result.
}
public Account getAccountAtIndex(int selectedAccountIndex) {
	return (Account) fCustInfo.getAccounts().elementAt(selectedAccountIndex);
}
/**
 * Create and answer the validator to be used for checking the
 * input to the account balance field.
 */
public IDataType getAccountBalanceValidator() {
	return new ULCRangeValidator(0.0, 1000000000.0);
}
/**
 * Create and answer the validator to be used for checking the
 * input to the account charges field.
 */
public IDataType getAccountChargesValidator() {
	return new ULCRangeValidator(0.0, 75.0);
}
/**
 * Create and answer the validator to be used for checking the
 * input to the account limit field.
 */
public IDataType getAccountLimitValidator() {
	return new ULCRangeValidator(0.0, 5000.0);	//Balance should stay above this value.
}
public ULCApplication getApplication() {
	return fCustInfo.getApplication();
}
/**
 * Answer the browser context which is to be used for
 * viewing the receiver's documents inside an external Html
 * browser
 */
private ULCBrowserContext getBrowserContext() {
	return new ULCBrowserContext(fCustInfo);
}
public Customer getCurrentCustomer() {
	return fCustInfo.getCustomer();
}
public CustomerInformation getCustInfo() {
	return fCustInfo;
}
protected ULCIcon getCutIcon() {
	return new ULCIcon(getClass().getResource("cut.gif"));
}
/**
 * Get the default URL string which will be used for displaying
 * the documents for this account.
 */
protected String getDefaultUrlString() {
	return "file:///AccountDocumentsPage.html";
}
protected ULCIcon getPasteIcon() {
	return new ULCIcon(getClass().getResource("paste.gif"));
}
public String getResource(String identifier) {
	return fCustInfo.getResourceString(identifier);
}
protected ULCIcon getSearchIcon() {
	return new ULCIcon(getClass().getResource("search.gif"));
}
public ULCShell getShell() {
	return fShell;
}
/**
 * Answer boolean whether the given customer is a different
 * one from the one being currently held on to by the
 * receiver.
 * For now, we compare just the lastNames right here, instead
 * of over-riding the equals() method for the customer.
 */
public boolean isDifferentCustomer(Customer customer) {
	return !(customer.equals(fCustInfo.getCustomer()));
}
/**
 * Since a new account is now going to be added, clear up
 * the account details fields for that input. Also, have the
 * corresponding page (showing the 'Add' button) set up in the
 * pagebook, and de-select the currently selected account from 
 * the accounts table. Also note the focus-setting to the account
 * number field.
 */

public void performNewAccountPreprocessing() {
	fAccountAdapter.setNewDummyAccount();
	fCreditAdapter.setNewDummyCredit();
	setAccountToBeAddedPage();
	fAccountsTable.setSelectedIndex(-1);
	fAccountNumberField.requestFocus();
}
/**
 * Use an Alert to confirm if the user really wants to exit.
 */
public void quit() {
	fCustInfo.beep();
	ULCAlert alert = new ULCAlert(fShell, "Please Confirm", "Do you really want to quit?", "Yes", "No");
	if (alert.ask().equals("Yes"))
		fCustInfo.terminateCustInfo();
}
/**
 * Use an 'alert' to reconfirm that the user indeed wishes 
 * to delete the given account. 
 * Note that a 'beep' done here can only be called on a 
 * ULCContext type of object...
 */
public String reconfirmDeletion() {
	ULCAlert reconfirm= new ULCAlert(fShell, "Please Confirm",
					"Do you really want to delete ?", "Yes", "No");
	fCustInfo.beep();
	return reconfirm.ask();
}
/**
 * Refresh the accountAdapter, to show any new values for the
 * individual account fields
 */
public void refreshAccountAdapter () {
	fAccountAdapter.notifyChange();
}
/**
 * Refresh all the different 'parts' of the receiver, to display
 * the relevant details for the given (newly selected) customer.
 */
public void refreshForNewCustomer(Customer customer) {
	// first set the customer in the model itself
	fCustInfo.setCustomerSelection(customer);

	// Now update the name-address fields
	fNameAddressAdapter.refreshForNewCustomer(customer);
	fNameAddressAdapter.notify(IDefaults.FORM_MODEL_CHANGED, null);

	// Now update the accounts fields. 
	fAccountsAdapter.setAccounts(fCustInfo.getAccounts());


	// And finally put dummy values into the account and credit forms.
	fAccountAdapter.setNewDummyAccount();
	fCreditAdapter.setNewDummyCredit();
}
/**
 * Saving the modifications involves two steps : First the storing of the 
 * changed values for the selected account :
 * and secondly, to inform the list that this account has been changed...
 * Also, note that the receiver is being passed in the flush call. This is in
 * order to enable the accountAdapter to show any alerts that it needs to,
 * while validating the user-inputs.
 *
 * @see #saveNameAddressModifications()
 */
public void saveAccountModifications() {
	fAccountAdapter.validateAndSaveChanges();	
	fAccountsAdapter.accountModified(fAccountAdapter.getAccount());
	setAccountSelectionIrrespective(fAccountAdapter.getAccount());
}
/**
 * Saving the changes means updating the real model with the
 * data modified in the view. The way this happens in ULC is :
 * (i) The adapter (a FormModel) is sent the flush() message
 * (ii) This message tells the UI that the model needs the updated
 * 		values (which have been entered by the user in the UI !)
 * (iii) The UI then sends each of the values (i.e. for each of the
 *			fields in the formModel) in sequence, and the formModel has
 *			to 'grab' them and incorporate them.
 *	(iv)	This last part is done by over-writing the setValueAt(object, key)
 *			method in the formModel.
 *
 * @see NameAddressAdapter#setValueAt(Object,String)
 *
 * Additionally, the changes done to the customer need to be stored in the
 * testCustomers list correctly, so that these can show up the next time the
 * user selects the same customer (in the same run).
 *
 * A bit of a hack out here, since the new values (after the flush) go into
 * the nameAddressAdapter, we need to pass this adapter to the customersAdapter
 * in order that the latter can update itself with the new values...
 */
public void saveNameAddressModifications() { 
	fNameAddressAdapter.saveInput();
	fCustomersAdapter.customerModified(fCustInfo.getCustomer(), fNameAddressAdapter);
	fNameAddressAdapter.refresh();
}
/**
 * Now the receiver knows exactly which index has been selected.
 * So it first finds out the account at this index, and uses this
 * for updating both the account and the credit details.
 *
 * And of course, don't do this if the index is undefined (which
 * happens at startup time...
 */
public void setAccountSelection (int selectedAccountIndex) {
	if (selectedAccountIndex != UNDEFINED_SELECTION_INDEX) {
		Account newlySelectedAccount= getAccountAtIndex(selectedAccountIndex); 
		setAccountSelection(newlySelectedAccount);
		setCreditSelection(newlySelectedAccount);
	}	
}
/**
 * Now the receiver knows exactly which element has been selected.
 * So it uses the adapter for updating the data, to display the
 * selected account. Also, the pagebook page is changed here, so 
 * that any changes made to this account can be saved.
 */
public void setAccountSelection(Account account) {
	fAccountAdapter.setAccount(account);
	setAccountSelectionPostProcessing(account);
}
/**
 * Normally, the account-setting process checks if the account
 * has really changes (in order to prevent excessive updates).
 * However, if we need to 'force' it to be set, even if notionally
 * it is the same value, this method is required.
 */
public void setAccountSelectionIrrespective(Account account) {
	fAccountAdapter.basicSetAccount(account);
	setAccountSelectionPostProcessing(account);
}
/**
 * Perform any steps required for ensuring that the whole
 * UI is in a 'consistent' state. Namely that all the buttons
 * are enabled-disabled as expected; the account selected in 
 * the table is the same as the one displayed in the detail, etc..
 */
public void setAccountSelectionPostProcessing (Account account) {
	setAccountSpecificDetails(account);	 
	setAccountToBeModifiedPage();

	/* setAccountsTableSelectedAccount(account); */
	// Creates Stack Overflow in R1.3.3, since the set method now
	// has a callback. Hence moved into addNewAccount(), which is
	// where it should have been in the first place.
}
/**
 * From the pageBook, set the page which contains
 * the (additional) details, if any, for the selected
 * account type
 */
public void setAccountSpecificDetails(Account account) {
	fAccountSpecificFields.setActivePage(account.getType());
}
/**
 * Set the given account as the selected account in the
 * accounts table.
 */
public void setAccountsTableSelectedAccount(Account account) {
	int index= fAccountsAdapter.getIndex(account);
	fAccountsTable.setSelectedIndex(index);
}
/**
 * From the pageBook, set the page which allows
 * for adding a new account.
 */
public void setAccountToBeAddedPage() {
	fAccountActions.setActivePage("Add");
}
/**
 * From the pageBook, set the page which allows
 * for saving the modifications on the selected account.
 */
public void setAccountToBeModifiedPage() {
	fAccountActions.setActivePage("Save");
}
/**
 * Now the receiver knows exactly which element has been selected.
 * So it uses the adapter for updating the data.
 */
public void setCreditSelection(Account account) {
	fCreditAdapter.setCreditForAccount(account);
}
/**
 * The user has selected a different customer in the UI.
 * refresh all the views...
 * First hide the selection window, and then do the refreshing
 * if the customer selected is indeed a different one...
 */
public void setCustomerSelection(Customer customer) {
	fCustomerSelector.hide();
	if (isDifferentCustomer(customer))
		refreshForNewCustomer(customer);
}
/**
 * The user has opted to see the next customer.
 * Set and Refresh all the views...
 */
public void setNextCustomer() {
	refreshForNewCustomer(fCustomersAdapter.getNextCustomer(fCustInfo.getCustomer()));
}
/**
 * The user has opted to see the previous customer.
 * Set and Refresh all the views...
 */
public void setPreviousCustomer() {
	refreshForNewCustomer(fCustomersAdapter.getPreviousCustomer(getCurrentCustomer()));
}
public void startCustomerSelector() {
	if (fCustomerSelector == null)
		fCustomerSelector= new CustomerSelector(this);
	fCustomerSelector.select();
}
/**
 * From the pageBook, set the page which contains
 * the (additional) details, if any, for the selected
 * account type
 */
public void updateAccountSpecificPagebook() {
	setAccountSpecificDetails(fAccountAdapter.getAccount());
}
}
