package com.ibm.ulc.comm;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.net.*;
import java.io.*;
import java.util.Vector;
import java.util.Enumeration;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;

/**
 * A connection with an associated object space. Objects in the object
 * space conform to ICallable.
 * An ORBConnection can optionally send interesting events to a registered
 * listener.
 *
 * @see Registry
 * @see ICallable
 * @see IORBListener
 */
public class ORBConnection extends UlcConnection {
	private Vector fListener = null;
	private Registry fRegistry = null;
	private ORBRequest fLastRequest = null;
	private IAnythingWriter fWriter = null;
	private IAnythingReader fReader = null;
	private boolean fDispose;
	private long fContextId = 0;
	private ORBRequest fCurrentRequest = null;
	private Thread fCurrentRequestThread = null;
	private int fRequestId = 0;
/**
 * ORBConnection constructor.
 *
 * @param transport com.ibm.ulc.comm.UlcTransport
 * @param rp com.ibm.ulc.comm.IRequestProcessor
 * @param connId java.lang.String
 * @param clientData java.lang.String
 * @param dispose boolean
 * @param r Registry
 */
public ORBConnection(UlcTransport transport, IRequestProcessor rp,
						   String connId, String clientData,
						   boolean dispose, Registry r) {
	super(transport, rp, connId, clientData);
	fDispose= dispose;
	fRegistry= r;
}
/**
 * ORBConnection constructor.
 *
 * @param transport com.ibm.ulc.comm.UlcTransport
 * @param rp com.ibm.ulc.comm.IRequestProcessor
 * @param dispose boolean
 * @param r Registry
 */
public ORBConnection(UlcTransport transport, IRequestProcessor rp,
						   boolean dispose, Registry r) {
	this(transport, rp, null, null, dispose, r);
}
/**
 * ORBConnection constructor.
 *
 * @param urlString java.lang.String
 * @param rp com.ibm.ulc.comm.IRequestProcessor
 * @param connId java.lang.String
 * @param clientData java.lang.String
 * @param dispose boolean
 * @param r Registry
 */
public ORBConnection(String urlString, IRequestProcessor rp,
						   String connId, String clientData,
						   boolean dispose, Registry r) {
	super(urlString, rp, connId, clientData);
	fDispose= dispose;
	fRegistry= r;
}
/**
 * ORBConnection constructor.
 *
 * @param urlString java.lang.String
 * @param rp com.ibm.ulc.comm.IRequestProcessor
 * @param dispose boolean
 * @param r Registry
 */
public ORBConnection(String urlString, IRequestProcessor rp,
						   boolean dispose, Registry r) {
	this(urlString, rp, null, null, dispose, r);
}
	public void addListener(IORBListener listener) {
		if (fListener == null)
			fListener= new Vector();
		fListener.addElement(listener);
		if (listener != null)
			listener.addedConnection(this);
	}
/**
 * Creates a request to be used with this connection.
 * @returns an ORBRequest
 * @see UlcConnection
 */
public Request createRequest() {
	return new ORBRequest(this);;
}
/**
 * Disposing the object registry.
 */
public void disposeRegistry() {
	if (fRegistry != null) {
		fRegistry.shutdown();
		fRegistry= null;
		if (fDebug)
			System.out.println("ORBConnection.disposeRegistry executed");
	}			
}
/**
 * Finds a given object in this connection's registry.
 */
public ICallable find(int oid) {
	if (fRegistry != null)
		return fRegistry.find(oid);
	return null;
}
/**
 * Creates an AnythingReader used for streaming Anythings.
 * Remembers the received format.
 */
public IAnythingReader getAnythingReader(InputStream is) {
	if (fReader == null) {
		int c;
		try {
			c= is.read();
		} catch (IOException e) {
			fReader= new AnythingReader();
			return fReader;
		}
		if (c == 'V') {
	    	fReader= new AnythingReader2(c);
		} else {
	 		fReader= new AnythingReader(c);
		}
	}
	return fReader;
}
/**
 * Creates an AnythingReader used for streaming Anythings.
 * Remembers the received format.
 */
public IAnythingReader getAnythingReaderEx(InputStream is) throws IOException {
	if (fReader == null) {
		int c= is.read();
		if (c == 'V') {
	    	fReader= new AnythingReader2(c);
		} else {
	 		fReader= new AnythingReader(c);
		}
	}
	return fReader;
}
/**
 * Creates an AnythingWriter used for streaming Anythings.
 * What kind of writer to use depends on the format of received events.
 */
public IAnythingWriter getAnythingWriter() {
	if (fWriter == null) {
		if (fReader != null)
			fWriter= fReader.createWriter();
		else
			fWriter= new AnythingWriter2();
	}
	return fWriter;
}
public long getContextId() {
	return fContextId;
}
public ORBRequest getCurrentRequest() {
	return fCurrentRequest;
}
public int getCurrentRequestId() {
	if (fCurrentRequest == null)
		return -1;
	else
		if (Thread.currentThread() != fCurrentRequestThread)
			return -1;
		else
			return fCurrentRequest.getRequestId();
}
public String getLabel() {
	UlcTransport transport= getTransport();
	if (transport != null) {
		String url= transport.getLocalUrlString();
		return url;
	}
	return "***no label***";
}
/**
 * Returns the connection's registry.
 */
public Registry getRegistry() {
	return fRegistry;
}
/**
 * Handles receivedRequest notification by informing the listener.
 */
public void receivedRequest(int target, String request, Anything args) {
	if (fListener != null) {
		Enumeration e= fListener.elements();
		while (e.hasMoreElements()) {
			IORBListener l= (IORBListener) e.nextElement();
			if (l != null)
				l.receivedRequest(this, target, request, args);
		}
	}	
}
/**
 * Register an object in this object space with a specific ID.
 */
public void register(int oid, ICallable c) {
	if (fRegistry != null)
		fRegistry.register(oid, c);
	//else
	//	trouble("register", "no registry");
}
/**
 * Register an object in this connection's object space.
 * Returns its ID.
 */
public int register(ICallable c) {
	if (fRegistry != null)
		return fRegistry.register(c);
	//trouble("register", "no registry");
	return 0;
}
/**
 * Create an ORBRequest for the given target and arguments and send it
 * to the other communication end point.
 */
public void send(int oid, String request, Anything args) {
	if (request.equals("event")) {
		String t = args.get("type", null);
		if (t != null && t.equals("moved")) {
			fLastRequest = new ORBRequest(getAnythingWriter(), oid, request, args, getContextId(), this);
			return;
		}
	}
	if (fLastRequest != null) {
		fLastRequest.setRequestId(fRequestId++);
		send(fLastRequest);
		fLastRequest = null;
	}
	ORBRequest orbRequest = new ORBRequest(getAnythingWriter(), oid, request, args, getContextId(), this);
	orbRequest.setReplyId(getCurrentRequestId());
	orbRequest.setRequestId(fRequestId++);
	//System.out.println("Send Req id: " + orbRequest.getRequestId() + " reply id: " + orbRequest.getReplyId());
	send(orbRequest);
	if (fListener != null) {
		Enumeration e = fListener.elements();
		while (e.hasMoreElements()) {
			IORBListener l = (IORBListener) e.nextElement();
			if (l != null)
				l.sentRequest(this, oid, request, args);
		}
	}
}
public void setContextId(long contextId) {
	fContextId = contextId;
}
public void setCurrentRequest(ORBRequest request) {
	fCurrentRequest = request;
	if (request == null)
		fCurrentRequestThread = null;
	else
		fCurrentRequestThread = Thread.currentThread();
}
/**
 * Handles the shutdown of the connection by disposing the
 * object registry.
 */
protected void shutdownHook() {
	if (fDispose)
		disposeRegistry();
	if (fListener != null) {
		Enumeration e= fListener.elements();
		while (e.hasMoreElements()) {
			IORBListener l= (IORBListener) e.nextElement();
			if (l != null)
				l.removedConnection(this);
		}
		fListener.removeAllElements();
		fListener= null;
	}
	fLastRequest= null;
	fWriter= null;
	fReader= null;
}
/**
 * Remove an object from this connection's object space.
 */
public void unregister(int oid) {
	if (fRegistry != null)
		fRegistry.unregister(oid);
	//else
	//	trouble("unregister", "no registry");
}
}
