package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import java.awt.Color;

/**
 * Internal implementation detail of ULCTable and ULCList.
 * @see ULCTableModel
 */
public abstract class ULCTableList extends ULCAbstractList {
	/**
	 * Any valid ULCTrigger object which will be triggered in the UI when 
	 * this widgets actionPerformed method is called
	 * @serial	 
	 */
	protected ULCTrigger fTrigger = null;
	/**
	 * Internal flag to determine if some debug statements should be printed out.
	 * @serial	 
	 */
	protected static final boolean DEBUG = false;
/**
 * Constructs a new ULCTableList which displays values from the given ULCTableModel.
 *
 */
public ULCTableList() {
}
/**
 * Constructs a new ULCTableList which displays values from the given tableModel.
 *
 * @param tableModel 		The ULCAbstractTableModel that serves as the data source.
 */
public ULCTableList(ULCAbstractTableModel tableModel) {
	super(tableModel);
}
/**
 * Constructs a new ULCTableList which displays values from the given tableModel.
 *
 * @param tableModel 	The ULCAbstractTableModel that serves as the data source.
 * @param width			The width in pixels for this table.
 * @param heightInRows	The height specified as number of visible rows for this table.
 */
public ULCTableList(ULCAbstractTableModel tableModel, int width, int heightInRows) {
	super(tableModel, width, heightInRows);
}
/**
 * Constructs a new ULCTableList which displays values from the given tableModel.
 *
 * @param tableModel	The ULCAbstractTableModel that serves as the data source.
 * @param width			The width in pixels for this table.
 * @param heightInRows	The height specified as number of visible rows for this table.
 * @param selectionMode	The selection mode which can be one of:
 * <pre>
 *  					IDefaults.LIST_SINGLE_SELECTION
 *  					IDefaults.LIST_SINGLE_INTERVAL_SELECTION
 *  					IDefaults.LIST_MULTIPLE_INTERVAL_SELECTION
 * </pre>
 */
public ULCTableList(ULCAbstractTableModel tableModel, int width, int heightInRows, int selectionMode) {
	super(tableModel, width, heightInRows, selectionMode);
}
/**
 * Registers the given listener to begin receiving notifications
 * when selecting rows in the table or list.
 *
 * @param listener	The object interested in my selectionChangedEvents.
 */
public void addSelectionChangedListener(ISelectionChangedListener listener) {
	internalAddListener("select", listener);
}
/**
 * Return the index of the row within the item list
 *
 * @return index The <code>int</code> index within the item list
 */

public int convertFromTableModelIndex(int index) {
	int oid = internalGetModel().getOidForIndex(index);
	return getItemList().getIndexForOid(oid);
}
/**
 * Return the index of the row within the table model.
 *
 * @return index The <code>int</code> index within the table model
 */
public int convertToTableModelIndex(int index) {
	int oid = getItemList().getOidForIndex(index);
	int tableModelIndex = internalGetModel().getIndexForOid(oid);
	return tableModelIndex;
}
/*
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void distributeSelectionChangeEventToListeners(String type) {
	distributeToListeners("select", new ULCSelectionChangedEvent(this, type, getSelectedIndices()));
}
/**
 * Requests my UI proxy to ensure that the row at <code>index</code> is visible.
 * Note: This will upload the item even if it already has been uploaded
 *
 * @param index	The index of the row to be made visible.
 */
public void ensureIndexIsVisible(int index) {
	 // skip if node unavailable
	if (index >= 0 && index < getItemList().getRowCount()) {
		if (getItemList() instanceof ULCItemListAbstract) {
			/*
			 * Upload the item to be sure it is there
			 * This only works for ULCItemListAbstract because getData
			 * is not part of the interface and we don't want to break the
			 * interface.
			 */
			Anything rows= new Anything();
			rows.put("s", index);
			rows.put("e", index);
			((ULCItemListAbstract) getItemList()).getData(rows);
		}
		sendUI("ensureItemIsVisible", new Anything(getItemList().getOidForIndex(index)));
	}	
}
/**
 * Requests my UI proxy to ensure that the row at <code>index</code> in my TableModel is visible.
 *
 * @param index	The index of the row to be made visible.
 */
public void ensureTableModelIndexIsVisible(int index) {
	ensureIndexIsVisible(convertFromTableModelIndex(index));
}
/**
 * Answer the default item list for the receiver, 
 * which is the default IndexedItemList of the receiver's model.
 *
 */
protected IItemList getDefaultItemList() {
	return getModel().getIndexedItemList();
}
/**
 * Gets the <code>ULCTableModel</code> that will serve as my data source.
 *
 * @return	The <code>ULCTableModel</code>
 */
public ULCAbstractTableModel getModel() {
	return internalGetModel();
}
/**
 * Get the index of the selected row 
 *
 * @return index The index of the row selected or -1.
 */
public int getSelectedIndex() {
	Vector indices = getSelectedIndices();
	if (indices.size() > 0)
		return ((Integer) indices.elementAt(0)).intValue();
	return -1;
}
/**
 * Return the selected rows as the collection of indices
 *
 * @return indices The <code>Vector</code> of indices selected.
 */
public Vector getSelectedIndices() {
	int[] oids = getSelectedOids();
	Vector answer = new Vector();
	for (int i = 0; i < oids.length; i++) {
		int oid = getItemList().getIndexForOid(oids[i]);
		answer.addElement(new Integer(oid));
	}
	return answer;
}
/**
 * Return the selected rows as the collection of indices in the TableModel's row collection
 *
 * @return indices The <code>Vector</code> of indices selected.
 */
public Vector getSelectedTableModelIndices() {
	Vector selectedIndices = getSelectedIndices();
	if (selectedIndices.size() == 0)
		return selectedIndices;
	Vector answer = new Vector();
	for (int i = 0; i < selectedIndices.size(); i++) {
		int tableModelIndex = convertToTableModelIndex(((Integer) selectedIndices.elementAt(i)).intValue());
		answer.addElement(new Integer(tableModelIndex));
	}
	return answer;
}
/**
 * Gets the <code>ULCTableModel</code> that will serve as my data source.
 *
 * @return	The <code>ULCTableModel</code>
 */
public IIndexedTableModel getTableModel() {
	return (IIndexedTableModel) getModel();
}
/**
 * Returns the configured ULCTrigger object which will be triggered in the UI when 
 * this widget's actionPerformed method is called.
 *
 * @return ULCTrigger	 
 */
public ULCTrigger getTrigger() {
	return fTrigger;
}
private Anything prepareSelectedIndices() {
	Anything a = new Anything();
	Anything rowIds = new Anything();
	for (int i = 0; i < getSelectedOids().length; i++) {
		int index = getItemList().getIndexForOid(getSelectedOids()[i]);
		Anything any = new Anything(index);
		a.append(any);
		rowIds.append(new Anything(getSelectedOids()[i]));
	}
	Anything selection = new Anything();
	selection.put("s", a);
	selection.put("rowids", rowIds);
	return selection;
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive events. 
 *
 * @param listener	The object that was registered for my selectionChanged events
 */
public void removeSelectionChangedListener(ISelectionChangedListener listener) {
	internalRemoveListener("select", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fInitiallySelectedItems == null && fInitiallySelectedIndices != null && !fInitiallySelectedIndices.isEmpty()) {
		setSelectedIndices(fInitiallySelectedIndices);
		fInitiallySelectedIndices= null;
	}
	if (fSelectedOids != null)
		a.put("selectedIndices", prepareSelectedIndices());
	if (fTrigger != null)
		a.put("trigger", ((ULCProxy) fTrigger).getRef(fContext));
}
/**
 *  Update the UI with the newly selected items.
 *
 */

protected void sendSelectedItems() {
	sendUI("setSelectedIndices", prepareSelectedIndices());
}
/**
 * Set the <code>IIndexedItemList</code> that will serve as intermediary between
 * the receiver and its tableModel.
 *
 * IMPORTANT: If the itemList passed does not know its tableModel, this method will throw
 * an exception. When connecting widgets to an itemList visually the connection 
 * between the itemList and its tableModel must be fired prior to the connection to the
 * widget. The sequence of connection execution can be modified in the builder.
 *
 * if <code>itemList</code> is null, the default itemList of the receiver's tableModel
 * will be used by default.
 *
 * @see #getItemList
 *
 * @param itemList	The <code>IIndexedItemList</code> to be used from now on
 */
public void setItemList(IIndexedItemList itemList) {
	super.setItemList(itemList);
}
/**
 * Set the <code>IItemList</code> that will serve as intermediary between
 * the receiver and its tableModel.
 *
 * IMPORTANT: If the itemList passed does not know its tableModel, this method will throw
 * an exception. When connecting widgets to an itemList visually the connection 
 * between the itemList and its tableModel must be fired prior to the connection to the
 * widget. The sequence of connection execution can be modified in the builder.
 *
 * if <code>itemList</code> is null, the default itemList of the receiver's tableModel
 * will be used by default.
 *
 * @see #getItemList
 *
 * @param itemList	The <code>IItemList</code> to be used from now on
 */
public void setItemList(IItemList itemList) {
	setItemList((IIndexedItemList) itemList);
}
/**
 * Set the <code>IIndexedTableModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param tableModel	The <code>IIndexedTableModel</code>
 */
public void setModel(IIndexedTableModel tableModel) {
	internalSetModel((ULCAbstractTableModel) tableModel);
}
/**
 * Set the selected row at <code>index</code>
 *
 * @param index The index of the row to be selected.
 */
public void setSelectedIndex(int x) {
	Vector indices = new Vector(1);
	indices.addElement(new Integer(x));
	setSelectedIndices(indices);
}
/**
 * Set the selected rows to be the collection of indices
 *
 * @param indices The vector of indices to be selected.
 */
public void setSelectedIndices(Vector indices) {
	if (indices == null)
		setSelectedItems(null);
	else {
		Vector items= new Vector();
		for (int i= 0; i < indices.size(); i++) {
			int index= ((Integer) indices.elementAt(i)).intValue();
			int oid= getItemList().getOidForIndex(index);
			if (oid == -1) {
				// if one index is not known yet then all must be pending
				fInitiallySelectedIndices= (Vector) indices.clone();
				fInitiallySelectedItems= null;
				return;
			}
			Object object= getItemList().getRowAt(index);
			items.addElement(object);
		}
		setSelectedItems(items);
	}
}
/**
 * Return the selected rows as the collection of indices in the TableModel's row collection
 *
 * @return indices The <code>Vector</code> of indices selected.
 */
public void setSelectedTableModelIndices(Vector indices) {
	Vector selectedIndices = new Vector();
	for (int i = 0; i < indices.size(); i++) {
		int oid = internalGetModel().getOidForIndex(((Integer) indices.elementAt(i)).intValue());
		selectedIndices.addElement(new Integer(getItemList().getIndexForOid(oid)));
	}
	setSelectedIndices(selectedIndices);
}
/**
 * Set the <code>IIndexedTableModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param tableModel	The <code>IIndexedTableModel</code>
 */
public void setTableModel(IIndexedTableModel tableModel) {
	internalSetModel((ULCAbstractTableModel) tableModel);
}
/**
 * Set the <code>ULCTableModel</code> that will serve as my data source.
 * And the <code>ULCItemListAbstract</code> which will be my itemList.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param tableModel	The <code>ULCTableModel</code>
 * @param itemList		The <code>ULCItemListAbstract</code>
 */
private void setTableModelItemList(ULCAbstractTableModel tableModel, ULCItemListAbstract itemList) {
	setModelItemList(tableModel, itemList);
}
/**
 * Sets the configured ULCTrigger which will be triggered in the UI when 
 * this widgets actionPerformed method is called
 *
 * @param ULCTrigger	 
 */
public void setTrigger(ULCTrigger trigger) {
	if (trigger != fTrigger) {
		fTrigger = trigger;
		sendUI("setTrigger", fTrigger);
	}
}
}
