package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
/** 
 * An event that characterizes a change in the current selection.
 * This is either done by specifying the first and last index inclusive
 * (fFirstIndex <= fLastIndex) OR by a vector of
 * indices.
 *
 * @see ISelectionChangedListener 
 */
public class ULCSelectionChangedEvent extends ULCEvent {
	/** 
	 * The index of the first row/element selected that this event represents.
	 * Always valid - even if vector is used
	 * @serial	 
	 */
	protected int fFirstIndex= -1;
	/** 
	 * The index of the last row/element selected that this event represents.
	 * Always valid - even if vector is used
	 * @serial	 
	 */
	protected int fLastIndex= -1;
	/** 
	 * An optional command that describes this event.
	 * @serial	 
	 */
	protected String fCmd= null;
	/** 
	 * The currently selected indices. This value
	 * will be null if the constructor is used that
	 * specifies the range (fFirstIndex, fLastIndex)
	 * @serial	 	 
	 */
	protected Vector fIndices= null;
/**
 * Construct a ULCSelectionChangedEvent for the specified source object.
 */
public ULCSelectionChangedEvent(Object source) {
	super(source);
}
/**
 * Construct a ULCSelectionChangedEvent for the specified selection range.
 *
 * @param source 		The source of this event.
 * @param firstIndex 	start of range.
 * @param lastIndex 	end of range.
 */
public ULCSelectionChangedEvent(Object source, int firstIndex, int lastIndex) {
	super(source);
	fFirstIndex= Math.min(firstIndex, lastIndex);
	fLastIndex= Math.max(firstIndex, lastIndex);
}
/**
 * Construct a ULCSelectionChangedEvent for the specified selection range and command.
 *
 * @param source 		The source of this event.
 * @param cmd 			Specifies type of event.
 * @param firstIndex	start of range.
 * @param lastIndex 	end of range.
 */
public ULCSelectionChangedEvent(Object source, String cmd, int firstIndex, int lastIndex) {
	this(source, firstIndex, lastIndex);
	fCmd= cmd;
}
/**
 * Construct a ULCSelectionChangedEvent for the specified selection items.
 *
 * @param source The source of this event.
 * @param cmd The optional command associated with this event.
 * @param selectedItemIndices 	Vector of selected indices.
 */
public ULCSelectionChangedEvent(Object source, String cmd, Vector selectedItemIndices) {
	this(source, selectedItemIndices);
	fCmd= cmd;
}
/**
 * Construct a ULCSelectionChangedEvent for the specified selection items.
 *
 * @param source 				The source of this event.
 * @param selectedItemIndices 	Vector of selected indices.
 */
public ULCSelectionChangedEvent(Object source, Vector selectedItemIndices) {
	super(source);
	fIndices= selectedItemIndices;
	setFirstAndLastIndex();
}
/**
 * Dispatch this event to the specified object if it implements the ISelectionChangedListener interface.
 *
 * @param listener the <code>IListener</code> that is interested in this event.
 */
public void dispatch(IListener listener) {
	if (listener instanceof ISelectionChangedListener) {
		ISelectionChangedListener sl = (ISelectionChangedListener) listener;
		sl.selectionChanged(this);
	}
}
/**
 * Returns the event's optional command.
 *
 * @return The <code>String</code> command
 */
public String getCmd(String defaultString) {
	if (fCmd == null)
		return defaultString;
	else
		return fCmd;
}
/**
 * Returns the first index of the selection range.
 *
 * @return the <code>int</code> value that represents the first row selected.
 */
public int getFirstIndex() {
	return fFirstIndex;
}
/**
 * Returns all the indices of the selection.
 * If the vector is <code>null</code> the selection
 * is described as a range between fFirstIndex and
 * fLastIndex.
 *
 * @return  Vector of selected indices.
 */
public Vector getIndices() {
	return fIndices;
}
/**
 * Returns the last index of the selection range.
 *
 * @return the <code>int</code> value that represents the last row selected.
 */
public int getLastIndex() {
	return fLastIndex;
}
/**
 * Compute and set <code>fFirstIndex</code> and <code>fLastIndex</code>
 */
protected void setFirstAndLastIndex() {
	if (fIndices == null || fIndices.isEmpty())
		return;
		
	int index;
	fFirstIndex= Integer.MAX_VALUE;
	fLastIndex= Integer.MIN_VALUE;		
	for (Enumeration enum= fIndices.elements(); enum.hasMoreElements();) {
		index= ((Integer) enum.nextElement()).intValue();
		fFirstIndex= Math.min(fFirstIndex, index);
		fLastIndex= Math.max(fLastIndex, index);
	}
}
}
