package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import java.net.*;
import java.io.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.base.IDefaults;

/**
 * This class represents the context in which an ULC application runs.
 * It sets up and maintains the Connection object to communicate with the UIEngine
 * and it knows all the top level ULCProxies (i.e. the roots), which have to be transmitted to
 * the UIEngine.
 * All ULC applications have to subclass ULCContext and implement the <code>start</code>
 * method. Typically the code contained in 'main' goes there.
 */
abstract public class ULCContext extends UlcObject implements Serializable, ICallable, IConnectionController, IDefaults {
	/**
	 * The Connection on which I am active
	 * @serial
	 */
	private ORBConnection fConnection = null;
	/**
	 * @serial
	 */
	private Vector fRoots = null;
	/**
	 * If true this context will attempt to reconnect to the UI when the connection goes down.
	 * @serial
	 */
	private boolean fAutoReconnect = true;
	/**
	 * The list of all active widgets within this context.
	 * @serial
	 */
	private Registry fRegistry;
	/**
	 * @serial
	 */
	private long fSynchId = 4711;
	/**
	 * @serial
	 */
	private boolean fGotReply;
	/**
	 * @serial
	 */
	private int fSeed;
	/**
	 * @serial
	 */
	private String fUrlString = null;
	/**
	 * @serial
	 */
	private ThreadPool fRequestProcessor;
	/**
	 * @serial
	 */
	protected Locale fUILocale = null;
	/**
	 * @serial
	 */
	private static int fgSeed = 1;
	/**
	 * @serial
	 */
	private ResourceBundle fResourceBundle = null;
	/**
	 * @serial
	 */
	private Vector fLookAndFeels; //Vector of available LookAndFeel objects
	/**
	 * Current LAF class name of the UI Engine
	 * @serial
	 */
	private String fLook;

	/**
	 * @serial
	 */
	private String fUserParameter = null;
	/**
	 * If true this Context will exit when the last Application is terminated.
	 * @serial
	 */
	private static boolean fgExitOnLastApplicationClosed = true;
	/**
	 * If not null all requests will be kept pending till the
	 * stopBatching request is called
	 * @serial
	 */
	private Anything fBatchRequest = null;
	/**
	 * Contains any user specified properties.
	 * @serial
	 */
	private UlcHashtable fProperties = null;
	/**
	 * Tacks the last contextid used.
	 * @serial
	 */
	private static long fgContextId = 10000L;
	/**
	 * Temporary field to hold synch answers which are not stored in a field
	 * @since R3.1
	 * @serial
	 */
	private Anything fTempSyncAnswer= null;
/**
 * Construct a new context object and register it in its own registry so that 
 * the UIEngine can direct messages to it.
 */
public ULCContext() {
	fRegistry = new Registry(null);
	fRegistry.register(Common.FACTORY_ID, this);
	Server.addConnectionController(this);
}
/**
 * Adds a proxyable object to the set of known roots.
 *
 * @param proxy The object that implements <code>IProxy</code>
 */
public void add(IProxy proxy) {
	getRoots().addElement(proxy);
}
/**
 * The UI has notified us that a connection has been added.
 * This event will be received only if <code>addUIConnectionListener</code>
 * has been called once.
 * Default behaviour is to do nothing.
 * The primary use of this event is to track all connections made to the UI Engine
 * that this context is connected to.
 *
 * @param host			The host <code>String</code> that made the connection to the UI.
 * @param port			The port <code>int</code> on which the connection was made.
 * @param clientData	The optional <code>String</code> data that can be passed when requesting a connection.
 * @param connId		The unique connectionId within the UI Engine that identifies this connection.
 *
 * @see #addUIConnectionListener()
 */
protected void addedUIConnection(String host, int port, String clientData, String connId) {
}
/**
 * Send a request to the UI to be notified when connections are added or removed.
 *
 * @see #addedUIConnection
 * @see #removedUIConnection
 */
public void addUIConnectionListener() {
	send(Common.FACTORY_ID, "addUIConnectionListener", null);
}
/**
 * Perform any clean up necessary when a ULCApplication has been terminated 
 * within this context.
 *
 * @param app The <code> ULCApplication</code> that has been terminated.
 */
public void applicationTerminated(ULCApplication app) {
	remove(app);
	terminateIfLastApplication();
}
/**
 * Append the request to the collection of batched requests.
 *
 * @param oid 		The identification of this object in the UI
 * @param request	The request <code> String</code> to be made to this object.
 * @param args		The optional arguments to be sent for this request.
 */
private void batchRequest(int oid, String request, Anything args) {
	Anything r = new Anything();
	r.put("id", oid);
	r.put("req", request);
	r.put("a", args);
	fBatchRequest.append(r);
}
/**
 * Sends a request to the UI to Beep.
 */
public void beep() {
	send(Common.FACTORY_ID, "beep", null);
}
/**
 * Connect to a UIEngine via the given connection.
 * Events and call backs from the UIEngine are placed into the given ThreadPool.
 *
 * @param connection 	The connection to be used for this context
 * @param rp 			The <code> ThreadPool</code> in which all incoming 
 *						requests from the UI-Engine are placed.
 */
public void connect(ORBConnection connection, ThreadPool rp) {
	fSeed = fgSeed++;
	fRequestProcessor = rp;
	fConnection = connection;
}
/**
 * Connect to a peer via the specified socket.
 * Events and call backs from the UIEngine are placed into the given ThreadPool.
 *
 * @param socket 	The active socket of the UIEngine.
 * @param rp 		The <code> ThreadPool</code> in which all incoming requests
 *					from the UI-Engine are placed.
 */
public void connect(UlcTransport transport, ThreadPool rp)
throws UlcTransportException {
	fSeed= fgSeed++;
	fRequestProcessor= rp;
	fConnection= new ORBConnection(transport,fRequestProcessor,false,fRegistry);
	Server.propagateConnectionCreated(fConnection);   // notify connection controllers
	try {
		fConnection.start();
		Server.propagateConnectionStarted(fConnection,null);   // notify connection controllers
	}
	catch (UlcTransportException ex) {
		Server.propagateConnectionStarted(fConnection,ex);   // notify connection controllers
		fConnection= null;
		throw ex;
	}
}
/**
 * Connect to a UIEngine at the specified address and port.
 * Events and call backs from the UIEngine are placed into the given ThreadPool.
 *
 * @param urlString The url address of the UIEngine.
 * @param rp 		The <code> ThreadPool</code> in which all incoming 
 *					requests from the UI-Engine are placed.
 */
public void connect(String urlString, ThreadPool rp)
throws UlcTransportException {
	fUrlString = urlString;
	fSeed = fgSeed++;
	fRequestProcessor = rp;
	fConnection = new ORBConnection(fUrlString, fRequestProcessor, false, fRegistry);
	Server.propagateConnectionCreated(fConnection);   // notify connection controllers
	try {
		fConnection.start();
		Server.propagateConnectionStarted(fConnection,null);   // notify connection controllers
	}
	catch (UlcTransportException ex) {
		Server.propagateConnectionStarted(fConnection,ex);   // notify connection controllers
		fConnection= null;
		throw ex;
	}
}
/**
 * Connect to a peer via the specified socket.
 * Events and call backs from the UIEngine are placed into the given ThreadPool.
 *
 * @param socket 	The active socket of the UIEngine.
 * @param rp 		The <code> ThreadPool</code> in which all incoming requests
 *					from the UI-Engine are placed.
 * @deprecated
 */
public void connect(Socket socket, ThreadPool rp)
throws UlcTransportException {
	fSeed= fgSeed++;
	fRequestProcessor= rp;
	UlcSocketTransport transport= new UlcSocketTransport(socket);
	fConnection= new ORBConnection(transport,fRequestProcessor,false,fRegistry);
	Server.propagateConnectionCreated(fConnection);   // notify connection controllers
	try {
		fConnection.start();
		Server.propagateConnectionStarted(fConnection,null);   // notify connection controllers
	}
	catch (UlcTransportException ex) {
		Server.propagateConnectionStarted(fConnection,ex);   // notify connection controllers
		fConnection= null;
		throw ex;
	}
}
public void connectionCreated(IConnection connection) {
	if (ULC.fgDebug)
		System.out.println("ULCContext.connectionCreated(). URL=" + connection.getUrlString());
}
public void connectionEnded(IConnection connection, UlcTransportException ex) {
	if (connection != fConnection)
		return;

	if (ULC.fgDebug) {
		System.out.println("ULCContext.connectionEnded(). URL=" + connection.getUrlString());
		if (ex == null)
			System.out.print("                        Exception is null");
		else
			System.out.print("                        Exception=" + ex);
	}
	fConnection = null;
	if (fAutoReconnect && fRequestProcessor != null) {
		fRequestProcessor.addRequest(new ULCContextReconnectRequest(this, ex));
	} else {
		Server.removeConnectionController(this);
		shutdown();
		fConnection = null;
	}
}
public void connectionStarted(IConnection connection, UlcTransportException ex) {
	if (ULC.fgDebug) {
		System.out.println("ULCContext.connectionStarted(). URL=" + connection.getUrlString());
		if (ex == null)
			System.out.print("                      Exception is null");
		else
			System.out.print("                      Exception=" + ex);
		System.out.println();
	}
}
/**
 * Send a request to the UI to establish a new connection to the app
 * referred to in url.
 *
 * @param urlString 	The url address of the running ULC application server.
 * @param clientData	Any optional data that will be passed unchanged to the server.
 */
public void connectToApp(String urlString, String clientData) {
	Anything a = new Anything();
	a.put("URL", urlString);
	a.put("clientData", clientData);
	send(Common.FACTORY_ID, "connectToApp", a);
}
boolean connectWithRetries(String urlString, ThreadPool reqProc) {
	try {
		connect(urlString,reqProc);
	}
	catch (UlcTransportException ex) {
		if (fAutoReconnect && fRequestProcessor != null) {
			reconnect(ex);
			return true;
		}
		else {
			shutdown();
			return false;
		}
	}
	return true;
}
/**
 * Sends a 'new' request to the corresponding connection in the UIEngine.
 * This causes the object(s) referred to in the <code>Anything</code> to be instantiated
 * in the UI.
 *
 * @param a The marshalled state of the new object.
 */
public void create(Anything a) {
	send(Common.FACTORY_ID, "new", a);
}
/**
 * Disconnect from the UIEngine by closing the connection.
 */
public void disconnect() {
	if (fConnection != null)
		fConnection.close();
}
/**
 * Send a request to the UI to disconnect the connection 
 * referred to in url and connectionId.
 *
 * @param urlString		The url that references the running ULC application server.
 * @param connId		The unique connectionId within the UI Engine that identifies the
 *						previously activated connection.
 * @see #addedUIConnection
 */
public void disconnectFromApp(String urlString, String connectionId) {
	Anything a = new Anything();
	a.put("URL", urlString);
	a.put("connectionId", connectionId);
	send(Common.FACTORY_ID, "disconnectFromApp", a);
}
/**
 * Sends a echo request to the UI. 
 * The answer will be processed in the handleEchoAnswer.
 */
public void echo(String parameter) {
	send(Common.FACTORY_ID, "echo", new Anything(parameter));
}
/**
 * Check if there are running ULCApplication instances within this context. 
 * If there are no running ULCApplications and this context has not been started
 * in server mode then call System.exit. 
 */
public void exitIfLastApp() {
	if (ULC.isServerMode())
		return;
	Enumeration e = getRoots().elements();
	while (e.hasMoreElements()) {
		Object o = e.nextElement();
		if (o instanceof ULCApplication) {
			return;
		}
	}
	if (getExitOnLastApplicationClosed())
		System.exit(1);
}
/**
  * Empty implementation of method from ICallable protocol.
  * perform any resource deallocation necessary after this object has been destroyed.
  * Subclasses can override this method but MUST call super.free() to ensure proper cleanup.
  */
public void free() {
	fLookAndFeels = null;
	if (fRegistry != null)
		fRegistry.unregister(Common.FACTORY_ID);
}
/**
 * Return all the applications active within this context.
 *
 * @return Vector 
 */
public Vector getActiveApplications() {
	Vector apps = new Vector();
	Enumeration e = getRoots().elements();
	while (e.hasMoreElements()) {
		try {
			ULCApplication app = (ULCApplication) e.nextElement();
			apps.addElement(app);
		} catch (Exception ex) {
		}
	}
	return apps;
}
/**
 * Return the reconect mode. If true then this context will retry the connection
 * to the UI engine if the connection ever goes down.
 *
 * @return mode 		If true retry the failed connection.
 */
public boolean getAutoReconnect() {
	return fAutoReconnect;
}
/**
 * Return a user defined error message for connection
 * down errors.
 *
 * This message will be displayed to the user in a dialog box
 * if the connection or the ULC side goes down.
 *
 * @since	R3.1
 * @return	The <code>String</code> which should be used as error message
 *			if <code>null</code> is returned, then no message will be displayed
 */
public String getConnDownErrorMsg() {
	return null;
}
/**
  * Return the active connection upon which this context is running.
  *
  * @return <code> UlcConnection</code>
  */
public UlcConnection getConnection() {
	return fConnection;
}
/**
 * Return the default application for this Context.
 * The default implementation returns the first active application.
 *
 * @return com.ibm.ulc.application.ULCApplication
 */
public ULCApplication getDefaultApplication() {
	Vector apps = getActiveApplications();
	if (apps.isEmpty()) {
		new ULCApplication(this); // this actually adds the resulting application to the receiver's roots
	}
	return (ULCApplication) getActiveApplications().firstElement();
}
/**
  * Returns true if the System exit is called when the last application is closed
  *
  * @see #setExitOnLastApplicationClosed
  */
public static boolean getExitOnLastApplicationClosed() {
	return fgExitOnLastApplicationClosed;
}
/**
 * Returns the registered id for this proxy.
 * The context always has the FACTORY_ID
 */
public int getId() {
	return Common.FACTORY_ID;
}
/**
 * Request the list of installed lookAndFeels by sending a request to the UIEngine
 */
public Vector getInstalledLookAndFeels() {
	boolean success = false;
	if (fLookAndFeels == null) {
		send(Common.FACTORY_ID, "getInstalledLookAndFeels", null);
		success = synch(0);
	}
	if (!success && fLookAndFeels == null)
		return new Vector();
	return fLookAndFeels;
}
/**
 * Return the current LAF class name of the UI Engine
 *
 * @return	The current LAF class name in the UI Enginge as <code>String</code>
 */
public String getLook() {
	if (fLook == null) {
		send(Common.FACTORY_ID, "getLook", null);
		synch(0);
	}
	return fLook;
}
/**
 * The contextId is a number and must be unique within a single connection between a 
 * UI Engine and a ULC Application Server VM. 
 * It is not interpreted by the UI but will be passed along unchanged on all subsequent messages to the
 * application. The contextId will be generated on the ULC side.
 */
public long getNextContextId() {
	return fgContextId++;
}
/**
 * User applications can store and retrieve property values in this context.
 *
 * @param key 			The <code>String</code> key used to access the property.
 * @param defaultValue 	The <code>Object</code> that is returned if the key was not found.
 */
public Object getProperty(String key, Object defaultValue) {
	if (fProperties == null)
		return defaultValue;
	return fProperties.get(key, defaultValue);
}
/**
 * Returns the active registry for this context.
 *
 * @return <code>Registry</code>
 */
public Registry getRegistry() {
	return fRegistry;
}
/**
 * Returns the Resource Bundle base file name.
 * Subclasses must implement this method to return the correct file name.
 *
 * @return <code>String</code> or null if the subclass has not implemented this method.
 * @see #getResourceString(String)
 */
public String getResourceBundleBaseFileName() {
	return null;
}
/**
 * Returns the locale to be used for resource lookup.
 * The locale used will be by default the locale of the UI Engine.
 * If subclasses want to override the locale used they should
 * implement this method and return the locale to be used for the context.
 *
 * @param key <code>String</code> lookup key.
 * @return <code>String</code>
 * @see #getResourceBundleBaseFileName
 * @see #getResourceString
 */
public Locale getResourceLocale() {
	return getUILocale();
}
/**
 * Gets the resource object identified by the specified key from the specified ResourceBundle.
 * The resourceBundle is identified by the <code>getResourceBundleBaseFileName</code>
 * Subclasses must implement <code>getResourceBundleBaseFileName</code> if they
 * are using this method to retrieve the correct String for the specified locale.
 * The locale used will be by default the locale of the UI Engine.
 * If subclasses want to override the locale used they can implement 
 * the method <code>getResourceLocale</code>
 *
 * @param key <code>String</code> lookup key.
 * @return <code>Object</code>
 * @see #getResourceBundleBaseFileName
 * @see #getResourceLocale
 */
public Object getResourceObject(String key) {
	if (fResourceBundle == null) {
		try {
			String baseFileName = getResourceBundleBaseFileName();
			if (baseFileName != null)
				fResourceBundle = PropertyResourceBundle.getBundle(getResourceBundleBaseFileName(), getResourceLocale());
		}
		catch (MissingResourceException e) {
		}
	}
	Object value;
	if (fResourceBundle == null)
		return key + " no Resource bundle defined";
	try {
		value = fResourceBundle.getString(key);
	}
	catch (MissingResourceException e) {
		value = key + " not found";
	}
	return value;
}
/**
 * Gets a string from the specified ResourceBundle.
 * The resourceBundle is identified by the <code>getResourceBundleBaseFileName</code>
 * Subclasses must implement <code>getResourceBundleBaseFileName</code> if they
 * are using this method to retrieve the correct String for the specified locale.
 * The locale used will be by default the locale of the UI Engine.
 * If subclasses want to override the locale used they can implement 
 * the method <code>getResourceLocale</code>
 *
 * @param key <code>String</code> lookup key.
 * @return <code>String</code>
 * @see #getResourceBundleBaseFileName
 * @see #getResourceLocale
 */
public String getResourceString(String key) {
	return (String) getResourceObject(key);
}
/**
 * Returns the collection of root objects to this Context.
 *
 * @return Vector
 */
private Vector getRoots() {
	if (fRoots == null)
		fRoots = new Vector();
	return fRoots;
}
/**
 * Returns a pseudo-timestamp for this context.
 * This is used to check whether we have to upload a ULCProxy to the UIEngine.
 *
 * @return <code>int</code>
 */
public int getSeed() {
	return fSeed;
}
/**
 * Return the available font family names in the UI Engine.
 * Note: This method is SYNCHRONOUS and the result is NOT CACHED!
 *
 * @since	R3.1
 * @return	<code>String[]</code> with the names of the available fonts
 *			<code>null</code> will be returned, if the synch(0) faills
 */
public String[] getUIAvailableFontFamilyNames() {
	this.send(Common.FACTORY_ID, "getUIAvailableFontFamilyNames", null);
	if (synch(0)) {
		String fonts[]= new String[fTempSyncAnswer.size()];
		for (int i= 0; i < fTempSyncAnswer.size(); i++)
			fonts[i]= fTempSyncAnswer.get(i).asString("");
		return fonts;
	}
	else
		return null;
}
/**
 * Return the Locale of the UI Engine SYNCHRONOUSLY.
 * If the connection is down the default locale is returned.
 *
 * @return	The <code>Locale</code> of the UI Engine
 */
public Locale getUILocale() {
	boolean success = false;
	if (fUILocale == null) {
		if (fConnection != null) {
			this.send(Common.FACTORY_ID, "getUILocale", null);
			success = synch(0);
		} else
			return Locale.getDefault();
	}
	if (!success && fUILocale == null)
		return Locale.getDefault();
	return fUILocale;
}
/**
 * Return the screen height of the machine where the UI Engine runs.
 * Note: This method is SYNCHRONOUS and the result is NOT CACHED!
 *
 * @since	R3.1
 * @return	The screen height as <code>int</code>
 *			<code>-1</code> will be returned, if the value can not be retrieved or synch(0) faills
 */
public int getUIScreenHeight() {
	this.send(Common.FACTORY_ID, "getUIScreenSize", null);
	if (synch(0))
		return fTempSyncAnswer.get("h", -1);
	else
		return -1;
}
/**
 * Return the screen resolution (dots-per-inch) of the machine where the UI Engine runs.
 * Note: This method is SYNCHRONOUS and the result is NOT CACHED!
 *
 * @since	R3.1
 * @return	The screen resolution in dots-per-inch as <code>int</code>
 *			<code>-1</code> will be returned, if the value can not be retrieved or synch(0) faills
 */
public int getUIScreenResolution() {
	this.send(Common.FACTORY_ID, "getUIScreenResolution", null);
	if (synch(0))
		return fTempSyncAnswer.asInt(-1);
	else
		return -1;
}
/**
 * Return the screen width of the machine where the UI Engine runs.
 * Note: This method is SYNCHRONOUS and the result is NOT CACHED!
 *
 * @since	R3.1
 * @return	The screen width as <code>int</code>
 *			<code>-1</code> will be returned, if the value can not be retrieved or synch(0) faills
 */
public int getUIScreenWidth() {
	this.send(Common.FACTORY_ID, "getUIScreenSize", null);
	if (synch(0))
		return fTempSyncAnswer.get("w", -1);
	else
		return -1;
}
/**
 * Return the value of each system property in <code>keys</code> in the UI Engine's VM.
 * Note: The result is NOT CACHED!
 *
 * @since	R3.1
 * @return	The values of the UI Engine's system properties in <code>keys</code>
 *			<code>null</code> will be returned, if synch(0) faills,
 			if a key is not found or a SecurityException is thrown on the
 			UI side for a key then this key will not be added to the properties
 */
public Properties getUISystemProperties(Vector keys) {
	if (keys == null)
		return null;
	this.send(Common.FACTORY_ID, "getUISystemProperty", new Anything(keys));
	if (synch(0))
		if (fTempSyncAnswer.isNull())
			return null;
		else {
			Properties p= new Properties();
			int i= 0;
			for (Enumeration enum= fTempSyncAnswer.values(); enum.hasMoreElements();) {
				String value= ((Anything) enum.nextElement()).toString();
				if (value != null)
					p.setProperty((String) keys.elementAt(i), value);
				i++;
			}
			return p;
		}
	else
		return null;
}
/**
 * Return the value of the system property <code>key</code> in the UI Engine's VM.
 * Note: The result is NOT CACHED!
 *
 * @since	R3.1
 * @return	The value of the UI Engine's system property <code>key</code>
 *			<code>null</code> will be returned, if synch(0) faills,
 			the key is not found or a SecurityException is thrown on the
 			UI side
 */
public String getUISystemProperty(String key) {
	if (key == null)
		return null;
	Vector v= new Vector(1);
	v.add(key);
	return getUISystemProperties(v).getProperty(key);
}
/**
 * Gets the optional user parameter specified when starting the UI Engine.
 * 
 * @return The userParameter <code>String</code> or an empty string.
 */
public String getUserParameter() {
	if (fUserParameter == null) {
		this.send(Common.FACTORY_ID, "getUserParameter", null);
		synch(0);
	}
	return fUserParameter;
}
/**
 * Gets the optional user parameter specified as the command line <code>-userparm</code> when starting this context.
 * 
 * @return The <code>String</code> or an empty string.
 */
public String getUserServerParameter() {
	return (String) getProperty("userparm", "");
}
/**
 * The UI has sent an answer to an echo request. Do any processing necessary.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param args		Anything		The arguments associated with this request.
 */
protected void handleEchoAnswer(ORBConnection conn, Anything args) {
}
/**
 * The UI has sent a EXIT request to this object. Do all processing necessary.
 * This message can be received when this context is running in server mode.
 * The appName argument received was 'EXIT'.
 * The default behavior of handleExitRequest is to terminate this context.
 * subclasses can override this method to perform any processing necessary or can
 * choose to ignore the ExitRequest completely.
 * If this method returns the context requested will be started normally.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
protected void handleExitRequest(ORBConnection conn, String request,Anything args) {
	System.exit(0);
}
private void handleGetLook(ORBConnection conn, Anything args) {
	fLook= args.get("className", "Unknown");
}
/**
 * The UI has sent a init request to this object. Do all processing necessary.
 * This message will be received when this context is running in server mode.
 * If the appName argument is 'EXIT' then handleExitRequest is called.
 * Start the context on receiving this request.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
protected void handleInitRequest(ORBConnection conn, String request, Anything args) {
	if (ULC.isServerMode()) {
		if (Common.checkMajorVersion(args)) { // a major mismatch
			Anything a = new Anything();		
			a.put("reason", "ULC protocol major version mismatch");
			conn.send(Common.FACTORY_ID, "reject", a); // let application kill itself
			return;
		}
		else if (Common.checkMinorVersion(args)) // a minor mismatch
			sendAccept("ULC protocol minor version mismatch");
		else
			sendAccept(null);

		String appName = args.get("appName", "<default>");
		if (appName.toUpperCase().equals("EXIT"))
			handleExitRequest(conn, request, args);
		setUILocale(args);
		if (waitForInit())
			start();
	}
	else
		trouble("handleInitRequest", "<init> received while in client mode - do nothing");
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request write an error message to the console.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("synch")) {
		long synchId = args.asLong(0);
		if (fSynchId == synchId)
			fGotReply = true;
		return;
	}
	if (request.equals("init")) {
		handleInitRequest(conn, request, args);
		return;
	}
	if (request.equals("echo")) {
		handleEchoAnswer(conn, args);
		return;
	}
	if (request.equals("reject")) {
		if (!ULC.isServerMode()) {
			String reason = args.get("reason", "unknown");
			System.out.println("Problem between UI Engine and application server. Reason: " + reason + "; giving up");
			System.exit(-1); // give up
		}
		return;
	}
	if (request.equals("accept")) {
		String reason = args.get("reason", null);
		if (reason != null)
			System.out.println("Minor problem between UI Engine and application server. Reason: " + reason);
		return;
	}
	if (request.equals("removedUIConnection")) {
		String host = args.get("host", "unknown");
		int port = args.get("port", UlcConnection.INVALIDPORT);
		removedUIConnection(host, port);
		return;
	}
	if (request.equals("addedUIConnection")) {
		String host = args.get("host", "unknown");
		String clientData = args.get("clientData", "");
		int port = args.get("port", UlcConnection.INVALIDPORT);
		String connId = args.get("connectionId", "0");
		addedUIConnection(host, port, clientData, connId);
		return;
	}
	if (request.equals("getUILocale")) {
		setUILocale(args);
		return;
	}
	if (request.equals("getUserParameter")) {
		fUserParameter = args.asString("");
		return;
	}
	if (request.equals("getInstalledLookAndFeels")) {
		setLookAndFeels(conn, args);
		return;
	}
	if (request.equals("getLook")) {
		handleGetLook(conn, args);
		return;
	}
	if (request.equals("getUISystemProperty")) {
		fTempSyncAnswer= args;
		return;
	}
	if (request.equals("getUIScreenResolution")) {
		fTempSyncAnswer= args;
		return;
	}
	if (request.equals("getUIScreenSize")) {
		fTempSyncAnswer= args;
		return;
	}
	if (request.equals("getUIAvailableFontFamilyNames")) {
		fTempSyncAnswer= args;
		return;
	}
	if (request.equals("terminate")) {
		terminate();
		return;
	}
	trouble("handleRequest", "request " + request + " not handled");
}
/**
 * Helps an object to find out whether it was already uploaded and is still valid.
 *
 * @return boolean if true then this object has already been uploaded to the UI
 */
public boolean isRecent(int seed) {
	return fSeed == seed;
}
/**
 * Send a request to the UI Engine that will cause it to exit.
 */
public void killServer() {
	send(Common.FACTORY_ID, "kill", null);
}
/**
 * Posts a request to the connection's request queue.
 */
public void postRequest(Request r) {
	if (fConnection != null)
		fConnection.postRequest(r);
}
/**
 * Ensure that the next incoming message from the UI will be processed before control
 * is returned to the caller.
 */
public void processNextRequest() {
	processNextRequest(0);
}
/**
 * Ensure that the next incoming message from the UI will be processed before control
 * is returned to the caller. Wait a maximum of timeout period for the next message.
 *
 * @param timeout	The milliseconds to wait for the next message.
 */
public boolean processNextRequest(long timeout) {
	if (fRequestProcessor != null)
		return fRequestProcessor.processNextRequest(timeout);
	else
		return true;
}
void reconnect(UlcTransportException ex) {
	UlcTransportException exception= ex;
	while (fConnection == null) {
		if (!reconnectAfterException(exception))
			return;
		try {
			Thread.sleep(reconnectDelayInMillis());
		}
		catch(InterruptedException e) {
		}
		try {
			connect(fUrlString, fRequestProcessor);
		}
		catch(UlcTransportException ex2) {
			exception= ex2;
		}
	}
	upload();

	// Upload connection down error message
	sendConnDownErrorMsg();	
}
protected boolean reconnectAfterException(UlcTransportException ex) {
	return true;
}
protected long reconnectDelayInMillis() {
	return 2000;
}
/**
  * Disconnect any active connection to the UI and force the connection to be restablished.
  */
public void reconnectUI() {
	disconnect();
}
/**
 * Register an ICallable with this context's registry.
 *
 * @param c The object that implements <code>ICallable</code>
 */
public int register(ICallable c) {
	if (fRegistry != null)
		return fRegistry.register(c);
	return -1; // not a valid id!
}
/**
 * Removes a proxyable object from the set of known roots.
 *
 * @param c The object that implements <code>IProxy</code>
 */
public void remove(IProxy c) {
	getRoots().removeElement(c);
}
/**
 * The UI has notified us that a connection has been removed.
 * This event will be received only if <code>addUIConnectionListener</code>
 * has been called once.
 * Default behaviour is to do nothing.
 *
 * @param host			The host <code>String</code> that made the connection to the UI.
 * @param port			The port <code>int</code> on which the connection was made.
 *
 * @see #addUIConnectionListener
 */
protected void removedUIConnection(String host, int port) {
}
/**
 * Restarts the UI associated with this connection.
 * First the registry in the UI server is cleared.
 * Then the seed associated with the current connection is changed
 * thereby rendering all uploaded objects invalid.
 * Eventually upload everything.
 */
public void restartUI() {
	send(Common.FACTORY_ID, "freeAll", null);
	fSeed = fgSeed++;
	upload();
}
/**
 * Send a request to the object identified by <code> oid</code>.
 *
 * @param oid 		The identification of this object in the UI
 * @param request	The request <code> String</code> to be made to this object.
 * @param args		The optional arguments to be sent for this request.
 */
public void send(int oid, String request, Anything args) {
	if (fConnection != null) {
		if (fBatchRequest == null)
			fConnection.send(oid, request, args);
		else
			batchRequest(oid, request, args);
	}
}
/**
 * This method will be sent when accepting a connection request from a UI Engine.
 * If the contextId is set then every request sent from the UI will contain this contextId.
 * The contextId is an  number and must be unique within a single connection between a 
 * UI Engine and a ULC Application Server VM. 
 * It is not interpreted by the UI but will be passed along unchanged on all subsequent messages to the
 * application. The contextId will be generated on the ULC side.
 *
 * @param reason A <code>String</code> describing the error or null when everything is ok.
 *
 * @see #getNextContextId()
 */
public void sendAccept(String reason) {
	if (ULC.isServerMode()) {
		Anything accept = new Anything();
		if (reason != null)
			accept.put("reason", reason);
		if (getConnDownErrorMsg() != null)
			accept.put("connDownErrorMsg", new Anything(getConnDownErrorMsg()));		
		accept.put("contextId", getNextContextId());
		this.send(Common.FACTORY_ID, "accept", accept);
	}
	else
		troubleErr("sendAccept(String)", "illegal to send <accept> while in client mode");
}
/**
 * Send the connection down error message to the UI
 *
 * @see #getConnDownErrorMsg()
 */
private void sendConnDownErrorMsg() {
		if (getConnDownErrorMsg() != null)
			send(Common.FACTORY_ID, "setConnDownErrorMsg", new Anything(getConnDownErrorMsg()));
}
/**
 * This method will be the first message sent across an active connection.
 * It sends the current protocol version as well as any other startUp information.
 */
public void sendInit() {
	Anything info = new Anything();
	info.put("majorVersion", Common.PROTOCOL_VERSION_MAJOR);
	info.put("minorVersion", Common.PROTOCOL_VERSION_MINOR);
	if (getConnDownErrorMsg() != null && !ULC.isServerMode())
		info.put("connDownErrorMsg", new Anything(getConnDownErrorMsg()));
	this.send(Common.FACTORY_ID, "init", info);
}
/**
 * Set the reconect mode. If true then this context will retry the connection
 * to the UI engine if the connection ever goes down.
 *
 * @param mode 		If true retry the failed connection.
 */
public void setAutoReconnect(boolean mode) {
	fAutoReconnect = mode;
}
/**
  * Set to true if the UI should exit when the last application is closed.
  * If set to false the UI will remain active even after the last application 
  * is closed and an explicit System.exit will need to be called.
  *
  * @param exitOnLastApplicationClosed If true UI will exit on last connection closed.
  * @see #getExitOnLastApplicationClosed
  */
public static void setExitOnLastApplicationClosed(boolean exitOnLastApplicationClosed) {
	fgExitOnLastApplicationClosed = exitOnLastApplicationClosed;
}
/**
 * Set the lookAndFeel className by sending a request to the UIEngine
 *
 * @param lookClassName The fully qualified Java L&F class name
 *
 * @see #getLook
 * @see #getInstalledLookAndFeels
 */
public void setLook(String lookClassName) {
	if (fLook == null || fLook != lookClassName) {
		send(Common.FACTORY_ID, "setLook", new Anything(lookClassName));
		fLook = lookClassName;
	}
}
private void setLookAndFeels(ORBConnection conn, Anything args) {
	Enumeration allLooks= args.values();
	if (fLookAndFeels == null) fLookAndFeels= new Vector();
	while (allLooks.hasMoreElements()) {
		Anything aLook= (Anything) allLooks.nextElement();
		String lookName= aLook.get("name", "Unknown");
		String lookClass= (String) aLook.get("className", "Unknown");
		//mi.addActionListener(new SwitchLookAction(this, lookClass));
		fLookAndFeels.addElement(new Look(lookName, lookClass));
	}
}
/**
 * User applications can store and retrieve property values in this context.
 *
 * @param key 			The <code>String</code> key used to access the property.
 * @param value 		The <code>Object</code> that is stored at the specified key.
 */
public void setProperty(String key, Object value) {
	if (fProperties == null)
		fProperties = new UlcHashtable();
	fProperties.put(key, value);
}
/**
 * Internal
 */
private void setUILocale(Anything any) {
	Locale l = Locale.getDefault();
	Anything locale = any.get("locale");
	if (locale != null) {
		fUILocale = new Locale(locale.get("lang", l.getLanguage()),locale.get("country", l.getCountry()), locale.get("variant", l.getVariant()));
	}
	else
		fUILocale = l;
}
/**
 * Sets the optional user parameter specified on the command line <code>-userparm</code> when starting this context.
 * 
 * @param The <code>String</code> parameter specified on the command line.
 */
public void setUserServerParameter(String parameter) {
	setProperty("userparm", parameter);
}
public void shutdown() {
	if (ULC.fgDebug)
		System.out.println("ULCContext.shutdown()");
	if (fRegistry != null) {
		fRegistry.shutdown();
		fRegistry= null;
	}
	if (fRequestProcessor != null) {
		fRequestProcessor.shutdown();
		fRequestProcessor= null;
	}
	exitIfLastApp();
	getRoots().clear();
}
/*
 * Called when this context is run.
 * Clients have to implement this method with their 'main' code.
 */
abstract public void start();
/**
 * After this method is called all requests sent to the UI are kept pending till 
 * the stopBatchingRequests is called.
 */
public synchronized void startBatchingRequests() {
	if (fBatchRequest == null)
		fBatchRequest = new Anything();
}
/**
 * After this method is called any pending requests are sent to the UI and the batching 
 * of requests is stopped.
 */
public synchronized void stopBatchingRequests() {
	if (fBatchRequest != null) {
		Anything request = fBatchRequest;
		fBatchRequest = null;
		send(Common.FACTORY_ID, "handleBundledRequests", request);
	}
}
/**
 * Synchronize with UIEngine by sending a request to the UIEngine
 * and waiting for an echo.
 * It is possible that this call returns before the answer is received in the
 * case where the connection goes down before the answer is received.
 *
 * @param timeout Wait timeout milliseconds.
 * @return true on success false on timeout or connection down.
 */
public boolean synch(long timeout) {
	fSynchId++;
	send(Common.FACTORY_ID, "synch", new Anything(fSynchId));
	fGotReply = false;
	while (!fGotReply && (fConnection != null)) {
		processNextRequest(timeout);
	};
	if (fGotReply)
		return true;
	return false;
}
/**
 * Terminate this context
 */
public void terminate() {
	setAutoReconnect(false);
	disconnect();
}
/**
 * Check if there are no active ULCShells within my ULCApplication. 
 * If there are no active ULCShells close the connection
 */
protected void terminateIfLastApplication() {
	Enumeration e = getRoots().elements();
	while (e.hasMoreElements()) {
		Object o = e.nextElement();
		if (o instanceof ULCApplication) {
			return;
		}
	}
	terminate();
}
/**
 * DeRegister an ICallable with this context's registry.
 *
 * @param c The object that implements <code>ICallable</code>
 */
public void unregister(ICallable c) {
	if (fRegistry != null)
		fRegistry.unregister(c.getId());
}
/**
 * Uploads all objects which are reachable from roots.
 */
public void upload() {
	Enumeration e = getRoots().elements();
	while (e.hasMoreElements()) {
		IProxy c = (IProxy) e.nextElement();
		c.upload(this);
	}
}
/*
 * Called when the context is running as a server and a connection has been established.
 * By default this returns true which indicates that the context.start() 
 * is called only when the init message is received.
 * subclasses of ULCContext can override this method to return false which will cause 
 * the context to start directly without waiting for the init message.
 *
 * @return boolean
 * @see #handleInitRequest
 * @since R1.3
 */
public boolean waitForInit() {
	return true;
}
}
