package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import java.util.Vector;

/**
 * The combo-box is a combination of a text field and 
 * drop-down list that lets the user either type in a value or select it from 
 * a list that is displayed when the user asks for it. The editing capability 
 * can also be disabled so that the ComboBox acts only as a drop down list.
 * The list of items is specified via a ULCTableModel.
 * The currently selected item is either determined by the state of this widget or
 * by an attribute of a ULCFormModel.
 * 
 */
public class ULCComboBox extends ULCFormComponent implements IItemListOwner {
	/**
	 * The string key used to retrieve the values for the combo box from the TableModel
	 * @serial
	 */
	protected String fColId = null;
	/**
	 * The TableModel that will act as the data source for the combo box
	 	 * @serial
	 */
	protected ULCAbstractTableModel fModel = null;
	/**
	 * The itemlist that will define the order of the receiver's rows
	 	 * @serial
	 */
	protected IIndexedItemList fItemList = null;
	/**
	 * The widget that will be used to interpret/render the values within the combo box.
	 	 * @serial
	 */
	protected ULCComponent fRenderer = null;
	/**
	 * The currently selected item.
	 	 * @serial
	 */
	protected String fSelected = null;
	/**
	 * The currently selected item index.
	 	 * @serial
	 */
	protected int fSelectedIndex = -1;
	/**
	 * Set to true if the combo box should allow new values to be entered.
	 	 * @serial
	 */
	protected boolean fEditable = false;
	/**
	 * Any valid ULCTrigger object which will be triggered in the UI when 
	 * this widgets actionPerformed method is called
	 * @serial	 
	 */
	protected ITrigger fTrigger = null;
	/**
	 * The number of visible rows in the drop down
	 * @serial	 
	 */
	protected int fMaximumRowCount= 8;
/**
 * Construct a combobox
 */
public ULCComboBox() {
	super();
}
/**
 * Construct a combobox which tracks the state of the specified attribute of the given FormModel.
 * The value entered must be one of the values contained in the ULCTableModel.
 *
 * @param formModel 		The formModel where the attribute at key is tracked.
 * @param formAttributeName The key of the attribute used to track changes.
 * @param tableModel 		The Table model which is the data source for the combo box list.
 *							When the attribute at key in the formModel changes then the list
 *							is updated to display the matching item in the list if found.
 */
public ULCComboBox(IForm formModel, String formAttributeName, ULCTableModel tableModel) {
	super(formModel, formAttributeName);
	fModel = tableModel;
	if (fModel != null) {
		fModel.addOwner(this);
		internalUpdatePreloadColumn();
	}
	setItemList(null);
}
/**
 * Construct a combobox from the values contained in the given ULCTableModel.
 *
 * @param tableModel 	The Table model which is the data source for the combo box list.
 */
public ULCComboBox(ULCTableModel tableModel) {
	fModel = tableModel;
	if (fModel != null) {
		fModel.addOwner(this);
		internalUpdatePreloadColumn();
	}
	setItemList(null);
}
/**
 * Construct a combobox from the values contained
 * in the column with the specified key.
 *
 * @param model 			The Table model which is the data source for the combo box list.
 * @param attributeName		The attribute of the Table model that is used to populate
 *							the contents of the combobox
 */
public ULCComboBox(ULCTableModel tableModel, String attributeName) {
	fModel = tableModel;
	if (fModel != null) {
		fModel.addOwner(this);
		internalUpdatePreloadColumn();
	}
	fColId = attributeName;
	setItemList(null);
}
/**
 * Registers the given listener to begin receiving notifications
 * when a new item was added to the list.
 *
 * @param listener	The object interested in my actionEvents.
 */
public void addActionListener(IActionListener listener) {
	internalAddListener("itemInsert", listener);
}
/**
 * Registers the given listener to begin receiving notifications
 * when an item was selected.
 *
 * @param listener	The object interested in my selectionChangedEvents.
 */
public void addSelectionChangedListener(ISelectionChangedListener listener) {
	internalAddListener("selectionChanged", listener);
}
/**
 * Answer a one-element collection. If the attribute is <code>null</code>, answer an
 * array with <code>"null"</code> to indicate the tableModel's row itself (typically a string), rather
 * than an attribute of the row should be uploaded.
 *
 * see IItemListOwner
 */
public String[] computePreloadedAttributes() {
	return new String[] {getAttributeName()};
}
/**
 * Force the receiver's itemlist to upload all rows defined by it.
 *
 * @see IItemListOwner
 */
public String[] computePreloadedColumns() {
	return computePreloadedAttributes();
}
/**
 * Always answer a string. If the attribute is <code>null</code>, answer <code>"null"</code> to indicate 
 * the tableModel's row itself (typically a string), rather than an attribute of the row should be uploaded.
 */
public String getAttributeName() {
	if (fColId == null)
		return "null";
	else
		return fColId;
}
/**
 * Answer the receiver's current itemList.
 *
 * The receiver's <code>ULCItemlist</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 * @see ULCDefaultItemList
 * @see ULCSortedItemList
 *
 */
public IIndexedItemList getItemList() {
	if (fItemList == null)
		return getTableModel().getIndexedItemList();
	else
		return fItemList;
}
/**
 * Returns the maximum number of items that the combo box
 * displays. If there are more items a scrollbar is shown
 *
 * @return an int specifying the maximum number of items that are displayed
 *         in the list before using a scrollbar
 */
public int getMaximumRowCount() {
	return fMaximumRowCount;
}
/**
 * Returns the currently selected item as a String.
 *
 * @return The currently selected item as a <code> String</code>
 */
public String getSelected() {
	return fSelected;
}
/**
 * Returns the index of the currently selected item in the list.
 *
 * @return The currently selected item index.
 */
public int getSelectedIndex() {
	return fSelectedIndex;
}
/**
 * Return the currently selected item or null if nothing is selected
 *
 * @return  The <code>Object</code> currently selected.
 */
public Object getSelectedItem() {
	if (getSelectedIndex() < 0)
		return null;
	else {
		return getItemList().getRowAt(getSelectedIndex());
	}
}
/**
 * Gets the <code>ULCTableModel</code> that will serve as my data source.
 *
 * @return	The <code>ULCTableModel</code>
 */
public ULCAbstractTableModel getTableModel() {
	return fModel;
}
/**
 * Returns the configured ULCTrigger object which will be triggered in the UI when 
 * this widget's actionPerformed method is called.
 *
 * @return ITrigger	 
 */
public ITrigger getTrigger() {
	return fTrigger;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("itemStateChange")) {
			fSelected = args.get("selected", fSelected);
			fSelectedIndex = args.get("selectedIndex", fSelectedIndex);
			distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, fSelectedIndex, fSelectedIndex));
		}
		if (type.equals("itemInsert")) {
			fSelected = args.get("item", "");
			distributeToListeners("itemInsert", new ULCActionEvent(this, "insert")); // FIXME
		}
		return;
	}
	super.handleRequest(conn, request, args);
}
protected void internalUpdatePreloadColumn() {
	if ((getTableModel() != null) && (getAttributeName() != null)) {
		getTableModel().addPreloadColumn(getAttributeName());
	}
}
/**
 * Returns true if the ULCComboBox is editable.
 * 
 * @return  if true allows direct edit in this comboBox.
 */
public boolean isEditable() {
	return fEditable;
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive action events. 
 *
 * @param listener	The object that is interested in my actionEvents.
 */
public void removeActionListener(IActionListener listener) {
	internalRemoveListener("itemInsert", listener);
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive events. 
 *
 * @param listener	The object that was registered for my selectionChanged events
 */
public void removeSelectionChangedListener(ISelectionChangedListener listener) {
	internalRemoveListener("selectionChanged", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fModel != null)
		a.put("model", fModel.getRef(fContext));
	if (fItemList != null)
		a.put("itemList", fItemList.getRef(fContext));
	a.put("colId", getAttributeName());
	if (fRenderer != null)
		a.put("renderer", fRenderer.getRef(fContext));
	if (fEditable)
		a.put("editable", fEditable);
	if (fMaximumRowCount != 8)
		a.put("maximumRowCount", fMaximumRowCount);
	if (fFormModel == null) {
		if (fSelected != null)
			a.put("selected", fSelected);
		else
			if (fSelectedIndex != -1)
				a.put("selectedIndex", fSelectedIndex);
	}
	if (fTrigger != null)
		a.put("trigger", ((ULCProxy) fTrigger).getRef(fContext));
}
/**
 * Update the UI with the new itemlist.
 */
protected void sendItemList() {
	if (!isUploaded())
		return;
	Anything args = new Anything();
	if (fItemList != null)
		args.put("itemList", fItemList.getRef(fContext));
	args.put("model", fModel.getRef(fContext));
	sendUI("setItemList", args);
}
/**
 * Sets the attributeName which is used as a key to query values from 
 * the ULCTableModel of this widget.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @return	The <code>String</code> to be used as a key
 */
public void setAttributeName(String attributeName) {
	fColId = attributeName;
	internalUpdatePreloadColumn();
}
/**
 * Determines whether the ULCComboBox is editable.
 * 
 * @param editable if true allows direct edit in this comboBox.
 */
public void setEditable(boolean editable) {
	if (fEditable != editable) {
		fEditable = editable;
		sendUI("setEditable", new Anything(fEditable));
	}
}
/**
 * Set the <code>ULCItemListAbstract</code> that will serve as intermediary between
 * the receiver and its ULCTableModel.
 *
 * if <code>itemList</code> is null, the default itemList of the receiver's <code>ULCTableModel</code>
 * will be used by default.
 *
 * @see #getItemList
 *
 * @param itemList	The <code>ULCItemListAbstract</code> to be used from now on
 */
public void setItemList(IIndexedItemList itemList) {
	if (fItemList != null)
		fItemList.removeOwner(this);
	else {
		if (getTableModel() != null)
			getItemList().removeOwner(this);
	}
	if (itemList == null) {
		fItemList = null;
		if (getTableModel() != null)
			getItemList().addOwner(this);
	}
	else {
		fItemList = itemList;
		fItemList.addOwner(this);
		if (getTableModel() != fItemList.getModel())
			setTableModel(fItemList.getModel());
	}
	sendItemList();
}
/**
 * Set the maximum number of items that the combo box
 * displays in the pop up. If there are more items
 * a scrollbar is shown
 *
 * @param count the maximum number of items that are displayed
 *         in the list before using a scrollbar
 */
public void setMaximumRowCount(int count) {
	if (count != fMaximumRowCount) {
		fMaximumRowCount= count;
		sendUI("setMaximumRowCount", new Anything(fMaximumRowCount));
	}
}
/**
 * Sets the selected item in the ULCComboBox by specifying the String value of an object in the list.
 * 
 * @param s The string to be displayed as the current item
 */
public void setSelected(String s) {
	if (fSelected == null || !fSelected.equals(s)) {
		fSelected = s;
		sendUI("setSelected", new Anything(fSelected));
	}
}
/**
 * Selects the item at the specified index.
 * If the index is out of bounds the selection is cleared
 *
 * @param index The index of the item to be selected
 */
public void setSelectedIndex(int index) {
	if (index != fSelectedIndex) {
		fSelected= null;
		fSelectedIndex= index;
		sendUI("setSelectedIndex", new Anything(fSelectedIndex));
	}
}
/**
 * Set the currently selected item to be the object specified
 * Clear the selection if the object is not in the list
 *
 * @param  The <code>Object</code>  to be selected.
 */
public void setSelectedItem(Object row) {
	int rowCount = getItemList().getRowCount();
	for (int i = 0; i < rowCount; i++)
		if (row == getItemList().getRowAt(i)) {
			setSelectedIndex(i);
			return;
		}
	setSelectedIndex(-1);
}
/**
 * Set the <code>ULCTableModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * when the <code>ULCTableModel</code> is set explicitly, any existing itemList will be reset to null,
 * and the tableModel's default item list will be used by the receiver.
 *
 * @param tableModel	The <code>ULCTableModel</code>
 */
public void setTableModel(ULCAbstractTableModel tableModel) {
	if (fModel != null) {
		fSelectedIndex = -1;
		fModel.removeOwner(this);
	}
	fModel = tableModel;
	if (fModel != null)
		fModel.addOwner(this);
	internalUpdatePreloadColumn();
	setItemList(null);
	distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, fSelectedIndex, fSelectedIndex));
}
/**
 * Sets the configured ULCTrigger which will be triggered in the UI when 
 * this widgets actionPerformed method is called
 *
 * @param ULCTrigger	 
 */
public void setTrigger(ITrigger trigger) {
	if (trigger != fTrigger) {
		fTrigger = trigger;
		sendUI("setTrigger", (ULCTrigger)fTrigger);
	}
}
/**
 * returns the fully qualified class name of my UI class
 */
protected String typeString() {
	return "ComboBox";
}
}
